"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const ShareService_1 = require("@joplin/lib/services/share/ShareService");
const mockShareService_1 = require("@joplin/lib/testing/share/mockShareService");
const test_utils_1 = require("@joplin/lib/testing/test-utils");
const testUtils_1 = require("./utils/testUtils");
const Note_1 = require("@joplin/lib/models/Note");
const Folder_1 = require("@joplin/lib/models/Folder");
const Setting_1 = require("@joplin/lib/models/Setting");
const Command = require('./command-publish');
const setUpCommand = () => {
    const onStdout = jest.fn();
    const command = (0, testUtils_1.setupCommandForTesting)(Command, onStdout);
    return { command, onStdout };
};
describe('command-publish', () => {
    beforeEach(async () => {
        await (0, test_utils_1.setupDatabaseAndSynchronizer)(1);
        await (0, test_utils_1.switchClient)(1);
        await (0, testUtils_1.setupApplication)();
        (0, mockShareService_1.default)({
            getShares: async () => {
                return { items: [] };
            },
            postShares: async () => ({ id: 'test-id' }),
            getShareInvitations: async () => null,
        }, ShareService_1.default.instance());
    });
    test('should publish a note', async () => {
        const { command, onStdout } = setUpCommand();
        const testFolder = await Folder_1.default.save({ title: 'Test' });
        const testNote = await Note_1.default.save({ title: 'test', parent_id: testFolder.id });
        await command.action({
            note: testNote.id,
            options: {
                force: true,
            },
        });
        // Should be shared
        await (0, test_utils_1.waitFor)(async () => {
            expect(await Note_1.default.load(testNote.id)).toMatchObject({
                is_shared: 1,
            });
        });
        // Should have logged the publication URL
        expect(onStdout).toHaveBeenCalled();
        expect(onStdout.mock.lastCall[0]).toMatch(/Published at URL:/);
    });
    test('should be enabled for Joplin Server and Cloud sync targets', () => {
        const { command } = setUpCommand();
        Setting_1.default.setValue('sync.target', 1);
        expect(command.enabled()).toBe(false);
        const supportedSyncTargets = [9, 10, 11];
        for (const id of supportedSyncTargets) {
            Setting_1.default.setValue('sync.target', id);
            expect(command.enabled()).toBe(true);
        }
    });
    test('should not ask for confirmation if a note is already published', async () => {
        const { command } = setUpCommand();
        const promptMock = jest.fn(() => true);
        command.setPrompt(promptMock);
        await (0, test_utils_1.createFolderTree)('', [
            {
                title: 'folder 1',
                children: [
                    {
                        title: 'note 1',
                        body: 'test',
                    },
                ],
            },
        ]);
        const noteId = (await Note_1.default.loadByTitle('note 1')).id;
        // Should ask for confirmation when first sharing
        await command.action({
            note: noteId,
            options: {},
        });
        expect(promptMock).toHaveBeenCalledTimes(1);
        expect(await Note_1.default.load(noteId)).toMatchObject({ is_shared: 1 });
        // Should not ask for confirmation if called again for the same note
        await command.action({
            note: noteId,
            options: {},
        });
        expect(promptMock).toHaveBeenCalledTimes(1);
    });
});
//# sourceMappingURL=command-publish.test.js.map