"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const convertHtmlToMarkdown = require("./convertNoteToMarkdown");
const reducer_1 = require("../reducer");
const Note_1 = require("../models/Note");
const renderer_1 = require("@joplin/renderer");
const test_utils_1 = require("../testing/test-utils");
const Folder_1 = require("../models/Folder");
const shim_1 = require("../shim");
describe('convertNoteToMarkdown', () => {
    let state = undefined;
    beforeEach(async () => {
        state = reducer_1.defaultState;
        await (0, test_utils_1.setupDatabaseAndSynchronizer)(1);
        await (0, test_utils_1.switchClient)(1);
        shim_1.default.showToast = jest.fn();
    });
    it('should set the original note to be trashed', async () => {
        const folder = await Folder_1.default.save({ title: 'test_folder' });
        const htmlNote = await Note_1.default.save({ title: 'test', body: '<p>Hello</p>', parent_id: folder.id, markup_language: renderer_1.MarkupLanguage.Html });
        state.selectedNoteIds = [htmlNote.id];
        await convertHtmlToMarkdown.runtime().execute({ state, dispatch: () => { } });
        const refreshedNote = await Note_1.default.load(htmlNote.id);
        expect(htmlNote.deleted_time).toBe(0);
        expect(refreshedNote.deleted_time).not.toBe(0);
    });
    it('should recreate a new note that is a clone of the original', async () => {
        const folder = await Folder_1.default.save({ title: 'test_folder' });
        const htmlNoteProperties = {
            title: 'test',
            body: '<p>Hello</p>',
            parent_id: folder.id,
            markup_language: renderer_1.MarkupLanguage.Html,
            author: 'test-author',
            is_todo: 1,
            todo_completed: 1,
        };
        const htmlNote = await Note_1.default.save(htmlNoteProperties);
        state.selectedNoteIds = [htmlNote.id];
        await convertHtmlToMarkdown.runtime().execute({ state, dispatch: jest.fn() });
        const notes = await Note_1.default.previews(folder.id);
        expect(notes).toHaveLength(1);
        const noteConvertedToMarkdownId = notes[0].id;
        const markdownNote = await Note_1.default.load(noteConvertedToMarkdownId);
        const fields = ['parent_id', 'title', 'author', 'is_todo', 'todo_completed'];
        for (const field of fields) {
            expect(htmlNote[field]).toEqual(markdownNote[field]);
        }
    });
    it('should generate action to trigger notification', async () => {
        const folder = await Folder_1.default.save({ title: 'test_folder' });
        const htmlNoteProperties = {
            title: 'test',
            body: '<p>Hello</p>',
            parent_id: folder.id,
            markup_language: renderer_1.MarkupLanguage.Html,
            author: 'test-author',
            is_todo: 1,
            todo_completed: 1,
        };
        const htmlNote = await Note_1.default.save(htmlNoteProperties);
        state.selectedNoteIds = [htmlNote.id];
        await convertHtmlToMarkdown.runtime().execute({ state, dispatch: jest.fn() });
        expect(shim_1.default.showToast).toHaveBeenCalled();
    });
});
//# sourceMappingURL=convertNoteToMarkdown.test.js.map