"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const registry_1 = require("../../registry");
const Folder_1 = require("../../models/Folder");
const BaseModel_1 = require("../../BaseModel");
const Note_1 = require("../../models/Note");
const Resource_1 = require("../../models/Resource");
const ResourceFetcher_1 = require("../../services/ResourceFetcher");
const DecryptionWorker_1 = require("../../services/DecryptionWorker");
const Setting_1 = require("../../models/Setting");
const async_mutex_1 = require("async-mutex");
const readOnly_1 = require("../../models/utils/readOnly");
const ItemChange_1 = require("../../models/ItemChange");
const BaseItem_1 = require("../../models/BaseItem");
const shared = {};
// If saveNoteButton_press is called multiple times in short intervals, it might result in
// the same new note being created twice, so we need to a mutex to access this function.
const saveNoteMutex_ = new async_mutex_1.Mutex();
shared.noteExists = async function (noteId) {
    const existingNote = await Note_1.default.load(noteId);
    return !!existingNote;
};
// Note has been deleted while user was modifying it. In that case, we
// just save a new note so that user can keep editing.
shared.handleNoteDeletedWhileEditing_ = async (note) => {
    if (await shared.noteExists(note.id))
        return null;
    registry_1.reg.logger().info('Note has been deleted while it was being edited - recreating it.');
    let newNote = Object.assign({}, note);
    delete newNote.id;
    newNote = await Note_1.default.save(newNote);
    return Note_1.default.load(newNote.id);
};
// eslint-disable-next-line @typescript-eslint/no-explicit-any -- Old code before rule was applied
shared.saveNoteButton_press = async function (comp, state, folderId = null, options = null) {
    options = Object.assign({ autoTitle: true }, options);
    state = Object.assign(Object.assign({}, comp.state), state);
    const releaseMutex = await saveNoteMutex_.acquire();
    let note = Object.assign({}, state.note);
    const recreatedNote = await shared.handleNoteDeletedWhileEditing_(note);
    if (recreatedNote)
        note = recreatedNote;
    if (folderId) {
        note.parent_id = folderId;
    }
    else if (!note.parent_id) {
        const activeFolderId = Setting_1.default.value('activeFolderId');
        let folder = await Folder_1.default.load(activeFolderId);
        if (!folder)
            folder = await Folder_1.default.defaultFolder();
        if (!folder)
            return releaseMutex();
        note.parent_id = folder.id;
    }
    const isProvisionalNote = comp.props.provisionalNoteIds.includes(note.id);
    const saveOptions = {
        userSideValidation: true,
        fields: BaseModel_1.default.diffObjectsFields(state.lastSavedNote, note),
        dispatchOptions: { preserveSelection: true },
    };
    const hasAutoTitle = state.newAndNoTitleChangeNoteId || (isProvisionalNote && !note.title);
    if (hasAutoTitle && options.autoTitle) {
        note.title = Note_1.default.defaultTitle(note.body);
        if (saveOptions.fields && saveOptions.fields.indexOf('title') < 0)
            saveOptions.fields.push('title');
    }
    const savedNote = 'fields' in saveOptions && !saveOptions.fields.length ? Object.assign({}, note) : await Note_1.default.save(note, saveOptions);
    const stateNote = comp.state.note;
    // Note was reloaded while being saved.
    if (!recreatedNote && (!stateNote || stateNote.id !== savedNote.id))
        return releaseMutex();
    // Re-assign any property that might have changed during saving (updated_time, etc.)
    note = Object.assign(Object.assign({}, note), savedNote);
    if (stateNote.id === note.id) {
        // But we preserve the current title and body because
        // the user might have changed them between the time
        // saveNoteButton_press was called and the note was
        // saved (it's done asynchronously).
        //
        // If the title was auto-assigned above, we don't restore
        // it from the state because it will be empty there.
        if (!hasAutoTitle)
            note.title = stateNote.title;
        note.body = stateNote.body;
    }
    // eslint-disable-next-line @typescript-eslint/no-explicit-any -- Old code before rule was applied
    const newState = {
        lastSavedNote: Object.assign(Object.assign({}, note), savedNote),
        note: note,
    };
    if (isProvisionalNote && hasAutoTitle)
        newState.newAndNoTitleChangeNoteId = note.id;
    if (!options.autoTitle)
        newState.newAndNoTitleChangeNoteId = null;
    comp.setState(newState);
    if (isProvisionalNote) {
        const updateGeoloc = async () => {
            const geoNote = await Note_1.default.updateGeolocation(note.id);
            const stateNote = state.note;
            if (!stateNote || !geoNote)
                return;
            if (stateNote.id !== geoNote.id)
                return; // Another note has been loaded while geoloc was being retrieved
            // Geo-location for this note has been saved to the database however the properties
            // are not in the state so set them now.
            const geoInfo = {
                longitude: geoNote.longitude,
                latitude: geoNote.latitude,
                altitude: geoNote.altitude,
            };
            const modNote = Object.assign(Object.assign({}, stateNote), geoInfo);
            const modLastSavedNote = Object.assign(Object.assign({}, state.lastSavedNote), geoInfo);
            comp.setState({ note: modNote, lastSavedNote: modLastSavedNote });
        };
        // We don't wait because it can be done in the background
        void updateGeoloc();
    }
    releaseMutex();
};
// eslint-disable-next-line @typescript-eslint/no-explicit-any -- Old code before rule was applied
shared.saveOneProperty = async function (comp, name, value) {
    let note = Object.assign({}, comp.state.note);
    const recreatedNote = await shared.handleNoteDeletedWhileEditing_(note);
    if (recreatedNote)
        note = recreatedNote;
    // eslint-disable-next-line @typescript-eslint/no-explicit-any -- Old code before rule was applied
    let toSave = { id: note.id };
    toSave[name] = value;
    toSave = await Note_1.default.save(toSave);
    note[name] = toSave[name];
    comp.setState({
        lastSavedNote: Object.assign({}, note),
        note: note,
    });
};
// eslint-disable-next-line @typescript-eslint/no-explicit-any -- Old code before rule was applied
shared.noteComponent_change = function (comp, propName, propValue) {
    // eslint-disable-next-line @typescript-eslint/no-explicit-any -- Old code before rule was applied
    const newState = {};
    const note = Object.assign({}, comp.state.note);
    note[propName] = propValue;
    newState.note = note;
    comp.setState(newState);
    comp.scheduleSave(newState);
};
// eslint-disable-next-line @typescript-eslint/no-explicit-any -- Old code before rule was applied
let resourceCache_ = {};
shared.clearResourceCache = function () {
    resourceCache_ = {};
};
shared.attachedResources = async function (noteBody) {
    if (!noteBody)
        return {};
    const resourceIds = await Note_1.default.linkedItemIdsByType(BaseModel_1.default.TYPE_RESOURCE, noteBody);
    // eslint-disable-next-line @typescript-eslint/no-explicit-any -- Old code before rule was applied
    const output = {};
    for (let i = 0; i < resourceIds.length; i++) {
        const id = resourceIds[i];
        if (resourceCache_[id]) {
            output[id] = resourceCache_[id];
        }
        else {
            const resource = await Resource_1.default.load(id);
            const localState = await Resource_1.default.localState(resource);
            const o = {
                item: resource,
                localState: localState,
            };
            // eslint-disable-next-line require-atomic-updates
            resourceCache_[id] = o;
            output[id] = o;
        }
    }
    return output;
};
shared.isModified = function (comp) {
    if (!comp.state.note || !comp.state.lastSavedNote)
        return false;
    const diff = BaseModel_1.default.diffObjects(comp.state.lastSavedNote, comp.state.note);
    delete diff.type_;
    return !!Object.getOwnPropertyNames(diff).length;
};
shared.reloadNote = async (comp) => {
    const isProvisionalNote = comp.props.provisionalNoteIds.includes(comp.props.noteId);
    const note = await Note_1.default.load(comp.props.noteId);
    let mode = 'view';
    if (isProvisionalNote && !comp.props.sharedData) {
        mode = 'edit';
        comp.scheduleFocusUpdate();
    }
    const fromShare = !!comp.props.sharedData;
    if (note) {
        const folder = Folder_1.default.byId(comp.props.folders, note.parent_id);
        comp.setState({
            lastSavedNote: Object.assign({}, note),
            note: note,
            mode: mode,
            folder: folder,
            isLoading: false,
            fromShare: !!comp.props.sharedData,
            noteResources: await shared.attachedResources(note ? note.body : ''),
            readOnly: (0, readOnly_1.itemIsReadOnlySync)(BaseModel_1.ModelType.Note, ItemChange_1.default.SOURCE_UNSPECIFIED, note, Setting_1.default.value('sync.userId'), BaseItem_1.default.syncShareCache),
            noteLastLoadTime: Date.now(),
        });
    }
    else {
        // Handle the case where a non-existent note is loaded. This can happen briefly after deleting a note.
        comp.setState({
            lastSavedNote: {},
            note: {},
            mode,
            folder: null,
            isLoading: true,
            fromShare,
            noteResources: [],
            readOnly: true,
            noteLastLoadTime: Date.now(),
        });
    }
    return note;
};
shared.initState = async function (comp) {
    const note = await shared.reloadNote(comp);
    if (comp.props.sharedData) {
        if (comp.props.sharedData.title) {
            this.noteComponent_change(comp, 'title', comp.props.sharedData.title);
        }
        if (comp.props.sharedData.text) {
            this.noteComponent_change(comp, 'body', comp.props.sharedData.text);
        }
        if (comp.props.sharedData.resources) {
            for (let i = 0; i < comp.props.sharedData.resources.length; i++) {
                const resource = comp.props.sharedData.resources[i];
                registry_1.reg.logger().info(`about to attach resource ${JSON.stringify(resource)}`);
                await comp.attachFile({
                    uri: resource.uri,
                    type: resource.mimeType,
                    fileName: resource.name,
                }, null);
            }
        }
    }
    // eslint-disable-next-line require-atomic-updates
    comp.lastLoadedNoteId_ = note === null || note === void 0 ? void 0 : note.id;
};
shared.toggleIsTodo_onPress = function (comp) {
    return Note_1.default.toggleIsTodo(comp.state.note);
};
function toggleCheckboxLine(ipcMessage, noteBody) {
    const newBody = noteBody.split('\n');
    const p = ipcMessage.split(':');
    const lineIndex = Number(p[p.length - 1]);
    if (lineIndex >= newBody.length) {
        registry_1.reg.logger().warn('Checkbox line out of bounds: ', ipcMessage);
        return newBody.join('\n');
    }
    let line = newBody[lineIndex];
    const noCrossIndex = line.trim().indexOf('- [ ] ');
    let crossIndex = line.trim().indexOf('- [x] ');
    if (crossIndex < 0)
        crossIndex = line.trim().indexOf('- [X] ');
    if (noCrossIndex < 0 && crossIndex < 0) {
        registry_1.reg.logger().warn('Could not find matching checkbox for message: ', ipcMessage);
        return newBody.join('\n');
    }
    let isCrossLine = false;
    if (noCrossIndex >= 0 && crossIndex >= 0) {
        isCrossLine = crossIndex < noCrossIndex;
    }
    else {
        isCrossLine = crossIndex >= 0;
    }
    if (!isCrossLine) {
        line = line.replace(/- \[ \] /, '- [x] ');
    }
    else {
        line = line.replace(/- \[x\] /i, '- [ ] ');
    }
    return [newBody, lineIndex, line];
}
shared.toggleCheckboxRange = function (ipcMessage, noteBody) {
    const [lineIndex, line] = toggleCheckboxLine(ipcMessage, noteBody).slice(1);
    const from = { line: lineIndex, ch: 0 };
    // eslint-disable-next-line @typescript-eslint/no-explicit-any -- Old code before rule was applied
    const to = { line: lineIndex, ch: line.length };
    return { line, from, to };
};
shared.toggleCheckbox = function (ipcMessage, noteBody) {
    const [newBody, lineIndex, line] = toggleCheckboxLine(ipcMessage, noteBody);
    // eslint-disable-next-line @typescript-eslint/no-explicit-any -- Old code before rule was applied
    newBody[lineIndex] = line;
    // eslint-disable-next-line @typescript-eslint/no-explicit-any -- Old code before rule was applied
    return newBody.join('\n');
};
// eslint-disable-next-line @typescript-eslint/no-explicit-any -- Old code before rule was applied
shared.installResourceHandling = function (refreshResourceHandler) {
    ResourceFetcher_1.default.instance().on('downloadComplete', refreshResourceHandler);
    ResourceFetcher_1.default.instance().on('downloadStarted', refreshResourceHandler);
    DecryptionWorker_1.default.instance().on('resourceDecrypted', refreshResourceHandler);
};
// eslint-disable-next-line @typescript-eslint/no-explicit-any -- Old code before rule was applied
shared.uninstallResourceHandling = function (refreshResourceHandler) {
    ResourceFetcher_1.default.instance().off('downloadComplete', refreshResourceHandler);
    ResourceFetcher_1.default.instance().off('downloadStarted', refreshResourceHandler);
    DecryptionWorker_1.default.instance().off('resourceDecrypted', refreshResourceHandler);
};
exports.default = shared;
//# sourceMappingURL=note-screen-shared.js.map