"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const isLowercaseHexadecimalString = (text) => {
    return text.match(/^[a-f0-9]+$/) && text.length % 2 === 0;
};
class StringToBufferDecorator {
    constructor(publicKeyCrypto_) {
        this.publicKeyCrypto_ = publicKeyCrypto_;
    }
    async generateKeyPair() {
        return this.publicKeyCrypto_.generateKeyPair();
    }
    async loadKeys(publicKeySource, privateKeySource, keySizeBits) {
        return this.publicKeyCrypto_.loadKeys(publicKeySource, privateKeySource, keySizeBits);
    }
    textToBuffer_(textUtf8) {
        // To avoid data loss when restoring (whether everything is capital or lowercase), work only
        // with lowercase hexadecimal.
        const isHex = isLowercaseHexadecimalString(textUtf8);
        // RSA can only encrypt a limited amount of data. If given hexadecimal data (e.g. when given
        // a Joplin master key), try to encrypt in that format.
        let data = isHex ? Buffer.from(textUtf8, 'hex') : Buffer.from(textUtf8, 'utf8');
        // Store the original data format with a single-byte suffix.
        if (isHex) {
            data = Buffer.concat([data, new Uint8Array([1])]);
        }
        else {
            data = Buffer.concat([data, new Uint8Array([0])]);
        }
        return data;
    }
    bufferToText_(buffer) {
        let encoding;
        const isHex = buffer.byteLength > 0 && buffer.readUInt8(buffer.byteLength - 1) === 1;
        if (isHex) {
            encoding = 'hex';
        }
        else {
            encoding = 'utf8';
        }
        // Use .slice for mobile compatibility:
        buffer = buffer.slice(0, buffer.byteLength - 1);
        return Buffer.from(buffer).toString(encoding);
    }
    get maximumPlaintextLengthBytes() {
        const baseMaximumLength = this.publicKeyCrypto_.maximumPlaintextLengthBytes;
        if (!baseMaximumLength)
            return baseMaximumLength;
        // Subtract an additional -1 to account for metadata:
        return baseMaximumLength - 1;
    }
    async encrypt(plaintextUtf8, rsaKeyPair) {
        const plaintextBuffer = this.textToBuffer_(plaintextUtf8);
        const internalMaximumLength = this.publicKeyCrypto_.maximumPlaintextLengthBytes;
        if (plaintextBuffer.byteLength > internalMaximumLength) {
            throw new Error(`Data too long (longer than ${internalMaximumLength})`);
        }
        const ciphertext = await this.publicKeyCrypto_.encrypt(plaintextBuffer, rsaKeyPair);
        return Buffer.from(ciphertext);
    }
    async decrypt(ciphertext, rsaKeyPair) {
        const plaintextBuffer = await this.publicKeyCrypto_.decrypt(ciphertext, rsaKeyPair);
        return this.bufferToText_(plaintextBuffer);
    }
    async publicKey(rsaKeyPair) {
        return this.publicKeyCrypto_.publicKey(rsaKeyPair);
    }
    async privateKey(rsaKeyPair) {
        return this.publicKeyCrypto_.privateKey(rsaKeyPair);
    }
}
exports.default = StringToBufferDecorator;
//# sourceMappingURL=StringToBufferWrapper.js.map