"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const InteropService_1 = require("../../services/interop/InteropService");
const types_1 = require("../../services/interop/types");
const shim_1 = require("../../shim");
const test_utils_1 = require("../../testing/test-utils");
const Folder_1 = require("../../models/Folder");
const Note_1 = require("../../models/Note");
const Tag_1 = require("../../models/Tag");
const Resource_1 = require("../../models/Resource");
const fs = require("fs-extra");
const BaseModel_1 = require("../../BaseModel");
const ArrayUtils = require("../../ArrayUtils");
const InteropService_Importer_Custom_1 = require("./InteropService_Importer_Custom");
const InteropService_Exporter_Custom_1 = require("./InteropService_Exporter_Custom");
const Module_1 = require("./Module");
async function recreateExportDir() {
    const dir = (0, test_utils_1.exportDir)();
    await fs.remove(dir);
    await fs.mkdirp(dir);
}
// eslint-disable-next-line @typescript-eslint/no-explicit-any -- Old code before rule was applied
function fieldsEqual(model1, model2, fieldNames) {
    for (let i = 0; i < fieldNames.length; i++) {
        const f = fieldNames[i];
        expect(model1[f]).toBe(model2[f]);
    }
}
function memoryExportModule() {
    const result = {
        destPath: '',
        items: [],
        resources: [],
    };
    const module = (0, Module_1.makeExportModule)({
        description: 'Memory Export Module',
        format: types_1.ExportModuleOutputFormat.Memory,
        fileExtensions: ['memory'],
    }, () => {
        return new InteropService_Exporter_Custom_1.default({
            onInit: async (context) => {
                result.destPath = context.destPath;
            },
            // eslint-disable-next-line @typescript-eslint/no-explicit-any -- Old code before rule was applied
            onProcessItem: async (_context, itemType, item) => {
                result.items.push({
                    type: itemType,
                    object: item,
                });
            },
            // eslint-disable-next-line @typescript-eslint/no-explicit-any -- Old code before rule was applied
            onProcessResource: async (_context, resource, filePath) => {
                result.resources.push({
                    filePath: filePath,
                    object: resource,
                });
            },
            onClose: async (_context) => {
                // nothing
            },
        });
    });
    return { result, module };
}
describe('services_InteropService', () => {
    beforeEach(async () => {
        await (0, test_utils_1.setupDatabaseAndSynchronizer)(1);
        await (0, test_utils_1.switchClient)(1);
        await recreateExportDir();
    });
    it('should export and import folders', (async () => {
        const service = InteropService_1.default.instance();
        let folder1 = await Folder_1.default.save({ title: 'folder1' });
        folder1 = await Folder_1.default.load(folder1.id);
        const filePath = `${(0, test_utils_1.exportDir)()}/test.jex`;
        await service.export({ path: filePath });
        await Folder_1.default.delete(folder1.id);
        await service.import({ path: filePath });
        // Check that a new folder, with a new ID, has been created
        expect(await Folder_1.default.count()).toBe(1);
        const folder2 = (await Folder_1.default.all())[0];
        expect(folder2.id).not.toBe(folder1.id);
        expect(folder2.title).toBe(folder1.title);
        await service.import({ path: filePath });
        // As there was already a folder with the same title, check that the new one has been renamed
        await Folder_1.default.delete(folder2.id);
        const folder3 = (await Folder_1.default.all())[0];
        expect(await Folder_1.default.count()).toBe(1);
        expect(folder3.title).not.toBe(folder2.title);
        let fieldNames = Folder_1.default.fieldNames();
        fieldNames = ArrayUtils.removeElement(fieldNames, 'id');
        fieldNames = ArrayUtils.removeElement(fieldNames, 'title');
        fieldsEqual(folder3, folder1, fieldNames);
    }));
    it('should import folders and de-duplicate titles when needed', (async () => {
        const service = InteropService_1.default.instance();
        const folder1 = await Folder_1.default.save({ title: 'folder' });
        const folder2 = await Folder_1.default.save({ title: 'folder' });
        const filePath = `${(0, test_utils_1.exportDir)()}/test.jex`;
        await service.export({ path: filePath });
        await Folder_1.default.delete(folder1.id);
        await Folder_1.default.delete(folder2.id);
        await service.import({ path: filePath });
        const allFolders = await Folder_1.default.all();
        // eslint-disable-next-line @typescript-eslint/no-explicit-any -- Old code before rule was applied
        expect(allFolders.map((f) => f.title).sort().join(' - ')).toBe('folder - folder (1)');
    }));
    it('should import folders, and only de-duplicate titles when needed', (async () => {
        const service = InteropService_1.default.instance();
        const folder1 = await Folder_1.default.save({ title: 'folder1' });
        const folder2 = await Folder_1.default.save({ title: 'folder2' });
        await Folder_1.default.save({ title: 'Sub', parent_id: folder1.id });
        await Folder_1.default.save({ title: 'Sub', parent_id: folder2.id });
        const filePath = `${(0, test_utils_1.exportDir)()}/test.jex`;
        await service.export({ path: filePath });
        await Folder_1.default.delete(folder1.id);
        await Folder_1.default.delete(folder2.id);
        await service.import({ path: filePath });
        const importedFolder1 = await Folder_1.default.loadByTitle('folder1');
        const importedFolder2 = await Folder_1.default.loadByTitle('folder2');
        const importedSub1 = await Folder_1.default.load((await Folder_1.default.childrenIds(importedFolder1.id))[0]);
        const importedSub2 = await Folder_1.default.load((await Folder_1.default.childrenIds(importedFolder2.id))[0]);
        expect(importedSub1.title).toBe('Sub');
        expect(importedSub2.title).toBe('Sub');
    }));
    it('should export and import folders and notes', (async () => {
        const service = InteropService_1.default.instance();
        const folder1 = await Folder_1.default.save({ title: 'folder1' });
        let note1 = await Note_1.default.save({ title: 'ma note', parent_id: folder1.id });
        note1 = await Note_1.default.load(note1.id);
        const filePath = `${(0, test_utils_1.exportDir)()}/test.jex`;
        await service.export({ path: filePath });
        await Folder_1.default.delete(folder1.id);
        await Note_1.default.delete(note1.id);
        await service.import({ path: filePath });
        expect(await Note_1.default.count()).toBe(1);
        let note2 = (await Note_1.default.all())[0];
        const folder2 = (await Folder_1.default.all())[0];
        expect(note1.parent_id).not.toBe(note2.parent_id);
        expect(note1.id).not.toBe(note2.id);
        expect(note2.parent_id).toBe(folder2.id);
        let fieldNames = Note_1.default.fieldNames();
        fieldNames = ArrayUtils.removeElement(fieldNames, 'id');
        fieldNames = ArrayUtils.removeElement(fieldNames, 'parent_id');
        fieldsEqual(note1, note2, fieldNames);
        await service.import({ path: filePath });
        note2 = (await Note_1.default.all())[0];
        const note3 = (await Note_1.default.all())[1];
        expect(note2.id).not.toBe(note3.id);
        expect(note2.parent_id).not.toBe(note3.parent_id);
        fieldsEqual(note2, note3, fieldNames);
    }));
    it('should export and import notes to specific folder', (async () => {
        const service = InteropService_1.default.instance();
        const folder1 = await Folder_1.default.save({ title: 'folder1' });
        let note1 = await Note_1.default.save({ title: 'ma note', parent_id: folder1.id });
        note1 = await Note_1.default.load(note1.id);
        const filePath = `${(0, test_utils_1.exportDir)()}/test.jex`;
        await service.export({ path: filePath });
        await Note_1.default.delete(note1.id);
        await service.import({ path: filePath, destinationFolderId: folder1.id });
        expect(await Note_1.default.count()).toBe(1);
        expect(await Folder_1.default.count()).toBe(1);
        expect(await (0, test_utils_1.checkThrowAsync)(async () => await service.import({ path: filePath, destinationFolderId: 'oops' }))).toBe(true);
    }));
    it('should export and import tags', (async () => {
        const service = InteropService_1.default.instance();
        const filePath = `${(0, test_utils_1.exportDir)()}/test.jex`;
        const folder1 = await Folder_1.default.save({ title: 'folder1' });
        const note1 = await Note_1.default.save({ title: 'ma note', parent_id: folder1.id });
        let tag1 = await Tag_1.default.save({ title: 'mon tag' });
        tag1 = await Tag_1.default.load(tag1.id);
        await Tag_1.default.addNote(tag1.id, note1.id);
        await service.export({ path: filePath });
        await Folder_1.default.delete(folder1.id);
        await Note_1.default.delete(note1.id);
        await Tag_1.default.delete(tag1.id);
        await service.import({ path: filePath });
        expect(await Tag_1.default.count()).toBe(1);
        const tag2 = (await Tag_1.default.all())[0];
        const note2 = (await Note_1.default.all())[0];
        expect(tag1.id).not.toBe(tag2.id);
        let fieldNames = Note_1.default.fieldNames();
        fieldNames = ArrayUtils.removeElement(fieldNames, 'id');
        fieldsEqual(tag1, tag2, fieldNames);
        let noteIds = await Tag_1.default.noteIds(tag2.id);
        expect(noteIds.length).toBe(1);
        expect(noteIds[0]).toBe(note2.id);
        await service.import({ path: filePath });
        // If importing again, no new tag should be created as one with
        // the same name already existed. The newly imported note should
        // however go under that already existing tag.
        expect(await Tag_1.default.count()).toBe(1);
        noteIds = await Tag_1.default.noteIds(tag2.id);
        expect(noteIds.length).toBe(2);
    }));
    it('should export and import resources', (async () => {
        const service = InteropService_1.default.instance();
        const filePath = `${(0, test_utils_1.exportDir)()}/test.jex`;
        const folder1 = await Folder_1.default.save({ title: 'folder1' });
        let note1 = await Note_1.default.save({ title: 'ma note', parent_id: folder1.id });
        await shim_1.default.attachFileToNote(note1, `${test_utils_1.supportDir}/photo.jpg`);
        note1 = await Note_1.default.load(note1.id);
        let resourceIds = await Note_1.default.linkedResourceIds(note1.body);
        const resource1 = await Resource_1.default.load(resourceIds[0]);
        await service.export({ path: filePath });
        await Note_1.default.delete(note1.id);
        await service.import({ path: filePath });
        expect(await Resource_1.default.count()).toBe(2);
        const note2 = (await Note_1.default.all())[0];
        expect(note2.body).not.toBe(note1.body);
        resourceIds = await Note_1.default.linkedResourceIds(note2.body);
        expect(resourceIds.length).toBe(1);
        const resource2 = await Resource_1.default.load(resourceIds[0]);
        expect(resource2.id).not.toBe(resource1.id);
        let fieldNames = Note_1.default.fieldNames();
        fieldNames = ArrayUtils.removeElement(fieldNames, 'id');
        fieldsEqual(resource1, resource2, fieldNames);
        const resourcePath1 = Resource_1.default.fullPath(resource1);
        const resourcePath2 = Resource_1.default.fullPath(resource2);
        expect(resourcePath1).not.toBe(resourcePath2);
        expect((0, test_utils_1.fileContentEqual)(resourcePath1, resourcePath2)).toBe(true);
    }));
    it('should export and import single notes', (async () => {
        const service = InteropService_1.default.instance();
        const filePath = `${(0, test_utils_1.exportDir)()}/test.jex`;
        const folder1 = await Folder_1.default.save({ title: 'folder1' });
        const note1 = await Note_1.default.save({ title: 'ma note', parent_id: folder1.id });
        await service.export({ path: filePath, sourceNoteIds: [note1.id] });
        await Note_1.default.delete(note1.id);
        await Folder_1.default.delete(folder1.id);
        await service.import({ path: filePath });
        expect(await Note_1.default.count()).toBe(1);
        expect(await Folder_1.default.count()).toBe(1);
        const folder2 = (await Folder_1.default.all())[0];
        expect(folder2.title).toBe('test');
    }));
    it('should export and import single folders', (async () => {
        const service = InteropService_1.default.instance();
        const filePath = `${(0, test_utils_1.exportDir)()}/test.jex`;
        const folder1 = await Folder_1.default.save({ title: 'folder1' });
        const note1 = await Note_1.default.save({ title: 'ma note', parent_id: folder1.id });
        await service.export({ path: filePath, sourceFolderIds: [folder1.id] });
        await Note_1.default.delete(note1.id);
        await Folder_1.default.delete(folder1.id);
        await service.import({ path: filePath });
        expect(await Note_1.default.count()).toBe(1);
        expect(await Folder_1.default.count()).toBe(1);
        const folder2 = (await Folder_1.default.all())[0];
        expect(folder2.title).toBe('folder1');
    }));
    it('should export and import folder and its sub-folders', (async () => {
        const service = InteropService_1.default.instance();
        const filePath = `${(0, test_utils_1.exportDir)()}/test.jex`;
        const folder1 = await Folder_1.default.save({ title: 'folder1' });
        const folder2 = await Folder_1.default.save({ title: 'folder2', parent_id: folder1.id });
        const folder3 = await Folder_1.default.save({ title: 'folder3', parent_id: folder2.id });
        const folder4 = await Folder_1.default.save({ title: 'folder4', parent_id: folder2.id });
        const note1 = await Note_1.default.save({ title: 'ma note', parent_id: folder4.id });
        await service.export({ path: filePath, sourceFolderIds: [folder1.id] });
        await Note_1.default.delete(note1.id);
        await Folder_1.default.delete(folder1.id);
        await Folder_1.default.delete(folder2.id);
        await Folder_1.default.delete(folder3.id);
        await Folder_1.default.delete(folder4.id);
        await service.import({ path: filePath });
        expect(await Note_1.default.count()).toBe(1);
        expect(await Folder_1.default.count()).toBe(4);
        const folder1_2 = await Folder_1.default.loadByTitle('folder1');
        const folder2_2 = await Folder_1.default.loadByTitle('folder2');
        const folder3_2 = await Folder_1.default.loadByTitle('folder3');
        const folder4_2 = await Folder_1.default.loadByTitle('folder4');
        const note1_2 = await Note_1.default.loadByTitle('ma note');
        expect(folder2_2.parent_id).toBe(folder1_2.id);
        expect(folder3_2.parent_id).toBe(folder2_2.id);
        expect(folder4_2.parent_id).toBe(folder2_2.id);
        expect(note1_2.parent_id).toBe(folder4_2.id);
    }));
    it('should export and import links to notes', (async () => {
        const service = InteropService_1.default.instance();
        const filePath = `${(0, test_utils_1.exportDir)()}/test.jex`;
        const folder1 = await Folder_1.default.save({ title: 'folder1' });
        const note1 = await Note_1.default.save({ title: 'ma note', parent_id: folder1.id });
        const note2 = await Note_1.default.save({ title: 'ma deuxième note', body: `Lien vers première note : ${Note_1.default.markdownTag(note1)}`, parent_id: folder1.id });
        await service.export({ path: filePath, sourceFolderIds: [folder1.id] });
        await Note_1.default.delete(note1.id);
        await Note_1.default.delete(note2.id);
        await Folder_1.default.delete(folder1.id);
        await service.import({ path: filePath });
        expect(await Note_1.default.count()).toBe(2);
        expect(await Folder_1.default.count()).toBe(1);
        const note1_2 = await Note_1.default.loadByTitle('ma note');
        const note2_2 = await Note_1.default.loadByTitle('ma deuxième note');
        expect(note2_2.body.indexOf(note1_2.id) >= 0).toBe(true);
    }));
    it('should export selected notes in md format', (async () => {
        const service = InteropService_1.default.instance();
        const folder1 = await Folder_1.default.save({ title: 'folder1' });
        let note11 = await Note_1.default.save({ title: 'title note11', parent_id: folder1.id });
        note11 = await Note_1.default.load(note11.id);
        const note12 = await Note_1.default.save({ title: 'title note12', parent_id: folder1.id });
        await Note_1.default.load(note12.id);
        let folder2 = await Folder_1.default.save({ title: 'folder2', parent_id: folder1.id });
        folder2 = await Folder_1.default.load(folder2.id);
        let note21 = await Note_1.default.save({ title: 'title note21', parent_id: folder2.id });
        note21 = await Note_1.default.load(note21.id);
        await Folder_1.default.save({ title: 'folder3', parent_id: folder1.id });
        await Folder_1.default.load(folder2.id);
        const outDir = (0, test_utils_1.exportDir)();
        await service.export({ path: outDir, format: types_1.ExportModuleOutputFormat.Markdown, sourceNoteIds: [note11.id, note21.id] });
        // verify that the md files exist
        expect(await shim_1.default.fsDriver().exists(`${outDir}/folder1`)).toBe(true);
        expect(await shim_1.default.fsDriver().exists(`${outDir}/folder1/title note11.md`)).toBe(true);
        expect(await shim_1.default.fsDriver().exists(`${outDir}/folder1/title note12.md`)).toBe(false);
        expect(await shim_1.default.fsDriver().exists(`${outDir}/folder1/folder2`)).toBe(true);
        expect(await shim_1.default.fsDriver().exists(`${outDir}/folder1/folder2/title note21.md`)).toBe(true);
        expect(await shim_1.default.fsDriver().exists(`${outDir}/folder3`)).toBe(false);
    }));
    it('should export MD with unicode filenames', (async () => {
        const service = InteropService_1.default.instance();
        const folder1 = await Folder_1.default.save({ title: 'folder1' });
        const folder2 = await Folder_1.default.save({ title: 'ジョプリン' });
        await Note_1.default.save({ title: '生活', parent_id: folder1.id });
        await Note_1.default.save({ title: '生活', parent_id: folder1.id });
        await Note_1.default.save({ title: '生活', parent_id: folder1.id });
        await Note_1.default.save({ title: '', parent_id: folder1.id });
        await Note_1.default.save({ title: '', parent_id: folder1.id });
        await Note_1.default.save({ title: 'salut, ça roule ?', parent_id: folder1.id });
        await Note_1.default.save({ title: 'ジョプリン', parent_id: folder2.id });
        const outDir = (0, test_utils_1.exportDir)();
        await service.export({ path: outDir, format: types_1.ExportModuleOutputFormat.Markdown });
        expect(await shim_1.default.fsDriver().exists(`${outDir}/folder1/生活.md`)).toBe(true);
        expect(await shim_1.default.fsDriver().exists(`${outDir}/folder1/生活-1.md`)).toBe(true);
        expect(await shim_1.default.fsDriver().exists(`${outDir}/folder1/生活-2.md`)).toBe(true);
        expect(await shim_1.default.fsDriver().exists(`${outDir}/folder1/Untitled.md`)).toBe(true);
        expect(await shim_1.default.fsDriver().exists(`${outDir}/folder1/Untitled-1.md`)).toBe(true);
        expect(await shim_1.default.fsDriver().exists(`${outDir}/folder1/salut, ça roule _.md`)).toBe(true);
        expect(await shim_1.default.fsDriver().exists(`${outDir}/ジョプリン/ジョプリン.md`)).toBe(true);
    }));
    it('should export a notebook as MD', (async () => {
        const folder1 = await Folder_1.default.save({ title: 'testexportfolder' });
        await Note_1.default.save({ title: 'textexportnote1', parent_id: folder1.id });
        await Note_1.default.save({ title: 'textexportnote2', parent_id: folder1.id });
        const service = InteropService_1.default.instance();
        await service.export({
            path: (0, test_utils_1.exportDir)(),
            format: types_1.ExportModuleOutputFormat.Markdown,
            sourceFolderIds: [folder1.id],
        });
        expect(await shim_1.default.fsDriver().exists(`${(0, test_utils_1.exportDir)()}/testexportfolder/textexportnote1.md`)).toBe(true);
        expect(await shim_1.default.fsDriver().exists(`${(0, test_utils_1.exportDir)()}/testexportfolder/textexportnote2.md`)).toBe(true);
    }));
    it('should export conflict notes', (async () => {
        const folder1 = await Folder_1.default.save({ title: 'testexportfolder' });
        await Note_1.default.save({ title: 'textexportnote1', parent_id: folder1.id, is_conflict: 1 });
        await Note_1.default.save({ title: 'textexportnote2', parent_id: folder1.id });
        const service = InteropService_1.default.instance();
        await service.export({
            path: (0, test_utils_1.exportDir)(),
            format: types_1.ExportModuleOutputFormat.Markdown,
            sourceFolderIds: [folder1.id],
            includeConflicts: false,
        });
        expect(await shim_1.default.fsDriver().exists(`${(0, test_utils_1.exportDir)()}/testexportfolder/textexportnote1.md`)).toBe(false);
        expect(await shim_1.default.fsDriver().exists(`${(0, test_utils_1.exportDir)()}/testexportfolder/textexportnote2.md`)).toBe(true);
        await recreateExportDir();
        await service.export({
            path: (0, test_utils_1.exportDir)(),
            format: types_1.ExportModuleOutputFormat.Markdown,
            sourceFolderIds: [folder1.id],
            includeConflicts: true,
        });
        expect(await shim_1.default.fsDriver().exists(`${(0, test_utils_1.exportDir)()}/testexportfolder/textexportnote1.md`)).toBe(true);
        expect(await shim_1.default.fsDriver().exists(`${(0, test_utils_1.exportDir)()}/testexportfolder/textexportnote2.md`)).toBe(true);
    }));
    it('should not try to export folders with a non-existing parent', (async () => {
        // Handles and edge case where user has a folder but this folder with a parent
        // that doesn't exist. Can happen for example in this case:
        //
        // - folder1/folder2
        // - Client 1 sync folder2, but not folder1
        // - Client 2 sync and get folder2 only
        // - Client 2 export data
        // => Crash if we don't handle this case
        await Folder_1.default.save({ title: 'orphan', parent_id: '0c5bbd8a1b5a48189484a412a7e534cc' });
        const service = InteropService_1.default.instance();
        const result = await service.export({
            path: (0, test_utils_1.exportDir)(),
            format: types_1.ExportModuleOutputFormat.Markdown,
        });
        expect(result.warnings.length).toBe(0);
    }));
    it('should not export certain note properties', (async () => {
        const folder = await Folder_1.default.save({ title: 'folder', share_id: 'some_id', is_shared: 1 });
        let note = await Note_1.default.save({ title: 'note', is_shared: 1, share_id: 'someid', parent_id: folder.id });
        note = await shim_1.default.attachFileToNote(note, `${test_utils_1.supportDir}/photo.jpg`);
        const resourceId = (await Note_1.default.linkedResourceIds(note.body))[0];
        await Resource_1.default.save({ id: resourceId, share_id: 'some_id', is_shared: 1 });
        const service = InteropService_1.default.instance();
        const { result, module } = memoryExportModule();
        service.registerModule(module);
        await service.export({
            format: types_1.ExportModuleOutputFormat.Memory,
        });
        const exportedNote = (result.items.find(i => i.type === BaseModel_1.ModelType.Note)).object;
        expect(exportedNote.share_id).toBe('');
        expect(exportedNote.is_shared).toBe(0);
        const exportedFolder = (result.items.find(i => i.type === BaseModel_1.ModelType.Folder)).object;
        expect(exportedFolder.share_id).toBe('');
        expect(exportedFolder.is_shared).toBe(0);
        const exportedResource = (result.items.find(i => i.type === BaseModel_1.ModelType.Resource)).object;
        expect(exportedResource.share_id).toBe('');
        expect(exportedResource.is_shared).toBe(0);
    }));
    it('should allow registering new import modules', (async () => {
        const testImportFilePath = `${(0, test_utils_1.exportDir)()}/testImport${Math.random()}.test`;
        await shim_1.default.fsDriver().writeFile(testImportFilePath, 'test', 'utf8');
        const result = {
            hasBeenExecuted: false,
            sourcePath: '',
        };
        const module = (0, Module_1.makeImportModule)({
            type: types_1.ModuleType.Importer,
            description: 'Test Import Module',
            format: 'testing',
            fileExtensions: ['test'],
        }, () => {
            return new InteropService_Importer_Custom_1.default({
                onExec: async (context) => {
                    result.hasBeenExecuted = true;
                    result.sourcePath = context.sourcePath;
                },
            });
        });
        const service = InteropService_1.default.instance();
        service.registerModule(module);
        await service.import({
            format: 'testing',
            path: testImportFilePath,
        });
        expect(result.hasBeenExecuted).toBe(true);
        expect(result.sourcePath).toBe(testImportFilePath);
    }));
    it('should allow registering new export modules', (async () => {
        const folder1 = await Folder_1.default.save({ title: 'folder1' });
        const note1 = await Note_1.default.save({ title: 'note1', parent_id: folder1.id });
        await Note_1.default.save({ title: 'note2', parent_id: folder1.id });
        await shim_1.default.attachFileToNote(note1, `${test_utils_1.supportDir}/photo.jpg`);
        const filePath = `${(0, test_utils_1.exportDir)()}/example.test`;
        // eslint-disable-next-line @typescript-eslint/no-explicit-any -- Old code before rule was applied
        const result = {
            destPath: '',
            itemTypes: [],
            items: [],
            resources: [],
            filePaths: [],
            closeCalled: false,
        };
        const module = (0, Module_1.makeExportModule)({
            type: types_1.ModuleType.Exporter,
            description: 'Test Export Module',
            // eslint-disable-next-line @typescript-eslint/no-explicit-any -- Old code before rule was applied
            format: 'testing',
            fileExtensions: ['test'],
        }, () => {
            return new InteropService_Exporter_Custom_1.default({
                onInit: async (context) => {
                    result.destPath = context.destPath;
                },
                // eslint-disable-next-line @typescript-eslint/no-explicit-any -- Old code before rule was applied
                onProcessItem: async (_context, itemType, item) => {
                    result.itemTypes.push(itemType);
                    result.items.push(item);
                },
                // eslint-disable-next-line @typescript-eslint/no-explicit-any -- Old code before rule was applied
                onProcessResource: async (_context, resource, filePath) => {
                    result.resources.push(resource);
                    result.filePaths.push(filePath);
                },
                onClose: async (_context) => {
                    result.closeCalled = true;
                },
            });
        });
        const service = InteropService_1.default.instance();
        service.registerModule(module);
        await service.export({
            // eslint-disable-next-line @typescript-eslint/no-explicit-any -- Old code before rule was applied
            format: 'testing',
            path: filePath,
        });
        expect(result.destPath).toBe(filePath);
        expect(result.itemTypes.sort().join('_')).toBe('1_1_2_4');
        expect(result.items.length).toBe(4);
        // eslint-disable-next-line @typescript-eslint/no-explicit-any -- Old code before rule was applied
        expect(result.items.map((o) => o.title).sort().join('_')).toBe('folder1_note1_note2_photo.jpg');
        expect(result.resources.length).toBe(1);
        expect(result.resources[0].title).toBe('photo.jpg');
        expect(result.filePaths.length).toBe(1);
        expect(!!result.filePaths[0]).toBe(true);
        expect(result.closeCalled).toBe(true);
    }));
});
//# sourceMappingURL=InteropService.test.js.map