"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const time_1 = require("../../time");
const shim_1 = require("../../shim");
const Setting_1 = require("../../models/Setting");
const test_utils_synchronizer_1 = require("../../testing/test-utils-synchronizer");
const test_utils_1 = require("../../testing/test-utils");
const Folder_1 = require("../../models/Folder");
const Note_1 = require("../../models/Note");
const Resource_1 = require("../../models/Resource");
const ResourceFetcher_1 = require("../../services/ResourceFetcher");
const BaseItem_1 = require("../../models/BaseItem");
const BaseModel_1 = require("../../BaseModel");
const syncInfoUtils_1 = require("../synchronizer/syncInfoUtils");
const utils_1 = require("../e2ee/utils");
let insideBeforeEach = false;
describe('Synchronizer.resources', () => {
    beforeEach(async () => {
        insideBeforeEach = true;
        await (0, test_utils_1.setupDatabaseAndSynchronizer)(1);
        await (0, test_utils_1.setupDatabaseAndSynchronizer)(2);
        await (0, test_utils_1.switchClient)(1);
        insideBeforeEach = false;
    });
    it('should sync resources', async () => {
        while (insideBeforeEach)
            await time_1.default.msleep(500);
        const folder1 = await Folder_1.default.save({ title: 'folder1' });
        const note1 = await Note_1.default.save({ title: 'ma note', parent_id: folder1.id });
        await shim_1.default.attachFileToNote(note1, `${test_utils_1.supportDir}/photo.jpg`);
        const resource1 = (await Resource_1.default.all())[0];
        const resourcePath1 = Resource_1.default.fullPath(resource1);
        await (0, test_utils_1.synchronizerStart)();
        expect((await (0, test_utils_synchronizer_1.remoteNotesFoldersResources)()).length).toBe(3);
        await (0, test_utils_1.switchClient)(2);
        await (0, test_utils_1.synchronizerStart)();
        const allResources = await Resource_1.default.all();
        expect(allResources.length).toBe(1);
        let resource1_2 = allResources[0];
        let ls = await Resource_1.default.localState(resource1_2);
        expect(resource1_2.id).toBe(resource1.id);
        expect(ls.fetch_status).toBe(Resource_1.default.FETCH_STATUS_IDLE);
        const fetcher = new ResourceFetcher_1.default(() => { return (0, test_utils_1.synchronizer)().api(); });
        fetcher.queueDownload_(resource1_2.id);
        await fetcher.waitForAllFinished();
        resource1_2 = await Resource_1.default.load(resource1.id);
        ls = await Resource_1.default.localState(resource1_2);
        expect(ls.fetch_status).toBe(Resource_1.default.FETCH_STATUS_DONE);
        const resourcePath1_2 = Resource_1.default.fullPath(resource1_2);
        expect((0, test_utils_1.fileContentEqual)(resourcePath1, resourcePath1_2)).toBe(true);
    });
    it('should handle resource download errors', async () => {
        while (insideBeforeEach)
            await time_1.default.msleep(500);
        const folder1 = await Folder_1.default.save({ title: 'folder1' });
        const note1 = await Note_1.default.save({ title: 'ma note', parent_id: folder1.id });
        await shim_1.default.attachFileToNote(note1, `${test_utils_1.supportDir}/photo.jpg`);
        let resource1 = (await Resource_1.default.all())[0];
        await (0, test_utils_1.synchronizerStart)();
        await (0, test_utils_1.switchClient)(2);
        await (0, test_utils_1.synchronizerStart)();
        const fetcher = new ResourceFetcher_1.default(() => {
            return {
                // Simulate a failed download
                // eslint-disable-next-line @typescript-eslint/ban-types -- Old code before rule was applied
                get: () => { return new Promise((_resolve, reject) => { reject(new Error('did not work')); }); },
            };
        });
        fetcher.queueDownload_(resource1.id);
        await fetcher.waitForAllFinished();
        resource1 = await Resource_1.default.load(resource1.id);
        const ls = await Resource_1.default.localState(resource1);
        expect(ls.fetch_status).toBe(Resource_1.default.FETCH_STATUS_ERROR);
        expect(ls.fetch_error).toBe('did not work');
    });
    it('should set the resource file size if it is missing', async () => {
        while (insideBeforeEach)
            await time_1.default.msleep(500);
        const folder1 = await Folder_1.default.save({ title: 'folder1' });
        const note1 = await Note_1.default.save({ title: 'ma note', parent_id: folder1.id });
        await shim_1.default.attachFileToNote(note1, `${test_utils_1.supportDir}/photo.jpg`);
        await (0, test_utils_1.synchronizerStart)();
        await (0, test_utils_1.switchClient)(2);
        await (0, test_utils_1.synchronizerStart)();
        let r1 = (await Resource_1.default.all())[0];
        await Resource_1.default.setFileSizeOnly(r1.id, -1);
        r1 = await Resource_1.default.load(r1.id);
        expect(r1.size).toBe(-1);
        const fetcher = new ResourceFetcher_1.default(() => { return (0, test_utils_1.synchronizer)().api(); });
        fetcher.queueDownload_(r1.id);
        await fetcher.waitForAllFinished();
        r1 = await Resource_1.default.load(r1.id);
        expect(r1.size).toBe(2720);
    });
    it('should delete resources', async () => {
        while (insideBeforeEach)
            await time_1.default.msleep(500);
        const folder1 = await Folder_1.default.save({ title: 'folder1' });
        const note1 = await Note_1.default.save({ title: 'ma note', parent_id: folder1.id });
        await shim_1.default.attachFileToNote(note1, `${test_utils_1.supportDir}/photo.jpg`);
        const resource1 = (await Resource_1.default.all())[0];
        const resourcePath1 = Resource_1.default.fullPath(resource1);
        await (0, test_utils_1.synchronizerStart)();
        await (0, test_utils_1.switchClient)(2);
        await (0, test_utils_1.synchronizerStart)();
        let allResources = await Resource_1.default.all();
        expect(allResources.length).toBe(1);
        expect((await (0, test_utils_synchronizer_1.remoteNotesFoldersResources)()).length).toBe(3);
        await Resource_1.default.delete(resource1.id);
        await (0, test_utils_1.synchronizerStart)();
        expect((await (0, test_utils_synchronizer_1.remoteNotesFoldersResources)()).length).toBe(2);
        const remoteBlob = await (0, test_utils_1.fileApi)().stat(`.resource/${resource1.id}`);
        expect(!remoteBlob).toBe(true);
        await (0, test_utils_1.switchClient)(1);
        expect(await shim_1.default.fsDriver().exists(resourcePath1)).toBe(true);
        await (0, test_utils_1.synchronizerStart)();
        allResources = await Resource_1.default.all();
        expect(allResources.length).toBe(0);
        expect(await shim_1.default.fsDriver().exists(resourcePath1)).toBe(false);
    });
    it('should encrypt resources', async () => {
        (0, syncInfoUtils_1.setEncryptionEnabled)(true);
        const masterKey = await (0, test_utils_1.loadEncryptionMasterKey)();
        const folder1 = await Folder_1.default.save({ title: 'folder1' });
        const note1 = await Note_1.default.save({ title: 'ma note', parent_id: folder1.id });
        await shim_1.default.attachFileToNote(note1, `${test_utils_1.supportDir}/photo.jpg`);
        const resource1 = (await Resource_1.default.all())[0];
        const resourcePath1 = Resource_1.default.fullPath(resource1);
        await (0, test_utils_1.synchronizerStart)();
        await (0, test_utils_1.switchClient)(2);
        await (0, test_utils_1.synchronizerStart)();
        Setting_1.default.setObjectValue('encryption.passwordCache', masterKey.id, '123456');
        await (0, utils_1.loadMasterKeysFromSettings)((0, test_utils_1.encryptionService)());
        const fetcher = new ResourceFetcher_1.default(() => { return (0, test_utils_1.synchronizer)().api(); });
        fetcher.queueDownload_(resource1.id);
        await fetcher.waitForAllFinished();
        let resource1_2 = (await Resource_1.default.all())[0];
        resource1_2 = await Resource_1.default.decrypt(resource1_2);
        const resourcePath1_2 = Resource_1.default.fullPath(resource1_2);
        expect((0, test_utils_1.fileContentEqual)(resourcePath1, resourcePath1_2)).toBe(true);
    });
    it('should sync resource blob changes', async () => {
        const tempFile = (0, test_utils_1.tempFilePath)('txt');
        await shim_1.default.fsDriver().writeFile(tempFile, '1234', 'utf8');
        const folder1 = await Folder_1.default.save({ title: 'folder1' });
        const note1 = await Note_1.default.save({ title: 'ma note', parent_id: folder1.id });
        await shim_1.default.attachFileToNote(note1, tempFile);
        await (0, test_utils_1.synchronizerStart)();
        await (0, test_utils_1.switchClient)(2);
        await (0, test_utils_1.synchronizerStart)();
        await (0, test_utils_1.resourceFetcher)().start();
        await (0, test_utils_1.resourceFetcher)().waitForAllFinished();
        let resource1_2 = (await Resource_1.default.all())[0];
        const modFile = (0, test_utils_1.tempFilePath)('txt');
        await shim_1.default.fsDriver().writeFile(modFile, '1234 MOD', 'utf8');
        await Resource_1.default.updateResourceBlobContent(resource1_2.id, modFile);
        const originalSize = resource1_2.size;
        resource1_2 = (await Resource_1.default.all())[0];
        const newSize = resource1_2.size;
        expect(originalSize).toBe(4);
        expect(newSize).toBe(8);
        await (0, test_utils_1.synchronizerStart)();
        await (0, test_utils_1.switchClient)(1);
        await (0, test_utils_1.synchronizerStart)();
        await (0, test_utils_1.resourceFetcher)().start();
        await (0, test_utils_1.resourceFetcher)().waitForAllFinished();
        const resource1_1 = (await Resource_1.default.all())[0];
        expect(resource1_1.size).toBe(newSize);
        expect(await Resource_1.default.resourceBlobContent(resource1_1.id, 'utf8')).toBe('1234 MOD');
    });
    it('should handle resource conflicts', async () => {
        {
            const tempFile = (0, test_utils_1.tempFilePath)('txt');
            await shim_1.default.fsDriver().writeFile(tempFile, '1234', 'utf8');
            const folder1 = await Folder_1.default.save({ title: 'folder1' });
            const note1 = await Note_1.default.save({ title: 'ma note', parent_id: folder1.id });
            await shim_1.default.attachFileToNote(note1, tempFile);
            await (0, test_utils_1.synchronizerStart)();
        }
        await (0, test_utils_1.switchClient)(2);
        {
            await (0, test_utils_1.synchronizerStart)();
            await (0, test_utils_1.resourceFetcher)().start();
            await (0, test_utils_1.resourceFetcher)().waitForAllFinished();
            const resource = (await Resource_1.default.all())[0];
            const modFile2 = (0, test_utils_1.tempFilePath)('txt');
            await shim_1.default.fsDriver().writeFile(modFile2, '1234 MOD 2', 'utf8');
            await Resource_1.default.updateResourceBlobContent(resource.id, modFile2);
            await (0, test_utils_1.synchronizerStart)();
        }
        await (0, test_utils_1.switchClient)(1);
        {
            // Going to modify a resource without syncing first, which will cause a conflict
            const resource = (await Resource_1.default.all())[0];
            const modFile1 = (0, test_utils_1.tempFilePath)('txt');
            await shim_1.default.fsDriver().writeFile(modFile1, '1234 MOD 1', 'utf8');
            await Resource_1.default.updateResourceBlobContent(resource.id, modFile1);
            await (0, test_utils_1.synchronizerStart)(); // CONFLICT
            // If we try to read the resource content now, it should throw because the local
            // content has been moved to the conflict notebook, and the new local content
            // has not been downloaded yet.
            await (0, test_utils_1.checkThrowAsync)(async () => await Resource_1.default.resourceBlobContent(resource.id));
            // Now download resources, and our local content would have been overwritten by
            // the content from client 2
            await (0, test_utils_1.resourceFetcher)().start();
            await (0, test_utils_1.resourceFetcher)().waitForAllFinished();
            const localContent = await Resource_1.default.resourceBlobContent(resource.id, 'utf8');
            expect(localContent).toBe('1234 MOD 2');
            // Check that the Conflict note has been generated, with the conflict resource
            // attached to it, and check that it has the original content.
            const allNotes = await Note_1.default.all();
            expect(allNotes.length).toBe(2);
            const resourceConflictFolderId = await Resource_1.default.resourceConflictFolderId();
            const conflictNote = allNotes.find((v) => {
                return v.parent_id === resourceConflictFolderId;
            });
            expect(!!conflictNote).toBe(true);
            const resourceIds = await Note_1.default.linkedResourceIds(conflictNote.body);
            expect(resourceIds.length).toBe(1);
            const conflictContent = await Resource_1.default.resourceBlobContent(resourceIds[0], 'utf8');
            expect(conflictContent).toBe('1234 MOD 1');
            // Also check that the conflict folder has been created and that it
            // is a top folder.
            const resourceConflictFolder = await Folder_1.default.load(resourceConflictFolderId);
            expect(resourceConflictFolder).toBeTruthy();
            expect(resourceConflictFolder.parent_id).toBeFalsy();
        }
    });
    it('should handle resource conflicts if a resource is changed locally but deleted remotely', async () => {
        {
            const tempFile = (0, test_utils_1.tempFilePath)('txt');
            await shim_1.default.fsDriver().writeFile(tempFile, '1234', 'utf8');
            const folder1 = await Folder_1.default.save({ title: 'folder1' });
            const note1 = await Note_1.default.save({ title: 'ma note', parent_id: folder1.id });
            await shim_1.default.attachFileToNote(note1, tempFile);
            await (0, test_utils_1.synchronizerStart)();
        }
        await (0, test_utils_1.switchClient)(2);
        {
            await (0, test_utils_1.synchronizerStart)();
            await (0, test_utils_1.resourceFetcher)().startAndWait();
        }
        await (0, test_utils_1.switchClient)(1);
        {
            const resource = (await Resource_1.default.all())[0];
            await Resource_1.default.delete(resource.id);
            await (0, test_utils_1.synchronizerStart)();
        }
        await (0, test_utils_1.switchClient)(2);
        {
            const originalResource = (await Resource_1.default.all())[0];
            await Resource_1.default.save({ id: originalResource.id, title: 'modified resource' });
            await (0, test_utils_1.synchronizerStart)(); // CONFLICT
            const deletedResource = await Resource_1.default.load(originalResource.id);
            expect(!deletedResource).toBe(true);
            const allResources = await Resource_1.default.all();
            expect(allResources.length).toBe(1);
            const conflictResource = allResources[0];
            expect(originalResource.id).not.toBe(conflictResource.id);
            expect(conflictResource.title).toBe('modified resource');
        }
    });
    it('should not upload a resource if it has not been fetched yet', async () => {
        // In some rare cases, the synchronizer might try to upload a resource even though it
        // doesn't have the resource file. It can happen in this situation:
        // - C1 create resource
        // - C1 sync
        // - C2 sync
        // - C2 resource metadata is received but ResourceFetcher hasn't downloaded the file yet
        // - C2 enables E2EE - all the items are marked for forced sync
        // - C2 sync
        // The synchronizer will try to upload the resource, even though it doesn't have the file,
        // so we need to make sure it doesn't. But also that once it gets the file, the resource
        // does get uploaded.
        const note1 = await Note_1.default.save({ title: 'note' });
        await shim_1.default.attachFileToNote(note1, `${test_utils_1.supportDir}/photo.jpg`);
        const resource = (await Resource_1.default.all())[0];
        await Resource_1.default.setLocalState(resource.id, { fetch_status: Resource_1.default.FETCH_STATUS_IDLE });
        await (0, test_utils_1.synchronizerStart)();
        expect((await (0, test_utils_synchronizer_1.remoteResources)()).length).toBe(0);
        await Resource_1.default.setLocalState(resource.id, { fetch_status: Resource_1.default.FETCH_STATUS_DONE });
        await (0, test_utils_1.synchronizerStart)();
        // At first, the resource is marked as cannot sync, so even after
        // synchronisation, nothing should happen.
        expect((await (0, test_utils_synchronizer_1.remoteResources)()).length).toBe(0);
        // The user can retry the item, in which case sync should happen.
        await BaseItem_1.default.saveSyncEnabled(BaseModel_1.ModelType.Resource, resource.id);
        await (0, test_utils_1.synchronizerStart)();
        expect((await (0, test_utils_synchronizer_1.remoteResources)()).length).toBe(1);
    });
    it('should not download resources over the limit', async () => {
        const note1 = await Note_1.default.save({ title: 'note' });
        await shim_1.default.attachFileToNote(note1, `${test_utils_1.supportDir}/photo.jpg`);
        await (0, test_utils_1.synchronizer)().start();
        await (0, test_utils_1.switchClient)(2);
        const previousMax = (0, test_utils_1.synchronizer)().maxResourceSize_;
        (0, test_utils_1.synchronizer)().maxResourceSize_ = 1;
        await (0, test_utils_1.synchronizerStart)();
        (0, test_utils_1.synchronizer)().maxResourceSize_ = previousMax;
        const syncItems = await BaseItem_1.default.allSyncItems((0, test_utils_1.syncTargetId)());
        expect(syncItems.length).toBe(2);
        expect(syncItems[1].item_location).toBe(BaseItem_1.default.SYNC_ITEM_LOCATION_REMOTE);
        expect(syncItems[1].sync_disabled).toBe(1);
    });
    it('should not upload blob if it has not changed', async () => {
        const note = await Note_1.default.save({});
        await shim_1.default.attachFileToNote(note, `${test_utils_1.supportDir}/sample.txt`);
        const resource = (await Resource_1.default.all())[0];
        const resourcePath = `.resource/${resource.id}`;
        await (0, test_utils_1.synchronizer)().api().mkdir('.resource/');
        await (0, test_utils_1.synchronizer)().api().put(resourcePath, 'before upload');
        expect(await (0, test_utils_1.synchronizer)().api().get(resourcePath)).toBe('before upload');
        await (0, test_utils_1.synchronizerStart)();
        expect(await (0, test_utils_1.synchronizer)().api().get(resourcePath)).toBe('just testing');
        // ----------------------------------------------------------------------
        // Change metadata only and check that blob is not uploaded. To do this,
        // we manually overwrite the data on the sync target, then sync. If the
        // synchronizer doesn't upload the blob, this manually changed data
        // should remain.
        // ----------------------------------------------------------------------
        await Resource_1.default.save({ id: resource.id, title: 'my new title' });
        await (0, test_utils_1.synchronizer)().api().put(resourcePath, 'check if changed');
        await (0, test_utils_1.synchronizerStart)();
        expect(await (0, test_utils_1.synchronizer)().api().get(resourcePath)).toBe('check if changed');
        // ----------------------------------------------------------------------
        // Now change the blob, and check that the remote item has been
        // overwritten.
        // ----------------------------------------------------------------------
        await Resource_1.default.updateResourceBlobContent(resource.id, `${test_utils_1.supportDir}/sample.txt`);
        await (0, test_utils_1.synchronizerStart)();
        expect(await (0, test_utils_1.synchronizer)().api().get(resourcePath)).toBe('just testing');
        // ----------------------------------------------------------------------
        // Change the blob, then change the metadata, and sync. Even though
        // blob_updated_time is earlier than updated_time, it should still
        // update everything on the sync target, because both times are after
        // the item sync_time.
        // ----------------------------------------------------------------------
        await Resource_1.default.updateResourceBlobContent(resource.id, `${test_utils_1.supportDir}/sample2.txt`);
        await (0, test_utils_1.msleep)(1);
        await Resource_1.default.save({ id: resource.id, title: 'my new title 2' });
        await (0, test_utils_1.synchronizerStart)();
        expect(await (0, test_utils_1.synchronizer)().api().get(resourcePath)).toBe('just testing 2');
        expect(await (0, test_utils_1.synchronizer)().api().get(`${resource.id}.md`)).toContain('my new title 2');
    });
});
//# sourceMappingURL=Synchronizer.resources.test.js.map