@doc raw"""
    LogEuclideanMetric <: RiemannianMetric

The LogEuclidean Metric consists of the Euclidean metric applied to all elements after mapping them
into the Lie Algebra, i.e. performing a matrix logarithm beforehand.
"""
struct LogEuclideanMetric <: RiemannianMetric end

@doc raw"""
    distance(M::MetricManifold{ℝ,<:SymmetricPositiveDefinite,LogEuclideanMetric}, p, q)

Compute the distance on the [`SymmetricPositiveDefinite`](@ref) manifold between
`p` and `q` as a [`MetricManifold`](@extref ManifoldsBase.MetricManifold) with [`LogEuclideanMetric`](@ref).
The formula reads

```math
    d_{\mathcal P(n)}(p,q) = \lVert \operatorname{Log} p - \operatorname{Log} q \rVert_{\mathrm{F}}
```

where $\operatorname{Log}$ denotes the matrix logarithm and
$\lVert⋅\rVert_{\mathrm{F}}$ denotes the matrix Frobenius norm.
"""
function distance(::MetricManifold{ℝ, <:SymmetricPositiveDefinite, LogEuclideanMetric}, p, q)
    return norm(log(Symmetric(p)) - log(Symmetric(q)))
end

"""
    is_flat(::MetricManifold{ℝ,<:SymmetricPositiveDefinite,LogEuclideanMetric})

Return false. [`SymmetricPositiveDefinite`](@ref) with [`LogEuclideanMetric`](@ref)
is not a flat manifold.
"""
is_flat(M::MetricManifold{ℝ, <:SymmetricPositiveDefinite, LogEuclideanMetric}) = false
