using Base.MathConstants

@testset "exponential integral" begin
    @testset "E1 real" begin
        @test expint(0) == Inf
        @test expint(Inf) == 0
        @test expint(1) ≅ 0.219383934395520273677163775460121649031047293406908207577
        @test expint(2) ≅ 0.048900510708061119567239835228049522314492184963023116327
        @test expint(4) ≅ 0.0037793524098489064788748601324664148517165470424895803607340203
        @test expint(10) ≅ 4.1569689296853242774028598102781803843462900824195331e-6
        @test expint(20) ≅ 9.8355252906498816903969871088947760743563240711060318495238e-11
        @test expint(100) ≅ 3.6835977616820321802351926205081189876552201369095676e-46
        @test expint(500) ≅ 1.422076782253638422098193936057278281607864840846107e-220
        @test expint(800) ≅ 0.0

        @test_throws DomainError expint(-1)

        inputs = exp.(-20:0.1:10)
        setprecision(BigFloat, 256) do
            for x in inputs
                @test expint(x) ≅ Float64(expint(big(x)))
            end
        end
    end

    @testset "E1 complex" begin
        #angle = -π/2:0.4:π/2
        #radius = exp.(-20:0.5:10)
        #inputs = [r*exp(im*t) for t in angle for r in radius]
        inputs = Complex{Float64}[1.262092593135176e-25 - 2.061153622438558e-9im, 2.080838903895047e-25 - 3.398267819495071e-9im, 3.430723361752104e-25 - 5.602796437537268e-9im, 5.6563065804085445e-25 - 9.237449661970594e-9im, 9.325672972720673e-25 - 1.522997974471263e-8im, 1.5375435393717868e-24 - 2.510999155743982e-8im, 2.534980737989825e-24 - 4.139937718785167e-8im, 4.179476663538933e-24 - 6.82560337633487e-8im, 6.89079207557144e-24 - 1.1253517471925912e-7im, 1.1360995466966519e-23 - 1.855391362615978e-7im, 1.8731114882715435e-23 - 3.059023205018258e-7im, 3.088238753106067e-23 - 5.04347662567888e-7im, 5.091644921246415e-23 - 8.315287191035679e-7im, 8.394703284511243e-23 - 1.3709590863840842e-6im, 1.3840525866389915e-22 - 2.2603294069810542e-6im, 2.2819169393592376e-22 - 3.726653172078671e-6im, 3.7622449958925094e-22 - 6.14421235332821e-6im, 6.202893350313096e-22 - 1.013009359863071e-5im, 1.0226842206545583e-21 - 1.670170079024566e-5im, 1.6861212278025537e-21 - 2.7536449349747158e-5im, 2.779943933257086e-21 - 4.539992976248485e-5im, 4.583352694114736e-21 - 7.48518298877006e-5im, 7.556671077907703e-21 - 0.00012340980408667956im, 1.2458844341830895e-20 - 0.00020346836901064417im, 2.0541161674718535e-20 - 0.00033546262790251185im, 3.386665017799872e-20 - 0.0005530843701478336im, 5.583666651582677e-20 - 0.0009118819655545162im, 9.205909976963319e-20 - 0.0015034391929775724im, 1.5177979595169953e-19 - 0.0024787521766663585im, 2.502425780480922e-19 - 0.004086771438464067im, 4.1258026126272655e-19 - 0.006737946999085467im, 6.802298526148734e-19 - 0.011108996538242306im, 1.121509426971355e-18 - 0.01831563888873418im, 1.849056447538385e-18 - 0.0301973834223185im, 3.048578695781751e-18 - 0.049787068367863944im, 5.026256541138628e-18 - 0.0820849986238988im, 8.286896071370909e-18 - 0.1353352832366127im, 1.3662781820950545e-17 - 0.22313016014842982im, 2.2526119005136194e-17 - 0.36787944117144233im, 3.713929155009045e-17 - 0.6065306597126334im, 6.123233995736766e-17 - 1.0im, 1.0095506134245344e-16 - 1.6487212707001282im, 1.664467570201392e-16 - 2.718281828459045im, 2.744243087381594e-16 - 4.4816890703380645im, 4.524491950137825e-16 - 7.38905609893065im, 7.459626117303735e-16 - 12.182493960703473im, 1.2298844251068877e-15 - 20.085536923187668im, 2.0277366121765243e-15 - 33.11545195869231im, 3.3431724838728525e-15 - 54.598150033144236im, 5.5119595857805524e-15 - 90.01713130052181im, 9.087685012315865e-15 - 148.4131591025766im, 1.4983059581227923e-14 - 244.69193226422038im, 2.470288903173783e-14 - 403.4287934927351im, 4.0728178594371055e-14 - 665.1416330443618im, 6.71494143654132e-14 - 1096.6331584284585im, 1.107106677793135e-13 - 1808.0424144560632im, 1.8253103286116948e-13 - 2980.9579870417283im, 3.009427964410742e-13 - 4914.768840299134im, 4.961707897563779e-13 - 8103.083927575384im, 8.180473349714214e-13 - 13359.726829661871im, 1.3487320416069354e-12 - 22026.465794806718im, 8.026510268934933e-10 - 1.8984482042759055e-9im, 1.3233478209886028e-9 - 3.130011935712147e-9im, 2.181831700998575e-9 - 5.1605172559538995e-9im, 3.5972323345241926e-9 - 8.508254567706251e-9im, 5.930833465580316e-9 - 1.402774028230882e-8im, 9.778291287682422e-9 - 2.312783378329957e-8im, 1.6121676837103755e-8 - 3.813135150374302e-8im, 2.6580151520686527e-8 - 6.286797030476444e-8im, 4.382326119058823e-8 - 1.0365175988720914e-7im, 7.225234287637024e-8 - 1.70892861271544e-7im, 1.191239745581905e-7 - 2.817546953892008e-7im, 1.9640223070442955e-7 - 4.645349594078106e-7im, 3.2381253537534684e-7 - 7.658886685594781e-7im, 5.33876614792672e-7 - 1.2627369388422118e-6im, 8.802137307380571e-7 - 2.0819012503679215e-6im, 1.45122710063015e-6 - 3.4324748749787854e-6im, 2.3926689894254037e-6 - 5.659194337521287e-6im, 3.944844256610243e-6 - 9.330434079297066e-6im, 6.503948635472542e-6 - 1.5383285131402438e-5im, 1.0723198458844655e-5 - 2.5362749409388216e-5im, 1.7679565389036012e-5 - 4.1816104434695464e-5im, 2.9148675513637465e-5 - 6.894310083930038e-5im, 4.805804133207007e-5 - 0.00011366795682177839im, 7.923431497236984e-5 - 0.00018740677820908974im, 0.0001306353004642998 - 0.00030898154150670753im, 0.00021538119857979343 - 0.0005094244397358232im, 0.0003551035634073936 - 0.0008398989096069473im, 0.0005854667982911815 - 0.0013847591975068181im, 0.0009652715636313724 - 0.0022830819437271313im, 0.0015914637589610158 - 0.0037641657633743137im, 0.0026238801509474085 - 0.006206060160516417im, 0.004326047016634855 - 0.010232063393888068im, 0.007132445734374717 - 0.0168698205606554im, 0.011759414994377992 - 0.027813631991246923im, 0.019387997632221026 - 0.045856926679394366im, 0.031965404092526524 - 0.07560529042525369im, 0.05270204165387341 - 0.12465205050157649im, 0.08689097708406524 - 0.2055164870983357im, 0.14325900215041576 - 0.33883940375859456im, 0.23619416406466587 - 0.5586517323281438im, 0.38941834230865047 - 0.9210609940028851im, 0.6420423041650557 - 1.51857285242476im, 1.0585488035662487 - 2.503703362900468im, 1.745251928513846 - 4.127908989937444im, 2.8774339772711977 - 6.805771355224145im, 4.744086603362292 - 11.220819996879685im, 7.821676493006933 - 18.500004603552885im, 12.895764406555712 - 30.50135109792793im, 21.261521079026018 - 50.28822634024648im, 35.054322050429334 - 82.91126843294684im, 57.794806394515376 - 136.69757184612754im, 95.28752663863328 - 225.37619435576946im, 157.10257200151977 - 371.5825255438033im, 259.01835214060407 - 612.6360136845421im, 427.04906667590996 - 1010.0660269586393im, 704.0848798612101 - 1665.317343458278im, 1160.8397178055213 - 2745.6441266254938im, 1913.9011346194973 - 4526.801873340328im, 3155.489510664275 - 7463.434536821385im, 5202.522675703328 - 12305.123273335375im, 8577.509796731823 - 20287.718479335224im, 1.4785811053359149e-9 - 1.4360195577481574e-9im, 2.4377681188226294e-9 - 2.367595990000778e-9im, 4.0192001505375065e-9 - 3.903505869138611e-9im, 6.626540779392344e-9 - 6.435793156751618e-9im, 1.0925318734145964e-8 - 1.0610829071362718e-8im, 1.8012805386165048e-8 - 1.7494299589719e-8im, 2.969809538515215e-8 - 2.884322384957024e-8im, 4.896388156078167e-8 - 4.755443667635169e-8im, 8.072779302530253e-8 - 7.840401126446333e-8im, 1.330976294974937e-7 - 1.2926636107993315e-7im, 2.1944089283228268e-7 - 2.1312419909848897e-7im, 3.6179686767401174e-7 - 3.513824003546078e-7im, 5.965021914068228e-7 - 5.793316376143102e-7im, 9.834658509916678e-7 - 9.551563937242516e-7im, 1.6214610675371656e-6 - 1.5747866631784001e-6im, 2.673337351660662e-6 - 2.5963842683971063e-6im, 4.407588155440082e-6 - 4.2807139702175e-6im, 7.266884344360006e-6 - 7.057704176480787e-6im, 1.1981066790264096e-5 - 1.1636186998073005e-5im, 1.9753439662787326e-5 - 1.9184829013567234e-5im, 3.256791614152903e-5 - 3.163043566941325e-5im, 5.369541608491697e-5 - 5.214977208967369e-5im, 8.85287746382964e-5 - 8.598043850640888e-5im, 0.0001459592738151773 - 0.00014175777782964065im, 0.00024064615939502708 - 0.0002337190635949116im, 0.0003967584417068747 - 0.00038533759151704674im, 0.000654144082171961 - 0.0006353142835345123im, 0.0010785012623795245 - 0.0010474561728429624im, 0.0017781479717620622 - 0.001726963272292342im, 0.0029316703834964025 - 0.0028472810807462813im, 0.004833507319952121 - 0.0046943728814884435im, 0.007969106330489832 - 0.007739712422307849im, 0.01313883511554963 - 0.012760628499760963im, 0.02166227692722845 - 0.021038719635058167im, 0.03571505674171816 - 0.034686984570616836im, 0.05888417373433275 - 0.05718916927802313im, 0.0970835897433962 - 0.09428899984234702im, 0.1600637794458621 - 0.15545627963311856im, 0.2639005578410468 - 0.2563040748950297im, 0.4350984630621634 - 0.42257398004655417im, 0.7173560908995228 - 0.6967067093471654im, 1.1827202457323378 - 1.1486751711401635im, 1.9499760264265877 - 1.893845187783897im, 3.2149669521248305 - 3.1224228445123896im, 5.300584398566169 - 5.148004959867577im, 8.739186245057288 - 8.487625279003433im, 14.408482250835934 - 13.99372833522507im, 23.755571165458463 - 23.071757562684663im, 39.166315478122 - 38.03889744603475im, 64.57433742373141 - 62.71553933325827im, 106.46508365187327 - 103.40044370217345im, 175.53124800371194 - 170.47851093160446im, 289.4021022562593 - 281.0715471702206im, 477.14340177522826 - 463.4086384081371im, 786.6764756810361 - 764.03167916968im, 1297.0102386147364 - 1259.6752809357874im, 2138.4083687199645 - 2076.8534298539926im, 3525.639362951768 - 3424.152425926794im, 5812.7966105162295 - 5645.472938744951im, 9583.681414011715 - 9307.81131727076im, 15800.819398894597 - 15345.986502447684im, 1.921075738295671e-9 - 7.468749982582968e-10im, 3.1673184323540253e-9 - 1.2313886961825751e-9im, 5.2220252705026676e-9 - 2.0302167358959095e-9im, 8.609664139611336e-9 - 3.34726151660297e-9im, 1.4194936400561332e-8 - 5.518701261019287e-9im, 2.340349357984098e-8 - 9.098800155682118e-9im, 3.8585837673777715e-8 - 1.5001385354522745e-8im, 6.361729132053966e-8 - 2.4733103123971034e-8im, 1.0488718138450039e-7 - 4.077799321091083e-8im, 1.729297269724083e-7 - 6.72315447832941e-8im, 2.8511291919577525e-7 - 1.1084607794624523e-7im, 4.7007173442948146e-7 - 1.8275428648365886e-7im, 7.750172673087879e-7 - 3.0131087943723333e-7im, 1.2777874537718856e-6 - 4.967776560215284e-7im, 2.106715354467465e-6 - 8.190478882912455e-7im, 3.473386416221069e-6 - 1.350381675147799e-6im, 5.726646065784566e-6 - 2.2264029913798468e-6im, 9.441643178430218e-6 - 3.6707179690383474e-6im, 1.5566637938638668e-5 - 6.051990794294698e-6im, 2.5665047082721167e-5 - 9.978045952635031e-6im, 4.2314509038802655e-5 - 1.64510166021327e-5im, 6.976483111150678e-5 - 2.7123140996577133e-5im, 0.0001150227610003433 - 4.471849948925539e-5im, 0.00018964047267592312 - 7.372834130172817e-5im, 0.00031266428108642094 - 0.00012155748455759801im, 0.000515496250815346 - 0.0002004144104029142im, 0.0008499096336854291 - 0.0003304275013861097im, 0.001401264091230121 - 0.0005447828499595751im, 0.0023102939130793857 - 0.0008981950726409879im, 0.0038090307160630154 - 0.0014808733215012433im, 0.0062800299623232345 - 0.0024415473443714494im, 0.010354018979516441 - 0.004025431040086619im, 0.01707089132876159 - 0.00663681377952735im, 0.02814514164353961 - 0.010942256047982454im, 0.046403493694571724 - 0.018040730295755793im, 0.07650642708903968 - 0.02974413577757679im, 0.1261377736869682 - 0.04903978933508355im, 0.20796603051646337 - 0.08085294378740554im, 0.3428780180955652 - 0.13330396822101728im, 0.5653102816896617 - 0.21978108787472514im, 0.9320390859672264 - 0.36235775447667334im, 1.5366726661580716 - 0.5974269374088259im, 2.5335449107982893 - 0.9849904993951654im, 4.177109384707198 - 1.6239747877903508im, 6.886889092607883 - 2.6774817757106786im, 11.35456053593532 - 4.414421155526145im, 18.72050547504881 - 7.278150056944594im, 30.86489557497118 - 11.9996408102319im, 50.88760985239325 - 19.784063044590653im, 83.89948477873016 - 32.61840556248895im, 138.32686515547405 - 53.77865906719891im, 228.06244489109844 - 88.66601911382115im, 376.0114039398298 - 146.18555170126106im, 619.9379997014153 - 241.01922855890243im, 1022.1049666230131 - 397.3735287727982im, 1685.1661993596063 - 655.1581893008818im, 2778.3693575490756 - 1080.1732423737449im, 4580.756657652611 - 1780.9046007427182im, 7552.390937373084 - 2936.215296332239im, 12451.787583089881 - 4841.000614418042im, 20529.52704648003 - 7981.460684463416im, 2.0602747528029592e-9 + 6.018470116477703e-11im, 3.3968188084326873e-9 + 9.922779698109866e-11im, 5.600407422177236e-9 + 1.6359897952745132e-10im, 9.23351084153048e-9 + 2.6972911741174376e-10im, 1.5223485727671545e-8 + 4.447081332039143e-10im, 2.5099284733411895e-8 + 7.331997584666395e-10im, 4.138172461933518e-8 + 1.208842037456145e-9im, 6.822692959815308e-8 + 1.9930435800704268e-9im, 1.1248719006303513e-7 + 3.285973343894447e-9im, 1.854600229382141e-7 + 5.417654147032401e-9im, 3.057718846827673e-7 + 8.93220162950908e-9im, 5.04132610258545e-7 + 1.4726710820753964e-8im, 8.311741577868409e-7 + 2.4280241377626806e-8im, 1.3703745135994291e-6 + 4.0031350417026694e-8im, 2.259365609396721e-6 + 6.600053892740236e-8im, 3.7250641385007314e-6 + 1.088164924072801e-7im, 6.141592479868404e-6 + 1.794080656348617e-7im, 1.0125774157530985e-5 + 2.9579389394736116e-7im, 1.6694579235827007e-5 + 4.876816846942322e-7im, 2.7524707891496678e-5 + 8.040511668862538e-7im, 4.538057137051824e-5 + 1.325656261576625e-6im, 7.48199132950987e-5 + 2.185637676098195e-6im, 0.00012335718252156854 + 3.6035073266266913e-6im, 0.0002033816107169481 + 5.941179178533179e-6im, 0.0003353195876582855 + 9.795348484688367e-6im, 0.0005528485366546116 + 1.614979940062598e-5im, 0.0009114931418578975 + 2.6626517789352233e-5im, 0.0015027981310784047 + 4.389970624398038e-5im, 0.0024776952442773655 + 7.237837946193768e-5im, 0.004085028851552642 + 0.00011933177375770194im, 0.006735073958978557 + 0.00019674483366469855im, 0.01110425969590647 + 0.00032437739216334714im, 0.018307829156019133 + 0.0005348079061939475im, 0.03018450734987272 + 0.0008817491706805601im, 0.04976583931333951 + 0.0014537586131232374im, 0.0820499978301475 + 0.0023968427479198im, 0.13527757668346357 + 0.00395172562101872im, 0.22303501812679405 + 0.006515294087344236im, 0.36772257849663403 + 0.01074190394667122im, 0.606272036884098 + 0.017710405524694495im, 0.9995736030415051 + 0.029199522301288878im, 1.648018260964896 + 0.04814187351241773im, 2.7171227613550584 + 0.0793725308712782im, 4.479778091749552 + 0.13086317995677893im, 7.385905427883919 + 0.21575690834620012im, 12.177299382331746 + 0.35572300409087815im, 20.076972511333885 + 0.5864880833219795im, 33.10133163069794 + 0.9669553779850966im, 54.574869548030655 + 1.5942398995019111im, 89.97874826952284 + 2.628457232907636im, 148.34987618293465 + 4.33359334902042im, 244.5875963685347 + 7.144887533094571im, 403.25677268222057 + 11.779928052573185im, 664.8580186750634 + 19.421817947594548im, 1096.1655573851199 + 32.021164365864635im, 1807.2714706697093 + 52.79397480258601im, 2979.6869156226526 + 87.04254922183014im, 4912.673197813925 + 143.50890235799426im, 8099.628797234237 + 236.6061798524529im, 13354.03028277538 + 390.0976415018392im, 22017.07377678542 + 643.1622791940354im, 1.8742016851758135e-9 + 8.577425596154891e-10im, 3.090036183931388e-9 + 1.4141784028228296e-9im, 5.0946083836807336e-9 + 2.3315860132987334e-9im, 8.399589208061625e-9 + 3.844135454592533e-9im, 1.3848581392474446e-8 + 6.337907891439217e-9im, 2.283245071079462e-8 + 1.0449443552354035e-8im, 3.764434714909935e-8 + 1.7228219851746405e-8im, 6.206503586633982e-8 + 2.840453252587251e-8im, 1.0232794479960082e-7 + 4.683115695969964e-8im, 1.687102591781304e-7 + 7.721152461095314e-8im, 2.7815619289231517e-7 + 1.2730028296926488e-7im, 4.5860203179852776e-7 + 2.0988268429757225e-7im, 7.561069246125293e-7 + 3.460380459530472e-7im, 1.2466095695323354e-6 + 5.705202868342973e-7im, 2.055311713546292e-6 + 9.406289322696442e-7im, 3.3886361400429e-6 + 1.5508349284689127e-6im, 5.586916482751908e-6 + 2.556894533911408e-6im, 9.211268042738215e-6 + 4.215606404996628e-6im, 1.5186813552182832e-5 + 6.9503599488176404e-6im, 2.5038822537640806e-5 + 1.1459206286637898e-5im, 4.128203931109416e-5 + 1.889303715012053e-5im, 6.80625763100798e-5 + 3.1149352217531456e-5im, 0.00011221621730107923 + 5.1356599569574315e-5im, 0.000185013264381797 + 8.46727181011862e-5im, 0.00030503530434793514 + 0.00013960171138142148im, 0.000502918194592928 + 0.00023016431098068975im, 0.0008291719248474666 + 0.00037947679526990224im, 0.001367073389563386 + 0.0006256514640986055im, 0.0022539229759812773 + 0.0010315248769040487im, 0.0037160907530200656 + 0.0017006970058080363im, 0.006126797868356239 + 0.002803975328491729im, 0.010101381966839134 + 0.004622973766602693im, 0.016654363312194376 + 0.007621995182886548im, 0.02745840304278271 + 0.012566545583198967im, 0.045271253156092976 + 0.020718731002242884im, 0.0746396780297008 + 0.03415941250531203im, 0.12306002480577674 + 0.0563193499921279im, 0.2028916804701695 + 0.09285491028402634im, 0.33451182923926226 + 0.15309186567422634im, 0.5515167681675808 + 0.2524058153082637im, 0.9092974268256817 + 0.41614683654714246im, 1.4991780090003948 + 0.6861101411498433im, 2.4717266720048188 + 1.1312043837568138im, 4.075188339491183 + 1.8650407290090893im, 6.71884969742825 + 3.0749323206393595im, 11.077510410787076 + 5.0697063230014185im, 18.263727040666765 + 8.358532650935373im, 30.1117952542084 + 13.780890573438677im, 49.645957334580565 + 22.720847417619236im, 81.85234586179402 + 37.46034442576091im, 134.9517036790434 + 61.76166666250499im, 222.4977443728596 + 101.82777354036297im, 366.8367638303334 + 167.88561618403213im, 604.8115754018701 + 276.79658644721144im, 997.1657091307175 + 456.3604197327043im, 1644.048315056591 + 752.4111311189481im, 2710.5774270925076 + 1240.516236187353im, 4468.986659927043 + 2045.2655052509529im, 7368.113364696836 + 3372.0727427364905im, 12147.965229305562 + 5559.608057297773im, 20028.608669281643 + 9166.244060822655im, 1.3922333814168738e-9 + 1.5198817279512663e-9im, 2.2954047897207643e-9 + 2.505861333821718e-9im, 3.784482701679579e-9 + 4.131466882496861e-9im, 6.239557128855809e-9 + 6.811637328365722e-9im, 1.0287290558093194e-8 + 1.123049135157156e-8im, 1.696087476100084e-8 + 1.8515949971749862e-8im, 2.7963754988143037e-8 + 3.052764056564343e-8im, 4.6104437657598235e-8 + 5.033157034486442e-8im, 7.60133670397502e-8 + 8.298273061531775e-8im, 1.253248550959722e-7 + 1.3681539306625308e-7im, 2.066257543441407e-7 + 2.255704487075303e-7im, 3.406682762616441e-7 + 3.719027968254774e-7im, 5.616670333253203e-7 + 6.131640517590328e-7im, 9.260323848944932e-7 + 1.0109366145637917e-6im, 1.5267692903327191e-6 + 1.6667526997609002e-6im, 2.5172170044232934e-6 + 2.748010629092661e-6im, 4.1501892181607424e-6 + 4.5307035762951104e-6im, 6.842505241411951e-6 + 7.469867357474889e-6im, 1.1281383936392999e-5 + 1.2315729201577408e-5im, 1.859985765886588e-5 + 2.030520469882338e-5im, 3.066598095416686e-5 + 3.347762289287029e-5im, 5.055965508601992e-5 + 5.5195268955952815e-5im, 8.335877877958296e-5 + 9.100161396969386e-5im, 0.0001374353916734849 + 0.00015003629661987616im, 0.00022659265359907782 + 0.0002473680336142636im, 0.0003735881277731856 + 0.00040784093871110074im, 0.0006159426927406883 + 0.000672416030715299im, 0.0010155178190538862 + 0.0011086266126000641im, 0.001674305829049146 + 0.0018278162774579565im, 0.0027604636340105393 + 0.00301355957557686im, 0.00455123511038735 + 0.00496851977277562im, 0.00750371813445287 + 0.00819170423326933im, 0.01237153969761073 + 0.01350583701267543im, 0.020397220650761844 + 0.02226736076140706im, 0.03362933155007496 + 0.03671267132968522im, 0.055445394246035506 + 0.06052896212547478im, 0.09141400085579324 + 0.09979538734967272im, 0.150716207650746 + 0.16453477784116388im, 0.24848901739304236 + 0.27127198799664703im, 0.40968912851128303 + 0.44725189675518173im, 0.6754631805511506 + 0.7373937155412458im, 1.1136505133494432 + 1.2157567036934516im, 1.8360992894853438 + 2.0044439373756666im, 3.0272159536918783 + 3.304769355477177im, 4.991035333854574 + 5.448643531133175im, 8.228826117741953 + 8.983294486241922im, 13.567040653213914 + 14.810948700430236im, 22.36826850540714 + 24.41912616164775im, 36.879040073596514 + 40.26033271461862im, 60.803257812340995 + 66.37806691205596im, 100.24762448307034 + 109.43893082586303im, 165.280390822397 + 180.4342930952803im, 272.50129597851617 + 297.4858569899299im, 449.2786829731309 + 490.4712601517538im, 740.7353210899404 + 808.6503992792926im, 1221.266079839874 + 1333.2391138519215im, 2013.5273630165614 + 2198.139685937053im, 3319.7453925421432 + 3624.1196561745182im, 5473.334841992979 + 5975.163164697364im, 9024.003575657947 + 9851.378605540434im, 14878.066642061274 + 16242.177452674685im, 6.904620391678344e-10 + 1.942064990611742e-9im, 1.1383794505869934e-9 + 3.2019238591036235e-9im, 1.8768704143107015e-9 + 5.279079973666385e-9im, 3.0944361744218156e-9 + 8.70373144231084e-9im, 5.10186274159318e-9 + 1.4350027163399389e-8im, 8.411549622257147e-9 + 2.3659195019421195e-8im, 1.3868300781764986e-8 + 3.900741807616225e-8im, 2.2864962487363148e-8 + 6.431235989726138e-8im, 3.769795000667613e-8 + 1.0603315573153675e-7im, 6.215341203779698e-8 + 1.7481911925404382e-7im, 1.0247365247330527e-7 + 2.882280004392044e-7im, 1.6895049051907118e-7 + 4.752076351354821e-7im, 2.78552267414013e-7 + 7.83484936046975e-7im, 4.592550482872334e-7 + 1.291748279333777e-6im, 7.571835667875762e-7 + 2.129732864527889e-6im, 1.2483846523872681e-6 + 3.5113358746562456e-6im, 2.0582383304064746e-6 + 5.789214145118191e-6im, 3.3934613155114725e-6 + 9.54480050169442e-6im, 5.594871852181803e-6 + 1.5736715611732845e-5im, 9.224384229533563e-6 + 2.594545776002272e-5im, 1.5208438488342797e-5 + 4.2776828087001156e-5im, 2.5074476029865277e-5 + 7.05270663601215e-5im, 4.1340821982099384e-5 + 0.00011627947446801177im, 6.815949255011468e-5 + 0.00019171244290124345im, 0.000112376005167501 + 0.00031608038246916387im, 0.00018527671003596642 + 0.0005211284498279424im, 0.0003054696528016377 + 0.0008591955599983131im, 0.000503634314127443 + 0.001416573995460327im, 0.0008303526063563855 + 0.0023355356778361077im, 0.001369020004281063 + 0.0038506473505274326im, 0.002257132401072169 + 0.00634864419277967im, 0.003721382200434138 + 0.010467144720742687im, 0.006135521990260611 + 0.01725740414458502im, 0.010115765612191055 + 0.028452649290245876im, 0.016678077934236295 + 0.04691048809259928im, 0.027497501844569835 + 0.07734231953719352im, 0.0453357161822583 + 0.12751592734625705im, 0.07474595959211325 + 0.21023822176882612im, 0.1232352534784094 + 0.3466242281444344im, 0.20318058370997552 + 0.571486737881743im, 0.3349881501559046 + 0.9422223406686583im, 0.5523020885945283 + 1.5534620147892793im, 0.9105922013179055 + 2.561225867007762im, 1.501312731246484 + 4.222747566003075im, 2.4752462339789836 + 6.962133732866462im, 4.080991116181536 + 11.47861797483582im, 6.7284168587867565 + 18.925041613352647im, 11.093283993219066 + 31.202118656819575im, 18.28973328153753 + 51.44359672040775im, 30.15467229670298 + 84.8161521542556im, 49.71664962656609 + 139.83819415565972im, 81.9688977472651 + 230.5542051607305im, 135.1438652517598 + 380.11962209785764im, 222.81456524520829 + 626.7113063632323im, 367.35911314157636 + 1033.2722613893256im, 605.672783822052 + 1703.5779557770043im, 998.5856017715777 + 2808.7252119853893im, 1646.3893222556876 + 4630.805000552037im, 2714.43709545652 + 7634.906704874663im, 4475.350177256639 + 12587.833084137881im, 7378.6050310746095 + 20753.828157840926im]
        sympy_outputs = Complex{Float64}[19.422784335098466 + 1.5707963247337429im, 18.922784335098466 + 1.5707963233966287im, 18.422784335098466 + 1.5707963211921im, 17.922784335098466 + 1.5707963175574469im, 17.422784335098466 + 1.5707963115649168im, 16.922784335098466 + 1.570796301684905im, 16.422784335098466 + 1.5707962853955193im, 15.922784335098468 + 1.5707962585388628im, 15.42278433509847 + 1.5707962142597218im, 14.922784335098475 + 1.5707961412557603im, 14.422784335098491 + 1.570796020892576im, 13.92278433509853 + 1.570795822447234im, 13.42278433509864 + 1.5707954952661773im, 12.922784335098937 + 1.57079495583581im, 12.422784335099745 + 1.5707940664654896im, 11.92278433510194 + 1.5707926001417245im, 11.422784335107904 + 1.5707901825825432im, 10.922784335124122 + 1.570786196701298im, 10.422784335168204 + 1.5707796250941066im, 9.922784335288032 + 1.570768790345548im, 9.422784335613756 + 1.5707509268651392im, 8.922784336499166 + 1.5707214749650322im, 8.422784338905963 + 1.5706729169909144im, 7.922784345448312 + 1.570592858426354im, 7.42278436323226 + 1.5704608641690914im, 6.922784411574046 + 1.5702432424341481im, 6.42278454298064 + 1.5698844448714673im, 5.9227849001807655 + 1.5692928877907115im, 5.422785871151162 + 1.5683175754643401im, 4.922788510520759 + 1.5667095591484324im, 4.422795685059437 + 1.5640583967903614im, 3.9228151873908437 + 1.559687406420766im, 3.42286819958321 + 1.552481029247858im, 2.923012296928274 + 1.5406004731334704im, 2.423403959143947 + 1.5210161140174439im, 1.9244683490031078 + 1.4887420488700736im, 1.4273597518401993 + 1.4355986763695583im, 0.935205310403304 + 1.3482824120936219im, 0.4564279407490047 + 1.2056716297636936im, 0.013355921045092107 + 0.9765258313021719im, -0.3374039229009681 + 0.6247132564277136im, -0.4701303568325964 + 0.15166525880822357im, -0.2139580013403798 - 0.25024394235267033im, 0.1925951566377415 - 0.08732176854725335im, -0.10970134875285979 + 0.07377808268843321im, 0.03637865461071485 + 0.07272301498238866im, -0.04599286822415459 + 0.018588783008943673im, -0.030009923051911254 - 0.0029700840357942456im, 0.016875515986581567 - 0.007095313732401066im, -0.009899389140895167 - 0.0050364951777518915im, 0.00460025625971065 - 0.004922539221206865im, 0.0014257779601906833 + 0.0038298852912730026im, -0.0023900206416381307 + 0.0006571913276295817im, 0.0011563691377997423 + 0.000960793680932573im, 0.0001958037831750238 - 0.0008906107585813064im, 0.0005522389705165051 + 3.056401833994714e-5im, -0.00013487948519259028 - 0.00030715250479356376im, -0.0001969875635336234 + 5.0943810803580305e-5im, 9.791153713985444e-5 - 7.512196711849147e-5im, -7.444743088567554e-5 - 7.77022379836572e-6im, 3.126563309460034e-5 - 3.291828970165017e-5im, 19.422784335901117 + 1.1707963248964484im, 18.922784336421813 + 1.1707963236648846im, 18.422784337280298 + 1.1707963216343793im, 17.9227843386957 + 1.170796318286642im, 17.4227843410293 + 1.1707963127671563im, 16.92278434487676 + 1.170796303667063im, 16.422784351220145 + 1.1707962886635455im, 15.92278436167862 + 1.1707962639269271im, 15.422784378921731 + 1.170796223143139im, 14.922784407350816 + 1.1707961559020414im, 14.422784454222459 + 1.170796045040218im, 13.922784531500742 + 1.1707958622599828im, 13.422784658911123 + 1.170795560906352im, 12.922784868975409 + 1.170795064058295im, 12.422785215313088 + 1.1707942448945625im, 11.922785786327987 + 1.1707928943225123im, 11.422786727774032 + 1.1707906676073294im, 10.922788279960598 + 1.170786996379221im, 10.422790839095688 + 1.1707809435597913im, 9.922795058428996 + 1.1707709641814725im, 9.422802015022857 + 1.1707545110601092im, 8.922813484749835 + 1.170727384698866im, 8.422832395792408 + 1.1706826615694423im, 7.92286357662381 + 1.170608927441381im, 7.422914989997979 + 1.170487365436098im, 6.922999769569336 + 1.170286957218789im, 6.423139583455522 + 1.169956577026095im, 5.923370195417464 + 1.169411973031014im, 5.423750676051995 + 1.16851434705441im, 4.92437870436997 + 1.1670351576693776im, 4.425416107019373 + 1.1645984147500794im, 3.9271318062414253 + 1.1605864230349665im, 3.429974892489083 + 1.153986790292839im, 2.934701153118623 + 1.1431467765195484im, 2.442597685725193 + 1.125386357709132im, 1.9558947131349638 + 1.0964100772511418im, 1.4785483539973987 + 1.049475424312998im, 1.0177734058791648 + 0.9744068317835769im, 0.5870529668396864 + 0.8570444133156812im, 0.21165768283269532 + 0.6812663393762747im, -0.06375818164860747 + 0.43961677930494425im, -0.17665198319401504 + 0.16373879982184414im, -0.10158832111504654 - 0.032891822965524856im, 0.014604560918506343 - 0.03176620188634744im, -0.00012696100584579475 + 0.007145485410252403im, 0.0006682437303066253 - 0.00016555660456651515im, 1.391474407497085e-5 + 1.3714620416232118e-5im, 7.284599637541871e-8 + 1.7011702048131712e-8im, 4.0716373070923785e-12 + 9.80045749388423e-12im, -4.827823301236994e-18 + 4.506918869538679e-18im, 4.981146835881918e-28 - 1.919780844389047e-28im, 1.5881747583113524e-44 + 5.771630300821964e-45im, -6.652649019587505e-72 + 1.3021866797571364e-71im, -1.7841249535445187e-116 - 4.520112327567821e-116im, 2.9273273379414775e-189 - 1.0913398530092691e-189im, 1.1643391475637e-310 + 9.0989791031947e-310im, 0.0 + 0.0im, -0.0 - 0.0im, 0.0 + 0.0im, -0.0 - 0.0im, 0.0 + 0.0im, 19.42278433657705 + 0.770796325358877im, 18.922784337536235 + 0.7707963244273006im, 18.42278433911767 + 0.7707963228913908im, 17.92278434172501 + 0.7707963203591034im, 17.422784346023786 + 0.7707963161840675im, 16.922784353111272 + 0.7707963093005971im, 16.422784364796563 + 0.7707962979516731im, 15.922784384062348 + 0.770796279240461im, 15.42278441582626 + 0.7707962483908884im, 14.922784468196097 + 0.7707961975285441im, 14.422784554539358 + 0.7707961136707209im, 13.922784696895333 + 0.7707959754125597im, 13.422784931600653 + 0.7707957474634318im, 12.922785318564305 + 0.7707953716389725im, 12.422785956559498 + 0.7707947520095101im, 11.922787008435717 + 0.7707937304140987im, 11.422788742686347 + 0.7707920460903601im, 10.922791601982063 + 0.7707892691163638im, 10.422796316163222 + 0.7707846906776054im, 9.922804088532594 + 0.7707771421553653im, 9.422816902999559 + 0.770764696874292im, 8.922838030473637 + 0.7707441784228916im, 8.422872863761858 + 0.7707103501621846im, 7.922930294069756 + 0.770654579362153im, 7.423024980434953 + 0.7705626358515526im, 6.923181091300775 + 0.7704110656394196im, 6.423438473082132 + 0.770161220273839im, 5.923862819733243 + 0.7697494353242446im, 5.424562437647089 + 0.7690708982968311im, 4.925715881003084 + 0.7679532165629317im, 4.427617499546561 + 0.7661132865258276im, 3.9307524892640435 + 0.7630873975145185im, 3.4359204919784716 + 0.7581192765569613im, 2.944438930686154 + 0.7499843529111785im, 2.458476729709604 + 0.7367237144559594im, 1.9815990351439752 + 0.7152682997726029im, 1.519644637192796 + 0.6809829909359293im, 1.0820928833147325 + 0.6273284942292389im, 0.6840106945996433 + 0.5462899789782621im, 0.34812949624953893 + 0.43117306152769475im, 0.10448768579612328 + 0.2845050380623101im, -0.02019310582833093 + 0.1316317496287615im, -0.03390206007899803 + 0.024468333558091217im, -0.006185943956706402 - 0.00465351808740371im, 0.0005573538396185417 - 0.00026091718771225215im, -1.2127363973340944e-5 + 2.648963041676963e-6im, -1.4909980987569136e-8 + 2.1981496267986093e-8im, 3.674106345109149e-13 - 1.3764124162649113e-12im, 8.045024699411008e-20 + 1.5740478477669898e-19im, 7.941649183373506e-31 + 5.998814897443223e-31im, -3.4191629803200705e-49 - 1.841589772545219e-49im, -7.084856305388888e-81 + 2.385963269000293e-79im, 3.1604864227015654e-129 - 4.004980686927174e-129im, 6.4392500278435185e-211 - 6.3360731369744935e-211im, -0.0 - 0.0im, -0.0 - 0.0im, -0.0 - 0.0im, 0.0 + 0.0im, -0.0 - 0.0im, -0.0 - 0.0im, -0.0 - 0.0im, 19.422784337019543 + 0.37079632604802143im, 18.922784338265785 + 0.3707963255635077im, 18.422784340320494 + 0.37079632476467966im, 17.92278434370813 + 0.3707963234476349im, 17.422784349293405 + 0.3707963212761951im, 16.92278435850196 + 0.37079631769609633im, 16.422784373684305 + 0.37079631179351125im, 15.922784398715757 + 0.37079630206179404im, 15.422784439985646 + 0.3707962860169053im, 14.922784508028188 + 0.37079625956335743im, 14.422784620211369 + 0.3707962159488343im, 13.922784805170155 + 0.37079614404065286im, 13.422785110115607 + 0.3707960254841337im, 12.922785612885574 + 0.3707958300175578im, 12.42278644181288 + 0.3707955077478708im, 11.922787808482322 + 0.37079497641556647im, 11.422790061737574 + 0.37079410039827987im, 10.922793776722727 + 0.3707926560942561im, 10.422799901684982 + 0.3707902748512064im, 9.922810000005768 + 0.3707863488769862im, 9.422826649227538 + 0.37077987612634794im, 8.922854098896723 + 0.3707692045999996im, 8.422899355051891 + 0.3707516108671361im, 7.92297396793944 + 0.37072260544411373im, 7.423096978634799 + 0.3706747883117999im, 6.92329977496083 + 0.370595964032504im, 6.42363409145978 + 0.3704660396724995im, 5.924185182585062 + 0.370251925468011im, 5.425093496710039 + 0.36989916851119525im, 4.926590288560013 + 0.36931827042176235im, 4.429056003173629 + 0.36836243072710256im, 3.933115637364833 + 0.3667916673062103im, 3.4397935353674995 + 0.36421585623910385im, 2.950762048933833 + 0.36000669314510736im, 2.4687339029477275 + 0.3531680879608159im, 1.9980621864572428 + 0.3421629103745032im, 1.5456062622848659 + 0.324729340632693im, 1.1218428064741537 + 0.2978221251478481im, 0.7419179914022016 + 0.25804518711426633im, 0.4256069112675396 + 0.2033028574871891im, 0.19380169407424014 + 0.13643047349091159im, 0.05828702534666404 + 0.06953126800908208im, 0.006435262826788192 + 0.021782230595068865im, -0.0010500489781439266 + 0.002695470666228301im, -0.00012277528135197163 + 1.60337993285212e-5im, 4.332279544871967e-8 - 8.956341340659355e-7im, 7.737730575706709e-11 + 3.444740978708735e-10im, 1.1335041185365489e-15 - 2.369163256309525e-16im, 3.8394027592288557e-25 + 1.3775631289716995e-24im, 5.7333170913660523e-42 + 4.019319913618439e-39im, -4.1645305067192524e-63 - 3.802463833645413e-63im, 1.7547902518690728e-102 + 3.2118212436713416e-102im, -5.632962376223908e-167 + 1.1050974912204096e-166im, -7.602292840066867e-273 + 4.2839237359859864e-273im, -0.0 + 0.0im, -0.0 + 0.0im, 0.0 - 0.0im, -0.0 + 0.0im, -0.0 + 0.0im, -0.0 - 0.0im, -0.0 + 0.0im, 19.422784337158742 - 0.029203673144918833im
, 18.922784338495287 - 0.029203673105875738im, 18.422784340698875 - 0.029203673041504553im, 17.92278434433198 - 0.029203672935374417im, 17.422784350321955 - 0.029203672760395406im, 16.922784360197753 - 0.029203672471903787im, 16.42278437648019 - 0.029203671996261525im, 15.922784403325396 - 0.02920367121206002im, 15.422784447585654 - 0.029203669919130378im, 14.922784520558482 - 0.029203667787449886im, 14.42278464087033 - 0.029203664272903267im, 13.922784839231014 - 0.029203658478396427im, 13.422785166272453 - 0.02920364892487225im, 12.922785705472512 - 0.029203633173780545im, 12.422786594462801 - 0.029203607204639166im, 11.922788060159139 - 0.029203564388813802im, 11.422790476681525 - 0.029203493797588825im, 10.922794460847014 - 0.029203377412707156im, 10.422801029608085 - 0.029203185527489635im, 9.92281185961712 - 0.029202869165002185im, 9.422829715155434 - 0.029202347578921022im, 8.922859153613475 - 0.02920148764919001im, 8.422907688480091 - 0.02920006992002703im, 7.922987706377455 - 0.02919773263004735im, 7.4231196266023955 - 0.029193879498721455im, 6.923337107299312 - 0.02918752786907706im, 6.423695620754588 - 0.029177058818573274im, 5.924286569298801 - 0.029159806468544838im, 5.425260497751445 - 0.02913138441743693im, 4.926865199419432 - 0.029084584836757257im, 4.429508095337361 - 0.029007589432432816im, 3.9338578706686858 - 0.02888109015248326im, 3.4410087804823353 - 0.028673731153572076im, 2.9527427760716067 - 0.028335098763862032im, 2.4719383098140226 - 0.02778549583478644im, 2.0031828640491396 - 0.026902526581439178im, 1.5536245933117467 - 0.025507584670758247im, 1.1339838503079331 - 0.023363827088483065im, 0.7593072088540906 - 0.020215469144043718im, 0.4483165720604514 - 0.015923527983522815im, 0.21922704942196994 - 0.010743431043978405im, 0.08026140877749337 - 0.0056148853587022im, 0.01865598084904698 - 0.0019258134104280294im, 0.0020975029211462598 - 0.00032968586251067337im, 7.260678405534787e-5 - 1.7926254192086143e-5im, 3.6394029588254425e-7 - 1.4661826414313507e-7im, 7.41111524749911e-11 - 5.228655363875185e-11im, 6.721230570877014e-17 - 1.0361363222368277e-16im, -1.864169823362996e-27 - 3.5720968579640937e-26im, -8.139775526992696e-42 - 4.2816798952068195e-42im, -8.5705278203875e-68 + 2.3494757832291492e-67im, 1.5313896916407397e-109 - 1.893701952982838e-109im, 1.3253076410048056e-178 + 1.2528740964916723e-178im, 2.2308653435430693e-292 - 1.5308786810615063e-292im, 0.0 - 0.0im, -0.0 - 0.0im, 0.0 + 0.0im, 0.0 + 0.0im, -0.0 + 0.0im, 0.0 - 0.0im, -0.0 - 0.0im, 19.42278433697267 - 0.42920367234736084im, 18.922784338188503 - 0.42920367179092506im, 18.422784340193076 - 0.42920367087351746im, 17.922784343498055 - 0.42920366936096793im, 17.422784348947047 - 0.42920366686719563im, 16.92278435793092 - 0.42920366275565996im, 16.422784372742814 - 0.42920365597688387im, 15.922784397163502 - 0.4292036448005718im, 15.42278443742641 - 0.4292036263739489im, 14.922784503808721 - 0.4292035959935854im, 14.422784613254645 - 0.42920354590483817im, 13.922784793700458 - 0.4292034633224673im, 13.422785091205279 - 0.42920332716718834im, 12.92278558170773 - 0.4292031026851722im, 12.422786390409346 - 0.42920273257713776im, 11.922787723732338 - 0.4292021223728026im, 11.42278992200878 - 0.4292011163177121im, 10.922793546349741 - 0.4291994576181139im, 10.422799521866436 - 0.4291967228979313im, 9.922809373797097 - 0.4291922141422782im, 9.422825616800965 - 0.4291847805579199im, 8.922852396759225 - 0.4291725249129162im, 8.422896548827053 - 0.42915231948695526im, 7.92296934159787 - 0.4291190083193409im, 7.4230893520139265 - 0.4290640927834337im, 6.923287203307911 - 0.4289735667620085im, 6.423613371154225 - 0.4288243536951455im, 5.924151039178316 - 0.42857844921551824im, 5.425037254279619 - 0.42817331000498887im, 4.926497697671408 - 0.4275061325334096im, 4.428903718874863 - 0.42640827127518255im, 3.93286557186195 - 0.4246039756763536im, 3.4393839757047893 - 0.42164482116812363im, 2.95009415537207 - 0.41680819592206714im, 2.467652457856141 - 0.4089474035324931im, 1.9963316101738395 - 0.39629004411950314im, 1.5428903310988162 - 0.37622082443650384im, 1.117715990264714 - 0.345200709719881im, 0.7359758618950316 - 0.29924118518393067im, 0.41778887902644557 - 0.23578449257048073im, 0.1849728717554206 - 0.15794164682101264im, 0.050624891660021015 - 0.07977458336698368im, 0.002271877217438789 - 0.02412005211237192im, -0.001975490551566147 - 0.0025306436416951427im, -0.0001392658852773198 + 4.621052039983524e-5im, 8.145281857187584e-7 + 8.59924915803196e-7im, -4.4236745140561634e-10 - 3.40054561329938e-10im, -1.5072272094878889e-16 - 2.45592289751446e-15im, -2.0153450436042867e-24 + 4.5231338334005084e-24im, 3.0603115497994336e-38 - 5.757093007467674e-39im, 1.3186180582171105e-61 + 9.894484316990687e-62im, -1.4612609528592482e-100 - 9.442607791153564e-100im, 2.828756024728357e-163 + 1.1630396096583835e-162im, 2.335195331306485e-266 - 2.241880282600362e-266im, -0.0 + 0.0im, 0.0 + 0.0im, -0.0 + 0.0im, -0.0 + 0.0im, 0.0 + 0.0im, 0.0 + 0.0im, 0.0 + 0.0im, 19.4227843364907 - 0.8292036716852221im, 18.92278433739387 - 0.8292036706992425im, 18.42278433888295 - 0.829203669073637im, 17.922784341338023 - 0.8292036663934665im, 17.422784345385757 - 0.8292036619746125im, 16.92278435205934 - 0.829203654689154im, 16.42278436306222 - 0.8292036426774636im, 15.922784381202906 - 0.8292036228735347im, 15.422784411111834 - 0.8292035902223763im, 14.922784460423323 - 0.8292035363897192im, 14.422784541724223 - 0.8292034476346783im, 13.92278467576675 - 0.8292033013023703im, 13.422784896765515 - 0.8292030600412242im, 12.922785261130894 - 0.8292026622689573im, 12.422785861867869 - 0.8292020064536764im, 11.922786852315776 - 0.8292009251979334im, 11.42278848528851 - 0.8291991425109292im, 10.922791177605953 - 0.8291962033633026im, 10.422795616488505 - 0.8291913575453713im, 9.922802934972712 - 0.8291833681892412im, 9.422815001124505 - 0.8291701960955198im, 8.922834894876095 - 0.8291484793314605im, 8.422867694210316 - 0.8291126753839623im, 7.92292177139537 - 0.8290536472183477im, 7.42301093021208 - 0.8289563331961032im, 6.9231579299103165 - 0.8287959084429407im, 6.4234002959472605 - 0.8285314642336306im, 5.923799902211866 - 0.8280956093926821im, 5.424458774659457 - 0.8273773865745692im, 4.925545161071208 - 0.8261942707329614im, 4.427336549853862 - 0.8242464495407521im, 3.9302906924978545 - 0.8210426567295104im, 3.4351629431935944 - 0.8157811723325826im, 2.9432002972868325 - 0.8071624764467379im, 2.4564625100037736 - 0.7931041315254844im, 1.9783531955862408 - 0.7703339671393372im, 1.5144930910715089 - 0.7338853118696992im, 1.0741247206259925 - 0.6766887196739264im, 0.6722202469513102 - 0.58992874694513im, 0.3319947929095889 - 0.4658866726525998im, 0.08535470889369866 - 0.30643466528170005im, -0.03733900572449072 - 0.1386367619198214im, -0.042067480553635336 - 0.02120741184151642im, -0.005991077985344485 + 0.007220357363965503im, 0.0008386668749499684 + 7.043588351797712e-5im, -1.9573199354809665e-5 + 6.832060917729574e-6im, -6.140424743950199e-8 - 6.232131309598241e-9im, 5.6850215965543594e-12 - 5.394718311174526e-13im, -1.6938945500157091e-18 + 4.0701113104347157e-19im, -1.462310657221111e-29 + 4.0692910148679145e-29im, -1.856502255835392e-46 + 5.895551561127694e-47im, 3.920757198528301e-75 + 5.504396946923525e-75im, -1.1054632599207572e-121 - 1.5359750079774523e-122im, 4.0192755581761105e-199 - 1.0681629965033397e-198im, 0.0 + 0.0im, -0.0 - 0.0im, 0.0 + 0.0im, 0.0 + 0.0im, 0.0 - 0.0im, 0.0 - 0.0im, 0.0 - 0.0im, 19.42278433578893 - 1.2292036712630388im, 18.922784336236848 - 1.2292036700031799im, 18.422784336975337 - 1.2292036679260239im, 17.922784338192905 - 1.2292036645013722im, 17.42278434020033 - 1.2292036588550765im, 16.922784343510017 - 1.2292036495459089im, 16.42278434896677 - 1.229203634197686im, 15.92278435796343 - 1.2292036088927445im, 15.42278437279642 - 1.2292035671719501im, 14.922784397251887 - 1.2292034983859899im, 14.422784437572137 - 1.229203384977118im, 13.922784504049007 - 1.2292031979975087im, 13.422784613650869 - 1.2292028897202767im, 12.92278479435388 - 1.229202381457121im, 12.422785092283025 - 1.2292015434730454im, 11.922785583485812 - 1.2292001618714208im, 11.422786393344117 - 1.2291978839969164im, 10.92278772857968 - 1.229194128420797im, 10.422789930024404 - 1.2291879365335145im, 9.922793559629715 - 1.2291777278670097im, 9.422799543936591 - 1.2291608967023038im, 8.922809410660811 - 1.2291331470229703im, 8.422825678873323 - 1.2290873961342346im, 7.922852502617603 - 1.229011967295957im, 7.422896732921453 - 1.2288876105836488im, 6.9229696711123205 - 1.2286825930366383im, 6.4230899659415925 - 1.2283446088960583im, 5.923288407509832 - 1.2277874560252655im, 5.423615878292022 - 1.2268691076253266im, 4.924156590178831 - 1.2253556636275011im, 4.4250502556307625 - 1.2228622026816482im, 3.926529580285248 - 1.2187560440033898im, 3.428984608482909 - 1.211999386761129im, 2.9330755977603307 - 1.2008957200141752im, 2.4399371502299614 - 1.182687871199021im, 1.9515619183211097 - 1.1529402079332955im, 1.4715529768315847 - 1.1046463591926041im, 1.006651810570212 - 1.0271180359240253im, 0.5698682817468509 - 0.9051891431813432im, 0.18654339164699657 - 0.7208595063511073im, -0.09640051243934544 - 0.46385927728183574im, -0.2087824175708163 - 0.16483426811518412im, -0.11405381653019356 + 0.05081491251114011im, 0.024538092462979437 + 0.037639224571643835im, -0.0024341445399757335 - 0.010461008300708324im, 0.0013375970782092976 - 9.523009242235569e-5im, 1.7854684906309634e-5 - 5.564494645760924e-5im, 2.501216238190226e-7 - 3.793280250670107e-7im, -1.5160273071553551e-10 - 1.415734833400537e-10im, -3.115103669354081e-16 + 8.306464570043674e-16im, -1.6389552556716997e-24 - 5.260446033283102e-25im, 7.876560123250489e-39 + 6.611239210483262e-39im, -1.7581312513562253e-62 + 4.713557266756052e-62im, 2.386089044922975e-100 + 9.514717594582031e-101im, -1.5923279422546296e-163 + 2.07656250727342e-163im, -2.3832441726055056e-267 - 4.44002680709751e-267im, 0.0 - 0.0im, 0.0 - 0.0im, -0.0 - 0.0im, -0.0 + 0.0im, -0.0 - 0.0im]
        for (x, y) in zip(inputs, sympy_outputs)
            if y == 5.7333170913660523e-42 + 4.019319913618439e-39im
                @test relerr(expint(x), y) ≤ 5e-12
            else
                @test expint(x) ≅ y
            end
        end
    end

    @test expint(3.0f0) == Float32(expint(3.0))
    @test expint(Float16(3.0)) == Float16(expint(3.0))
    @test expint(3.0f0+4.0f0im) == ComplexF32(expint(3.0+4.0im))
    @test expint(ComplexF16(3+4im)) == ComplexF16(expint(3.0+4.0im))

    @testset "En" begin
        # ≅ but worse
        ⩭(x, y) = relerr(x, y) ≤ 5e-12

        @test expint(0, 1) ⩭ 1/e
        @test expint(0, 2) ⩭ 1/(2e^2)
        @test expint(2, 0) == 1.0
        @test expint(3, 0) == 0.5
        @test_throws DomainError expint(1, -5)
        @test_throws DomainError expint(1, -5.5)
        @test expint(1, -5+0im) ⩭ -40.185275355803177455091421793795867095419087399195930434 - 3.1415926535897932384626433832795028841971693993751058209im
        @test expint(1, -5+0.0001im) ⩭ -40.185275237072650246247297527983285156145761611565973243 - 3.1386243904111057379992440632406314204488813709668965035im
        @test expint(1, -5+0.001im) ⩭ -40.185263482751181420528529573820113544087988149917372800 - 3.1119100251333093938956809011637485625455065579106720262im
        @test expint(1, -5+0.01im) ⩭ -40.184088057852053166778027787056086608566881324182021075 - 2.8447696994032256009548700952561340708768881138416215357im
        @test expint(1, -5+1im) ⩭ -29.025158144992710296353782915633718625839577127679530594 + 23.304702810489904370776539518061890789478578493028994502im
        @test expint(10im, -10+10im) ⩭ 842.196601731195121156507301117535342516890218271788126699 + 481.296521260231635469781278283025739948078026046933687102im
        @test expint(-10, 5) ⩭ 0.073300021438672650112689843609495288768872386736573672606
        @test expint(-50, 5) ⩭ 6.8486449405023952496492865790416225809622023723598328e28
        @test expint(-10, 5+5im) ⩭ -0.0008950056371723383212263766350696833266683028128620772 - 0.002011873719014935561647124110993999073670119015512577im
        @test expint(0.0001, 5+5im) ⩭ 0.00083723619432046535141814660262350691841213819828142586 + 0.00045499057287120330778306472378501814547107632922018099im

        @test expint(100, 10+10im) ⩭ -3.254610677616123e-7 + 2.564627766537454e-7im
        @test expint(1000, 10+10im) ⩭ -3.750710414994654e-8 + 2.485006012671188e-8im
        @test expint(10000, 10+10im) ⩭ -3.803483654815381e-9 + 2.4714313502801793e-9im
        @test expint(100000, 10+10im) ⩭ -3.8087890502240534e-10 + 2.470010601657554e-10im
        @test expint(1000000, 10+10im) ⩭ -3.8093198659567175e-11 + 2.4698678867697097e-11im

        @test expint(-2, 3.3) ≅ 0.0200031425174657791728529428002375552480803611197081245457722070
        @test expint(-2, 3.3) isa Real
        @test expint(-2, -3.3) ≅ -4.745485121488663825992363318400378114254141655964565963
        @test expint(-2, -3.3) isa Real
        @test expint(2.2, 3.3) ≅ 0.0071305694486185747267607978483436184475047798769491326205
        @test expint(1.2, 3.3) ≅ 0.0085837830493629441737850749465161073231031009810293477318

        @test expint(-2.1+3im, 4.2 +0im) ⩭ 0.003387565626497080536855067510744646235748203589175196673362104 - 0.003355838600814825542988272289126728631565803782052141768321840im
        @test expint(-4.3, -10+0.0im) ⩭ -1505.2167858114105916067190408869152177781925766866637213104 + 0.00015439438628349460776476190593323691084185406009267711345841im
        @test expint(-4.3, -10-0.0im) ⩭ -1505.2167858114105916067190408869152177781925766866637213104 - 0.00015439438628349460776476190593323691084185406009267711345841im
        @test expint(-4.3+2im, -10+0.0im) ⩭ -1473.52621376586428327992147406861600035297263354271049988247182 - 210.712752794950368400059514411922967744944073260391136148642008im
        @test expint(-4.3+2im, -10-0.0im) ⩭ -1473.47816311715810169554659685200228583538709701122499753349598 - 210.761182017526309473296237005277228612276963738826032570499367im

        # don't hang on large integer order
        @test expint(1e15, 2.5) ⩭ 8.20849986238987e-17
        @test expint(1e63, 2.5) ⩭ 8.20849986238988e-65

        # handle gamma pole
        near_pole5_ans = [0.0120728157429194, 0.0119237297690539, 0.0119090129077972, 0.0119075431153846, 0.0119073961550572, 0.0119073814592135, 0.0119073799896311, 0.0119073798426728, 0.011907379827977, 0.0119073798265074, 0.0119073798263605, 0.0119073798263458, 0.0119073798263443, 0.0119073798263441, 0.0119073798263441]
        for (i, ans) in zip(1:15, near_pole5_ans)
            @test expint(5 - 10.0^(-i), 2.5) ⩭ ans
        end

        @test expint(2.000001, 4.3) ⩭ 0.0022470372637239675640390482141893761501016099509821867746
        @test expint(1.999999, 4.3) ⩭ 0.0022470379319936678143503120160911823783417969321242814713

        @test isnan(expint(NaN))
        @test isnan(expint(NaN+NaN*im))
        @test isnan(expint(NaN, 1.0))
        @test isnan(expint(NaN, NaN))
        @test isnan(expint(NaN+NaN*im, NaN+NaN*im))
        @test expint(4, Inf) == expint(4, Inf + 3im) == expint(-4, Inf) == expint(4+1im, Inf) == expint(4+1im, Inf + Inf*im) == 0
        @test isnan(expint(-Inf+0im))
        @test isnan(expint(4, -Inf+3im))
        @test isnan(expint(4, -Inf+Inf*im))

        # |ν| > 50 not currently supported
        @test_throws ArgumentError expint(-100, 5) ⩭ 2.3661006604908971561634280511260954613666084788690790e87
        @test_throws ArgumentError expint(-50+100im, 5+5im) ⩭ 0.00004495913300988524775299480665674170202100859018442234 - 0.00003791061255142431660134050245626121520797056806394134im
        @test_throws ArgumentError expint(0.0001-100im, 5+5im) ⩭ -0.0000670979311229384710017493655410977414867979779532195 + 0.0000229212453762433221017163627753752485911057259666081im
        @test_throws ArgumentError expint(-100-100im, 5+5im) ⩭ -6.08640887739723453551143736571517308449904372104665e86 - 1.89339595225376542869023397967131390592967463726985e87im
        @test_throws ArgumentError expint(-67.7427839203913 - 53.20793098548715im, -18.33277989291684 + 5.734337337627761im) ⩭ -8.98320096334454197835241406421715692910936300308912e64 + 5.24063785033491832811625264321940307636652348236306e64im

        #complex_orders = Complex{Float64}[0.11700811829315724 + 0.07237176593469438im, 7060.980036993411 - 312.2822336024412im, -0.07810440094845131 + 0.10991560623276703im, 2.8681931397557494 + 0.48191076677812916im, -0.0002042494767828777 + 0.000719718956048358im, -400127.741078456 - 83770.66876515116im, -0.8231592923815496 - 0.01222209198755957im, -80.41967451594795 - 752.8162696161975im, 0.10410878103045253 - 0.0013958151919657923im, -0.38512195673750343 + 0.20206360780641064im, -0.021559310259815515 + 0.04020156763174565im, -33.35512540338605 + 3.293786848513468im, -0.15934133049439234 + 0.05437900550596498im, 13.817529409421939 + 52.41321363857554im, -0.0004516865200750638 + 0.003975469137649451im, -0.002151698387984082 + 0.014337878896312752im, 0.0005602052948508679 + 0.061320514646617474im, 330.92354237063444 - 182.51618354765046im, 0.0005005349906190552 - 0.00013165964481444474im, -3334.175760325154 + 7824.580069405813im, 0.01842251014668398 + 0.010760499844447187im, 0.0011084140346222726 + 0.001953662139374601im, 0.020169857496750918 - 0.045065767612523476im, 1.4460472915347165e6 + 895074.0965224776im, -0.12749575765105228 - 0.4553983030175986im, 399724.64732882136 - 53580.13956919416im, 252975.30715852365 + 15971.727674338406im, -0.002492179269702592 + 0.0010533608694487323im, 4195.2470174539185 - 5537.775306756674im, -127.84924549551631 - 207.75150691379213im, -1.675536694112286 + 3.4585547714424942im, 20638.272768478386 - 4791.209077746527im, -0.17712412619876042 + 0.009853348540808222im, 0.10873797200673406 + 0.1944500658043079im, 138501.5212070995 + 694936.9572399684im, -156.40654085616225 - 132.69221541805703im, -1.102447196077966e-5 - 0.00012170632236745253im, 13.05629141567096 - 24.050312416577697im, 2.295640998333424e6 - 736606.8806920773im, -0.3839366454191859 - 4.37817129529046im, 30.22106249803574 - 0.18728249683191728im, 512655.20252162614 + 80958.69024333062im, -112.0017631027479 - 92.78862513293855im, 1158.1840663247697 - 82.42659915838665im, 8.87671660388369e-6 - 0.0002003301413775308im, -0.16989015567640423 + 0.1941080880713674im, -161.80759482900206 - 19.690029297617116im, 548891.3962344378 + 10337.673037942415im, 4.149524542013174 + 54.43341063381733im, -1.9874599100841606 + 1.9948683281786093im, -5.024922729216726e-5 + 0.0008466555975749811im, 0.4881523324761429 + 2.563961427853926im, -27682.1151360095 - 776042.0396914122im, 9.86235324084743 - 36.33462376200989im, 2078.5705027707995 - 2669.107212124583im, 0.002942289960625623 - 0.00011770652344320951im, -0.33094951292679964 + 3.906568104973893im, -12607.563125739734 - 72501.69664367771im, 122902.41413826183 + 87450.88167727407im, -0.0002415408690273834 - 0.0032289196193279026im, -0.005577921264665741 - 0.002072905248768769im, -1415.0862235831416 + 1508.3431748185496im, -0.002603367751993649 - 0.006273555785896362im, -0.00011121140793028205 + 0.0008404268497870498im, 0.9095355293307844 - 1.0245161371335396im, -0.2046192599818901 + 0.24667918315747153im, 22.774867598149317 + 1.900758427256582im, -3.59132022124858e-5 - 0.000294211480358384im, -0.7934160560251374 + 0.7806878283955511im, -0.0010600542785636404 - 0.002642313847253561im, -0.0007468303718782575 + 1.3567158107127604e-5im, 41.283239652476446 - 23.007003570437853im, 5.380074072434925 + 2.01195740967198im, 0.040576253922537074 - 0.026195925748000146im, 103343.75310630476 + 20655.782403968642im, 258.09087614630647 - 5.528385770156952im, 47.56500926334721 + 387.7836211633227im, 0.039796462256580345 - 0.03200357837889448im, 0.0005962928623526773 + 0.006733608111553142im, -0.0004733470409932336 + 0.0001677738508892556im, -0.00297517714897401 + 0.0012386879869359798im, 1142.3270665304437 - 780.2771640547608im, 0.01211170879351441 - 0.006484987921698502im, 420.39106532057366 - 1723.8556429496075im, 16.63539915599837 - 21.99607564760897im, -0.0001318242613437761 + 3.25824489013223e-5im, -0.003057657277208497 + 0.0028141198241151727im, 0.00019653233574934824 - 0.00046942893063092544im, 111883.09502992724 + 70735.30427661471im, 9623.860576465842 + 2459.9092318513385im, 3.443451501410717e-5 - 0.00021937879248955415im, 473.48577997227403 + 89.09414622927443im, -39902.92001856412 - 6376.883819451536im, 0.01357717594529387 + 0.010508843492887535im, -16323.44933488832 - 552136.7057999639im, 1322.2767797926003 - 13040.882735851055im, -0.20626630047528827 - 0.19454577850593724im, -250840.6163672714 + 832373.7819987085im, -9.864541858794732e-5 - 0.0006164526935575966im, 12.147097299498396 + 16.489982931454442im]
        #complex_inputs = Complex{Float64}[-9.87069213822445 - 3.811529250764424im, -0.001705041173194617 + 0.00028768089717449085im, -0.0005103378337774433 + 6.540003301806718e-5im, -8642.219081484756 + 12619.484208969243im, 133303.34854905694 + 129776.70505985153im, 0.0007285562983420401 - 0.001870154685167422im, -4.7807929095254735 + 14.970103108117732im, -0.0020162267187235827 + 0.0011017259520041823im, -0.005567018545282227 + 0.0013172834963094643im, -0.030603943229841557 + 0.04533028498349393im, -7309.664117121871 + 1543.314431079508im, -2.624391318397948e-5 + 4.856897064551571e-5im, -172.1269770580335 + 98.59414167777635im, -0.15632258852321199 - 0.06820279030303825im, 0.003982151070928306 + 0.0035947342237090244im, -13830.550621767192 + 3295.977400749607im, 0.3430029208414258 + 0.20597825634119798im, 2.9388348367202883e-5 - 0.0001133890098776906im, -1.9180735325735792 + 3.0100133063087937im, -10.962025365681605 + 1.3001100242056045im, -1.361179810203242e-5 - 0.00010031232426490721im, 0.00011325970888214695 - 4.713354721127061e-5im, -0.00024275021929945895 - 4.458056930602084e-5im, -3898.047681054125 - 801.5033058434637im, 229735.22931104113 + 1.0257663021896945e6im, 0.0028307367691868908 - 0.009936186656056525im, -5.538310189970093 + 4.235897490481438im, 336.6162651819914 - 524.7576189291967im, -0.0010171368975154993 + 0.0009596576965268554im, 7885.630769970493 - 4284.640933291882im, 140.90730072596008 + 492.89734961932214im, 2.7364781953713666 + 4.7519005116359im, 6.15896302490673e-5 + 3.649091895250442e-6im, 0.0005400146220176668 - 0.0016439409929813224im, 0.00017752440582812228 - 0.00013455434772818406im, 4.759175597438777e-5 + 0.0002802438966108739im, 0.2363431749902789 - 1.8354862517633548im, 0.05167624975784817 - 0.0008355404619955311im, 12607.17410787538 - 178432.04508315283im, -3362.0253933160325 + 2117.9558097801587im, -0.4188343597698366 + 0.12159034969928785im, -0.0010928452013149932 - 0.005110608890415864im, 0.00021224077721321234 + 8.785905680712027e-5im, -1.4808810763394369 - 5.683208346600066im, 0.0040052830261459935 - 0.004652235196639043im, 4.37293178024758e-5 - 0.00013664005019595895im, 1.5890373788027961e6 + 1.2835932190154726e6im, -5.683398724429634 + 0.38723461776129914im, 0.19760960078084472 - 0.1891331304089848im, 482.5973009675434 - 2965.1204314748466im, -1.234102507117772 - 0.553682229133905im, -271373.9205825498 - 215889.54925839137im, -0.0033337132073958157 - 0.0017764552242406021im, -654.263879305446 + 4353.276804135042im, 99216.14820899564 - 80796.58975140747im, -1.5141216529691028e6 - 2.12290662169282e6im, 2268.6405809396074 + 6235.216208183325im, 3066.4003024244917 - 12631.772417991686im, 0.8262308885914383 - 0.4403424797640886im, 0.00013254588444464393 - 7.886521034191459e-5im, -663.5982190679889 - 707.5738763839958im, -0.00569427042132876 + 0.021037874823364226im, 0.00705925574561114 - 0.044869385146464644im, 8775.993127279115 + 78.70715978432729im, 16.224683207063578 - 5.802231001895036im, 0.09282242923641955 - 0.04135553549155076im, -163743.697616407 - 70563.48664880193im, -2.831049008605197 - 0.4742924738975423im, 0.0030761343493198394 + 5.212282764900067e-5im, 2.7678092665823977e-5 - 9.073094376456747e-5im, 463.8760228975407 - 3538.8628897641406im, 30.422696305407126 + 404.52535888063954im, -893.1437470386834 + 176.44682356031322im, -11.945400063421758 - 6.940811741142993im, 5.245404086922615 - 1.8304077438400932im, 4.469400112242066 + 3.0969731226854273im, -31858.146173885514 - 1.0166587388138341e6im, -0.0011359983095499083 - 7.345270844056152e-5im, 7.217691488232666 - 2.3628623812018255im, 0.14861503812405671 - 0.17467699447653054im, 0.033202330265079544 - 0.09980198365752208im, -0.066155487803214 - 0.10233815839204513im, -0.36896426383231107 + 0.5192933054221572im, 0.0001079470795289078 + 0.00025565584613346356im, 2254.9290319277266 - 4888.926279609392im, 0.010487584961220826 + 0.01461581191197843im, -0.0022516805439078513 + 0.0014963774108310952im, -0.00012910056603527905 - 0.0006035212025072887im, 0.006572868540660047 + 0.0010405025571699048im, 4.3696576356636865e-5 - 1.7709457046068167e-5im, -38.069863856300444 - 676.8264162612595im, -4.482727435760739e-5 - 2.443994182357729e-5im, -1.6358507299609564e6 - 514782.787871262im, -0.0004009091726208727 - 0.0002649579495562792im, 0.07975688436726693 - 0.3289262262553162im, 111191.29860653849 - 89518.45577911162im, -0.0005801792134045477 - 0.00915232608475763im, 0.4311128647760165 - 0.31918775593582177im, -0.9910121488177311 + 2.24926989700125im, -5294.963723864245 + 28476.35935417037im]
        #sympy_complex_outputs = Complex{Float64}[1770.869216265717 + 554.4068819291233im, 0.00014160994339329223 + 6.222973328932327e-6im, -1510.4350442384573 + 1865.9593092936857im, Inf + Inf*im, -0.0 + 0.0im, Inf + Inf*im, -2.758774073817244 + 6.956291190285107im, Inf + Inf*im, -95.30916839323694 - 56.633371940797055im, -29.301983426918113 + 13.014783135569653im, Inf - Inf*im, -1.1811794866337162e180 - 5.374876068838714e180im, 2.21325783461214e72 - 1.807249415025919e72im, 0.006528344209529012 - 0.020704996160824215im, 134.88086649348097 - 127.38393745254174im, Inf - Inf*im, 1.1994636374154315 - 1.2570889264524734im, 0.002320547906415499 + 0.0012840889009836463im, 0.8049560596179197 + 1.72937225703151im, -6.712452156186971 + 0.900530142196031im, -95.56570525918356 + 8584.870188116387im, 7512.052790828632 + 2978.6520450507883im, -3013.836914503322 - 319.10776156888227im, -Inf + Inf*im, 0.0 + 0.0im, 2.4472401579645007e-6 + 3.5282233150079064e-7im, -0.0004029735518824845 + 0.0009185525731563604im, -4.579472271719604e-150 - 9.273198745737493e-150im, 8.711044833155677e-5 + 0.00011478520143434018im, 0.0 + 0.0im, -7.049616210023507e-65 + 1.0169210123882166e-64im, -5.726656678048583e-7 + 3.004191414293308e-6im, 82281.89363631312 - 13450.630309751185im, 381.9282225576668 + 9.987794659805122im, 2.759688701744655e-7 - 1.3837235738621324e-6im, Inf + Inf*im, -0.42261765359562453 - 0.05811702097910586im, 0.015834396639172608 + 0.03151431917087795im, -0.0 + 0.0im, -Inf - Inf*im, 0.052431438784845574 - 0.00629339241717934im, 1.9067633783230262e-6 - 2.911375363267587e-7im, Inf - Inf*im, 0.0032846855189000825 - 0.001897648867316045im, 105.15023427481417 + 123.51559856336264im, 35616.09866962017 - 6506.757962109077im, -0.0 - 0.0im, 0.0004918984911762192 - 0.00021149828239007466im, 0.0037465596047789825 - 0.014629799676874826im, 5.549660094163257e-214 + 6.537799556811112e-214im, -2.519174939182313 - 0.3352167692073511im, -Inf + Inf*im, -4.835548129427574e-8 + 1.2911659536274306e-6im, 2.3346408232219433e280 - 2.167798678690289e280im, -0.0 + 0.0im, -Inf + Inf*im, -0.0 + 0.0im, -0.0 - 0.0im, 2.8558394893941187e-6 - 5.140414884209146e-7im, 5482.069446787932 + 3463.212428401809im, 1.614292678079022e285 - 1.651116913488533e284im, Inf + Inf*im, 1.9966466147266195 + 21.743730178419092im, -0.0 + 0.0im, 4.865993254091446e-9 - 4.993008051409113e-10im, 14.627775183270892 - 0.0662742838511591im, Inf - Inf*im, -5.6305880769143215 - 1.7923589441031695im, 2146.338255672743 + 23559.8580137768im, 2845.5506584302184 + 10214.460595445147im, -9.384782380262446e-206 + 2.613201919539122e-206im, -1.261950931410683e-16 + 9.491854192640718e-17im, -Inf + Inf*im, -11080.602857557576 - 1424.4836554952888im, -3.1172992952761996e-9 + 4.99227876269434e-8im, -4.372351969364027e-5 - 2.3575981994601973e-6im, -Inf - Inf*im, -623.263343821411 - 5.742740603681383im, -8.641994591470621e-5 + 4.315499470010166e-5im, 1.9019744085061088 + 3.2444817107826123im, 2.0270730727622457 + 9.035560420770867im, 0.0005899964318237499 + 0.0004990688554819831im, -2.053235921941274 - 0.9887100775403173im, 0.0001333686121284535 + 0.0005475856120983248im, -0.0 + 0.0im, 31.416337256389784 - 45.18425872668647im, -316.5209925543502 - 199.44920967470182im, -343.69636602818434 + 1580.028503971555im, 6.339557682547958e-6 - 4.0173023881608425e-6im, 9.754105618705067e-5 - 2.4932755825784563e-5im, 4.996356489474852e13 - 6.563218450038547e12im, 0.0020438965529309774 - 0.00038535497223109026im, -Inf - Inf*im, -1495.6219481974294 + 1240.3695649394147im, -5.864789251075764e-7 + 1.5653113462896759e-6im, -0.0 + 0.0im, -178.8058240701435 + 57.26066712466002im, Inf - Inf*im, -0.5035913133675121 + 0.9737934487486525im, -Inf - Inf*im]
        complex_orders = Complex{Float64}[2802.1972265790796 + 489.65328426250437im, 628.438840695858 - 915.0219018005914im, 0.9205635309256506 + 3.847600868872933im, 0.07621717454833514 + 0.07804752265660679im, -539.226849555439 - 2904.946863748866im, -5.951373683232884e-5 - 4.4111751536179756e-5im, -9883.82040671479 - 9775.199628008308im, 0.0076816103288650855 - 0.005618823000091182im, -2708.862591090647 - 4038.945651650799im, -3144.9639051559693 - 1065.705423571788im, 8.515563646681104 + 64.61472602030352im, 2.7129475726320504 + 7.7037779273687566im, 1.1133596463737567 + 0.09837945909714703im, 0.5830372069981893 + 0.42696219944833974im, 0.04007330622018601 - 1.7927238115192128e-5im, 2397.9641771511874 - 473.84145486894454im, 7.229956069821563e-5 - 4.695026730804693e-5im, -255173.32172923465 + 1.3354132696058152e6im, 15158.263108139012 - 1577.1801452604332im, 13983.542267829083 + 17798.6972568714im, -577.9287788390171 - 458.33730833921686im, -9.269489924843825e-5 + 0.00013976459650405217im, -0.011314437769676645 - 0.045698056023012225im, -11344.36160811529 + 3511.9567859355625im, 0.0037545286412588716 - 0.014264454142893186im, -9983.616121053377 + 44435.7595164082im, 21.87541793819557 + 31.798785615324316im, -0.0038566122828622863 - 0.0004292207024557002im, -1.264812200002861 + 11.62886677547152im, 241.11121341041613 + 192.80513339097755im, 1418.2995257317884 + 10152.664505485192im, 422596.710395312 - 2662.8898688565437im, 1.5558472697038352e6 - 2.551446091483643e6im, 423314.2930417994 + 255066.65407270033im, -3.1269683488669363e6 - 796108.1639580812im, -0.00859115324357219 - 0.003947287482336855im, -1900.384575177788 - 2672.458062823582im, 275.4217906864241 + 9295.468711701149im, -2.8461538722046056 + 5.742478560130701im, -0.0016504890553343389 + 0.0011998250415108742im, -0.00012852494667295927 + 9.670225099804868e-5im, -1615.2343318310689 - 1187.590761545176im, -102.2655106076873 - 87.84771196351194im, 0.05381264774798834 - 4.232066645692217im, 0.00010275586830650973 + 0.00029964068401128267im, 1224.7724685491312 + 162.11260255172604im, -0.8328755684700762 + 2.3622872818276677im, 0.9502873139367974 + 0.8854028747337215im, -335.06589088128754 + 345.6141813980166im, -515.5644058284929 + 550.5941961047416im, 0.4099771652420862 + 3.1014852493360388im, -433465.3106243227 + 243799.5124972474im, 0.00025700051727091475 + 0.0003414173857018965im, 38130.190811125816 - 1940.0240603093557im, -0.5025166126733999 - 0.026311563991047674im, 0.20027340012182135 + 0.4166151932931847im, -6036.696846400663 - 85492.93459644109im, -0.0011440818102474125 + 0.0008659481651084397im, -513.1529862377799 - 1313.567721782508im, -9.70419687238446e-5 + 5.400104181076698e-5im, -7.871107405455864e-5 + 0.00037310844636180725im, 0.18211519955687072 - 0.06828488851656798im, 515064.96633688797 - 597166.2692176442im, -13.7511926263185 + 2.3297012391282967im, -91406.68444089571 - 25077.082998045185im, 0.0008371347284858105 - 0.0004798953636379388im, -0.000592573708486316 - 0.00031886957045595206im, -0.0008267561975659352 + 0.0020223390160103772im, 0.09601274886690776 + 0.0540912317114096im, 1603.1132857611055 + 3278.618501670071im, -141312.51440471527 + 793134.2580019227im, -548154.1943496682 - 36373.99232359876im, -0.027005847680486954 - 0.012441456116240103im, 1.2741513225120402 - 2.6052364151903014im, -0.02152835840158428 - 0.11522799721073314im, -299.1472341409817 - 1845.48192743865im, -10361.70953092055 - 10067.585163905518im, 0.025885157302375193 - 0.03356967675766488im, -0.00013348692876335918 + 1.9869391520310315e-5im, -8.284790703341877e-6 + 0.00011787582904872539im, 0.0037211817417881067 - 0.006210089933374363im, 0.004090501101145849 + 0.001500528663219177im, -32293.2006646442 - 46015.727590306116im, 12.746312099531268 + 5.5831837094694885im, 0.002837890224942538 - 0.007468074422484914im, -189.32269251662828 + 43.65408228507107im, 4702.998476156974 - 455.34015906023im, 2.5487558538685346e-5 - 0.0001067530385430015im, -0.12273479859918503 + 0.06376567444441754im, -0.011489710253367093 + 0.007573609319939926im, 0.007297936126043878 + 0.0024623041649804757im, -0.0001567358003216213 + 0.00023494628916334787im, -0.5530968573908251 - 0.021902888599685318im, 2.478184220558431e6 + 174806.09162442965im, 0.2656422120259019 + 0.22404080774214408im, 1266.0586643921988 + 1332.5997533353714im, 1.2833089969455989e6 + 180134.37316097235im, 1.0515546695655487e6 + 366807.79781428276im, -0.07542199534324494 - 0.09929153067428087im, -553.8553130120799 + 6012.484180585329im, -69648.41802009603 - 87075.62257704904im, 62162.830096138285 + 213193.4650340032im, 7370.967228317164 - 1586.2073686769959im, 119.41609916793017 + 49.21486723035074im, 4.233259188191879e-5 + 0.0037143577565349516im, -0.0271366620074447 - 0.007836773529832565im, 31.449884929284927 + 27.861077430590157im, -1267.2033813735982 - 1730.1717354477596im, -1.1925743562966085e6 + 303097.22077287204im, -27196.455452054215 + 1799.216357568888im, 1621.5480437409938 + 3498.550097219721im, -0.1406462748893811 + 0.07354054293638994im, -8.925911232122603 - 1.5760382312547334im, 1.0857819763511322e6 + 845857.9915079646im, 17.569460048405016 + 23.6490713003959im, 0.3097675397699137 - 0.32783877868345807im, 0.0058646034607998734 + 0.043172156441739266im, -8.228249505499015 - 14.929856851501656im, -0.000948844261311448 + 0.006491397628655822im, -19082.24390278327 + 13146.687963159553im, 0.006129951143050223 + 0.06649636838592937im, -3.3476787485232644 + 2.803781658392517im, 0.5217526435312246 - 0.18203485796365454im, -2.345683984875169e6 - 1.4466604236352525e6im, 89.89019339003488 + 427.6131648895045im, 0.016701317099086836 + 0.03795878362299086im, -1112.5162488081203 - 10580.074890967255im, 0.001488068363420001 - 0.004011153811015184im, -0.0004058933842516511 - 7.907544711979595e-5im, 0.10691919091578121 - 0.02046231014811242im, -79.61255094481233 + 164.2488087973983im, -1.9469006885904642e6 + 1.0396328350051036e6im, 0.01411170386231728 + 0.018219740221736377im, 1.5042037854791434e-5 + 0.00023852039409563664im, 91.87986569433626 - 6.363688160911657im, 0.001956446692592356 - 0.00549714113002722im, 0.12902390645641867 + 0.1538474270770209im, 9.314851768724729e-5 - 0.00018240996634662148im, 2.150389025423486 + 134.54514917714096im, -0.014129282039218658 - 0.031078623550182347im, 8.924409481413715 - 6.214249787446488im, -0.05532702485867124 + 0.07623487602446968im, 16.78823823238913 + 10.33293463768194im, -1.0247902314638373 - 0.549735377541302im, -14256.283161640198 + 17607.616719983915im, -68.37739861667906 - 167.95272555015327im, 0.07537879278101972 - 0.06278151803089908im, 401.17970018718125 + 2045.4451840774932im, -19394.165467050385 + 5700.543175824435im, -1.0531667040061785e6 + 743893.6128476112im, -3.0283749111823575e-5 - 3.8531676121386384e-5im, 8.45699230610461e-6 + 8.263135095778822e-5im, -0.00014311465640073012 + 0.00017016929796709334im, -60.60776862422261 - 16.8377889217509im, 0.06982353902165782 - 0.027606807230802128im, -31028.044295538148 + 110541.07959296223im, 282488.856643454 + 671777.9285392588im, -0.0002449367989738247 + 0.0009351785892345893im, 0.06968258149425399 - 0.16578716972717303im, 0.005959780105143905 + 0.0068727814569324586im, -0.00022703784406149144 - 0.011118916219708628im, -127.36502303768238 - 85.03910040011138im, 313426.7664022941 - 881308.8382685778im, -1.4885274065242733e6 - 463622.16749089526im, 0.00016192504238007866 - 0.00012174226216565271im, 0.35763226334220294 + 0.25001208360341076im, -344.4033718716871 + 16758.556279508386im, -1.2854476403610515 + 5.5400455149115375im, -4.0489159967629215e-5 - 3.055559011133855e-5im, -0.006936496820585996 + 0.012986925494491115im, -803358.5388206418 + 532300.6425033197im, 110.91064579809569 + 152.5235814061417im, 5.014620456732865e-5 - 4.2994067572670836e-5im, -68982.99854003216 - 227962.4356876082im, -3.877950262704879e-5 + 0.00010272496795159568im, 0.0006954611955076009 - 0.0002650688439967194im, 0.004737840922121714 + 0.01725483879933454im, 0.00012059820944269894 + 2.4462686670088754e-5im, -0.001225807234704899 + 0.001689866535920248im, -11253.393386242487 + 33154.588929805206im, 22.740188371533765 + 12.600475871331662im, 2.7972571545206085e6 - 451984.8857764654im, 0.24898568562685447 + 0.05691335628928623im, -12534.737506218535 + 12926.216542039087im, -34.03107084400432 - 21.975441107348292im, -0.00018529220760455945 + 0.00017643149241582592im, -15785.52916245016 + 11835.529420525856im, 0.007679466467623727 + 0.0009209123421362982im, 362.2790477992112 - 911.3223916460166im, -4.702642983469814e-5 + 0.0005517874315317168im, 691.3641622056458 - 10081.546652771853im, 12265.266570730175 + 2820.2259865935434im, 252848.19327269573 + 343575.5215621514im, 0.27801448984563787 - 0.09113756225339412im, 1.6042214538050472 - 2.188165181516539im, 0.07026895667711258 + 0.16186710738523874im, 57755.731129483254 + 70591.69138182167im, 4.960463865690599 + 5.117595357548634im, 0.0019423103501152001 - 0.0004386352195522862im, -31.4459663876656 + 38.502917550654736im]
        complex_inputs = Complex{Float64}[-0.003231641157940195 + 0.003511503981821159im, 0.0003511875705101058 - 0.0002967678971063248im, -0.0011997759012423563 + 0.0014055812127151172im, -0.1107704613629459 + 0.6489701534959887im, -0.0010622395142610265 + 0.020466085875253982im, 661.5303401228165 - 358.82298882599764im, 0.0015536617396754238 + 0.0010862446567976869im, -0.00023553540847409367 + 0.0003612021000561223im, -227.338852219818 + 257.3442356814361im, 42987.991940107604 - 25153.270709165885im, 19489.662321273565 - 10293.356466075164im, 4.717378925253497 - 4.836564023265535im, 0.0001955686701341741 + 0.00031815077277291305im, -154.601776016573 + 561.7781743197932im, -1021.6141566547319 - 2145.623118802056im, 0.006667186835074078 - 0.02769360539402995im, 2.386754711640224 - 4.840084366369915im, 15480.042298599987 - 5737.205766842911im, 196565.752334228 + 157239.22739506228im, 0.0034594867763828945 - 0.0010804921857260515im, 0.0012971158299212892 + 0.0019392250499218773im, -5350.853916583974 + 107564.11332407437im, 2242.6038624279313 - 33762.75968229416im, -9.793707932056002e-6 - 0.00014226859375174764im, -62.92257621227457 + 100.6454077023004im, -3342.3857942593468 - 639.1870314594953im, -4.340852418211193e-5 + 5.036352882236416e-5im, 0.0020575609016097547 - 0.0043092882095990236im, -2.4356023884510613e-5 - 0.00015178629775195802im, 220343.78481646266 - 272824.14605786034im, 6.397855827447737 - 17.13464875170837im, 0.8387463773482948 - 0.8408332743650528im, -22219.152388431325 - 101898.04105888026im, -576.7307940101958 - 567.3765246753663im, 0.0008090526197006311 - 0.000712464888902046im, 10138.04131285203 + 19764.952728761345im, 72.76785381454488 + 498.1192140720909im, 145315.54712050984 + 34163.14555096257im, -2.030037472217121 + 2.621951027160376im, 0.0001492044760707073 + 0.00016379846887561525im, -1.0704121031952167e6 + 202605.18705347553im, 1.2567435306132886 - 0.889602855970317im, -0.12183937557285143 + 0.10220442719281003im, 1.8601637230846178e-5 - 4.9187941623357136e-5im, -0.009547966173388477 - 0.04480771811590404im, -27.850229524144744 - 35.63993877844333im, 0.0009763056352848922 - 0.08799980452932837im, -8.04649016355904 + 5.5544517292797275im, -7.38028624776299e-5 - 0.00011190440553632226im, -0.3421414257900438 - 0.06591555294331265im, 126.56032566669457 - 313.72006499923395im, 6.008485043350022 + 11.565435364865085im, -21.889359607440966 + 3066.032285305223im, -0.00010051116532222963 - 8.387947298324234e-5im, 0.04392571264735291 + 0.6973671946930264im, -678.3437177873299 + 4951.123963655774im, 2.5594758624872624 + 0.282137065669307im, -0.0020114304904337308 - 0.004471561787747692im, -5364.933764222839 + 60298.673360939196im, -11.235562679284698 + 28.216002956994775im, 2.2282148174783285e6 - 1.4676088556971357e6im, 7.864032404227804e-5 - 1.6283977635520136e-5im, -0.009902605372796951 + 0.002683451583094278im, 0.002637546015038562 - 0.007447965191034699im, 10560.476499823517 - 1467.3012857066146im, -3.095744186967231e-5 - 4.456275503003482e-5im, -0.14809902527249308 + 0.23424096202159628im, 75369.0254296317 + 38934.014044443764im, 0.0017555240655655335 - 0.001106230255485918im, 0.14465398759464673 + 0.08519039904992252im, 0.0037511425684028077 + 0.29740145794976im, -0.025432990545857462 - 0.010671703350796589im, 85407.3529308972 + 470087.5923765763im, -31.991039227457257 - 30.327905761770378im, 0.000916958921245299 + 1.7492318891004739e-6im, 0.004130626499872159 + 0.00197966790586771im, 85712.69632649605 - 2.783167017494447e6im, -0.0009600441493114725 - 0.0011410461205073085im, 50.799226656768944 - 2748.04019456035im, -51.15868229564822 + 38.069416017053825im, -0.15963585095122673 - 0.14512735272712549im, 656407.9883880756 + 2.422673620789157e6im, -6.658755936419926 + 4.134284890354727im, 4.5925340940967046e-5 + 3.788300636038225e-5im, 1.6717506496199008 + 2.2221923636997896im, 940687.8706080102 + 2.751791684213872e6im, -1877.0481075341086 + 22928.489283234027im, -0.038920788309846986 - 0.04509350659501414im, 0.24450561982126318 + 0.07810557149423096im, 1567.5783929724387 + 11862.52410931837im, 0.012005429010318227 - 0.02297081949365693im, -0.013421583720940494 - 0.004803127379407543im, -0.019033968561341357 + 0.000631223817863445im, 55295.89124463145 + 6740.020094133191im, 0.0002551546386793933 - 0.0002859253250508869im, -0.002636213985506195 + 0.0025607001644823405im, -15.023732164451083 - 44.19435641344513im, -0.00025391088698225553 - 0.00018572076197385563im, 7525.509431310812 + 12151.276146144994im, -0.4341922640387619 - 1.5902020278720301im, -20051.263595654418 + 1995.853221858759im, -0.31825339098921046 - 0.10499246880309132im, 0.02300378817791373 - 0.07741290006365245im, -0.05713821781061089 + 0.06385706752273053im, -0.0003502262191962101 - 1.8934923983836158e-5im, -1.010114494126819 + 2.9987864270228797im, 7339.3546634409895 - 4126.171861530353im, 0.694758494392984 - 2.2161604407965547im, -0.053878719943814046 + 0.23528927170422687im, 0.0016468481878420063 - 0.0016171051600341955im, 0.03163948086029663 - 0.004619442788337659im, -191856.04710216835 - 570626.1133265463im, -0.14451969419707317 - 0.005300566718325074im, 81486.32956262786 - 162002.7304682272im, 9.438514135794877 - 6.3199010450035775im, 285.1849990652621 + 369.38838846215504im, -137300.28699585414 + 28931.575797021298im, 362182.20820944436 + 1.0212497733460643e6im, 4904.234467592263 + 15150.814739841733im, -323623.83059668884 + 544765.7452887103im, -42.22389775218731 - 32.191597439485825im, 0.0018926890361006836 + 0.0006728403559828019im, 5.180109206721071e-6 + 5.9933413519461305e-5im, -28.305489018144346 + 56.7097704095492im, 0.0038682196182607104 + 0.43544452946304807im, -111.29244740084053 - 0.9216475609178916im, -0.4194418821612568 + 0.2064698303323375im, 207.71998122546375 + 533.4606434920877im, -1600.7887842125976 + 769.1016961769956im, -0.15318376771089137 - 0.05331892522477287im, 11936.53119289084 + 111835.26047033673im, 12128.373690025977 + 1971.9357607247im, 443580.06995126535 - 531236.1626410707im, 0.031975295612073656 + 0.04968151241688353im, -0.016534543248573248 - 0.03600611615934087im, -0.024019424099834232 - 0.10906417547650298im, 1059.365892128607 - 57.261285205149335im, -0.0036687783546047376 - 0.00022916479663408703im, 0.00806379533604528 - 0.016149966546914918im, 0.3488009414492556 - 0.3406504906556247im, 10372.244002895437 + 17861.144776437242im, 425962.72620403947 + 925899.1924033904im, 2388.5070294955012 + 229.34653460396913im, -0.5492665665717658 + 0.31637531122290724im, 28524.410421122404 + 145204.50393559248im, 0.009154689622756215 + 0.012427375804957813im, 1.6775422803477512e6 - 568834.7207671275im, -42980.563533413566 + 8142.590365140961im, -0.0008645249308521061 + 0.000980669643415004im, -259.6339386469725 - 60.14873975542695im, 0.014698267534705095 - 0.0002449786955401577im, -106482.70508193965 - 47434.80649148568im, 0.05724892769683904 - 0.011679293637887245im, -8.619491526232522 - 1.3887072258995061im, 0.17529541415136488 - 0.23302523936964542im, 42.515741270662 - 85.2868792130808im, -3.088169597375103e-5 + 6.813138719847177e-5im, -468.5204287767817 + 81.21286235090714im, 206.0366461492981 - 533.0871580548867im, 1.1653615882706856e6 - 2.5029650800275235e6im, -0.011328848284755085 - 0.017503827913642453im, -47.63724434666232 - 15276.100846790563im, -14764.567022745674 + 871.2460372392112im, -0.02633088399522823 + 0.5553185384603425im, 0.009511480764735106 - 0.01932811786525881im, -393.0626910403413 + 1022.4230074061436im, 11.62402411004559 - 14.2598913469714im, 1849.8408639229465 + 2291.102078508775im, 13.83236334911497 + 62.51547362952426im, -8241.348475987978 + 22275.533870393374im, -47.35512677268483 + 58.07920903845608im, 0.11588058912183748 - 0.27090478359963605im, 0.0012274964873045953 - 0.0066465330966636186im, -15576.66589875276 + 4541.871578846958im, -28959.965637172478 - 37004.11186225787im, -0.22035738528918408 + 0.14593183577059718im, -11165.772792029973 - 4853.319934862472im, -0.003054514904389196 + 0.001428219541062602im, 0.0443191146393288 - 0.00499831131669166im, 18.240693496278023 - 21.641873150197327im, -0.1433209459592103 + 0.4014155050245423im, 396.7073612548034 + 1334.305423604222im, -0.02878452228989258 + 0.04079056212230407im, 142004.10971716506 - 146280.59107096883im, 0.010546500688335805 - 0.00789315501937964im, 39835.39202144819 + 12493.163792009991im, 0.001897330182495805 - 0.0012400750064291148im, -181722.61409264608 + 376384.544617873im, 117.80273946243129 + 186.42889560433372im, -30.58037745580521 - 20.784488503536593im, 9.361684414784484e-5 - 0.00015687754655726473im, 0.24038133248349158 - 0.06813904434751981im, 0.0016294860088156729 + 0.0014030156428294146im, -86.89476009975759 - 28.108421013042783im, -5405.023520128045 + 362.8859155673847im, -0.23557724837241384 - 0.10453432544108202im, -0.2631208716674161 - 0.6563246624884234im, -25672.914629162533 + 47365.61885552393im, -178125.5431598209 - 1.5107503983163633e6im, -0.0027550933064815165 + 0.0009320724661870437im]
        sympy_complex_outputs = Complex{Float64}[0.000347311641703001 - 6.196870673288563e-5im, 0.0005093217922380722 + 0.0007432392102401933im, -0.005782779590804852 - 0.2599852449912216im, -1.0906435952705433 - 1.082532748835804im, -Inf + Inf*im, 2.552498725836081e-291 + 6.168018593351761e-291im, -Inf - Inf*im, -1106.9439074258312 - 1926.5835470017116im, Inf + Inf*im, -0.0 + 0.0im, -0.0 + 0.0im, -0.00019679683439802184 - 0.0012137969061975847im, 4.325636709158466 - 1.828592666515844im, -7.030649478070732e63 + 2.2767178257523656e64im, Inf - Inf*im, 0.00039649702905695237 + 8.986082525245331e-5im, 0.01611263557121976 - 0.005526916933931195im, 0.0 - 0.0im, -0.0 + 0.0im, 2.7236527114453656e-5 - 3.4592978986141784e-5im, Inf + Inf*im, -Inf + Inf*im, -0.0 - 0.0im, -Inf + Inf*im, -1.1152748960360749e25 - 1.3984616814023706e25im, Inf - Inf*im, 0.014426702870409059 - 0.02197817989993727im, 89.6347469483598 + 192.97167369676566im, -4.092421386299666e10 + 5.2574320183550684e11im, -0.0 + 0.0im, -1.6268699886321503e-7 + 7.415365350263168e-10im, 6.772467544196321e-7 + 7.664942552902726e-7im, -Inf - Inf*im, 1.3249573432390717e244 + 5.8465264954706e244im, -Inf - Inf*im, 0.0 + 0.0im, -Inf + Inf*im, -0.0 - 0.0im, 0.05045086418396209 + 0.7562207519676277im, 3038.2136397986906 - 3409.6017506063918im, Inf - Inf*im, Inf + Inf*im, -Inf - Inf*im, 0.011493596836882043 + 0.635153175912361im, -5.534908352449168 + 21.335527589395216im, -5.792316370726783e8 - 8.581790172358298e8im, -280.7144972853392 + 640.3903534829549im, -30.524934134503994 - 320.6030693266838im, Inf + Inf*im, Inf + Inf*im, 2.3805846038477596e-58 + 2.18918469308163e-58im, Inf - Inf*im, 161602.03777277056 - 1.0341937071643792e6im, 2.616142274214311e-5 + 1.3333007788212049e-6im, -1.5785891632631261 - 0.9549043215694846im, -8.76747131517371e289 - 7.934555238866346e290im, 1.8959190891449007e-7 + 8.823150735519941e-7im, -84.92569464907133 + 187.48472486207157im, Inf - Inf*im, 786.3494678284752 + 2367.939712801408im, -0.0 + 0.0im, 1839.4463331780562 + 1707.9415602529832im, 8.390539097824342e-7 + 9.675391395584626e-7im, 5.2460449398784086e42 + 4.1791127009075564e42im, Inf - Inf*im, -10462.92066316738 + 14976.119278818956im, -3.000628974745907 - 2.9222301898428515im, -0.0 + 0.0im, 283.894915188409 + 62.01305137751021im, 8.560675984836002e-5 - 0.00022112990901402868im, -5.641502178883139e-7 - 1.1004430171222618e-6im, Inf + Inf*im, 0.0 + 0.0im, 5.753290358143331e11 + 1.637199449810527e12im, 912.7911451864636 + 837.9573943115561im, Inf - Inf*im, 0.0 + 0.0im, -397.14300889127094 + 356.27072910193806im, -2.4156817000277403e-26 - 2.0303088473241418e-26im, -2.496686598236872e20 - 6.89472863776364e19im, -4.441822381253849 + 2.9967591507081437im, -0.0 + 0.0im, -Inf - Inf*im, 0.06943927005548686 - 0.03300875258943384im, -0.06763592333349325 + 0.0002880659687178219im, -0.0 - 0.0im, -Inf - Inf*im, -11.987399776563475 + 12.679812899149804im, 3.110747094377466 - 1.7030660599974514im, 0.0 - 0.0im, 17.037817787979765 + 33.23628065522699im, -67.12621382018601 + 23.690978327658406im, 11.277939020182458 + 446.6491273271883im, -0.0 + 0.0im, 281.61376901204716 - 358.0982770672371im, 0.00037467313687986834 - 0.000396707920908419im, 2.576419794779388 + 0.1874689391552637im, 8.480833139742938e-7 - 2.956556700377875e-7im, 0.0 - 0.0im, Inf + Inf*im, Inf - Inf*im, 2.346105993331903e-6 - 5.728681887624256e-6im, 0.0001242398605150691 + 3.699553056690425e-5im, 0.007405951646553123 - 0.0036543698960870255im, -2874.505280436935 + 234.84627703734748im, 0.16473814136706316 + 0.8538671532715492im, 0.0 - 0.0im, Inf - Inf*im, -Inf + Inf*im, -Inf - Inf*im, 0.00010666882035159822 - 0.00022751745157940256im, -Inf + Inf*im, 4.1934622024637476e11 - 1.2446095289130148e11im, -0.0 + 0.0im, 2.190336110261323e-6 - 1.3106128603876925e-6im, 2.7526144822191053e-127 + 1.1871037530641815e-127im, Inf - Inf*im, 0.0 - 0.0im, -0.0 + 0.0im, -Inf - Inf*im, -4.069237607357043e16 - 5.121883866683438e15im, -4.638072450018913e11 + 5.910948878313285e11im, 176.50029191710502 + 150.64983404648825im, -Inf - Inf*im, -0.0005209660930780471 - 0.0022181897234975047im, -1.1835604123135578e46 - 1.5342069244861926e46im, -Inf - Inf*im, 8.919877741421664e-94 - 5.964143944574984e-94im, Inf + Inf*im, -5.411685237011981 + 2.9584097712081023im, -0.0 - 0.0im, Inf + Inf*im, 0.0 + 0.0im, 8.16649197467827 - 14.209478203240712im, 0.01109835723203463 + 0.001184519046876223im, -2.959431484942619 + 8.571578007866592im, 0.0 + 0.0im, -272.24395626093826 + 16.771136380484187im, 0.0001825463322217604 - 0.007371111571509799im, 0.5938228333973165 + 1.3182042008203683im, 0.0 + 0.0im, -0.0 + 0.0im, -0.0 + 0.0im, 0.22669089731084147 + 10.045360934157072im, -0.0 - 0.0im, -5.921143641702298e233 + 3.228979035246926e233im, 0.0 - 0.0im, -Inf - Inf*im, Inf - Inf*im, -Inf + Inf*im, 67.03097746148622 + 1.1432641846816967im, Inf - Inf*im, 15.805103315474259 + 3.4106139293045143im, -200.871876164505 - 53.55288993736505im, 1.0698370220445224 + 2.3848782174442884im, Inf - Inf*im, 5.318373194182988e-7 - 1.2649913896164938e-6im, -5.0448344423026584e200 - 3.75752815927432e200im, 5.646971956911101e-93 + 1.253185968841339e-93im, -0.0 + 0.0im, -27.90923119421513 + 38.37096088172996im, -3.1558209616087104e16 - 3.740005073579051e15im, Inf - Inf*im, -Inf - Inf*im, 19.478947488473118 + 41.61876711241857im, 4.534187589809618e167 - 9.249570031426717e166im, 5.314945268141859e-10 + 5.482720464234228e-11im, 0.0 + 0.0im, 7.818331073228955e-9 - 1.321817829337252e-8im, -Inf + Inf*im, -Inf - Inf*im, 0.003703952732548812 - 0.0029593914002336527im, 25.845286319356376 + 145.45103840480738im, Inf + Inf*im, -Inf - Inf*im, -4.266189592222933 - 2.0108908540450066im, Inf - Inf*im, -269.44287039871546 - 125.58188853177215im, 21.387051866126086 + 2.424266507224452im, Inf - Inf*im, 0.026847191191350574 - 0.03705685643220223im, -9.346247859641681e-180 - 1.560490702518522e-179im, -3.1662182044992435 - 10.136374557052884im, -0.0 + 0.0im, -1.1230217118908584e95 + 2.2237610241425842e95im, -0.0 - 0.0im, Inf - Inf*im, Inf + Inf*im, -7.913789764283478e-55 + 6.996414838202153e-56im, -1.1913417296235515e11 - 5.0246687039306903e11im, 6.7445886602781425e-6 + 9.871998524576315e-5im, 6.170640707547878e-5 - 9.8237641482289e-6im, 1.3845464797877178e-6 - 1.886898574839961e-6im, 5.323672526776602e35 - 2.776211500167626e35im, Inf - Inf*im, -5.386643065074706 + 3.425319656351208im, 1.3892521379276805e-5 - 3.234642262315766e-6im, Inf + Inf*im, -Inf - Inf*im, -1.3955825094679286e61 + 1.683354197233064e61im]
        for (ν, z, E) in zip(complex_orders, complex_inputs, sympy_complex_outputs)
            abs(ν) ≤ 50 ? (@test expint(ν, z) ⩭ E) : (@test_throws ArgumentError expint(ν, z) ⩭ E)
        end

        xs = range(2.0, stop=5.0, step=0.05)
        ys = [0.05702612399289205, 0.05308665097847697, 0.04944294741593074, 
            0.04607036967463369, 0.04294659589019342, 0.04005137577047593, 
            0.03736631128005022, 0.03487466387112418, 0.03256118461138553, 
            0.03041196412241836, 0.02841429970928976, 0.02655657745053252, 
            0.02482816734246761, 0.0232193298641301, 0.02172113255824923, 
            0.02032537541727814, 0.01902452402744482, 0.01781164956316038, 
            0.01668037484292225, 0.01562482575941789, 0.01463958748361088, 
            0.01371966491744429, 0.01286044693430924, 0.01205767400216573, 
            0.01130740883247835, 0.01060600974003776, 0.00995010643520129, 
            0.00933657800187881, 0.00876253284236996, 0.00822529039448117, 
            0.007722364447687571, 0.007251447903858146, 0.006810398844575473, 
            0.006397227781646947, 0.006010085980275144, 0.005647254755746198, 
            0.005307135654593942, 0.004988241440163916, 0.004689187810475, 
            0.004408685783377122, 0.004145534690336333, 0.003898615725833893, 
            0.003666886004423625, 0.003449373082020276, 0.00324516990205213, 
            0.003053430130755408, 0.002873363849164126, 0.0027042335722962, 
            0.002545350568691675, 0.002396071455853417, 0.002255795049301819, 
            0.002123959444908601, 0.00200003931594166, 0.001883543407852843, 
            0.001774012215290864, 0.001671015827136935, 0.00157415192655548, 
            0.001483043934137846, 0.001397339283204469, 0.001316707817230031, 
            0.001240840300175119]

        @testset "En_cf_nogamma" begin
            for (x, y) in zip(xs, ys)
                cf, iter = SpecialFunctions.En_cf_nogamma(1 / 2, x)
                @test cf*exp(-x) ≈ y
            end
        end
        @testset "En_cf_gamma" begin
            for (x, y) in zip(xs, ys)
                Γ, cf, iter = SpecialFunctions.En_cf_gamma(1 / 2, x)
                @test Γ + cf*exp(-x) ≈ y
            end
            # type stability
            @test @inferred(SpecialFunctions.En_cf_gamma(1, 1.0 + 2.1im, 1000)) isa Tuple{ComplexF64,ComplexF64,Int}
            @test @inferred(SpecialFunctions.En_cf_gamma(1, 1.0f0, 1000)) isa Tuple{Float32,Float32,Int}
        end
        @testset "En_expand_origin" begin
            for (x, y) in zip(xs, ys)
                res = SpecialFunctions.En_expand_origin(1 / 2, x, 1000)
                @test res ≈ y
            end
        end
    end

    # issue #259: positive vs. negative integer order for small/larger args
    @test expint(-4, 0.7) ≈ 142.685471227614252544 rtol=1e-14
    @test expint(+4, 0.7) ≈ 0.1267808300929215658  rtol=1e-14
    @test expint(-4, 5.7) ≈ 0.0013051784816041558 rtol=1e-14
    @test expint(+4, 5.7) ≈ 0.0003585161998373381 rtol=1e-13
    setprecision(BigFloat, 256) do
        @test expint(-4, big"0.7") ≈ big"142.68547122761425254403329916322251663998829971602071820534499366398779205282" rtol=1e-76
        @test expint(+4, big"0.7") ≈ big"0.12678083009292156580635697928870772316601035828975150623334903100558297079388567" rtol=1e-76
        @test expint(-4, big"5.7") ≈ big"0.0013051784816041557674219259305064008704427556690521012788554962045721241678128274" rtol=1e-76
        @test expint(+4, big"5.7") ≈ big"0.00035851619983733810063957822090566023258844652925986297947736355731994111784745604432" rtol=1e-74
    end


    @testset "expintx" begin
        @test expintx(1e-10) ≅ 22.448635267383787506197038031047821505309344119883144316155006107
        @test expintx(1e-100) ≅ 229.68129363450303554119263337835401832906798952693737400452702286
        @test expintx(1e10) ≅ 9.999999999000000000199999999940000000023999999988000000007e-11
        @test_broken expintx(1e10, 1e10im) ≅ 4.99999999999999999995000000001499999999700000000000000000e-11 - 4.99999999950000000004999999999999999999700000000149999999e-11im
        @test expintx(2, 1e10) ≅ 9.999999998000000000599999999760000000119999999928000000050e-11
        @test expintx(5+5im, 1e10+1e10im) ≅ 4.99999999750000000124999999940000000023624999998587499986e-11 - 4.99999999750000000149999999897500000075624999942337500043e-11im
        @test expintx(2, 1e5+1e5im) ≅ 4.999999998500059998500000003149748011339999937637483913514e-6 - 4.999900001499999998500089996850000011338866062369999513582e-6im

        for ν in (1,2,-2,2+3im,2-3im,-2+3im,-2-3im), z in (0.1+0.2im, -0.1+0.2im, 4+5im, -4+5im, -4-5im, -10+1e-4im, -10-1e-4im)
            @test expintx(ν, z) ≅ exp(z) * expint(ν, z)
        end

        @test expintx(4, 1e300) == 1e-300
    end

    # issue #272: more cases with real results
    @test expint(-2, 2.2) ≈ 0.11696351427428933590118175681379776 rtol=1e-14
    @test expint(-2, -2.2) ≈ -2.0680909972407264331884989 rtol=1e-14
    @test expint(-2.2, 3.2) ≈ 0.024950173497409329191241353395358 rtol=1e-14
    @test expint(+2.2, 3.2) ≈ 0.008044603700773423319087602010 rtol=1e-12
    @test expint(3.2, 1.3) ≈ 0.070147692224611216675759479422283315452559216337905 rtol=1e-13
    @test expint(-20, -1000) === -Inf
end

expinti_real(x) = invoke(expinti, Tuple{Real}, x)

@testset "Ei" begin
    x = exp.(range(log(1e-6), log(300), length=1000))
    setprecision(BigFloat, 256) do
        for x in x
            @test expinti(x) ≈ expinti(big(x))     rtol=1e-13
            @test expinti(x) ≈ expinti_real(x)     rtol=5e-12
            @test expinti(-x) == -expint(x) == expinti_real(-x)
            @test expinti(-x) ≈ expinti(big(-x))   rtol=1e-13
            @test -expint(big(x)) == expinti(big(-x))
        end

        @test expinti(big(3)) ≈ big"9.933832570625416558008336019216765262990653022987582119693034065242496228332697" rtol=1e-76
        @test expinti(big(-3)) ≈ big"-0.013048381094197037412500745828645022948477634080223601176999031140859165434295263061813" rtol=1e-76
    end

    @test expinti(0) === -Inf == expinti(big(0))
    @test expinti(800) === expinti(Inf) === Inf
    @test expinti(NaN) === NaN
    @test expinti(3) == expinti(3.0)
    @test expinti(3.0f0) == Float32(expinti(3.0))
    @test expinti(Float16(3.0)) == Float16(expinti(3.0))

    # check points near the real root
    root = 0.37250741078136663446
    @test abs(expinti(root)) < 1e-15
    Δx = exp.(range(log(1e-6), log(0.2), length=20))
    setprecision(BigFloat, 256) do
        for Δx in Δx
            # error tolerance gets worse as we approach the root
            # because of cancellation errors in computing (root+Δx)-root
            x = root + Δx
            @test expinti(x) ≈ expinti(big(x))    rtol=max(1e-14, 1e-16/Δx)
        end
    end
end
