/*
 * This program source code file is part of KiCad, a free EDA CAD application.
 *
 * Copyright (C) 2024 Jon Evans <jon@craftyjon.com>
 * Copyright The KiCad Developers, see AUTHORS.txt for contributors.
 *
 * This program is free software: you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation, either version 3 of the License, or (at your
 * option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#ifndef KICAD_API_HANDLER_SCH_H
#define KICAD_API_HANDLER_SCH_H

#include <google/protobuf/empty.pb.h>

#include <api/api_handler_editor.h>
#include <api/common/commands/editor_commands.pb.h>
#include <api/schematic/schematic_commands.pb.h>
#include <api/schematic/schematic_types.pb.h>
#include <kiid.h>
#include <memory>
#include <optional>

#include <cpp_sdk/api_cpp_sdk_param.h>

using namespace kiapi;
using namespace kiapi::common;
using namespace kiapi::schematic::commands;

using google::protobuf::Empty;

class SCH_EDIT_FRAME;
class SCH_ITEM;
class API_SCH_DESIGN_RESOURCE_MGR;

struct COMPONENT_CONTENT;
struct PLACE_SYMBOL_OPTION;
struct PLACE_SYMBOL_RESOURCE_CMD;

struct DESIGN_BLOCK_CONTENT;
struct PLACE_DESIGN_BLOCK_OPTION;
struct PLACE_DESIGN_BLOCK_RESOURCE_CMD;
struct API_PLACE_NETLABEL_PARAMS;

class API_HANDLER_SCH : public API_HANDLER_EDITOR
{
public:
    API_HANDLER_SCH( SCH_EDIT_FRAME* aFrame );
    ~API_HANDLER_SCH() override;

protected:
    std::unique_ptr<COMMIT> createCommit() override;

    kiapi::common::types::DocumentType thisDocumentType() const override
    {
        return kiapi::common::types::DOCTYPE_SCHEMATIC;
    }

    bool validateDocumentInternal( const DocumentSpecifier& aDocument ) const override;

    HANDLER_RESULT<std::unique_ptr<EDA_ITEM>> createItemForType( KICAD_T aType, EDA_ITEM* aContainer );

    HANDLER_RESULT<types::ItemRequestStatus> handleCreateUpdateItemsInternal(
            bool aCreate, const std::string& aClientName, const types::ItemHeader& aHeader,
            const google::protobuf::RepeatedPtrField<google::protobuf::Any>&   aItems,
            std::function<void( commands::ItemStatus, google::protobuf::Any )> aItemHandler ) override;

    void deleteItemsInternal( std::map<KIID, ItemDeletionStatus>& aItemsToDelete,
                              const std::string&                  aClientName ) override;

    std::optional<EDA_ITEM*> getItemFromDocument( const DocumentSpecifier& aDocument, const KIID& aId ) override;

private:
    HANDLER_RESULT<commands::GetOpenDocumentsResponse>
    handleGetOpenDocuments( const HANDLER_CONTEXT<commands::GetOpenDocuments>& aCtx );

    HANDLER_RESULT<Empty> handlePlaceDesignBlock( const HANDLER_CONTEXT<PlaceDesignBlock>& aCtx );

    HANDLER_RESULT<Empty> handlePlaceSymbol( const HANDLER_CONTEXT<PlaceSymbol>& aCtx );

    SCH_EDIT_FRAME*                              m_frame;
    std::unique_ptr<API_SCH_DESIGN_RESOURCE_MGR> m_resourceMgr;

public:
    void placeSymbol( COMPONENT_CONTENT const& aComponent, std::optional<PLACE_SYMBOL_OPTION> const& aOption );

    void placeSymbol( PLACE_SYMBOL_RESOURCE_CMD const& aCommand );

    void placeDesignBlock( DESIGN_BLOCK_CONTENT const&                     aDesignBlock,
                           std::optional<PLACE_DESIGN_BLOCK_OPTION> const& aOption );

    void placeDesignBlock( PLACE_DESIGN_BLOCK_RESOURCE_CMD const& aCommand );

    void placeNetlabels( API_PLACE_NETLABEL_PARAMS const& aParams );

};


#endif //KICAD_API_HANDLER_SCH_H
