/*
    Copyright 2016-2026 melonDS team

    This file is part of melonDS.

    melonDS is free software: you can redistribute it and/or modify it under
    the terms of the GNU General Public License as published by the Free
    Software Foundation, either version 3 of the License, or (at your option)
    any later version.

    melonDS is distributed in the hope that it will be useful, but WITHOUT ANY
    WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
    FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.

    You should have received a copy of the GNU General Public License along
    with melonDS. If not, see http://www.gnu.org/licenses/.
*/

#include "ROMInfoDialog.h"
#include "ui_ROMInfoDialog.h"

#include <QFileDialog>

#include "gif-h/gif.h"

#include "NDS.h"
#include "NDSCart.h"
#include "Platform.h"
#include "Config.h"
#include "main.h"

using namespace melonDS;

QString IntToHex(u64 num)
{
    return ("0x" + QString::number(num, 16).toUpper());
}

QString QStringBytes(u64 num)
{
    return (QString::number(num) + " bytes");
}

ROMInfoDialog* ROMInfoDialog::currentDlg = nullptr;

ROMInfoDialog::ROMInfoDialog(QWidget* parent) : QDialog(parent), ui(new Ui::ROMInfoDialog)
{
    ui->setupUi(this);
    setAttribute(Qt::WA_DeleteOnClose);

    emuInstance = ((MainWindow*)parent)->getEmuInstance();

    auto rom = emuInstance->getNDS()->NDSCartSlot.GetCart();
    populateBannerInfo(rom->Banner());
    populateHeaderInfo(rom->GetHeader());
}

void ROMInfoDialog::populateBannerInfo(const NDSBanner* banner)
{
    if (!banner)
    {
        ui->iconImage->clear();
        ui->dsiIconImage->clear();
        ui->iconTitle->clear();
        ui->japaneseTitle->clear();
        ui->englishTitle->clear();
        ui->frenchTitle->clear();
        ui->germanTitle->clear();
        ui->italianTitle->clear();
        ui->spanishTitle->clear();
        ui->chineseTitle->clear();
        ui->koreanTitle->clear();
        ui->titles->setEnabled(false); // grey out the group box
        ui->saveIconButton->setEnabled(false);
        ui->saveAnimatedIconButton->setEnabled(false);
        return;
    }

    u32 iconData[32 * 32];
    emuInstance->romIcon(banner->Icon, banner->Palette, iconData);
    iconImage = QImage(reinterpret_cast<u8*>(iconData), 32, 32, QImage::Format_RGBA8888).copy();
    ui->iconImage->setPixmap(QPixmap::fromImage(iconImage));

    if (banner->Version == 0x103)
    {
        ui->saveAnimatedIconButton->setEnabled(true);

        emuInstance->animatedROMIcon(banner->DSiIcon, banner->DSiPalette, banner->DSiSequence, animatedIconData, animatedSequence);

        for (u32* image: animatedIconData)
        {
            if (!image)
                break;
            animatedIconImages.push_back(QPixmap::fromImage(QImage(reinterpret_cast<u8*>(image), 32, 32, QImage::Format_RGBA8888).copy()));
        }
        iconTimeline = new QTimeLine(animatedSequence.size() / 60 * 1000, this);
        iconTimeline->setFrameRange(0, animatedSequence.size() - 1);
        iconTimeline->setLoopCount(0);
        iconTimeline->setEasingCurve(QEasingCurve::Linear);
        connect(iconTimeline, &QTimeLine::frameChanged, this, &ROMInfoDialog::iconSetFrame);
        iconTimeline->start();
    }
    else
    {
        ui->dsiIconImage->setPixmap(QPixmap::fromImage(iconImage));
    }

    // It seems that setting line-height is not supported in QSS, but is supported in the subset of
    // CSS implemented in rich text areas. So this will have to be here instead of in the .ui file.
    ui->iconTitle->setText(
        "<div style='line-height: 85%;'>" +
        QString::fromUtf16(banner->EnglishTitle).replace('\n', "<br>") +
        "</div>"
    );

    ui->japaneseTitle->setText(QString::fromUtf16(banner->JapaneseTitle));
    ui->englishTitle->setText(QString::fromUtf16(banner->EnglishTitle));
    ui->frenchTitle->setText(QString::fromUtf16(banner->FrenchTitle));
    ui->germanTitle->setText(QString::fromUtf16(banner->GermanTitle));
    ui->italianTitle->setText(QString::fromUtf16(banner->ItalianTitle));
    ui->spanishTitle->setText(QString::fromUtf16(banner->SpanishTitle));

    if (banner->Version > 1)
        ui->chineseTitle->setText(QString::fromUtf16(banner->ChineseTitle));
    else
        ui->chineseTitle->setText("None");

    if (banner->Version > 2)
        ui->koreanTitle->setText(QString::fromUtf16(banner->KoreanTitle));
    else
        ui->koreanTitle->setText("None");
}

void ROMInfoDialog::populateHeaderInfo(const NDSHeader& header)
{
    ui->gameTitle->setText(QString::fromLatin1(header.GameTitle, 12));
    ui->gameCode->setText(QString::fromLatin1(header.GameCode, 4));
    ui->makerCode->setText(QString::fromLatin1(header.MakerCode, 2));
    ui->cardSize->setText(QString::number(128 << header.CardSize) + " KB");

    ui->arm9RomOffset->setText(IntToHex(header.ARM9ROMOffset));
    ui->arm9EntryAddress->setText(IntToHex(header.ARM9EntryAddress));
    ui->arm9RamAddress->setText(IntToHex(header.ARM9RAMAddress));
    ui->arm9Size->setText(QStringBytes(header.ARM9Size));

    ui->arm7RomOffset->setText(IntToHex(header.ARM7ROMOffset));
    ui->arm7EntryAddress->setText(IntToHex(header.ARM7EntryAddress));
    ui->arm7RamAddress->setText(IntToHex(header.ARM7RAMAddress));
    ui->arm7Size->setText(QStringBytes(header.ARM7Size));

    ui->fntOffset->setText(IntToHex(header.FNTOffset));
    ui->fntSize->setText(QStringBytes(header.FNTSize));
    ui->fatOffset->setText(IntToHex(header.FATOffset));
    ui->fatSize->setText(QStringBytes(header.FATSize));
}

ROMInfoDialog::~ROMInfoDialog()
{
    delete ui;
}

void ROMInfoDialog::done(int r)
{
    QDialog::done(r);

    closeDlg();
}

void ROMInfoDialog::on_saveIconButton_clicked()
{
    QString filename = QFileDialog::getSaveFileName(this,
                                                    "Save Icon",
                                                    emuInstance->getGlobalConfig().GetQString("LastROMFolder"),
                                                    "PNG Images (*.png)");
    if (filename.isEmpty())
        return;

    iconImage.save(filename, "PNG");
}

void ROMInfoDialog::on_saveAnimatedIconButton_clicked()
{
    QString filename = QFileDialog::getSaveFileName(this,
                                                    "Save Animated Icon",
                                                    emuInstance->getGlobalConfig().GetQString("LastROMFolder"),
                                                    "GIF Images (*.gif)");
    if (filename.isEmpty())
        return;


    GifWriter writer;

    // The GIF format only supports delays of 0.01 seconds, so 0.0166... (60fps)
    // is rounded up to 0.02 (50fps)
    GifBegin(&writer, filename.toStdString().c_str(), 32, 32, 2);
    for (int i: animatedSequence)
    {
        if (animatedIconData[i] == 0)
            break;
        GifWriteFrame(&writer, reinterpret_cast<u8*>(animatedIconData[i]), 32, 32, 2);
    }
    GifEnd(&writer);
}

void ROMInfoDialog::iconSetFrame(int frame)
{
    ui->dsiIconImage->setPixmap(animatedIconImages[animatedSequence[frame]]);
}
