/*
 * Copyright 2019-2025 The OpenSSL Project Authors. All Rights Reserved.
 *
 * Licensed under the Apache License 2.0 (the "License").  You may not use
 * this file except in compliance with the License.  You can obtain a copy
 * in the file LICENSE in the source distribution or at
 * https://www.openssl.org/source/license.html
 */
/* clang-format off */

/* clang-format on */

/*
 * Generic dispatch table functions for ciphers.
 */

/* For SSL3_VERSION */
#include <string.h>
#include <openssl/prov_ssl.h>
#include <openssl/proverr.h>
#include "ciphercommon_local.h"
#include "prov/provider_ctx.h"
#include "prov/providercommon.h"
#include "internal/skey.h"
#include "internal/e_os.h"
#include "crypto/types.h"

/*-
 * Generic cipher functions for OSSL_PARAM gettables and settables
 */
/* clang-format off */
/* Machine generated by util/perl/OpenSSL/paramnames.pm */
#ifndef ossl_cipher_generic_get_params_list
static const OSSL_PARAM ossl_cipher_generic_get_params_list[] = {
    OSSL_PARAM_uint(OSSL_CIPHER_PARAM_MODE, NULL),
    OSSL_PARAM_size_t(OSSL_CIPHER_PARAM_KEYLEN, NULL),
    OSSL_PARAM_size_t(OSSL_CIPHER_PARAM_IVLEN, NULL),
    OSSL_PARAM_size_t(OSSL_CIPHER_PARAM_BLOCK_SIZE, NULL),
    OSSL_PARAM_int(OSSL_CIPHER_PARAM_AEAD, NULL),
    OSSL_PARAM_int(OSSL_CIPHER_PARAM_CUSTOM_IV, NULL),
    OSSL_PARAM_int(OSSL_CIPHER_PARAM_CTS, NULL),
    OSSL_PARAM_int(OSSL_CIPHER_PARAM_TLS1_MULTIBLOCK, NULL),
    OSSL_PARAM_int(OSSL_CIPHER_PARAM_HAS_RAND_KEY, NULL),
    OSSL_PARAM_int(OSSL_CIPHER_PARAM_ENCRYPT_THEN_MAC, NULL),
    OSSL_PARAM_END
};
#endif

#ifndef ossl_cipher_generic_get_params_st
struct ossl_cipher_generic_get_params_st {
    OSSL_PARAM *aead;
    OSSL_PARAM *bsize;
    OSSL_PARAM *cts;
    OSSL_PARAM *custiv;
    OSSL_PARAM *etm;
    OSSL_PARAM *ivlen;
    OSSL_PARAM *keylen;
    OSSL_PARAM *mb;
    OSSL_PARAM *mode;
    OSSL_PARAM *rand;
};
#endif

#ifndef ossl_cipher_generic_get_params_decoder
static int ossl_cipher_generic_get_params_decoder
    (const OSSL_PARAM *p, struct ossl_cipher_generic_get_params_st *r)
{
    const char *s;

    memset(r, 0, sizeof(*r));
    if (p != NULL)
        for (; (s = p->key) != NULL; p++)
            switch(s[0]) {
            default:
                break;
            case 'a':
                if (ossl_likely(strcmp("ead", s + 1) == 0)) {
                    /* OSSL_CIPHER_PARAM_AEAD */
                    if (ossl_unlikely(r->aead != NULL)) {
                        ERR_raise_data(ERR_LIB_PROV, PROV_R_REPEATED_PARAMETER,
                                       "param %s is repeated", s);
                        return 0;
                    }
                    r->aead = (OSSL_PARAM *)p;
                }
                break;
            case 'b':
                if (ossl_likely(strcmp("locksize", s + 1) == 0)) {
                    /* OSSL_CIPHER_PARAM_BLOCK_SIZE */
                    if (ossl_unlikely(r->bsize != NULL)) {
                        ERR_raise_data(ERR_LIB_PROV, PROV_R_REPEATED_PARAMETER,
                                       "param %s is repeated", s);
                        return 0;
                    }
                    r->bsize = (OSSL_PARAM *)p;
                }
                break;
            case 'c':
                switch(s[1]) {
                default:
                    break;
                case 't':
                    if (ossl_likely(strcmp("s", s + 2) == 0)) {
                        /* OSSL_CIPHER_PARAM_CTS */
                        if (ossl_unlikely(r->cts != NULL)) {
                            ERR_raise_data(ERR_LIB_PROV, PROV_R_REPEATED_PARAMETER,
                                           "param %s is repeated", s);
                            return 0;
                        }
                        r->cts = (OSSL_PARAM *)p;
                    }
                    break;
                case 'u':
                    if (ossl_likely(strcmp("stom-iv", s + 2) == 0)) {
                        /* OSSL_CIPHER_PARAM_CUSTOM_IV */
                        if (ossl_unlikely(r->custiv != NULL)) {
                            ERR_raise_data(ERR_LIB_PROV, PROV_R_REPEATED_PARAMETER,
                                           "param %s is repeated", s);
                            return 0;
                        }
                        r->custiv = (OSSL_PARAM *)p;
                    }
                }
                break;
            case 'e':
                if (ossl_likely(strcmp("ncrypt-then-mac", s + 1) == 0)) {
                    /* OSSL_CIPHER_PARAM_ENCRYPT_THEN_MAC */
                    if (ossl_unlikely(r->etm != NULL)) {
                        ERR_raise_data(ERR_LIB_PROV, PROV_R_REPEATED_PARAMETER,
                                       "param %s is repeated", s);
                        return 0;
                    }
                    r->etm = (OSSL_PARAM *)p;
                }
                break;
            case 'h':
                if (ossl_likely(strcmp("as-randkey", s + 1) == 0)) {
                    /* OSSL_CIPHER_PARAM_HAS_RAND_KEY */
                    if (ossl_unlikely(r->rand != NULL)) {
                        ERR_raise_data(ERR_LIB_PROV, PROV_R_REPEATED_PARAMETER,
                                       "param %s is repeated", s);
                        return 0;
                    }
                    r->rand = (OSSL_PARAM *)p;
                }
                break;
            case 'i':
                if (ossl_likely(strcmp("vlen", s + 1) == 0)) {
                    /* OSSL_CIPHER_PARAM_IVLEN */
                    if (ossl_unlikely(r->ivlen != NULL)) {
                        ERR_raise_data(ERR_LIB_PROV, PROV_R_REPEATED_PARAMETER,
                                       "param %s is repeated", s);
                        return 0;
                    }
                    r->ivlen = (OSSL_PARAM *)p;
                }
                break;
            case 'k':
                if (ossl_likely(strcmp("eylen", s + 1) == 0)) {
                    /* OSSL_CIPHER_PARAM_KEYLEN */
                    if (ossl_unlikely(r->keylen != NULL)) {
                        ERR_raise_data(ERR_LIB_PROV, PROV_R_REPEATED_PARAMETER,
                                       "param %s is repeated", s);
                        return 0;
                    }
                    r->keylen = (OSSL_PARAM *)p;
                }
                break;
            case 'm':
                if (ossl_likely(strcmp("ode", s + 1) == 0)) {
                    /* OSSL_CIPHER_PARAM_MODE */
                    if (ossl_unlikely(r->mode != NULL)) {
                        ERR_raise_data(ERR_LIB_PROV, PROV_R_REPEATED_PARAMETER,
                                       "param %s is repeated", s);
                        return 0;
                    }
                    r->mode = (OSSL_PARAM *)p;
                }
                break;
            case 't':
                if (ossl_likely(strcmp("ls-multi", s + 1) == 0)) {
                    /* OSSL_CIPHER_PARAM_TLS1_MULTIBLOCK */
                    if (ossl_unlikely(r->mb != NULL)) {
                        ERR_raise_data(ERR_LIB_PROV, PROV_R_REPEATED_PARAMETER,
                                       "param %s is repeated", s);
                        return 0;
                    }
                    r->mb = (OSSL_PARAM *)p;
                }
            }
    return 1;
}
#endif
/* End of machine generated */
/* clang-format on */

const OSSL_PARAM *ossl_cipher_generic_gettable_params(ossl_unused void *provctx)
{
    return ossl_cipher_generic_get_params_list;
}

int ossl_cipher_generic_get_params(OSSL_PARAM params[], unsigned int md,
    uint64_t flags,
    size_t kbits, size_t blkbits, size_t ivbits)
{
    struct ossl_cipher_generic_get_params_st p;

    if (!ossl_cipher_generic_get_params_decoder(params, &p))
        return 0;

    if (p.mode != NULL && !OSSL_PARAM_set_uint(p.mode, md)) {
        ERR_raise(ERR_LIB_PROV, PROV_R_FAILED_TO_SET_PARAMETER);
        return 0;
    }
    if (p.aead != NULL
        && !OSSL_PARAM_set_int(p.aead, (flags & PROV_CIPHER_FLAG_AEAD) != 0)) {
        ERR_raise(ERR_LIB_PROV, PROV_R_FAILED_TO_SET_PARAMETER);
        return 0;
    }
    if (p.custiv != NULL
        && !OSSL_PARAM_set_int(p.custiv, (flags & PROV_CIPHER_FLAG_CUSTOM_IV) != 0)) {
        ERR_raise(ERR_LIB_PROV, PROV_R_FAILED_TO_SET_PARAMETER);
        return 0;
    }
    if (p.cts != NULL
        && !OSSL_PARAM_set_int(p.cts, (flags & PROV_CIPHER_FLAG_CTS) != 0)) {
        ERR_raise(ERR_LIB_PROV, PROV_R_FAILED_TO_SET_PARAMETER);
        return 0;
    }
    if (p.mb != NULL
        && !OSSL_PARAM_set_int(p.mb, (flags & PROV_CIPHER_FLAG_TLS1_MULTIBLOCK) != 0)) {
        ERR_raise(ERR_LIB_PROV, PROV_R_FAILED_TO_SET_PARAMETER);
        return 0;
    }
    if (p.rand != NULL
        && !OSSL_PARAM_set_int(p.rand, (flags & PROV_CIPHER_FLAG_RAND_KEY) != 0)) {
        ERR_raise(ERR_LIB_PROV, PROV_R_FAILED_TO_SET_PARAMETER);
        return 0;
    }
    if (p.etm != NULL
        && !OSSL_PARAM_set_int(p.etm, (flags & EVP_CIPH_FLAG_ENC_THEN_MAC) != 0)) {
        ERR_raise(ERR_LIB_PROV, PROV_R_FAILED_TO_SET_PARAMETER);
        return 0;
    }
    if (p.keylen != NULL && !OSSL_PARAM_set_size_t(p.keylen, kbits / 8)) {
        ERR_raise(ERR_LIB_PROV, PROV_R_FAILED_TO_SET_PARAMETER);
        return 0;
    }
    if (p.bsize != NULL && !OSSL_PARAM_set_size_t(p.bsize, blkbits / 8)) {
        ERR_raise(ERR_LIB_PROV, PROV_R_FAILED_TO_SET_PARAMETER);
        return 0;
    }
    if (p.ivlen != NULL && !OSSL_PARAM_set_size_t(p.ivlen, ivbits / 8)) {
        ERR_raise(ERR_LIB_PROV, PROV_R_FAILED_TO_SET_PARAMETER);
        return 0;
    }
    return 1;
}

#define cipher_generic_get_ctx_params_st ossl_cipher_get_ctx_param_list_st

/* clang-format off */
/* Machine generated by util/perl/OpenSSL/paramnames.pm */
#ifndef cipher_generic_get_ctx_params_list
static const OSSL_PARAM cipher_generic_get_ctx_params_list[] = {
    OSSL_PARAM_size_t(OSSL_CIPHER_PARAM_KEYLEN, NULL),
    OSSL_PARAM_size_t(OSSL_CIPHER_PARAM_IVLEN, NULL),
    OSSL_PARAM_uint(OSSL_CIPHER_PARAM_PADDING, NULL),
    OSSL_PARAM_uint(OSSL_CIPHER_PARAM_NUM, NULL),
    OSSL_PARAM_octet_string(OSSL_CIPHER_PARAM_IV, NULL, 0),
    OSSL_PARAM_octet_string(OSSL_CIPHER_PARAM_UPDATED_IV, NULL, 0),
    OSSL_PARAM_octet_string(OSSL_CIPHER_PARAM_TLS_MAC, NULL, 0),
    OSSL_PARAM_END
};
#endif

#ifndef cipher_generic_get_ctx_params_st
struct cipher_generic_get_ctx_params_st {
    OSSL_PARAM *iv;
    OSSL_PARAM *ivlen;
    OSSL_PARAM *keylen;
    OSSL_PARAM *num;
    OSSL_PARAM *pad;
    OSSL_PARAM *tlsmac;
    OSSL_PARAM *updiv;
};
#endif

#ifndef cipher_generic_get_ctx_params_decoder
static int cipher_generic_get_ctx_params_decoder
    (const OSSL_PARAM *p, struct cipher_generic_get_ctx_params_st *r)
{
    const char *s;

    memset(r, 0, sizeof(*r));
    if (p != NULL)
        for (; (s = p->key) != NULL; p++)
            switch(s[0]) {
            default:
                break;
            case 'i':
                switch(s[1]) {
                default:
                    break;
                case 'v':
                    switch(s[2]) {
                    default:
                        break;
                    case 'l':
                        if (ossl_likely(strcmp("en", s + 3) == 0)) {
                            /* OSSL_CIPHER_PARAM_IVLEN */
                            if (ossl_unlikely(r->ivlen != NULL)) {
                                ERR_raise_data(ERR_LIB_PROV, PROV_R_REPEATED_PARAMETER,
                                               "param %s is repeated", s);
                                return 0;
                            }
                            r->ivlen = (OSSL_PARAM *)p;
                        }
                        break;
                    case '\0':
                        if (ossl_unlikely(r->iv != NULL)) {
                            ERR_raise_data(ERR_LIB_PROV, PROV_R_REPEATED_PARAMETER,
                                           "param %s is repeated", s);
                            return 0;
                        }
                        r->iv = (OSSL_PARAM *)p;
                    }
                }
                break;
            case 'k':
                if (ossl_likely(strcmp("eylen", s + 1) == 0)) {
                    /* OSSL_CIPHER_PARAM_KEYLEN */
                    if (ossl_unlikely(r->keylen != NULL)) {
                        ERR_raise_data(ERR_LIB_PROV, PROV_R_REPEATED_PARAMETER,
                                       "param %s is repeated", s);
                        return 0;
                    }
                    r->keylen = (OSSL_PARAM *)p;
                }
                break;
            case 'n':
                if (ossl_likely(strcmp("um", s + 1) == 0)) {
                    /* OSSL_CIPHER_PARAM_NUM */
                    if (ossl_unlikely(r->num != NULL)) {
                        ERR_raise_data(ERR_LIB_PROV, PROV_R_REPEATED_PARAMETER,
                                       "param %s is repeated", s);
                        return 0;
                    }
                    r->num = (OSSL_PARAM *)p;
                }
                break;
            case 'p':
                if (ossl_likely(strcmp("adding", s + 1) == 0)) {
                    /* OSSL_CIPHER_PARAM_PADDING */
                    if (ossl_unlikely(r->pad != NULL)) {
                        ERR_raise_data(ERR_LIB_PROV, PROV_R_REPEATED_PARAMETER,
                                       "param %s is repeated", s);
                        return 0;
                    }
                    r->pad = (OSSL_PARAM *)p;
                }
                break;
            case 't':
                if (ossl_likely(strcmp("ls-mac", s + 1) == 0)) {
                    /* OSSL_CIPHER_PARAM_TLS_MAC */
                    if (ossl_unlikely(r->tlsmac != NULL)) {
                        ERR_raise_data(ERR_LIB_PROV, PROV_R_REPEATED_PARAMETER,
                                       "param %s is repeated", s);
                        return 0;
                    }
                    r->tlsmac = (OSSL_PARAM *)p;
                }
                break;
            case 'u':
                if (ossl_likely(strcmp("pdated-iv", s + 1) == 0)) {
                    /* OSSL_CIPHER_PARAM_UPDATED_IV */
                    if (ossl_unlikely(r->updiv != NULL)) {
                        ERR_raise_data(ERR_LIB_PROV, PROV_R_REPEATED_PARAMETER,
                                       "param %s is repeated", s);
                        return 0;
                    }
                    r->updiv = (OSSL_PARAM *)p;
                }
            }
    return 1;
}
#endif
/* End of machine generated */
/* clang-format on */

const OSSL_PARAM *ossl_cipher_generic_gettable_ctx_params(ossl_unused void *cctx, ossl_unused void *provctx)
{
    return cipher_generic_get_ctx_params_list;
}

#define cipher_generic_set_ctx_params_st ossl_cipher_set_ctx_param_list_st

/* clang-format off */
/* Machine generated by util/perl/OpenSSL/paramnames.pm */
#ifndef cipher_generic_set_ctx_params_list
static const OSSL_PARAM cipher_generic_set_ctx_params_list[] = {
    OSSL_PARAM_uint(OSSL_CIPHER_PARAM_PADDING, NULL),
    OSSL_PARAM_uint(OSSL_CIPHER_PARAM_NUM, NULL),
    OSSL_PARAM_uint(OSSL_CIPHER_PARAM_USE_BITS, NULL),
    OSSL_PARAM_uint(OSSL_CIPHER_PARAM_TLS_VERSION, NULL),
    OSSL_PARAM_size_t(OSSL_CIPHER_PARAM_TLS_MAC_SIZE, NULL),
    OSSL_PARAM_END
};
#endif

#ifndef cipher_generic_set_ctx_params_st
struct cipher_generic_set_ctx_params_st {
    OSSL_PARAM *bits;
    OSSL_PARAM *num;
    OSSL_PARAM *pad;
    OSSL_PARAM *tlsmacsize;
    OSSL_PARAM *tlsvers;
};
#endif

#ifndef cipher_generic_set_ctx_params_decoder
static int cipher_generic_set_ctx_params_decoder
    (const OSSL_PARAM *p, struct cipher_generic_set_ctx_params_st *r)
{
    const char *s;

    memset(r, 0, sizeof(*r));
    if (p != NULL)
        for (; (s = p->key) != NULL; p++)
            switch(s[0]) {
            default:
                break;
            case 'n':
                if (ossl_likely(strcmp("um", s + 1) == 0)) {
                    /* OSSL_CIPHER_PARAM_NUM */
                    if (ossl_unlikely(r->num != NULL)) {
                        ERR_raise_data(ERR_LIB_PROV, PROV_R_REPEATED_PARAMETER,
                                       "param %s is repeated", s);
                        return 0;
                    }
                    r->num = (OSSL_PARAM *)p;
                }
                break;
            case 'p':
                if (ossl_likely(strcmp("adding", s + 1) == 0)) {
                    /* OSSL_CIPHER_PARAM_PADDING */
                    if (ossl_unlikely(r->pad != NULL)) {
                        ERR_raise_data(ERR_LIB_PROV, PROV_R_REPEATED_PARAMETER,
                                       "param %s is repeated", s);
                        return 0;
                    }
                    r->pad = (OSSL_PARAM *)p;
                }
                break;
            case 't':
                switch(s[1]) {
                default:
                    break;
                case 'l':
                    switch(s[2]) {
                    default:
                        break;
                    case 's':
                        switch(s[3]) {
                        default:
                            break;
                        case '-':
                            switch(s[4]) {
                            default:
                                break;
                            case 'm':
                                if (ossl_likely(strcmp("ac-size", s + 5) == 0)) {
                                    /* OSSL_CIPHER_PARAM_TLS_MAC_SIZE */
                                    if (ossl_unlikely(r->tlsmacsize != NULL)) {
                                        ERR_raise_data(ERR_LIB_PROV, PROV_R_REPEATED_PARAMETER,
                                                       "param %s is repeated", s);
                                        return 0;
                                    }
                                    r->tlsmacsize = (OSSL_PARAM *)p;
                                }
                                break;
                            case 'v':
                                if (ossl_likely(strcmp("ersion", s + 5) == 0)) {
                                    /* OSSL_CIPHER_PARAM_TLS_VERSION */
                                    if (ossl_unlikely(r->tlsvers != NULL)) {
                                        ERR_raise_data(ERR_LIB_PROV, PROV_R_REPEATED_PARAMETER,
                                                       "param %s is repeated", s);
                                        return 0;
                                    }
                                    r->tlsvers = (OSSL_PARAM *)p;
                                }
                            }
                        }
                    }
                }
                break;
            case 'u':
                if (ossl_likely(strcmp("se-bits", s + 1) == 0)) {
                    /* OSSL_CIPHER_PARAM_USE_BITS */
                    if (ossl_unlikely(r->bits != NULL)) {
                        ERR_raise_data(ERR_LIB_PROV, PROV_R_REPEATED_PARAMETER,
                                       "param %s is repeated", s);
                        return 0;
                    }
                    r->bits = (OSSL_PARAM *)p;
                }
            }
    return 1;
}
#endif
/* End of machine generated */
/* clang-format on */

const OSSL_PARAM *ossl_cipher_generic_settable_ctx_params(ossl_unused void *cctx, ossl_unused void *provctx)
{
    return cipher_generic_set_ctx_params_list;
}

/*
 * Variable key length cipher functions for OSSL_PARAM settables
 */
#define cipher_var_keylen_set_ctx_params_st ossl_cipher_set_ctx_param_list_st

/* clang-format off */
/* Machine generated by util/perl/OpenSSL/paramnames.pm */
#ifndef cipher_var_keylen_set_ctx_params_list
static const OSSL_PARAM cipher_var_keylen_set_ctx_params_list[] = {
    OSSL_PARAM_uint(OSSL_CIPHER_PARAM_PADDING, NULL),
    OSSL_PARAM_uint(OSSL_CIPHER_PARAM_NUM, NULL),
    OSSL_PARAM_uint(OSSL_CIPHER_PARAM_USE_BITS, NULL),
    OSSL_PARAM_uint(OSSL_CIPHER_PARAM_TLS_VERSION, NULL),
    OSSL_PARAM_size_t(OSSL_CIPHER_PARAM_TLS_MAC_SIZE, NULL),
    OSSL_PARAM_size_t(OSSL_CIPHER_PARAM_KEYLEN, NULL),
    OSSL_PARAM_END
};
#endif

#ifndef cipher_var_keylen_set_ctx_params_st
struct cipher_var_keylen_set_ctx_params_st {
    OSSL_PARAM *bits;
    OSSL_PARAM *keylen;
    OSSL_PARAM *num;
    OSSL_PARAM *pad;
    OSSL_PARAM *tlsmacsize;
    OSSL_PARAM *tlsvers;
};
#endif

#ifndef cipher_var_keylen_set_ctx_params_decoder
static int cipher_var_keylen_set_ctx_params_decoder
    (const OSSL_PARAM *p, struct cipher_var_keylen_set_ctx_params_st *r)
{
    const char *s;

    memset(r, 0, sizeof(*r));
    if (p != NULL)
        for (; (s = p->key) != NULL; p++)
            switch(s[0]) {
            default:
                break;
            case 'k':
                if (ossl_likely(strcmp("eylen", s + 1) == 0)) {
                    /* OSSL_CIPHER_PARAM_KEYLEN */
                    if (ossl_unlikely(r->keylen != NULL)) {
                        ERR_raise_data(ERR_LIB_PROV, PROV_R_REPEATED_PARAMETER,
                                       "param %s is repeated", s);
                        return 0;
                    }
                    r->keylen = (OSSL_PARAM *)p;
                }
                break;
            case 'n':
                if (ossl_likely(strcmp("um", s + 1) == 0)) {
                    /* OSSL_CIPHER_PARAM_NUM */
                    if (ossl_unlikely(r->num != NULL)) {
                        ERR_raise_data(ERR_LIB_PROV, PROV_R_REPEATED_PARAMETER,
                                       "param %s is repeated", s);
                        return 0;
                    }
                    r->num = (OSSL_PARAM *)p;
                }
                break;
            case 'p':
                if (ossl_likely(strcmp("adding", s + 1) == 0)) {
                    /* OSSL_CIPHER_PARAM_PADDING */
                    if (ossl_unlikely(r->pad != NULL)) {
                        ERR_raise_data(ERR_LIB_PROV, PROV_R_REPEATED_PARAMETER,
                                       "param %s is repeated", s);
                        return 0;
                    }
                    r->pad = (OSSL_PARAM *)p;
                }
                break;
            case 't':
                switch(s[1]) {
                default:
                    break;
                case 'l':
                    switch(s[2]) {
                    default:
                        break;
                    case 's':
                        switch(s[3]) {
                        default:
                            break;
                        case '-':
                            switch(s[4]) {
                            default:
                                break;
                            case 'm':
                                if (ossl_likely(strcmp("ac-size", s + 5) == 0)) {
                                    /* OSSL_CIPHER_PARAM_TLS_MAC_SIZE */
                                    if (ossl_unlikely(r->tlsmacsize != NULL)) {
                                        ERR_raise_data(ERR_LIB_PROV, PROV_R_REPEATED_PARAMETER,
                                                       "param %s is repeated", s);
                                        return 0;
                                    }
                                    r->tlsmacsize = (OSSL_PARAM *)p;
                                }
                                break;
                            case 'v':
                                if (ossl_likely(strcmp("ersion", s + 5) == 0)) {
                                    /* OSSL_CIPHER_PARAM_TLS_VERSION */
                                    if (ossl_unlikely(r->tlsvers != NULL)) {
                                        ERR_raise_data(ERR_LIB_PROV, PROV_R_REPEATED_PARAMETER,
                                                       "param %s is repeated", s);
                                        return 0;
                                    }
                                    r->tlsvers = (OSSL_PARAM *)p;
                                }
                            }
                        }
                    }
                }
                break;
            case 'u':
                if (ossl_likely(strcmp("se-bits", s + 1) == 0)) {
                    /* OSSL_CIPHER_PARAM_USE_BITS */
                    if (ossl_unlikely(r->bits != NULL)) {
                        ERR_raise_data(ERR_LIB_PROV, PROV_R_REPEATED_PARAMETER,
                                       "param %s is repeated", s);
                        return 0;
                    }
                    r->bits = (OSSL_PARAM *)p;
                }
            }
    return 1;
}
#endif
/* End of machine generated */
/* clang-format on */

const OSSL_PARAM *ossl_cipher_var_keylen_settable_ctx_params(ossl_unused void *cctx, ossl_unused void *provctx)
{
    return cipher_var_keylen_set_ctx_params_list;
}

int ossl_cipher_var_keylen_set_ctx_params(void *vctx, const OSSL_PARAM params[])
{
    PROV_CIPHER_CTX *ctx = (PROV_CIPHER_CTX *)vctx;
    struct ossl_cipher_set_ctx_param_list_st p;

    if (ctx == NULL
        || !cipher_var_keylen_set_ctx_params_decoder(params, &p)
        || !ossl_cipher_common_set_ctx_params(ctx, &p))
        return 0;

    if (p.keylen != NULL) {
        size_t keylen;

        if (!OSSL_PARAM_get_size_t(p.keylen, &keylen)) {
            ERR_raise(ERR_LIB_PROV, PROV_R_FAILED_TO_GET_PARAMETER);
            return 0;
        }
        if (ctx->keylen != keylen) {
            ctx->keylen = keylen;
            ctx->key_set = 0;
        }
    }
    return 1;
}

void ossl_cipher_generic_reset_ctx(PROV_CIPHER_CTX *ctx)
{
    if (ctx != NULL && ctx->alloced) {
        OPENSSL_free(ctx->tlsmac);
        ctx->alloced = 0;
        ctx->tlsmac = NULL;
    }
}

static int cipher_generic_init_internal(PROV_CIPHER_CTX *ctx,
    const unsigned char *key, size_t keylen,
    const unsigned char *iv, size_t ivlen,
    const OSSL_PARAM params[], int enc)
{
    ctx->num = 0;
    ctx->bufsz = 0;
    ctx->updated = 0;
    ctx->enc = enc ? 1 : 0;

    if (!ossl_prov_is_running())
        return 0;

    if (iv != NULL && ctx->mode != EVP_CIPH_ECB_MODE) {
        if (!ossl_cipher_generic_initiv(ctx, iv, ivlen))
            return 0;
    }
    if (iv == NULL && ctx->iv_set
        && (ctx->mode == EVP_CIPH_CBC_MODE
            || ctx->mode == EVP_CIPH_CFB_MODE
            || ctx->mode == EVP_CIPH_OFB_MODE))
        /* reset IV for these modes to keep compatibility with 1.1.1 */
        memcpy(ctx->iv, ctx->oiv, ctx->ivlen);

    if (key != NULL) {
        if (ctx->variable_keylength == 0) {
            if (keylen != ctx->keylen) {
                ERR_raise(ERR_LIB_PROV, PROV_R_INVALID_KEY_LENGTH);
                return 0;
            }
        } else {
            ctx->keylen = keylen;
        }
        if (!ctx->hw->init(ctx, key, ctx->keylen))
            return 0;
        ctx->key_set = 1;
    }
    return ossl_cipher_generic_set_ctx_params(ctx, params);
}

int ossl_cipher_generic_einit(void *vctx, const unsigned char *key,
    size_t keylen, const unsigned char *iv,
    size_t ivlen, const OSSL_PARAM params[])
{
    return cipher_generic_init_internal((PROV_CIPHER_CTX *)vctx, key, keylen,
        iv, ivlen, params, 1);
}

int ossl_cipher_generic_dinit(void *vctx, const unsigned char *key,
    size_t keylen, const unsigned char *iv,
    size_t ivlen, const OSSL_PARAM params[])
{
    return cipher_generic_init_internal((PROV_CIPHER_CTX *)vctx, key, keylen,
        iv, ivlen, params, 0);
}

int ossl_cipher_generic_skey_einit(void *vctx, void *skeydata,
    const unsigned char *iv, size_t ivlen,
    const OSSL_PARAM params[])
{
    PROV_SKEY *key = skeydata;

    return cipher_generic_init_internal((PROV_CIPHER_CTX *)vctx,
        key->data, key->length,
        iv, ivlen, params, 1);
}

int ossl_cipher_generic_skey_dinit(void *vctx, void *skeydata,
    const unsigned char *iv, size_t ivlen,
    const OSSL_PARAM params[])
{
    PROV_SKEY *key = skeydata;

    return cipher_generic_init_internal((PROV_CIPHER_CTX *)vctx,
        key->data, key->length,
        iv, ivlen, params, 0);
}

/* Max padding including padding length byte */
#define MAX_PADDING 256

int ossl_cipher_generic_block_update(void *vctx, unsigned char *out,
    size_t *outl, size_t outsize,
    const unsigned char *in, size_t inl)
{
    size_t outlint = 0;
    PROV_CIPHER_CTX *ctx = (PROV_CIPHER_CTX *)vctx;
    size_t blksz = ctx->blocksize;
    size_t nextblocks;

    if (!ctx->key_set) {
        ERR_raise(ERR_LIB_PROV, PROV_R_NO_KEY_SET);
        return 0;
    }

    if (ctx->tlsversion > 0) {
        /*
         * Each update call corresponds to a TLS record and is individually
         * padded
         */

        /* Sanity check inputs */
        if (in == NULL
            || in != out
            || outsize < inl
            || !ctx->pad) {
            ERR_raise(ERR_LIB_PROV, PROV_R_CIPHER_OPERATION_FAILED);
            return 0;
        }

        if (ctx->enc) {
            unsigned char padval;
            size_t padnum, loop;

            /* Add padding */

            padnum = blksz - (inl % blksz);

            if (outsize < inl + padnum) {
                ERR_raise(ERR_LIB_PROV, PROV_R_CIPHER_OPERATION_FAILED);
                return 0;
            }

            if (padnum > MAX_PADDING) {
                ERR_raise(ERR_LIB_PROV, PROV_R_CIPHER_OPERATION_FAILED);
                return 0;
            }
            padval = (unsigned char)(padnum - 1);
            if (ctx->tlsversion == SSL3_VERSION) {
                if (padnum > 1)
                    memset(out + inl, 0, padnum - 1);
                *(out + inl + padnum - 1) = padval;
            } else {
                /* we need to add 'padnum' padding bytes of value padval */
                for (loop = inl; loop < inl + padnum; loop++)
                    out[loop] = padval;
            }
            inl += padnum;
        }

        if ((inl % blksz) != 0) {
            ERR_raise(ERR_LIB_PROV, PROV_R_CIPHER_OPERATION_FAILED);
            return 0;
        }

        /* Shouldn't normally fail */
        if (!ctx->hw->cipher(ctx, out, in, inl)) {
            ERR_raise(ERR_LIB_PROV, PROV_R_CIPHER_OPERATION_FAILED);
            return 0;
        }

        if (ctx->alloced) {
            OPENSSL_free(ctx->tlsmac);
            ctx->alloced = 0;
            ctx->tlsmac = NULL;
        }

        /* This only fails if padding is publicly invalid */
        *outl = inl;
        if (!ctx->enc
            && !ossl_cipher_tlsunpadblock(ctx->libctx, ctx->tlsversion,
                out, outl,
                blksz, &ctx->tlsmac, &ctx->alloced,
                ctx->tlsmacsize, 0)) {
            ERR_raise(ERR_LIB_PROV, PROV_R_CIPHER_OPERATION_FAILED);
            return 0;
        }
        return 1;
    }

    if (ctx->bufsz != 0)
        nextblocks = ossl_cipher_fillblock(ctx->buf, &ctx->bufsz, blksz,
            &in, &inl);
    else
        nextblocks = inl & ~(blksz - 1);

    /*
     * If we're decrypting and we end an update on a block boundary we hold
     * the last block back in case this is the last update call and the last
     * block is padded.
     */
    if (ctx->bufsz == blksz && (ctx->enc || inl > 0 || !ctx->pad)) {
        if (outsize < blksz) {
            ERR_raise(ERR_LIB_PROV, PROV_R_OUTPUT_BUFFER_TOO_SMALL);
            return 0;
        }
        if (!ctx->hw->cipher(ctx, out, ctx->buf, blksz)) {
            ERR_raise(ERR_LIB_PROV, PROV_R_CIPHER_OPERATION_FAILED);
            return 0;
        }
        ctx->bufsz = 0;
        outlint = blksz;
        out += blksz;
    }
    if (nextblocks > 0) {
        if (!ctx->enc && ctx->pad && nextblocks == inl) {
            if (!ossl_assert(inl >= blksz)) {
                ERR_raise(ERR_LIB_PROV, PROV_R_OUTPUT_BUFFER_TOO_SMALL);
                return 0;
            }
            nextblocks -= blksz;
        }
        outlint += nextblocks;
        if (outsize < outlint) {
            ERR_raise(ERR_LIB_PROV, PROV_R_OUTPUT_BUFFER_TOO_SMALL);
            return 0;
        }
    }
    if (nextblocks > 0) {
        if (!ctx->hw->cipher(ctx, out, in, nextblocks)) {
            ERR_raise(ERR_LIB_PROV, PROV_R_CIPHER_OPERATION_FAILED);
            return 0;
        }
        in += nextblocks;
        inl -= nextblocks;
    }
    if (inl != 0
        && !ossl_cipher_trailingdata(ctx->buf, &ctx->bufsz, blksz, &in, &inl)) {
        /* ERR_raise already called */
        return 0;
    }

    *outl = outlint;
    return inl == 0;
}

int ossl_cipher_generic_block_final(void *vctx, unsigned char *out,
    size_t *outl, size_t outsize)
{
    PROV_CIPHER_CTX *ctx = (PROV_CIPHER_CTX *)vctx;
    size_t blksz = ctx->blocksize;

    if (!ossl_prov_is_running())
        return 0;

    if (!ctx->key_set) {
        ERR_raise(ERR_LIB_PROV, PROV_R_NO_KEY_SET);
        return 0;
    }

    if (ctx->tlsversion > 0) {
        /* We never finalize TLS, so this is an error */
        ERR_raise(ERR_LIB_PROV, PROV_R_CIPHER_OPERATION_FAILED);
        return 0;
    }

    if (ctx->enc) {
        if (ctx->pad) {
            ossl_cipher_padblock(ctx->buf, &ctx->bufsz, blksz);
        } else if (ctx->bufsz == 0) {
            *outl = 0;
            return 1;
        } else if (ctx->bufsz != blksz) {
            ERR_raise(ERR_LIB_PROV, PROV_R_WRONG_FINAL_BLOCK_LENGTH);
            return 0;
        }

        if (outsize < blksz) {
            ERR_raise(ERR_LIB_PROV, PROV_R_OUTPUT_BUFFER_TOO_SMALL);
            return 0;
        }
        if (!ctx->hw->cipher(ctx, out, ctx->buf, blksz)) {
            ERR_raise(ERR_LIB_PROV, PROV_R_CIPHER_OPERATION_FAILED);
            return 0;
        }
        ctx->bufsz = 0;
        *outl = blksz;
        return 1;
    }

    /* Decrypting */
    if (ctx->bufsz != blksz) {
        if (ctx->bufsz == 0 && !ctx->pad) {
            *outl = 0;
            return 1;
        }
        ERR_raise(ERR_LIB_PROV, PROV_R_WRONG_FINAL_BLOCK_LENGTH);
        return 0;
    }

    if (!ctx->hw->cipher(ctx, ctx->buf, ctx->buf, blksz)) {
        ERR_raise(ERR_LIB_PROV, PROV_R_CIPHER_OPERATION_FAILED);
        return 0;
    }

    if (ctx->pad && !ossl_cipher_unpadblock(ctx->buf, &ctx->bufsz, blksz)) {
        /* ERR_raise already called */
        return 0;
    }

    if (outsize < ctx->bufsz) {
        ERR_raise(ERR_LIB_PROV, PROV_R_OUTPUT_BUFFER_TOO_SMALL);
        return 0;
    }
    memcpy(out, ctx->buf, ctx->bufsz);
    *outl = ctx->bufsz;
    ctx->bufsz = 0;
    return 1;
}

int ossl_cipher_generic_stream_update(void *vctx, unsigned char *out,
    size_t *outl, size_t outsize,
    const unsigned char *in, size_t inl)
{
    PROV_CIPHER_CTX *ctx = (PROV_CIPHER_CTX *)vctx;

    if (!ctx->key_set) {
        ERR_raise(ERR_LIB_PROV, PROV_R_NO_KEY_SET);
        return 0;
    }

    if (inl == 0) {
        *outl = 0;
        return 1;
    }

    if (outsize < inl) {
        ERR_raise(ERR_LIB_PROV, PROV_R_OUTPUT_BUFFER_TOO_SMALL);
        return 0;
    }

    if (!ctx->hw->cipher(ctx, out, in, inl)) {
        ERR_raise(ERR_LIB_PROV, PROV_R_CIPHER_OPERATION_FAILED);
        return 0;
    }

    *outl = inl;
    if (!ctx->enc && ctx->tlsversion > 0) {
        /*
         * Remove any TLS padding. Only used by cipher_aes_cbc_hmac_sha1_hw.c and
         * cipher_aes_cbc_hmac_sha256_hw.c
         */
        if (ctx->removetlspad) {
            /*
             * We should have already failed in the cipher() call above if this
             * isn't true.
             */
            if (!ossl_assert(*outl >= (size_t)(out[inl - 1] + 1)))
                return 0;
            /* The actual padding length */
            *outl -= out[inl - 1] + 1;
        }

        /* TLS MAC and explicit IV if relevant. We should have already failed
         * in the cipher() call above if *outl is too short.
         */
        if (!ossl_assert(*outl >= ctx->removetlsfixed))
            return 0;
        *outl -= ctx->removetlsfixed;

        /* Extract the MAC if there is one */
        if (ctx->tlsmacsize > 0) {
            if (*outl < ctx->tlsmacsize)
                return 0;

            ctx->tlsmac = out + *outl - ctx->tlsmacsize;
            *outl -= ctx->tlsmacsize;
        }
    }

    return 1;
}
int ossl_cipher_generic_stream_final(void *vctx, unsigned char *out,
    size_t *outl, size_t outsize)
{
    PROV_CIPHER_CTX *ctx = (PROV_CIPHER_CTX *)vctx;

    if (!ossl_prov_is_running())
        return 0;

    if (!ctx->key_set) {
        ERR_raise(ERR_LIB_PROV, PROV_R_NO_KEY_SET);
        return 0;
    }

    *outl = 0;
    return 1;
}

int ossl_cipher_generic_cipher(void *vctx, unsigned char *out, size_t *outl,
    size_t outsize, const unsigned char *in,
    size_t inl)
{
    PROV_CIPHER_CTX *ctx = (PROV_CIPHER_CTX *)vctx;

    if (!ossl_prov_is_running())
        return 0;

    if (!ctx->key_set) {
        ERR_raise(ERR_LIB_PROV, PROV_R_NO_KEY_SET);
        return 0;
    }

    if (outsize < inl) {
        ERR_raise(ERR_LIB_PROV, PROV_R_OUTPUT_BUFFER_TOO_SMALL);
        return 0;
    }

    if (!ctx->hw->cipher(ctx, out, in, inl)) {
        ERR_raise(ERR_LIB_PROV, PROV_R_CIPHER_OPERATION_FAILED);
        return 0;
    }

    *outl = inl;
    return 1;
}

int ossl_cipher_common_get_ctx_params(PROV_CIPHER_CTX *ctx, const struct ossl_cipher_get_ctx_param_list_st *p)
{
    if (p->ivlen != NULL && !OSSL_PARAM_set_size_t(p->ivlen, ctx->ivlen)) {
        ERR_raise(ERR_LIB_PROV, PROV_R_FAILED_TO_SET_PARAMETER);
        return 0;
    }

    if (p->pad != NULL && !OSSL_PARAM_set_uint(p->pad, ctx->pad)) {
        ERR_raise(ERR_LIB_PROV, PROV_R_FAILED_TO_SET_PARAMETER);
        return 0;
    }

    if (p->iv != NULL
        && !OSSL_PARAM_set_octet_string_or_ptr(p->iv, ctx->oiv, ctx->ivlen)) {
        ERR_raise(ERR_LIB_PROV, PROV_R_FAILED_TO_SET_PARAMETER);
        return 0;
    }

    if (p->updiv != NULL
        && !OSSL_PARAM_set_octet_string_or_ptr(p->updiv, ctx->iv, ctx->ivlen)) {
        ERR_raise(ERR_LIB_PROV, PROV_R_FAILED_TO_SET_PARAMETER);
        return 0;
    }

    if (p->num != NULL && !OSSL_PARAM_set_uint(p->num, ctx->num)) {
        ERR_raise(ERR_LIB_PROV, PROV_R_FAILED_TO_SET_PARAMETER);
        return 0;
    }

    if (p->keylen != NULL && !OSSL_PARAM_set_size_t(p->keylen, ctx->keylen)) {
        ERR_raise(ERR_LIB_PROV, PROV_R_FAILED_TO_SET_PARAMETER);
        return 0;
    }

    if (p->tlsmac != NULL
        && !OSSL_PARAM_set_octet_ptr(p->tlsmac, ctx->tlsmac, ctx->tlsmacsize)) {
        ERR_raise(ERR_LIB_PROV, PROV_R_FAILED_TO_SET_PARAMETER);
        return 0;
    }
    return 1;
}

int ossl_cipher_generic_get_ctx_params(void *vctx, OSSL_PARAM params[])
{
    PROV_CIPHER_CTX *ctx = (PROV_CIPHER_CTX *)vctx;
    struct ossl_cipher_get_ctx_param_list_st p;

    if (ctx == NULL || !cipher_generic_get_ctx_params_decoder(params, &p))
        return 0;
    return ossl_cipher_common_get_ctx_params(ctx, &p);
}

int ossl_cipher_common_set_ctx_params(PROV_CIPHER_CTX *ctx, const struct ossl_cipher_set_ctx_param_list_st *p)
{
    if (p->pad != NULL) {
        unsigned int pad;

        if (!OSSL_PARAM_get_uint(p->pad, &pad)) {
            ERR_raise(ERR_LIB_PROV, PROV_R_FAILED_TO_GET_PARAMETER);
            return 0;
        }
        ctx->pad = pad ? 1 : 0;
    }

    if (p->bits != NULL) {
        unsigned int bits;

        if (!OSSL_PARAM_get_uint(p->bits, &bits)) {
            ERR_raise(ERR_LIB_PROV, PROV_R_FAILED_TO_GET_PARAMETER);
            return 0;
        }
        ctx->use_bits = bits ? 1 : 0;
    }

    if (p->tlsvers != NULL) {
        if (!OSSL_PARAM_get_uint(p->tlsvers, &ctx->tlsversion)) {
            ERR_raise(ERR_LIB_PROV, PROV_R_FAILED_TO_GET_PARAMETER);
            return 0;
        }
    }

    if (p->tlsmacsize != NULL) {
        if (!OSSL_PARAM_get_size_t(p->tlsmacsize, &ctx->tlsmacsize)) {
            ERR_raise(ERR_LIB_PROV, PROV_R_FAILED_TO_GET_PARAMETER);
            return 0;
        }
    }

    if (p->num != NULL) {
        unsigned int num;

        if (!OSSL_PARAM_get_uint(p->num, &num)) {
            ERR_raise(ERR_LIB_PROV, PROV_R_FAILED_TO_GET_PARAMETER);
            return 0;
        }
        ctx->num = num;
    }
    return 1;
}

int ossl_cipher_generic_set_ctx_params(void *vctx, const OSSL_PARAM params[])
{
    PROV_CIPHER_CTX *ctx = (PROV_CIPHER_CTX *)vctx;
    struct ossl_cipher_set_ctx_param_list_st p;

    if (ossl_param_is_empty(params))
        return 1;

    if (ctx == NULL || !cipher_generic_set_ctx_params_decoder(params, &p))
        return 0;
    return ossl_cipher_common_set_ctx_params(ctx, &p);
}

int ossl_cipher_generic_initiv(PROV_CIPHER_CTX *ctx, const unsigned char *iv,
    size_t ivlen)
{
    if (ivlen != ctx->ivlen
        || ivlen > sizeof(ctx->iv)) {
        ERR_raise(ERR_LIB_PROV, PROV_R_INVALID_IV_LENGTH);
        return 0;
    }
    ctx->iv_set = 1;
    memcpy(ctx->iv, iv, ivlen);
    memcpy(ctx->oiv, iv, ivlen);
    return 1;
}

void ossl_cipher_generic_initkey(void *vctx, size_t kbits, size_t blkbits,
    size_t ivbits, unsigned int mode,
    uint64_t flags, const PROV_CIPHER_HW *hw,
    void *provctx)
{
    PROV_CIPHER_CTX *ctx = (PROV_CIPHER_CTX *)vctx;

    if ((flags & PROV_CIPHER_FLAG_INVERSE_CIPHER) != 0)
        ctx->inverse_cipher = 1;
    if ((flags & PROV_CIPHER_FLAG_VARIABLE_LENGTH) != 0)
        ctx->variable_keylength = 1;

    ctx->pad = 1;
    ctx->keylen = ((kbits) / 8);
    ctx->ivlen = ((ivbits) / 8);
    ctx->hw = hw;
    ctx->mode = mode;
    ctx->blocksize = blkbits / 8;
    if (provctx != NULL)
        ctx->libctx = PROV_LIBCTX_OF(provctx); /* used for rand */
}
