// Copyright (C) 2021 The Qt Company Ltd.
// Copyright (C) 2013 John Layt <jlayt@kde.org>
// Copyright © 2004-2023 Unicode, Inc.
// SPDX-License-Identifier: Unicode-3.0

#ifndef QTIMEZONEPRIVATE_DATA_P_H
#define QTIMEZONEPRIVATE_DATA_P_H

//
//  W A R N I N G
//  -------------
//
// This file is not a normal header file.
// It defines data tables for internal use by QTimeZone's L10n code.
// It may change incompatibly between versions, or even be removed.
//
// We mean it.
//

QT_REQUIRE_CONFIG(timezone);
QT_BEGIN_NAMESPACE

namespace QtTimeZoneCldr {
/*
    Recognized UTC-offset zones and CLDR-derived data on timezone IDs.

    The UTC-offset zone table is provided for generic UTC±HH:mm format time
    zones. The UTC backend can support arbitrary offsets in seconds, but only
    advertises a limited repertoire of offsets as "available" in the normal
    sense.

    Windows Zone ID support is included in the default base class, QTZP, so can
    be used on all platforms, since an app running on Linux may need to
    communicate with a Windows Outlook server. These tables can also be used to
    look up Region Codes and UTC Offsets on platforms whose backends don't
    directly support them. For example, Darwin does not support
    availableTimeZones() filtering by region or offset. This table is
    auto-generated from the CLDR supplemental/windowsZones.xml data file, with
    IDs converted from CLDR-stable to IANA-current using bcp47/timezone.xml
    data.

    Please do not edit this data directly. See the generated section for details
    of its last update and how to update it.
*/

struct AliasData
{
    // Table is sorted on aliasId(), then on ianaId().
    // These are both indices into the IANA table.
    quint16 aliasIdIndex;
    quint16 ianaIdIndex;
    // Values for the actual IDs:
    constexpr QByteArrayView aliasId() const;
    constexpr QByteArrayView ianaId() const;
};

struct ZoneData
{
    // Keys (table is sorted in Windows ID, then on territory enum value):
    quint16 windowsIdKey;      // Windows ID sequence number
    quint16 territory;         // QLocale::Territory, AnyTerritory means No Territory
    // Values for this Windows zone and territory:
    quint16 ianaIdIndex;       // Index in ianaListData of space-joined IANA IDs
    constexpr QLatin1StringView id() const; // Space-joined list of IANA IDs
    constexpr auto ids() const { return id().tokenize(u' '); } // Iterate IANA IDs
};

struct WindowsData
{
    // Table is sorted on key and this puts the windowsId()s in ascending order.
    quint16 windowsIdKey;      // Windows ID sequence number
    quint16 windowsIdIndex;    // Index of Windows ID in windowsIdData
    // Values for this Windows zone:
    quint16 ianaIdIndex;       // Index in ianaIdData of single IANA ID
    qint32 offsetFromUtc;      // Standard Time Offset from UTC, used for quick look-ups
    constexpr QByteArrayView windowsId() const;
    constexpr QByteArrayView ianaId() const; // Single IANA ID
};

struct UtcData
{
    quint16 ianaIdIndex;       // Index in ianaListData of space-joined IANA IDs
    qint32 offsetFromUtc;      // Offset form UTC in seconds
    constexpr QByteArrayView id() const; // Space-joined list of IANA IDs
};

#if QT_CONFIG(timezone_locale) && !QT_CONFIG(icu)
/* Data comes from CldrAccess::readMetaZoneMap(alias) */

struct TerritoryZone
{
    // Table is sorted by territory (should have no duplicates)
    quint16 territory; // QLocale::Territory value
    quint16 ianaIdIndex; // Index in alias table
    constexpr QByteArrayView ianaId() const;
};

struct MetaZoneData
{
    // Table is sorted by metazone key and name (monotonic in each other), then territory:
    quint16 metaZoneKey; // Index from 1 to number of metazones
    quint16 metaIdIndex; // Index in metazone ID data
    quint16 territory; // QLocale::Territory value
    quint16 ianaIdIndex; // Index in alias table
    constexpr QByteArrayView ianaId() const;
    constexpr QByteArrayView metaZoneId() const;
};

struct ZoneMetaHistory
{
    // Table is sorted on ianaId then, among those with same ianaId,
    // earlier end <= begin < end <= later begin.
    quint16 ianaIdIndex; // Index in alias table
    quint16 metaZoneKey; // See MetaZoneData
    // quint32 epoch minutes, with 0xffffffff +inf and 0xfffffffe reserved.
    quint32 begin; // zone is in metazone from this datetime
    quint32 end; // zone is no longer in metazone from this datetime
    constexpr QByteArrayView ianaId() const;
};
#endif // timezone_locale but not ICU

// GENERATED PART STARTS HERE

/*
    This part of the file was generated on 2025-10-24 from the
    Common Locale Data Repository v47

    http://www.unicode.org/cldr/

    Do not edit this section: instead regenerate it using
    cldr2qlocalexml.py and qlocalexml2cpp.py on updated (or
    edited) CLDR data; see qtbase/util/locale_database/.
*/

// IANA ID indices of alias and IANA ID
static inline constexpr AliasData aliasMappingTable[] = {
    {      0,    14 }, // Africa/Asmera -> Africa/Asmara
    {     28,    44 }, // Africa/Timbuktu -> Africa/Bamako
    {     58,    91 }, // America/Argentina/ComodRivadavia -> America/Argentina/Catamarca
    {    119,   132 }, // America/Atka -> America/Adak
    {    145,   166 }, // America/Buenos_Aires -> America/Argentina/Buenos_Aires
    {    197,    91 }, // America/Catamarca -> America/Argentina/Catamarca
    {    215,   237 }, // America/Coral_Harbour -> America/Atikokan
    {    254,   270 }, // America/Cordoba -> America/Argentina/Cordoba
    {    296,   313 }, // America/Ensenada -> America/Tijuana
    {    329,   348 }, // America/Fort_Wayne -> America/Indiana/Indianapolis
    {    377,   393 }, // America/Godthab -> America/Nuuk
    {    406,   348 }, // America/Indianapolis -> America/Indiana/Indianapolis
    {    427,   441 }, // America/Jujuy -> America/Argentina/Jujuy
    {    465,   481 }, // America/Knox_IN -> America/Indiana/Knox
    {    502,   521 }, // America/Louisville -> America/Kentucky/Louisville
    {    549,   565 }, // America/Mendoza -> America/Argentina/Mendoza
    {    591,   608 }, // America/Montreal -> America/Toronto
    {    624,   608 }, // America/Nipigon -> America/Toronto
    {    640,   660 }, // America/Pangnirtung -> America/Iqaluit
    {    676,   695 }, // America/Porto_Acre -> America/Rio_Branco
    {    714,   734 }, // America/Rainy_River -> America/Winnipeg
    {    751,   270 }, // America/Rosario -> America/Argentina/Cordoba
    {    767,   313 }, // America/Santa_Isabel -> America/Tijuana
    {    788,   805 }, // America/Shiprock -> America/Denver
    {    820,   608 }, // America/Thunder_Bay -> America/Toronto
    {    840,   855 }, // America/Virgin -> America/St_Thomas
    {    873,   893 }, // America/Yellowknife -> America/Edmonton
    {    910,   932 }, // Antarctica/South_Pole -> Pacific/Auckland
    {    949,   964 }, // Asia/Ashkhabad -> Asia/Ashgabat
    {    978,   992 }, // Asia/Calcutta -> Asia/Kolkata
    {   1005,  1021 }, // Asia/Choibalsan -> Asia/Ulaanbaatar
    {   1038,  1053 }, // Asia/Chongqing -> Asia/Shanghai
    {   1067,  1053 }, // Asia/Chungking -> Asia/Shanghai
    {   1082,  1093 }, // Asia/Dacca -> Asia/Dhaka
    {   1104,  1053 }, // Asia/Harbin -> Asia/Shanghai
    {   1116,  1130 }, // Asia/Istanbul -> Europe/Istanbul
    {   1146,  1159 }, // Asia/Kashgar -> Asia/Urumqi
    {   1171,  1185 }, // Asia/Katmandu -> Asia/Kathmandu
    {   1200,  1211 }, // Asia/Macao -> Asia/Macau
    {   1222,  1235 }, // Asia/Rangoon -> Asia/Yangon
    {   1247,  1259 }, // Asia/Saigon -> Asia/Ho_Chi_Minh
    {   1276,  1290 }, // Asia/Tel_Aviv -> Asia/Jerusalem
    {   1305,  1317 }, // Asia/Thimbu -> Asia/Thimphu
    {   1330,  1349 }, // Asia/Ujung_Pandang -> Asia/Makassar
    {   1363,  1021 }, // Asia/Ulan_Bator -> Asia/Ulaanbaatar
    {   1379,  1395 }, // Atlantic/Faeroe -> Atlantic/Faroe
    {   1410,  1429 }, // Atlantic/Jan_Mayen -> Arctic/Longyearbyen
    {   1449,  1463 }, // Australia/ACT -> Australia/Sydney
    {   1480,  1463 }, // Australia/Canberra -> Australia/Sydney
    {   1499,  1516 }, // Australia/Currie -> Australia/Hobart
    {   1533,  1547 }, // Australia/LHI -> Australia/Lord_Howe
    {   1567,  1583 }, // Australia/North -> Australia/Darwin
    {   1600,  1463 }, // Australia/NSW -> Australia/Sydney
    {   1614,  1635 }, // Australia/Queensland -> Australia/Brisbane
    {   1654,  1670 }, // Australia/South -> Australia/Adelaide
    {   1689,  1516 }, // Australia/Tasmania -> Australia/Hobart
    {   1708,  1727 }, // Australia/Victoria -> Australia/Melbourne
    {   1747,  1762 }, // Australia/West -> Australia/Perth
    {   1778,  1799 }, // Australia/Yancowinna -> Australia/Broken_Hill
    {   1821,   695 }, // Brazil/Acre -> America/Rio_Branco
    {   1833,  1850 }, // Brazil/DeNoronha -> America/Noronha
    {   1866,  1878 }, // Brazil/East -> America/Sao_Paulo
    {   1896,  1908 }, // Brazil/West -> America/Manaus
    {   1923,  1939 }, // Canada/Atlantic -> America/Halifax
    {   1955,   734 }, // Canada/Central -> America/Winnipeg
    {   1970,  1995 }, // Canada/East-Saskatchewan -> America/Regina
    {   2010,   608 }, // Canada/Eastern -> America/Toronto
    {   2025,   893 }, // Canada/Mountain -> America/Edmonton
    {   2041,  2061 }, // Canada/Newfoundland -> America/St_Johns
    {   2078,  2093 }, // Canada/Pacific -> America/Vancouver
    {   2111,  1995 }, // Canada/Saskatchewan -> America/Regina
    {   2131,  2144 }, // Canada/Yukon -> America/Whitehorse
    {   2163,  2167 }, // CET -> Europe/Brussels
    {   2183,  2201 }, // Chile/Continental -> America/Santiago
    {   2218,  2237 }, // Chile/EasterIsland -> Pacific/Easter
    {   2252,  2260 }, // CST6CDT -> America/Chicago
    {   2276,  2281 }, // Cuba -> America/Havana
    {   2296,  2300 }, // EET -> Europe/Athens
    {   2314,  2320 }, // Egypt -> Africa/Cairo
    {   2333,  2338 }, // Eire -> Europe/Dublin
    {   2352,  2356 }, // EST -> America/Panama
    {   2371,  2379 }, // EST5EDT -> America/New_York
    {   2396,  2406 }, // Etc/GMT+0 -> Etc/GMT
    {   2414,  2406 }, // Etc/GMT-0 -> Etc/GMT
    {   2424,  2406 }, // Etc/GMT0 -> Etc/GMT
    {   2433,  2406 }, // Etc/Greenwich -> Etc/GMT
    {   2447,  2455 }, // Etc/UCT -> Etc/UTC
    {   2463,  2455 }, // Etc/Universal -> Etc/UTC
    {   2477,  2455 }, // Etc/Zulu -> Etc/UTC
    {   2486,  2501 }, // Europe/Belfast -> Europe/London
    {   2515,  2527 }, // Europe/Kiev -> Europe/Kyiv
    {   2539,  2554 }, // Europe/Nicosia -> Asia/Nicosia
    {   2567,  2583 }, // Europe/Tiraspol -> Europe/Chisinau
    {   2599,  2527 }, // Europe/Uzhgorod -> Europe/Kyiv
    {   2615,  2527 }, // Europe/Zaporozhye -> Europe/Kyiv
    {   2633,  2641 }, // Factory -> Etc/Unknown
    {   2653,  2501 }, // GB -> Europe/London
    {   2656,  2501 }, // GB-Eire -> Europe/London
    {   2664,  2406 }, // GMT -> Etc/GMT
    {   2668,  2406 }, // GMT+0 -> Etc/GMT
    {   2674,  2406 }, // GMT-0 -> Etc/GMT
    {   2680,  2406 }, // GMT0 -> Etc/GMT
    {   2685,  2406 }, // Greenwich -> Etc/GMT
    {   2695,  2704 }, // Hongkong -> Asia/Hong_Kong
    {   2719,  2723 }, // HST -> Pacific/Honolulu
    {   2740,  2748 }, // Iceland -> Atlantic/Reykjavik
    {   2767,  2772 }, // Iran -> Asia/Tehran
    {   2784,  1290 }, // Israel -> Asia/Jerusalem
    {   2791,  2799 }, // Jamaica -> America/Jamaica
    {   2815,  2821 }, // Japan -> Asia/Tokyo
    {   2832,  2842 }, // Kwajalein -> Pacific/Kwajalein
    {   2860,  2866 }, // Libya -> Africa/Tripoli
    {   2881,  2167 }, // MET -> Europe/Brussels
    {   2885,   313 }, // Mexico/BajaNorte -> America/Tijuana
    {   2902,  2917 }, // Mexico/BajaSur -> America/Mazatlan
    {   2934,  2949 }, // Mexico/General -> America/Mexico_City
    {   2969,  2973 }, // MST -> America/Phoenix
    {   2989,   805 }, // MST7MDT -> America/Denver
    {   2997,   805 }, // Navajo -> America/Denver
    {   3004,   932 }, // NZ -> Pacific/Auckland
    {   3007,  3015 }, // NZ-CHAT -> Pacific/Chatham
    {   3031,  3049 }, // Pacific/Enderbury -> Pacific/Kanton
    {   3064,  2723 }, // Pacific/Johnston -> Pacific/Honolulu
    {   3081,  3096 }, // Pacific/Ponape -> Pacific/Pohnpei
    {   3112,  3126 }, // Pacific/Samoa -> Pacific/Pago_Pago
    {   3144,  3157 }, // Pacific/Truk -> Pacific/Chuuk
    {   3171,  3157 }, // Pacific/Yap -> Pacific/Chuuk
    {   3183,  3190 }, // Poland -> Europe/Warsaw
    {   3204,  3213 }, // Portugal -> Europe/Lisbon
    {   3227,  1053 }, // PRC -> Asia/Shanghai
    {   3231,  3239 }, // PST8PDT -> America/Los_Angeles
    {   3259,  3263 }, // ROC -> Asia/Taipei
    {   3275,  3279 }, // ROK -> Asia/Seoul
    {   3290,  3300 }, // Singapore -> Asia/Singapore
    {   3315,  1130 }, // Turkey -> Europe/Istanbul
    {   3322,  2455 }, // UCT -> Etc/UTC
    {   3326,  2455 }, // Universal -> Etc/UTC
    {   3336,  3346 }, // US/Alaska -> America/Anchorage
    {   3364,   132 }, // US/Aleutian -> America/Adak
    {   3376,  2973 }, // US/Arizona -> America/Phoenix
    {   3387,  2260 }, // US/Central -> America/Chicago
    {   3398,   348 }, // US/East-Indiana -> America/Indiana/Indianapolis
    {   3414,  2379 }, // US/Eastern -> America/New_York
    {   3425,  2723 }, // US/Hawaii -> Pacific/Honolulu
    {   3435,   481 }, // US/Indiana-Starke -> America/Indiana/Knox
    {   3453,  3465 }, // US/Michigan -> America/Detroit
    {   3481,   805 }, // US/Mountain -> America/Denver
    {   3493,  3239 }, // US/Pacific -> America/Los_Angeles
    {   3504,  3239 }, // US/Pacific-New -> America/Los_Angeles
    {   3519,  3126 }, // US/Samoa -> Pacific/Pago_Pago
    {   3528,  2455 }, // UTC -> Etc/UTC
    {   3532,  3537 }, // W-SU -> Europe/Moscow
    {   3551,  3213 }, // WET -> Europe/Lisbon
    {   3555,  2455 }, // Zulu -> Etc/UTC
};

// Windows ID Key, Territory Enum, IANA List Index
static inline constexpr ZoneData zoneDataTable[] = {
    {      1,     1,     0 }, // Afghanistan Standard Time / Afghanistan
    {      2,   248,    11 }, // Alaskan Standard Time / United States
    {      3,   248,   106 }, // Aleutian Standard Time / United States
    {      4,   193,   119 }, // Altai Standard Time / Russia
    {      5,    19,   132 }, // Arab Standard Time / Bahrain
    {      5,   127,   145 }, // Arab Standard Time / Kuwait
    {      5,   190,   157 }, // Arab Standard Time / Qatar
    {      5,   205,   168 }, // Arab Standard Time / Saudi Arabia
    {      5,   259,   180 }, // Arab Standard Time / Yemen
    {      6,     0,   190 }, // Arabian Standard Time / AnyTerritory
    {      6,   176,   200 }, // Arabian Standard Time / Oman
    {      6,   245,   212 }, // Arabian Standard Time / United Arab Emirates
    {      7,   113,   223 }, // Arabic Standard Time / Iraq
    {      8,    11,   236 }, // Argentina Standard Time / Argentina
    {      9,   193,   559 }, // Astrakhan Standard Time / Russia
    {     10,    26,   593 }, // Atlantic Standard Time / Bermuda
    {     10,    41,   610 }, // Atlantic Standard Time / Canada
    {     10,    95,   678 }, // Atlantic Standard Time / Greenland
    {     11,    15,   692 }, // AUS Central Standard Time / Australia
    {     12,    15,   709 }, // Aus Central W. Standard Time / Australia
    {     13,    15,   725 }, // AUS Eastern Standard Time / Australia
    {     14,    17,   762 }, // Azerbaijan Standard Time / Azerbaijan
    {     15,    95,   772 }, // Azores Standard Time / Greenland
    {     15,   188,   793 }, // Azores Standard Time / Portugal
    {     16,    32,   809 }, // Bahia Standard Time / Brazil
    {     17,    20,   823 }, // Bangladesh Standard Time / Bangladesh
    {     17,    27,   834 }, // Bangladesh Standard Time / Bhutan
    {     18,    22,   847 }, // Belarus Standard Time / Belarus
    {     19,   182,   860 }, // Bougainville Standard Time / Papua New Guinea
    {     20,    41,   881 }, // Canada Central Standard Time / Canada
    {     21,     0,   918 }, // Cape Verde Standard Time / AnyTerritory
    {     21,    43,   928 }, // Cape Verde Standard Time / Cape Verde
    {     22,    12,   948 }, // Caucasus Standard Time / Armenia
    {     23,    15,   961 }, // Cen. Australia Standard Time / Australia
    {     24,     0,  1002 }, // Central America Standard Time / AnyTerritory
    {     24,    24,  1012 }, // Central America Standard Time / Belize
    {     24,    59,  1027 }, // Central America Standard Time / Costa Rica
    {     24,    70,  1046 }, // Central America Standard Time / Ecuador
    {     24,    72,  1064 }, // Central America Standard Time / El Salvador
    {     24,    99,  1084 }, // Central America Standard Time / Guatemala
    {     24,   106,  1102 }, // Central America Standard Time / Honduras
    {     24,   168,  1122 }, // Central America Standard Time / Nicaragua
    {     25,     0,  1138 }, // Central Asia Standard Time / AnyTerritory
    {     25,     9,  1148 }, // Central Asia Standard Time / Antarctica
    {     25,    33,  1166 }, // Central Asia Standard Time / British Indian Ocean Territory
    {     25,    50,  1180 }, // Central Asia Standard Time / China
    {     25,   128,  1192 }, // Central Asia Standard Time / Kyrgyzstan
    {     26,    32,  1205 }, // Central Brazilian Standard Time / Brazil
    {     27,     3,  1241 }, // Central Europe Standard Time / Albania
    {     27,    64,  1255 }, // Central Europe Standard Time / Czechia
    {     27,   108,  1269 }, // Central Europe Standard Time / Hungary
    {     27,   157,  1285 }, // Central Europe Standard Time / Montenegro
    {     27,   207,  1302 }, // Central Europe Standard Time / Serbia
    {     27,   212,  1318 }, // Central Europe Standard Time / Slovakia
    {     27,   213,  1336 }, // Central Europe Standard Time / Slovenia
    {     28,    29,  1353 }, // Central European Standard Time / Bosnia and Herzegovina
    {     28,    60,  1369 }, // Central European Standard Time / Croatia
    {     28,   140,  1383 }, // Central European Standard Time / Macedonia
    {     28,   187,  1397 }, // Central European Standard Time / Poland
    {     29,     0,  1411 }, // Central Pacific Standard Time / AnyTerritory
    {     29,     9,  1422 }, // Central Pacific Standard Time / Antarctica
    {     29,   153,  1439 }, // Central Pacific Standard Time / Micronesia
    {     29,   166,  1470 }, // Central Pacific Standard Time / New Caledonia
    {     29,   214,  1485 }, // Central Pacific Standard Time / Solomon Islands
    {     29,   252,  1505 }, // Central Pacific Standard Time / Vanuatu
    {     30,    41,  1519 }, // Central Standard Time / Canada
    {     30,   152,  1574 }, // Central Standard Time / Mexico
    {     30,   248,  1608 }, // Central Standard Time / United States
    {     31,   152,  1776 }, // Central Standard Time (Mexico) / Mexico
    {     32,   167,  1870 }, // Chatham Islands Standard Time / New Zealand
    {     33,    50,  1886 }, // China Standard Time / China
    {     33,   107,  1900 }, // China Standard Time / Hong Kong
    {     33,   139,  1915 }, // China Standard Time / Macao
    {     34,    61,  1926 }, // Cuba Standard Time / Cuba
    {     35,     0,  1941 }, // Dateline Standard Time / AnyTerritory
    {     36,     0,  1952 }, // E. Africa Standard Time / AnyTerritory
    {     36,     9,  1962 }, // E. Africa Standard Time / Antarctica
    {     36,    55,  1979 }, // E. Africa Standard Time / Comoros
    {     36,    67,  1993 }, // E. Africa Standard Time / Djibouti
    {     36,    74,  2009 }, // E. Africa Standard Time / Eritrea
    {     36,    77,  2023 }, // E. Africa Standard Time / Ethiopia
    {     36,   124,  2042 }, // E. Africa Standard Time / Kenya
    {     36,   141,  2057 }, // E. Africa Standard Time / Madagascar
    {     36,   151,  2077 }, // E. Africa Standard Time / Mayotte
    {     36,   215,  2092 }, // E. Africa Standard Time / Somalia
    {     36,   230,  2109 }, // E. Africa Standard Time / Tanzania
    {     36,   243,  2130 }, // E. Africa Standard Time / Uganda
    {     37,    15,  2145 }, // E. Australia Standard Time / Australia
    {     38,   154,  2183 }, // E. Europe Standard Time / Moldova
    {     39,    32,  2199 }, // E. South America Standard Time / Brazil
    {     40,    49,  2217 }, // Easter Island Standard Time / Chile
    {     41,    18,  2232 }, // Eastern Standard Time / Bahamas
    {     41,    41,  2247 }, // Eastern Standard Time / Canada
    {     41,   248,  2279 }, // Eastern Standard Time / United States
    {     42,   152,  2445 }, // Eastern Standard Time (Mexico) / Mexico
    {     43,    71,  2460 }, // Egypt Standard Time / Egypt
    {     44,   193,  2473 }, // Ekaterinburg Standard Time / Russia
    {     45,    82,  2492 }, // Fiji Standard Time / Fiji
    {     46,     2,  2505 }, // FLE Standard Time / Aland Islands
    {     46,    36,  2522 }, // FLE Standard Time / Bulgaria
    {     46,    75,  2535 }, // FLE Standard Time / Estonia
    {     46,    83,  2550 }, // FLE Standard Time / Finland
    {     46,   131,  2566 }, // FLE Standard Time / Latvia
    {     46,   137,  2578 }, // FLE Standard Time / Lithuania
    {     46,   244,  2593 }, // FLE Standard Time / Ukraine
    {     47,    90,  2605 }, // Georgian Standard Time / Georgia
    {     48,    81,  2618 }, // GMT Standard Time / Faroe Islands
    {     48,   100,  2633 }, // GMT Standard Time / Guernsey
    {     48,   114,  2649 }, // GMT Standard Time / Ireland
    {     48,   115,  2663 }, // GMT Standard Time / Isle of Man
    {     48,   121,  2682 }, // GMT Standard Time / Jersey
    {     48,   188,  2696 }, // GMT Standard Time / Portugal
    {     48,   220,  2727 }, // GMT Standard Time / Spain
    {     48,   246,  2743 }, // GMT Standard Time / United Kingdom
    {     49,    95,  2757 }, // Greenland Standard Time / Greenland
    {     50,    37,  2770 }, // Greenwich Standard Time / Burkina Faso
    {     50,    89,  2789 }, // Greenwich Standard Time / Gambia
    {     50,    92,  2803 }, // Greenwich Standard Time / Ghana
    {     50,    95,  2816 }, // Greenwich Standard Time / Greenland
    {     50,   101,  2837 }, // Greenwich Standard Time / Guinea-Bissau
    {     50,   102,  2851 }, // Greenwich Standard Time / Guinea
    {     50,   109,  2866 }, // Greenwich Standard Time / Iceland
    {     50,   118,  2885 }, // Greenwich Standard Time / Ivory Coast
    {     50,   134,  2900 }, // Greenwich Standard Time / Liberia
    {     50,   145,  2916 }, // Greenwich Standard Time / Mali
    {     50,   149,  2930 }, // Greenwich Standard Time / Mauritania
    {     50,   196,  2948 }, // Greenwich Standard Time / Saint Helena
    {     50,   206,  2967 }, // Greenwich Standard Time / Senegal
    {     50,   209,  2980 }, // Greenwich Standard Time / Sierra Leone
    {     50,   233,  2996 }, // Greenwich Standard Time / Togo
    {     51,    63,  3008 }, // GTB Standard Time / Cyprus
    {     51,    94,  3036 }, // GTB Standard Time / Greece
    {     51,   192,  3050 }, // GTB Standard Time / Romania
    {     52,   104,  3067 }, // Haiti Standard Time / Haiti
    {     53,     0,  3090 }, // Hawaiian Standard Time / AnyTerritory
    {     53,    58,  3101 }, // Hawaiian Standard Time / Cook Islands
    {     53,    86,  3119 }, // Hawaiian Standard Time / French Polynesia
    {     53,   248,  3134 }, // Hawaiian Standard Time / United States
    {     54,   110,  3151 }, // India Standard Time / India
    {     55,   112,  3164 }, // Iran Standard Time / Iran
    {     56,   116,  3176 }, // Israel Standard Time / Israel
    {     57,   122,  3191 }, // Jordan Standard Time / Jordan
    {     58,   193,  3202 }, // Kaliningrad Standard Time / Russia
    {     59,   218,  3221 }, // Korea Standard Time / South Korea
    {     60,   135,  3232 }, // Libya Standard Time / Libya
    {     61,     0,  3247 }, // Line Islands Standard Time / AnyTerritory
    {     61,   125,  3258 }, // Line Islands Standard Time / Kiribati
    {     62,    15,  3277 }, // Lord Howe Standard Time / Australia
    {     63,   193,  3297 }, // Magadan Standard Time / Russia
    {     64,    49,  3310 }, // Magallanes Standard Time / Chile
    {     65,    86,  3331 }, // Marquesas Standard Time / French Polynesia
    {     66,   150,  3349 }, // Mauritius Standard Time / Mauritius
    {     66,   191,  3366 }, // Mauritius Standard Time / Reunion
    {     66,   208,  3381 }, // Mauritius Standard Time / Seychelles
    {     67,   132,  3393 }, // Middle East Standard Time / Lebanon
    {     68,   250,  3405 }, // Montevideo Standard Time / Uruguay
    {     69,   159,  3424 }, // Morocco Standard Time / Morocco
    {     69,   257,  3442 }, // Morocco Standard Time / Western Sahara
    {     70,    41,  3458 }, // Mountain Standard Time / Canada
    {     70,   152,  3512 }, // Mountain Standard Time / Mexico
    {     70,   248,  3534 }, // Mountain Standard Time / United States
    {     71,   152,  3563 }, // Mountain Standard Time (Mexico) / Mexico
    {     72,    53,  3580 }, // Myanmar Standard Time / Cocos Islands
    {     72,   161,  3593 }, // Myanmar Standard Time / Myanmar
    {     73,   193,  3605 }, // N. Central Asia Standard Time / Russia
    {     74,   162,  3622 }, // Namibia Standard Time / Namibia
    {     75,   164,  3638 }, // Nepal Standard Time / Nepal
    {     76,     9,  3653 }, // New Zealand Standard Time / Antarctica
    {     76,   167,  3672 }, // New Zealand Standard Time / New Zealand
    {     77,    41,  3689 }, // Newfoundland Standard Time / Canada
    {     78,   172,  3706 }, // Norfolk Standard Time / Norfolk Island
    {     79,   193,  3722 }, // North Asia East Standard Time / Russia
    {     80,   193,  3735 }, // North Asia Standard Time / Russia
    {     81,   174,  3770 }, // North Korea Standard Time / North Korea
    {     82,   193,  3785 }, // Omsk Standard Time / Russia
    {     83,    49,  3795 }, // Pacific SA Standard Time / Chile
    {     84,    41,  3812 }, // Pacific Standard Time / Canada
    {     84,   248,  3830 }, // Pacific Standard Time / United States
    {     85,   152,  3850 }, // Pacific Standard Time (Mexico) / Mexico
    {     86,   178,  3866 }, // Pakistan Standard Time / Pakistan
    {     87,   183,  3879 }, // Paraguay Standard Time / Paraguay
    {     88,   123,  3896 }, // Qyzylorda Standard Time / Kazakhstan
    {     89,    23,  3911 }, // Romance Standard Time / Belgium
    {     89,    65,  3927 }, // Romance Standard Time / Denmark
    {     89,    84,  3945 }, // Romance Standard Time / France
    {     89,   220,  3958 }, // Romance Standard Time / Spain
    {     90,   193,  3985 }, // Russia Time Zone 10 / Russia
    {     91,   193,  4004 }, // Russia Time Zone 11 / Russia
    {     92,   193,  4031 }, // Russia Time Zone 3 / Russia
    {     93,   193,  4045 }, // Russian Standard Time / Russia
    {     93,   244,  4072 }, // Russian Standard Time / Ukraine
    {     94,     0,  4090 }, // SA Eastern Standard Time / AnyTerritory
    {     94,     9,  4100 }, // SA Eastern Standard Time / Antarctica
    {     94,    32,  4137 }, // SA Eastern Standard Time / Brazil
    {     94,    80,  4216 }, // SA Eastern Standard Time / Falkland Islands
    {     94,    85,  4233 }, // SA Eastern Standard Time / French Guiana
    {     94,   223,  4249 }, // SA Eastern Standard Time / Suriname
    {     95,     0,  4268 }, // SA Pacific Standard Time / AnyTerritory
    {     95,    32,  4278 }, // SA Pacific Standard Time / Brazil
    {     95,    41,  4314 }, // SA Pacific Standard Time / Canada
    {     95,    45,  4331 }, // SA Pacific Standard Time / Cayman Islands
    {     95,    54,  4346 }, // SA Pacific Standard Time / Colombia
    {     95,    70,  4361 }, // SA Pacific Standard Time / Ecuador
    {     95,   119,  4379 }, // SA Pacific Standard Time / Jamaica
    {     95,   181,  4395 }, // SA Pacific Standard Time / Panama
    {     95,   184,  4410 }, // SA Pacific Standard Time / Peru
    {     96,     0,  4423 }, // SA Western Standard Time / AnyTerritory
    {     96,     8,  4433 }, // SA Western Standard Time / Anguilla
    {     96,    10,  4450 }, // SA Western Standard Time / Antigua and Barbuda
    {     96,    13,  4466 }, // SA Western Standard Time / Aruba
    {     96,    21,  4480 }, // SA Western Standard Time / Barbados
    {     96,    28,  4497 }, // SA Western Standard Time / Bolivia
    {     96,    32,  4512 }, // SA Western Standard Time / Brazil
    {     96,    34,  4565 }, // SA Western Standard Time / British Virgin Islands
    {     96,    41,  4581 }, // SA Western Standard Time / Canada
    {     96,    44,  4602 }, // SA Western Standard Time / Caribbean Netherlands
    {     96,    62,  4621 }, // SA Western Standard Time / Curacao
    {     96,    68,  4637 }, // SA Western Standard Time / Dominica
    {     96,    69,  4654 }, // SA Western Standard Time / Dominican Republic
    {     96,    96,  4676 }, // SA Western Standard Time / Grenada
    {     96,    97,  4692 }, // SA Western Standard Time / Guadeloupe
    {     96,   103,  4711 }, // SA Western Standard Time / Guyana
    {     96,   148,  4726 }, // SA Western Standard Time / Martinique
    {     96,   158,  4745 }, // SA Western Standard Time / Montserrat
    {     96,   189,  4764 }, // SA Western Standard Time / Puerto Rico
    {     96,   195,  4784 }, // SA Western Standard Time / Saint Barthelemy
    {     96,   197,  4806 }, // SA Western Standard Time / Saint Kitts and Nevis
    {     96,   198,  4823 }, // SA Western Standard Time / Saint Lucia
    {     96,   199,  4840 }, // SA Western Standard Time / Saint Martin
    {     96,   201,  4856 }, // SA Western Standard Time / Saint Vincent and Grenadines
    {     96,   211,  4875 }, // SA Western Standard Time / Sint Maarten
    {     96,   236,  4897 }, // SA Western Standard Time / Trinidad and Tobago
    {     96,   249,  4919 }, // SA Western Standard Time / United States Virgin Islands
    {     97,   200,  4937 }, // Saint Pierre Standard Time / Saint Pierre and Miquelon
    {     98,   193,  4954 }, // Sakhalin Standard Time / Russia
    {     99,   202,  4968 }, // Samoa Standard Time / Samoa
    {    100,   204,  4981 }, // Sao Tome Standard Time / Sao Tome and Principe
    {    101,   193,  4997 }, // Saratov Standard Time / Russia
    {    102,     0,  5012 }, // SE Asia Standard Time / AnyTerritory
    {    102,     9,  5022 }, // SE Asia Standard Time / Antarctica
    {    102,    39,  5039 }, // SE Asia Standard Time / Cambodia
    {    102,    51,  5055 }, // SE Asia Standard Time / Christmas Island
    {    102,   111,  5072 }, // SE Asia Standard Time / Indonesia
    {    102,   129,  5100 }, // SE Asia Standard Time / Laos
    {    102,   231,  5115 }, // SE Asia Standard Time / Thailand
    {    102,   255,  5128 }, // SE Asia Standard Time / Vietnam
    {    103,     0,  5145 }, // Singapore Standard Time / AnyTerritory
    {    103,    35,  5155 }, // Singapore Standard Time / Brunei
    {    103,   111,  5167 }, // Singapore Standard Time / Indonesia
    {    103,   143,  5181 }, // Singapore Standard Time / Malaysia
    {    103,   185,  5212 }, // Singapore Standard Time / Philippines
    {    103,   210,  5224 }, // Singapore Standard Time / Singapore
    {    104,     0,  5239 }, // South Africa Standard Time / AnyTerritory
    {    104,    30,  5249 }, // South Africa Standard Time / Botswana
    {    104,    38,  5265 }, // South Africa Standard Time / Burundi
    {    104,    57,  5282 }, // South Africa Standard Time / Congo - Kinshasa
    {    104,    76,  5300 }, // South Africa Standard Time / Eswatini
    {    104,   133,  5315 }, // South Africa Standard Time / Lesotho
    {    104,   142,  5329 }, // South Africa Standard Time / Malawi
    {    104,   160,  5345 }, // South Africa Standard Time / Mozambique
    {    104,   194,  5359 }, // South Africa Standard Time / Rwanda
    {    104,   216,  5373 }, // South Africa Standard Time / South Africa
    {    104,   260,  5393 }, // South Africa Standard Time / Zambia
    {    104,   261,  5407 }, // South Africa Standard Time / Zimbabwe
    {    105,   219,  5421 }, // South Sudan Standard Time / South Sudan
    {    106,   221,  5433 }, // Sri Lanka Standard Time / Sri Lanka
    {    107,   222,  5446 }, // Sudan Standard Time / Sudan
    {    108,   227,  5462 }, // Syria Standard Time / Syria
    {    109,   228,  5476 }, // Taipei Standard Time / Taiwan
    {    110,    15,  5488 }, // Tasmania Standard Time / Australia
    {    111,    32,  5526 }, // Tocantins Standard Time / Brazil
    {    112,     0,  5544 }, // Tokyo Standard Time / AnyTerritory
    {    112,   111,  5554 }, // Tokyo Standard Time / Indonesia
    {    112,   120,  5568 }, // Tokyo Standard Time / Japan
    {    112,   179,  5579 }, // Tokyo Standard Time / Palau
    {    112,   232,  5593 }, // Tokyo Standard Time / Timor-Leste
    {    113,   193,  5603 }, // Tomsk Standard Time / Russia
    {    114,   235,  5614 }, // Tonga Standard Time / Tonga
    {    115,   193,  5632 }, // Transbaikal Standard Time / Russia
    {    116,   239,  5643 }, // Turkey Standard Time / Turkey
    {    117,   241,  5659 }, // Turks And Caicos Standard Time / Turks and Caicos Islands
    {    118,   156,  5678 }, // Ulaanbaatar Standard Time / Mongolia
    {    119,   248,  5695 }, // US Eastern Standard Time / United States
    {    120,     0,  5770 }, // US Mountain Standard Time / AnyTerritory
    {    120,    41,  5780 }, // US Mountain Standard Time / Canada
    {    120,   152,  5837 }, // US Mountain Standard Time / Mexico
    {    120,   248,  5856 }, // US Mountain Standard Time / United States
    {    121,     0,  5872 }, // UTC / AnyTerritory
    {    122,     0,  5888 }, // UTC+12 / AnyTerritory
    {    122,   125,  5899 }, // UTC+12 / Kiribati
    {    122,   147,  5914 }, // UTC+12 / Marshall Islands
    {    122,   163,  5947 }, // UTC+12 / Nauru
    {    122,   242,  5961 }, // UTC+12 / Tuvalu
    {    122,   247,  5978 }, // UTC+12 / United States Outlying Islands
    {    122,   256,  5991 }, // UTC+12 / Wallis and Futuna
    {    123,     0,  6006 }, // UTC+13 / AnyTerritory
    {    123,   125,  6017 }, // UTC+13 / Kiribati
    {    123,   234,  6032 }, // UTC+13 / Tokelau
    {    124,     0,  6048 }, // UTC-02 / AnyTerritory
    {    124,    32,  6058 }, // UTC-02 / Brazil
    {    124,   217,  6074 }, // UTC-02 / South Georgia and South Sandwich Islands
    {    125,     0,  6097 }, // UTC-08 / AnyTerritory
    {    125,   186,  6107 }, // UTC-08 / Pitcairn
    {    126,     0,  6124 }, // UTC-09 / AnyTerritory
    {    126,    86,  6134 }, // UTC-09 / French Polynesia
    {    127,     0,  6150 }, // UTC-11 / AnyTerritory
    {    127,     5,  6161 }, // UTC-11 / American Samoa
    {    127,   171,  6179 }, // UTC-11 / Niue
    {    127,   247,  6192 }, // UTC-11 / United States Outlying Islands
    {    128,   254,  6207 }, // Venezuela Standard Time / Venezuela
    {    129,   193,  6223 }, // Vladivostok Standard Time / Russia
    {    130,   193,  6254 }, // Volgograd Standard Time / Russia
    {    131,    15,  6271 }, // W. Australia Standard Time / Australia
    {    132,     0,  6287 }, // W. Central Africa Standard Time / AnyTerritory
    {    132,     4,  6297 }, // W. Central Africa Standard Time / Algeria
    {    132,     7,  6312 }, // W. Central Africa Standard Time / Angola
    {    132,    25,  6326 }, // W. Central Africa Standard Time / Benin
    {    132,    40,  6344 }, // W. Central Africa Standard Time / Cameroon
    {    132,    46,  6358 }, // W. Central Africa Standard Time / Central African Republic
    {    132,    48,  6372 }, // W. Central Africa Standard Time / Chad
    {    132,    56,  6388 }, // W. Central Africa Standard Time / Congo - Brazzaville
    {    132,    57,  6407 }, // W. Central Africa Standard Time / Congo - Kinshasa
    {    132,    73,  6423 }, // W. Central Africa Standard Time / Equatorial Guinea
    {    132,    88,  6437 }, // W. Central Africa Standard Time / Gabon
    {    132,   169,  6455 }, // W. Central Africa Standard Time / Nigeria
    {    132,   170,  6468 }, // W. Central Africa Standard Time / Niger
    {    132,   238,  6482 }, // W. Central Africa Standard Time / Tunisia
    {    133,     6,  6495 }, // W. Europe Standard Time / Andorra
    {    133,    16,  6510 }, // W. Europe Standard Time / Austria
    {    133,    91,  6524 }, // W. Europe Standard Time / Germany
    {    133,    93,  6554 }, // W. Europe Standard Time / Gibraltar
    {    133,   117,  6571 }, // W. Europe Standard Time / Italy
    {    133,   136,  6583 }, // W. Europe Standard Time / Liechtenstein
    {    133,   138,  6596 }, // W. Europe Standard Time / Luxembourg
    {    133,   146,  6614 }, // W. Europe Standard Time / Malta
    {    133,   155,  6627 }, // W. Europe Standard Time / Monaco
    {    133,   165,  6641 }, // W. Europe Standard Time / Netherlands
    {    133,   175,  6658 }, // W. Europe Standard Time / Norway
    {    133,   203,  6670 }, // W. Europe Standard Time / San Marino
    {    133,   224,  6688 }, // W. Europe Standard Time / Svalbard and Jan Mayen
    {    133,   225,  6708 }, // W. Europe Standard Time / Sweden
    {    133,   226,  6725 }, // W. Europe Standard Time / Switzerland
    {    133,   253,  6739 }, // W. Europe Standard Time / Vatican City
    {    134,   156,  6754 }, // W. Mongolia Standard Time / Mongolia
    {    135,     0,  6764 }, // West Asia Standard Time / AnyTerritory
    {    135,     9,  6774 }, // West Asia Standard Time / Antarctica
    {    135,    87,  6792 }, // West Asia Standard Time / French Southern Territories
    {    135,   123,  6809 }, // West Asia Standard Time / Kazakhstan
    {    135,   144,  6880 }, // West Asia Standard Time / Maldives
    {    135,   229,  6896 }, // West Asia Standard Time / Tajikistan
    {    135,   240,  6910 }, // West Asia Standard Time / Turkmenistan
    {    135,   251,  6924 }, // West Asia Standard Time / Uzbekistan
    {    136,   180,  6953 }, // West Bank Standard Time / Palestinian Territories
    {    137,     0,  6975 }, // West Pacific Standard Time / AnyTerritory
    {    137,     9,  6986 }, // West Pacific Standard Time / Antarctica
    {    137,    98,  7012 }, // West Pacific Standard Time / Guam
    {    137,   153,  7025 }, // West Pacific Standard Time / Micronesia
    {    137,   173,  7039 }, // West Pacific Standard Time / Northern Mariana Islands
    {    137,   182,  7054 }, // West Pacific Standard Time / Papua New Guinea
    {    138,   193,  7075 }, // Yakutsk Standard Time / Russia
    {    139,    41,  7102 }, // Yukon Standard Time / Canada
};

// Windows ID Key, Windows ID Index, IANA ID Index, UTC Offset
static inline constexpr WindowsData windowsDataTable[] = {
    {      1,     0,  3560, 16200 }, // Afghanistan Standard Time
    {      2,    26,  3346,-32400 }, // Alaskan Standard Time
    {      3,    48,   132,-36000 }, // Aleutian Standard Time
    {      4,    71,  3571, 25200 }, // Altai Standard Time
    {      5,    91,  3584, 10800 }, // Arab Standard Time
    {      6,   110,  3596, 14400 }, // Arabian Standard Time
    {      7,   132,  3607, 10800 }, // Arabic Standard Time
    {      8,   153,   166,-10800 }, // Argentina Standard Time
    {      9,   177,  3620, 14400 }, // Astrakhan Standard Time
    {     10,   201,  1939,-14400 }, // Atlantic Standard Time
    {     11,   224,  1583, 34200 }, // AUS Central Standard Time
    {     12,   250,  3637, 31500 }, // Aus Central W. Standard Time
    {     13,   279,  1463, 36000 }, // AUS Eastern Standard Time
    {     14,   305,  3653, 14400 }, // Azerbaijan Standard Time
    {     15,   330,  3663, -3600 }, // Azores Standard Time
    {     16,   351,  3679,-10800 }, // Bahia Standard Time
    {     17,   371,  1093, 21600 }, // Bangladesh Standard Time
    {     18,   396,  3693, 10800 }, // Belarus Standard Time
    {     19,   418,  3706, 39600 }, // Bougainville Standard Time
    {     20,   445,  1995,-21600 }, // Canada Central Standard Time
    {     21,   474,  3727, -3600 }, // Cape Verde Standard Time
    {     22,   499,  3747, 14400 }, // Caucasus Standard Time
    {     23,   522,  1670, 34200 }, // Cen. Australia Standard Time
    {     24,   551,  3760,-21600 }, // Central America Standard Time
    {     25,   581,  3778, 21600 }, // Central Asia Standard Time
    {     26,   608,  3791,-14400 }, // Central Brazilian Standard Time
    {     27,   640,  3806,  3600 }, // Central Europe Standard Time
    {     28,   669,  3190,  3600 }, // Central European Standard Time
    {     29,   700,  3822, 39600 }, // Central Pacific Standard Time
    {     30,   730,  2260,-21600 }, // Central Standard Time
    {     31,   752,  2949,-21600 }, // Central Standard Time (Mexico)
    {     32,   783,  3015, 45900 }, // Chatham Islands Standard Time
    {     33,   813,  1053, 28800 }, // China Standard Time
    {     34,   833,  2281,-18000 }, // Cuba Standard Time
    {     35,   852,  3842,-43200 }, // Dateline Standard Time
    {     36,   875,  3853, 10800 }, // E. Africa Standard Time
    {     37,   899,  1635, 36000 }, // E. Australia Standard Time
    {     38,   926,  2583,  7200 }, // E. Europe Standard Time
    {     39,   950,  1878,-10800 }, // E. South America Standard Time
    {     40,   981,  2237,-21600 }, // Easter Island Standard Time
    {     41,  1009,  2379,-18000 }, // Eastern Standard Time
    {     42,  1031,  3868,-18000 }, // Eastern Standard Time (Mexico)
    {     43,  1062,  2320,  7200 }, // Egypt Standard Time
    {     44,  1082,  3883, 18000 }, // Ekaterinburg Standard Time
    {     45,  1109,  3902, 43200 }, // Fiji Standard Time
    {     46,  1128,  2527,  7200 }, // FLE Standard Time
    {     47,  1146,  3915, 14400 }, // Georgian Standard Time
    {     48,  1169,  2501,     0 }, // GMT Standard Time
    {     49,  1187,   393,-10800 }, // Greenland Standard Time
    {     50,  1211,  2748,     0 }, // Greenwich Standard Time
    {     51,  1235,  3928,  7200 }, // GTB Standard Time
    {     52,  1253,  3945,-18000 }, // Haiti Standard Time
    {     53,  1273,  2723,-36000 }, // Hawaiian Standard Time
    {     54,  1296,   992, 19800 }, // India Standard Time
    {     55,  1316,  2772, 12600 }, // Iran Standard Time
    {     56,  1335,  1290,  7200 }, // Israel Standard Time
    {     57,  1356,  3968,  7200 }, // Jordan Standard Time
    {     58,  1377,  3979,  7200 }, // Kaliningrad Standard Time
    {     59,  1403,  3279, 32400 }, // Korea Standard Time
    {     60,  1423,  2866,  7200 }, // Libya Standard Time
    {     61,  1443,  3998, 50400 }, // Line Islands Standard Time
    {     62,  1470,  1547, 37800 }, // Lord Howe Standard Time
    {     63,  1494,  4017, 36000 }, // Magadan Standard Time
    {     64,  1516,  4030,-10800 }, // Magallanes Standard Time
    {     65,  1541,  4051,-34200 }, // Marquesas Standard Time
    {     66,  1565,  4069, 14400 }, // Mauritius Standard Time
    {     67,  1589,  4086,  7200 }, // Middle East Standard Time
    {     68,  1615,  4098,-10800 }, // Montevideo Standard Time
    {     69,  1640,  4117,     0 }, // Morocco Standard Time
    {     70,  1662,   805,-25200 }, // Mountain Standard Time
    {     71,  1685,  2917,-25200 }, // Mountain Standard Time (Mexico)
    {     72,  1717,  1235, 23400 }, // Myanmar Standard Time
    {     73,  1739,  4135, 21600 }, // N. Central Asia Standard Time
    {     74,  1769,  4152,  3600 }, // Namibia Standard Time
    {     75,  1791,  1185, 20700 }, // Nepal Standard Time
    {     76,  1811,   932, 43200 }, // New Zealand Standard Time
    {     77,  1837,  2061,-12600 }, // Newfoundland Standard Time
    {     78,  1864,  4168, 39600 }, // Norfolk Standard Time
    {     79,  1886,  4184, 28800 }, // North Asia East Standard Time
    {     80,  1916,  4197, 25200 }, // North Asia Standard Time
    {     81,  1941,  4214, 30600 }, // North Korea Standard Time
    {     82,  1967,  4229, 21600 }, // Omsk Standard Time
    {     83,  1986,  2201,-10800 }, // Pacific SA Standard Time
    {     84,  2011,  3239,-28800 }, // Pacific Standard Time
    {     85,  2033,   313,-28800 }, // Pacific Standard Time (Mexico)
    {     86,  2064,  4239, 18000 }, // Pakistan Standard Time
    {     87,  2087,  4252,-14400 }, // Paraguay Standard Time
    {     88,  2110,  4269, 18000 }, // Qyzylorda Standard Time
    {     89,  2134,  4284,  3600 }, // Romance Standard Time
    {     90,  2156,  4297, 39600 }, // Russia Time Zone 10
    {     91,  2176,  4316, 43200 }, // Russia Time Zone 11
    {     92,  2196,  4331, 14400 }, // Russia Time Zone 3
    {     93,  2215,  3537, 10800 }, // Russian Standard Time
    {     94,  2237,  4345,-10800 }, // SA Eastern Standard Time
    {     95,  2262,  4361,-18000 }, // SA Pacific Standard Time
    {     96,  2287,  4376,-14400 }, // SA Western Standard Time
    {     97,  2312,  4391,-10800 }, // Saint Pierre Standard Time
    {     98,  2339,  4408, 39600 }, // Sakhalin Standard Time
    {     99,  2362,  4422, 46800 }, // Samoa Standard Time
    {    100,  2382,  4435,     0 }, // Sao Tome Standard Time
    {    101,  2405,  4451, 14400 }, // Saratov Standard Time
    {    102,  2427,  4466, 25200 }, // SE Asia Standard Time
    {    103,  2449,  3300, 28800 }, // Singapore Standard Time
    {    104,  2473,  4479,  7200 }, // South Africa Standard Time
    {    105,  2500,  4499,  7200 }, // South Sudan Standard Time
    {    106,  2526,  4511, 19800 }, // Sri Lanka Standard Time
    {    107,  2550,  4524,  7200 }, // Sudan Standard Time
    {    108,  2570,  4540,  7200 }, // Syria Standard Time
    {    109,  2590,  3263, 28800 }, // Taipei Standard Time
    {    110,  2611,  1516, 36000 }, // Tasmania Standard Time
    {    111,  2634,  4554,-10800 }, // Tocantins Standard Time
    {    112,  2658,  2821, 32400 }, // Tokyo Standard Time
    {    113,  2678,  4572, 25200 }, // Tomsk Standard Time
    {    114,  2698,  4583, 46800 }, // Tonga Standard Time
    {    115,  2718,  4601, 32400 }, // Transbaikal Standard Time
    {    116,  2744,  1130,  7200 }, // Turkey Standard Time
    {    117,  2765,  4612,-14400 }, // Turks And Caicos Standard Time
    {    118,  2796,  1021, 28800 }, // Ulaanbaatar Standard Time
    {    119,  2822,   348,-18000 }, // US Eastern Standard Time
    {    120,  2847,  2973,-25200 }, // US Mountain Standard Time
    {    121,  2873,  2455,     0 }, // UTC
    {    122,  2877,  4631, 43200 }, // UTC+12
    {    123,  2884,  4642, 46800 }, // UTC+13
    {    124,  2891,  4653, -7200 }, // UTC-02
    {    125,  2898,  4663,-28800 }, // UTC-08
    {    126,  2905,  4673,-32400 }, // UTC-09
    {    127,  2912,  4683,-39600 }, // UTC-11
    {    128,  2919,  4694,-16200 }, // Venezuela Standard Time
    {    129,  2943,  4710, 36000 }, // Vladivostok Standard Time
    {    130,  2969,  4727, 14400 }, // Volgograd Standard Time
    {    131,  2993,  1762, 28800 }, // W. Australia Standard Time
    {    132,  3020,  4744,  3600 }, // W. Central Africa Standard Time
    {    133,  3052,  4757,  3600 }, // W. Europe Standard Time
    {    134,  3076,  4771, 25200 }, // W. Mongolia Standard Time
    {    135,  3102,  4781, 18000 }, // West Asia Standard Time
    {    136,  3126,  4795,  7200 }, // West Bank Standard Time
    {    137,  3150,  4807, 36000 }, // West Pacific Standard Time
    {    138,  3177,  4828, 32400 }, // Yakutsk Standard Time
    {    139,  3199,  2144,-25200 }, // Yukon Standard Time
};

// IANA List Index, UTC Offset
static inline constexpr UtcData utcDataTable[] = {
    {   7136,-50400 }, // UTC-14:00
    {   7146,-46800 }, // UTC-13:00
    {   7156,-43200 }, // UTC-12:00
    {   7166,-39600 }, // UTC-11:00
    {   7176,-36000 }, // UTC-10:00
    {   7186,-34200 }, // UTC-09:30
    {   7196,-32400 }, // UTC-09:00
    {   7206,-28800 }, // UTC-08:00
    {   7216,-25200 }, // UTC-07:00
    {   7226,-21600 }, // UTC-06:00
    {   7236,-18000 }, // UTC-05:00
    {   7246,-16200 }, // UTC-04:30
    {   7256,-14400 }, // UTC-04:00
    {   7266,-12600 }, // UTC-03:30
    {   7276,-10800 }, // UTC-03:00
    {   7286, -7200 }, // UTC-02:00
    {   7296, -3600 }, // UTC-01:00
    {   7306,     0 }, // UTC
    {   7330,  3600 }, // UTC+01:00
    {   7340,  7200 }, // UTC+02:00
    {   7350, 10800 }, // UTC+03:00
    {   7360, 12600 }, // UTC+03:30
    {   7370, 14400 }, // UTC+04:00
    {   7380, 16200 }, // UTC+04:30
    {   7390, 18000 }, // UTC+05:00
    {   7400, 19800 }, // UTC+05:30
    {   7410, 20700 }, // UTC+05:45
    {   7420, 21600 }, // UTC+06:00
    {   7430, 23400 }, // UTC+06:30
    {   7440, 25200 }, // UTC+07:00
    {   7450, 28800 }, // UTC+08:00
    {   7460, 30600 }, // UTC+08:30
    {   7470, 31500 }, // UTC+08:45
    {   7480, 32400 }, // UTC+09:00
    {   7490, 34200 }, // UTC+09:30
    {   7500, 36000 }, // UTC+10:00
    {   7510, 37800 }, // UTC+10:30
    {   7520, 39600 }, // UTC+11:00
    {   7530, 43200 }, // UTC+12:00
    {   7540, 45900 }, // UTC+12:45
    {   7550, 46800 }, // UTC+13:00
    {   7560, 50400 }, // UTC+14:00
};

#if QT_CONFIG(timezone_locale) && !QT_CONFIG(icu)

// QLocale::Territory value, IANA ID Index
static inline constexpr TerritoryZone territoryZoneMap[] = {
    {     49,  2201 }, // Chile -> America/Santiago
    {     50,  1053 }, // China -> Asia/Shanghai
    {     70,  4841 }, // Ecuador -> America/Guayaquil
    {     91,  4757 }, // Germany -> Europe/Berlin
    {    143,  4859 }, // Malaysia -> Asia/Kuala_Lumpur
    {    147,  4877 }, // Marshall Islands -> Pacific/Majuro
    {    167,   932 }, // New Zealand -> Pacific/Auckland
    {    188,  3213 }, // Portugal -> Europe/Lisbon
    {    220,  4892 }, // Spain -> Europe/Madrid
    {    244,  2527 }, // Ukraine -> Europe/Kyiv
    {    251,  4781 }, // Uzbekistan -> Asia/Tashkent
};

// MetaZone Key, MetaZone Name Index, QLocale::Territory value, IANA ID Index
static inline constexpr MetaZoneData metaZoneTable[] = {
    {      1,     0,   258,   695 }, // Acre/world -> America/Rio_Branco
    {      2,     5,   258,  3560 }, // Afghanistan/world -> Asia/Kabul
    {      3,    17,    30,  4906 }, // Africa_Central/Botswana -> Africa/Gaborone
    {      3,    17,    38,  4922 }, // Africa_Central/Burundi -> Africa/Bujumbura
    {      3,    17,    57,  4939 }, // Africa_Central/Congo - Kinshasa -> Africa/Lubumbashi
    {      3,    17,   142,  4957 }, // Africa_Central/Malawi -> Africa/Blantyre
    {      3,    17,   194,  4973 }, // Africa_Central/Rwanda -> Africa/Kigali
    {      3,    17,   258,  4987 }, // Africa_Central/world -> Africa/Maputo
    {      3,    17,   260,  5001 }, // Africa_Central/Zambia -> Africa/Lusaka
    {      3,    17,   261,  5015 }, // Africa_Central/Zimbabwe -> Africa/Harare
    {      4,    32,    55,  5029 }, // Africa_Eastern/Comoros -> Indian/Comoro
    {      4,    32,    67,  5043 }, // Africa_Eastern/Djibouti -> Africa/Djibouti
    {      4,    32,    74,    14 }, // Africa_Eastern/Eritrea -> Africa/Asmara
    {      4,    32,    77,  5059 }, // Africa_Eastern/Ethiopia -> Africa/Addis_Ababa
    {      4,    32,   141,  5078 }, // Africa_Eastern/Madagascar -> Indian/Antananarivo
    {      4,    32,   151,  5098 }, // Africa_Eastern/Mayotte -> Indian/Mayotte
    {      4,    32,   215,  5113 }, // Africa_Eastern/Somalia -> Africa/Mogadishu
    {      4,    32,   230,  5130 }, // Africa_Eastern/Tanzania -> Africa/Dar_es_Salaam
    {      4,    32,   243,  5151 }, // Africa_Eastern/Uganda -> Africa/Kampala
    {      4,    32,   258,  3853 }, // Africa_Eastern/world -> Africa/Nairobi
    {      5,    47,   258,  5166 }, // Africa_FarWestern/world -> Africa/El_Aaiun
    {      6,    65,    76,  5182 }, // Africa_Southern/Eswatini -> Africa/Mbabane
    {      6,    65,   133,  5197 }, // Africa_Southern/Lesotho -> Africa/Maseru
    {      6,    65,   258,  4479 }, // Africa_Southern/world -> Africa/Johannesburg
    {      7,    81,     7,  5211 }, // Africa_Western/Angola -> Africa/Luanda
    {      7,    81,    25,  5225 }, // Africa_Western/Benin -> Africa/Porto-Novo
    {      7,    81,    40,  5243 }, // Africa_Western/Cameroon -> Africa/Douala
    {      7,    81,    46,  5257 }, // Africa_Western/Central African Republic -> Africa/Bangui
    {      7,    81,    48,  5271 }, // Africa_Western/Chad -> Africa/Ndjamena
    {      7,    81,    56,  5287 }, // Africa_Western/Congo - Brazzaville -> Africa/Brazzaville
    {      7,    81,    57,  5306 }, // Africa_Western/Congo - Kinshasa -> Africa/Kinshasa
    {      7,    81,    73,  5322 }, // Africa_Western/Equatorial Guinea -> Africa/Malabo
    {      7,    81,    88,  5336 }, // Africa_Western/Gabon -> Africa/Libreville
    {      7,    81,   170,  5354 }, // Africa_Western/Niger -> Africa/Niamey
    {      7,    81,   258,  4744 }, // Africa_Western/world -> Africa/Lagos
    {      8,    96,   258,  5368 }, // Aktyubinsk/world -> Asia/Aqtobe
    {      9,   107,   258,  5380 }, // Alaska/world -> America/Juneau
    {     10,   114,   258,  3346 }, // Alaska_Hawaii/world -> America/Anchorage
    {     11,   128,   258,  5395 }, // Almaty/world -> Asia/Almaty
    {     12,   135,   258,  1908 }, // Amazon/world -> America/Manaus
    {     13,   142,    24,  5407 }, // America_Central/Belize -> America/Belize
    {     13,   142,    41,   734 }, // America_Central/Canada -> America/Winnipeg
    {     13,   142,    59,  5422 }, // America_Central/Costa Rica -> America/Costa_Rica
    {     13,   142,    72,  5441 }, // America_Central/El Salvador -> America/El_Salvador
    {     13,   142,    99,  3760 }, // America_Central/Guatemala -> America/Guatemala
    {     13,   142,   106,  5461 }, // America_Central/Honduras -> America/Tegucigalpa
    {     13,   142,   152,  2949 }, // America_Central/Mexico -> America/Mexico_City
    {     13,   142,   258,  2260 }, // America_Central/world -> America/Chicago
    {     14,   158,    18,  5481 }, // America_Eastern/Bahamas -> America/Nassau
    {     14,   158,    41,   608 }, // America_Eastern/Canada -> America/Toronto
    {     14,   158,    45,  5496 }, // America_Eastern/Cayman Islands -> America/Cayman
    {     14,   158,   104,  3945 }, // America_Eastern/Haiti -> America/Port-au-Prince
    {     14,   158,   119,  2799 }, // America_Eastern/Jamaica -> America/Jamaica
    {     14,   158,   181,  2356 }, // America_Eastern/Panama -> America/Panama
    {     14,   158,   258,  2379 }, // America_Eastern/world -> America/New_York
    {     15,   174,    41,   893 }, // America_Mountain/Canada -> America/Edmonton
    {     15,   174,   258,   805 }, // America_Mountain/world -> America/Denver
    {     16,   191,    41,  2093 }, // America_Pacific/Canada -> America/Vancouver
    {     16,   191,   152,   313 }, // America_Pacific/Mexico -> America/Tijuana
    {     16,   191,   258,  3239 }, // America_Pacific/world -> America/Los_Angeles
    {     17,   207,   258,  5511 }, // Anadyr/world -> Asia/Anadyr
    {     18,   214,   258,  4422 }, // Apia/world -> Pacific/Apia
    {     19,   219,   258,  5523 }, // Aqtau/world -> Asia/Aqtau
    {     20,   225,   258,  5368 }, // Aqtobe/world -> Asia/Aqtobe
    {     21,   232,    19,  5534 }, // Arabian/Bahrain -> Asia/Bahrain
    {     21,   232,   113,  3607 }, // Arabian/Iraq -> Asia/Baghdad
    {     21,   232,   127,  5547 }, // Arabian/Kuwait -> Asia/Kuwait
    {     21,   232,   190,  5559 }, // Arabian/Qatar -> Asia/Qatar
    {     21,   232,   258,  3584 }, // Arabian/world -> Asia/Riyadh
    {     21,   232,   259,  5570 }, // Arabian/Yemen -> Asia/Aden
    {     22,   240,   258,   166 }, // Argentina/world -> America/Argentina/Buenos_Aires
    {     23,   250,   258,  5580 }, // Argentina_Western/world -> America/Argentina/San_Luis
    {     24,   268,   258,  3747 }, // Armenia/world -> Asia/Yerevan
    {     25,   276,   258,   964 }, // Ashkhabad/world -> Asia/Ashgabat
    {     26,   286,     8,  5607 }, // Atlantic/Anguilla -> America/Anguilla
    {     26,   286,    10,  5624 }, // Atlantic/Antigua and Barbuda -> America/Antigua
    {     26,   286,    13,  5640 }, // Atlantic/Aruba -> America/Aruba
    {     26,   286,    21,  5654 }, // Atlantic/Barbados -> America/Barbados
    {     26,   286,    26,  5671 }, // Atlantic/Bermuda -> Atlantic/Bermuda
    {     26,   286,    34,  5688 }, // Atlantic/British Virgin Islands -> America/Tortola
    {     26,   286,    44,  5704 }, // Atlantic/Caribbean Netherlands -> America/Kralendijk
    {     26,   286,    62,  5723 }, // Atlantic/Curacao -> America/Curacao
    {     26,   286,    68,  5739 }, // Atlantic/Dominica -> America/Dominica
    {     26,   286,    95,  5756 }, // Atlantic/Greenland -> America/Thule
    {     26,   286,    96,  5770 }, // Atlantic/Grenada -> America/Grenada
    {     26,   286,    97,  5786 }, // Atlantic/Guadeloupe -> America/Guadeloupe
    {     26,   286,   148,  5805 }, // Atlantic/Martinique -> America/Martinique
    {     26,   286,   158,  5824 }, // Atlantic/Montserrat -> America/Montserrat
    {     26,   286,   189,  5843 }, // Atlantic/Puerto Rico -> America/Puerto_Rico
    {     26,   286,   197,  5863 }, // Atlantic/Saint Kitts and Nevis -> America/St_Kitts
    {     26,   286,   198,  5880 }, // Atlantic/Saint Lucia -> America/St_Lucia
    {     26,   286,   199,  5897 }, // Atlantic/Saint Martin -> America/Marigot
    {     26,   286,   201,  5913 }, // Atlantic/Saint Vincent and Grenadines -> America/St_Vincent
    {     26,   286,   211,  5932 }, // Atlantic/Sint Maarten -> America/Lower_Princes
    {     26,   286,   236,  5954 }, // Atlantic/Trinidad and Tobago -> America/Port_of_Spain
    {     26,   286,   249,   855 }, // Atlantic/United States Virgin Islands -> America/St_Thomas
    {     26,   286,   258,  1939 }, // Atlantic/world -> America/Halifax
    {     27,   295,   258,  1670 }, // Australia_Central/world -> Australia/Adelaide
    {     28,   313,   258,  3637 }, // Australia_CentralWestern/world -> Australia/Eucla
    {     29,   338,   258,  1463 }, // Australia_Eastern/world -> Australia/Sydney
    {     30,   356,   258,  1762 }, // Australia_Western/world -> Australia/Perth
    {     31,   374,   258,  3653 }, // Azerbaijan/world -> Asia/Baku
    {     32,   385,   258,  3663 }, // Azores/world -> Atlantic/Azores
    {     33,   392,   258,  3653 }, // Baku/world -> Asia/Baku
    {     34,   397,   258,  1093 }, // Bangladesh/world -> Asia/Dhaka
    {     35,   408,   258,   132 }, // Bering/world -> America/Adak
    {     36,   415,   258,  1317 }, // Bhutan/world -> Asia/Thimphu
    {     37,   422,   258,  4376 }, // Bolivia/world -> America/La_Paz
    {     38,   430,   258,  5976 }, // Borneo/world -> Asia/Kuching
    {     39,   437,   258,  1878 }, // Brasilia/world -> America/Sao_Paulo
    {     40,   446,   258,  2501 }, // British/world -> Europe/London
    {     41,   454,   258,  5989 }, // Brunei/world -> Asia/Brunei
    {     42,   461,   258,  3727 }, // Cape_Verde/world -> Atlantic/Cape_Verde
    {     43,   472,   258,  6001 }, // Casey/world -> Antarctica/Casey
    {     44,   478,    98,  6018 }, // Chamorro/Guam -> Pacific/Guam
    {     44,   478,   258,  6031 }, // Chamorro/world -> Pacific/Saipan
    {     45,   487,   258,  3015 }, // Chatham/world -> Pacific/Chatham
    {     46,   495,   258,  2201 }, // Chile/world -> America/Santiago
    {     47,   501,   258,  1053 }, // China/world -> Asia/Shanghai
    {     48,   507,   258,  6046 }, // Christmas/world -> Indian/Christmas
    {     49,   517,   258,  6063 }, // Cocos/world -> Indian/Cocos
    {     50,   523,   258,  4361 }, // Colombia/world -> America/Bogota
    {     51,   532,   258,  6076 }, // Cook/world -> Pacific/Rarotonga
    {     52,   537,   258,  2281 }, // Cuba/world -> America/Havana
    {     53,   542,   258,  1093 }, // Dacca/world -> Asia/Dhaka
    {     54,   548,   258,  6094 }, // Davis/world -> Antarctica/Davis
    {     55,   554,   258,  6111 }, // Dominican/world -> America/Santo_Domingo
    {     56,   564,   258,  6133 }, // DumontDUrville/world -> Antarctica/DumontDUrville
    {     57,   579,   258,  6159 }, // Dushanbe/world -> Asia/Dushanbe
    {     58,   588,   258,  6173 }, // Dutch_Guiana/world -> America/Paramaribo
    {     59,   601,   258,  6192 }, // East_Timor/world -> Asia/Dili
    {     60,   612,   258,  2237 }, // Easter/world -> Pacific/Easter
    {     61,   619,   258,  4841 }, // Ecuador/world -> America/Guayaquil
    {     62,   627,     3,  6202 }, // Europe_Central/Albania -> Europe/Tirane
    {     62,   627,     6,  6216 }, // Europe_Central/Andorra -> Europe/Andorra
    {     62,   627,    16,  6231 }, // Europe_Central/Austria -> Europe/Vienna
    {     62,   627,    23,  2167 }, // Europe_Central/Belgium -> Europe/Brussels
    {     62,   627,    29,  6245 }, // Europe_Central/Bosnia and Herzegovina -> Europe/Sarajevo
    {     62,   627,    60,  6261 }, // Europe_Central/Croatia -> Europe/Zagreb
    {     62,   627,    64,  6275 }, // Europe_Central/Czechia -> Europe/Prague
    {     62,   627,    65,  6289 }, // Europe_Central/Denmark -> Europe/Copenhagen
    {     62,   627,    91,  4757 }, // Europe_Central/Germany -> Europe/Berlin
    {     62,   627,    93,  6307 }, // Europe_Central/Gibraltar -> Europe/Gibraltar
    {     62,   627,   108,  3806 }, // Europe_Central/Hungary -> Europe/Budapest
    {     62,   627,   117,  6324 }, // Europe_Central/Italy -> Europe/Rome
    {     62,   627,   126,  6336 }, // Europe_Central/Kosovo -> Europe/Belgrade
    {     62,   627,   136,  6352 }, // Europe_Central/Liechtenstein -> Europe/Vaduz
    {     62,   627,   138,  6365 }, // Europe_Central/Luxembourg -> Europe/Luxembourg
    {     62,   627,   140,  6383 }, // Europe_Central/Macedonia -> Europe/Skopje
    {     62,   627,   146,  6397 }, // Europe_Central/Malta -> Europe/Malta
    {     62,   627,   155,  6410 }, // Europe_Central/Monaco -> Europe/Monaco
    {     62,   627,   157,  6424 }, // Europe_Central/Montenegro -> Europe/Podgorica
    {     62,   627,   165,  6441 }, // Europe_Central/Netherlands -> Europe/Amsterdam
    {     62,   627,   175,  6458 }, // Europe_Central/Norway -> Europe/Oslo
    {     62,   627,   187,  3190 }, // Europe_Central/Poland -> Europe/Warsaw
    {     62,   627,   203,  6470 }, // Europe_Central/San Marino -> Europe/San_Marino
    {     62,   627,   207,  6336 }, // Europe_Central/Serbia -> Europe/Belgrade
    {     62,   627,   212,  6488 }, // Europe_Central/Slovakia -> Europe/Bratislava
    {     62,   627,   213,  6506 }, // Europe_Central/Slovenia -> Europe/Ljubljana
    {     62,   627,   220,  4892 }, // Europe_Central/Spain -> Europe/Madrid
    {     62,   627,   224,  1429 }, // Europe_Central/Svalbard and Jan Mayen -> Arctic/Longyearbyen
    {     62,   627,   225,  6523 }, // Europe_Central/Sweden -> Europe/Stockholm
    {     62,   627,   226,  6540 }, // Europe_Central/Switzerland -> Europe/Zurich
    {     62,   627,   238,  6554 }, // Europe_Central/Tunisia -> Africa/Tunis
    {     62,   627,   253,  6567 }, // Europe_Central/Vatican City -> Europe/Vatican
    {     62,   627,   258,  4284 }, // Europe_Central/world -> Europe/Paris
    {     63,   642,     2,  6582 }, // Europe_Eastern/Aland Islands -> Europe/Mariehamn
    {     63,   642,    36,  6599 }, // Europe_Eastern/Bulgaria -> Europe/Sofia
    {     63,   642,    63,  2554 }, // Europe_Eastern/Cyprus -> Asia/Nicosia
    {     63,   642,    71,  2320 }, // Europe_Eastern/Egypt -> Africa/Cairo
    {     63,   642,    83,  6612 }, // Europe_Eastern/Finland -> Europe/Helsinki
    {     63,   642,    94,  2300 }, // Europe_Eastern/Greece -> Europe/Athens
    {     63,   642,   132,  4086 }, // Europe_Eastern/Lebanon -> Asia/Beirut
    {     63,   642,   258,  3928 }, // Europe_Eastern/world -> Europe/Bucharest
    {     64,   657,   193,  3979 }, // Europe_Further_Eastern/Russia -> Europe/Kaliningrad
    {     64,   657,   258,  3693 }, // Europe_Further_Eastern/world -> Europe/Minsk
    {     65,   680,    81,  1395 }, // Europe_Western/Faroe Islands -> Atlantic/Faroe
    {     65,   680,   258,  6628 }, // Europe_Western/world -> Atlantic/Canary
    {     66,   695,   258,  6644 }, // Falkland/world -> Atlantic/Stanley
    {     67,   704,   258,  3902 }, // Fiji/world -> Pacific/Fiji
    {     68,   709,   258,  4345 }, // French_Guiana/world -> America/Cayenne
    {     69,   723,   258,  6661 }, // French_Southern/world -> Indian/Kerguelen
    {     70,   739,   258,  3778 }, // Frunze/world -> Asia/Bishkek
    {     71,   746,   258,  6678 }, // Galapagos/world -> Pacific/Galapagos
    {     72,   756,   258,  6696 }, // Gambier/world -> Pacific/Gambier
    {     73,   764,   258,  3915 }, // Georgia/world -> Asia/Tbilisi
    {     74,   772,   258,  6712 }, // Gilbert_Islands/world -> Pacific/Tarawa
    {     75,   788,    37,  6727 }, // GMT/Burkina Faso -> Africa/Ouagadougou
    {     75,   788,    89,  6746 }, // GMT/Gambia -> Africa/Banjul
    {     75,   788,    92,  6760 }, // GMT/Ghana -> Africa/Accra
    {     75,   788,   102,  6773 }, // GMT/Guinea -> Africa/Conakry
    {     75,   788,   114,  2338 }, // GMT/Ireland -> Europe/Dublin
    {     75,   788,   118,  6788 }, // GMT/Ivory Coast -> Africa/Abidjan
    {     75,   788,   145,    44 }, // GMT/Mali -> Africa/Bamako
    {     75,   788,   149,  6803 }, // GMT/Mauritania -> Africa/Nouakchott
    {     75,   788,   196,  6821 }, // GMT/Saint Helena -> Atlantic/St_Helena
    {     75,   788,   206,  6840 }, // GMT/Senegal -> Africa/Dakar
    {     75,   788,   209,  6853 }, // GMT/Sierra Leone -> Africa/Freetown
    {     75,   788,   233,  6869 }, // GMT/Togo -> Africa/Lome
    {     75,   788,   246,  2501 }, // GMT/United Kingdom -> Europe/London
    {     75,   788,   258,  2748 }, // GMT/world -> Atlantic/Reykjavik
    {     76,   792,   258,  6881 }, // Goose_Bay/world -> America/Goose_Bay
    {     77,   802,   258,   393 }, // Greenland/world -> America/Nuuk
    {     78,   812,   258,  6899 }, // Greenland_Central/world -> America/Scoresbysund
    {     79,   830,   258,  6899 }, // Greenland_Eastern/world -> America/Scoresbysund
    {     80,   848,   258,   393 }, // Greenland_Western/world -> America/Nuuk
    {     81,   866,   258,  6018 }, // Guam/world -> Pacific/Guam
    {     82,   871,   176,  6920 }, // Gulf/Oman -> Asia/Muscat
    {     82,   871,   258,  3596 }, // Gulf/world -> Asia/Dubai
    {     83,   876,   258,  6932 }, // Guyana/world -> America/Guyana
    {     84,   883,   258,  2723 }, // Hawaii_Aleutian/world -> Pacific/Honolulu
    {     85,   899,   258,  2704 }, // Hong_Kong/world -> Asia/Hong_Kong
    {     86,   909,   258,  4771 }, // Hovd/world -> Asia/Hovd
    {     87,   914,   221,  4511 }, // India/Sri Lanka -> Asia/Colombo
    {     87,   914,   258,   992 }, // India/world -> Asia/Kolkata
    {     88,   920,   258,  6947 }, // Indian_Ocean/world -> Indian/Chagos
    {     89,   933,    39,  6961 }, // Indochina/Cambodia -> Asia/Phnom_Penh
    {     89,   933,   129,  6977 }, // Indochina/Laos -> Asia/Vientiane
    {     89,   933,   258,  4466 }, // Indochina/world -> Asia/Bangkok
    {     90,   943,   258,  1349 }, // Indonesia_Central/world -> Asia/Makassar
    {     91,   961,   258,  6992 }, // Indonesia_Eastern/world -> Asia/Jayapura
    {     92,   979,   258,  7006 }, // Indonesia_Western/world -> Asia/Jakarta
    {     93,   997,   258,  2772 }, // Iran/world -> Asia/Tehran
    {     94,  1002,   258,  2338 }, // Irish/world -> Europe/Dublin
    {     95,  1008,   258,  4184 }, // Irkutsk/world -> Asia/Irkutsk
    {     96,  1016,   258,  1290 }, // Israel/world -> Asia/Jerusalem
    {     97,  1023,   258,  2821 }, // Japan/world -> Asia/Tokyo
    {     98,  1029,   258,  4316 }, // Kamchatka/world -> Asia/Kamchatka
    {     99,  1039,   258,  4239 }, // Karachi/world -> Asia/Karachi
    {    100,  1047,   258,  5395 }, // Kazakhstan/world -> Asia/Almaty
    {    101,  1058,   258,  5395 }, // Kazakhstan_Eastern/world -> Asia/Almaty
    {    102,  1077,   258,  5368 }, // Kazakhstan_Western/world -> Asia/Aqtobe
    {    103,  1096,   258,  4269 }, // Kizilorda/world -> Asia/Qyzylorda
    {    104,  1106,   258,  3279 }, // Korea/world -> Asia/Seoul
    {    105,  1112,   258,  7019 }, // Kosrae/world -> Pacific/Kosrae
    {    106,  1119,   258,  4197 }, // Krasnoyarsk/world -> Asia/Krasnoyarsk
    {    107,  1131,   258,  4331 }, // Kuybyshev/world -> Europe/Samara
    {    108,  1141,   258,  2842 }, // Kwajalein/world -> Pacific/Kwajalein
    {    109,  1151,   258,  3778 }, // Kyrgystan/world -> Asia/Bishkek
    {    110,  1161,   258,  4511 }, // Lanka/world -> Asia/Colombo
    {    111,  1167,   258,  7034 }, // Liberia/world -> Africa/Monrovia
    {    112,  1175,   258,  3998 }, // Line_Islands/world -> Pacific/Kiritimati
    {    113,  1188,   258,  1547 }, // Lord_Howe/world -> Australia/Lord_Howe
    {    114,  1198,   258,  1211 }, // Macau/world -> Asia/Macau
    {    115,  1204,   258,  4017 }, // Magadan/world -> Asia/Magadan
    {    116,  1212,   258,  4859 }, // Malaya/world -> Asia/Kuala_Lumpur
    {    117,  1219,   258,  5976 }, // Malaysia/world -> Asia/Kuching
    {    118,  1228,   258,  7050 }, // Maldives/world -> Indian/Maldives
    {    119,  1237,   258,  4051 }, // Marquesas/world -> Pacific/Marquesas
    {    120,  1247,   258,  4877 }, // Marshall_Islands/world -> Pacific/Majuro
    {    121,  1264,   258,  4069 }, // Mauritius/world -> Indian/Mauritius
    {    122,  1274,   258,  7066 }, // Mawson/world -> Antarctica/Mawson
    {    123,  1281,   258,  2917 }, // Mexico_Pacific/world -> America/Mazatlan
    {    124,  1296,   258,  1021 }, // Mongolia/world -> Asia/Ulaanbaatar
    {    125,  1305,   258,  3537 }, // Moscow/world -> Europe/Moscow
    {    126,  1312,   258,  1235 }, // Myanmar/world -> Asia/Yangon
    {    127,  1320,   258,  7084 }, // Nauru/world -> Pacific/Nauru
    {    128,  1326,   258,  1185 }, // Nepal/world -> Asia/Kathmandu
    {    129,  1332,   258,  7098 }, // New_Caledonia/world -> Pacific/Noumea
    {    130,  1346,     9,  7113 }, // New_Zealand/Antarctica -> Antarctica/McMurdo
    {    130,  1346,   258,   932 }, // New_Zealand/world -> Pacific/Auckland
    {    131,  1358,   258,  2061 }, // Newfoundland/world -> America/St_Johns
    {    132,  1371,   258,  7132 }, // Niue/world -> Pacific/Niue
    {    133,  1376,   258,  4168 }, // Norfolk/world -> Pacific/Norfolk
    {    134,  1384,   258,  1850 }, // Noronha/world -> America/Noronha
    {    135,  1392,   258,  6031 }, // North_Mariana/world -> Pacific/Saipan
    {    136,  1406,   258,  4135 }, // Novosibirsk/world -> Asia/Novosibirsk
    {    137,  1418,   258,  4229 }, // Omsk/world -> Asia/Omsk
    {    138,  1423,   258,  7145 }, // Oral/world -> Asia/Oral
    {    139,  1428,   258,  4239 }, // Pakistan/world -> Asia/Karachi
    {    140,  1437,   258,  7155 }, // Palau/world -> Pacific/Palau
    {    141,  1443,   258,  4807 }, // Papua_New_Guinea/world -> Pacific/Port_Moresby
    {    142,  1460,   258,  4252 }, // Paraguay/world -> America/Asuncion
    {    143,  1469,   258,  7169 }, // Peru/world -> America/Lima
    {    144,  1474,   258,  7182 }, // Philippines/world -> Asia/Manila
    {    145,  1486,   258,  3049 }, // Phoenix_Islands/world -> Pacific/Kanton
    {    146,  1502,   258,  4391 }, // Pierre_Miquelon/world -> America/Miquelon
    {    147,  1518,   258,  7194 }, // Pitcairn/world -> Pacific/Pitcairn
    {    148,  1527,   258,  3096 }, // Ponape/world -> Pacific/Pohnpei
    {    149,  1534,   258,  4214 }, // Pyongyang/world -> Asia/Pyongyang
    {    150,  1544,   258,  4269 }, // Qyzylorda/world -> Asia/Qyzylorda
    {    151,  1554,   258,  7211 }, // Reunion/world -> Indian/Reunion
    {    152,  1562,   258,  7226 }, // Rothera/world -> Antarctica/Rothera
    {    153,  1570,   258,  4408 }, // Sakhalin/world -> Asia/Sakhalin
    {    154,  1579,   258,  4331 }, // Samara/world -> Europe/Samara
    {    155,  1586,   258,  7245 }, // Samarkand/world -> Asia/Samarkand
    {    156,  1596,   258,  3126 }, // Samoa/world -> Pacific/Pago_Pago
    {    157,  1602,   258,  7260 }, // Seychelles/world -> Indian/Mahe
    {    158,  1613,   258,  5523 }, // Shevchenko/world -> Asia/Aqtau
    {    159,  1624,   258,  3300 }, // Singapore/world -> Asia/Singapore
    {    160,  1634,   258,  3822 }, // Solomon/world -> Pacific/Guadalcanal
    {    161,  1642,   258,  7272 }, // South_Georgia/world -> Atlantic/South_Georgia
    {    162,  1656,   258,  6173 }, // Suriname/world -> America/Paramaribo
    {    163,  1665,   258,  3883 }, // Sverdlovsk/world -> Asia/Yekaterinburg
    {    164,  1676,   258,  7295 }, // Syowa/world -> Antarctica/Syowa
    {    165,  1682,   258,  7312 }, // Tahiti/world -> Pacific/Tahiti
    {    166,  1689,   258,  3263 }, // Taipei/world -> Asia/Taipei
    {    167,  1696,   258,  6159 }, // Tajikistan/world -> Asia/Dushanbe
    {    168,  1707,   258,  4781 }, // Tashkent/world -> Asia/Tashkent
    {    169,  1716,   258,  3915 }, // Tbilisi/world -> Asia/Tbilisi
    {    170,  1724,   258,  7327 }, // Tokelau/world -> Pacific/Fakaofo
    {    171,  1732,   258,  4583 }, // Tonga/world -> Pacific/Tongatapu
    {    172,  1738,   258,  3157 }, // Truk/world -> Pacific/Chuuk
    {    173,  1743,   258,  1130 }, // Turkey/world -> Europe/Istanbul
    {    174,  1750,   258,   964 }, // Turkmenistan/world -> Asia/Ashgabat
    {    175,  1763,   258,  7343 }, // Tuvalu/world -> Pacific/Funafuti
    {    176,  1770,   258,  7145 }, // Uralsk/world -> Asia/Oral
    {    177,  1777,   258,  4098 }, // Uruguay/world -> America/Montevideo
    {    178,  1785,   258,  1159 }, // Urumqi/world -> Asia/Urumqi
    {    179,  1792,   258,  4781 }, // Uzbekistan/world -> Asia/Tashkent
    {    180,  1803,   258,  7360 }, // Vanuatu/world -> Pacific/Efate
    {    181,  1811,   258,  4694 }, // Venezuela/world -> America/Caracas
    {    182,  1821,   258,  4710 }, // Vladivostok/world -> Asia/Vladivostok
    {    183,  1833,   258,  4727 }, // Volgograd/world -> Europe/Volgograd
    {    184,  1843,   258,  7374 }, // Vostok/world -> Antarctica/Vostok
    {    185,  1850,   258,  7392 }, // Wake/world -> Pacific/Wake
    {    186,  1855,   258,  7405 }, // Wallis/world -> Pacific/Wallis
    {    187,  1862,   258,  4828 }, // Yakutsk/world -> Asia/Yakutsk
    {    188,  1870,   258,  3883 }, // Yekaterinburg/world -> Asia/Yekaterinburg
    {    189,  1884,   258,  3747 }, // Yerevan/world -> Asia/Yerevan
    {    190,  1892,   258,  2144 }, // Yukon/world -> America/Whitehorse
};

// IANA ID Index, MetaZone Key, interval start, end
static inline constexpr ZoneMetaHistory zoneHistoryTable[] = {
    {   6788,    75,         0,4294967295 },
    {   6760,    75,         0,4294967295 },
    {   5059,     4,         0,4294967295 },
    {   7420,    65,         0,   4103940 },
    {   7420,    62,   4103940,   5162340 },
    {   7420,    65,   5162340,   5958720 },
    {   7420,    62,   5958720,4294967295 },
    {     14,     4,         0,4294967295 },
    {     44,    75,         0,4294967295 },
    {   5257,     7,         0,4294967295 },
    {   6746,    75,         0,4294967295 },
    {   7435,     5,         0,   2629500 },
    {   7435,    75,   2629500,4294967295 },
    {   4957,     3,         0,4294967295 },
    {   5287,     7,         0,4294967295 },
    {   4922,     3,         0,4294967295 },
    {   2320,    63,         0,4294967295 },
    {   4117,    65,         0,   7470720 },
    {   4117,    62,   7470720,   8415300 },
    {   4117,    65,   8415300,  25678200 },
    {   7449,    65,         0,   7470720 },
    {   7449,    62,   7470720,4294967295 },
    {   6773,    75,         0,4294967295 },
    {   6840,    75,         0,4294967295 },
    {   5130,     4,         0,4294967295 },
    {   5043,     4,         0,4294967295 },
    {   5243,     7,         0,4294967295 },
    {   5166,     5,         0,   3304860 },
    {   5166,    65,   3304860,  25678200 },
    {   6853,    75,         0,4294967295 },
    {   4906,     3,         0,4294967295 },
    {   5015,     3,         0,4294967295 },
    {   4479,     6,         0,4294967295 },
    {   4499,     3,         0,  15798840 },
    {   4499,     4,  15798840,  26868780 },
    {   4499,     3,  26868780,4294967295 },
    {   5151,     4,         0,4294967295 },
    {   4524,     3,         0,  15798840 },
    {   4524,     4,  15798840,  25158060 },
    {   4524,     3,  25158060,4294967295 },
    {   4973,     3,         0,4294967295 },
    {   5306,     7,         0,4294967295 },
    {   4744,     7,         0,4294967295 },
    {   5336,     7,         0,4294967295 },
    {   6869,    75,         0,4294967295 },
    {   5211,     7,         0,4294967295 },
    {   4939,     3,         0,4294967295 },
    {   5001,     3,         0,4294967295 },
    {   5322,     7,         0,4294967295 },
    {   4987,     3,         0,4294967295 },
    {   5197,     6,         0,4294967295 },
    {   5182,     6,         0,4294967295 },
    {   5113,     4,         0,4294967295 },
    {   7034,   111,         0,   1225485 },
    {   7034,    75,   1225485,4294967295 },
    {   3853,     4,         0,4294967295 },
    {   5271,     7,         0,4294967295 },
    {   5354,     7,         0,4294967295 },
    {   6803,    75,         0,4294967295 },
    {   6727,    75,         0,4294967295 },
    {   5225,     7,         0,4294967295 },
    {   4435,    75,         0,  25246140 },
    {   4435,     7,  25246140,  25771740 },
    {   4435,    75,  25771740,4294967295 },
    {   2866,    63,         0,   6311400 },
    {   2866,    62,   6311400,  10696260 },
    {   2866,    63,  10696260,  14067240 },
    {   2866,    62,  14067240,  14598600 },
    {   2866,    63,  14598600,  22541760 },
    {   2866,    62,  22541760,  23044320 },
    {   2866,    63,  23044320,4294967295 },
    {   6554,    62,         0,4294967295 },
    {   4152,     6,         0,  10632840 },
    {   4152,     3,  10632840,  12736680 },
    {   4152,     7,  12736680,  25146600 },
    {   4152,     3,  25146600,4294967295 },
    {    132,    35,         0,   7272720 },
    {    132,    84,   7317240,4294967295 },
    {   3346,    10,         0,   7272660 },
    {   3346,     9,   7317180,4294967295 },
    {   5607,    26,         0,4294967295 },
    {   5624,    26,         0,4294967295 },
    {   4554,    39,         0,4294967295 },
    {    166,    22,         0,4294967295 },
    {     91,    22,         0,  11132760 },
    {     91,    22,  11465520,  18100980 },
    {     91,    23,  18100980,  18128400 },
    {     91,    22,  18128400,4294967295 },
    {    270,    22,         0,  11132760 },
    {    270,    22,  11465520,4294967295 },
    {    441,    22,         0,  10608600 },
    {    441,    22,  11445360,4294967295 },
    {   7462,    22,         0,  11129880 },
    {   7462,    23,  11129880,  11226480 },
    {   7462,    22,  11226480,  18100980 },
    {   7462,    23,  18100980,  18128400 },
    {   7462,    22,  18128400,4294967295 },
    {    565,    22,         0,  10608600 },
    {    565,    22,  11989680,  18088020 },
    {    565,    22,  18269520,4294967295 },
    {   7489,    22,         0,  18100980 },
    {   7489,    23,  18100980,  18128400 },
    {   7489,    22,  18128400,4294967295 },
    {   7520,    22,         0,  11132760 },
    {   7520,    22,  11465520,4294967295 },
    {   7544,    22,         0,  11129880 },
    {   7544,    23,  11129880,  11226480 },
    {   7544,    22,  11226480,  18099540 },
    {   7544,    23,  18099540,  18178800 },
    {   7544,    22,  18178800,4294967295 },
    {   5580,    22,         0,  10623000 },
    {   5580,    23,  10623000,  11262480 },
    {   5580,    22,  11262480,  15648660 },
    {   5580,    23,  15648660,  15867540 },
    {   5580,    22,  15867540,  18099540 },
    {   5580,    23,  18099540,  18178800 },
    {   5580,    22,  18178800,  20014680 },
    {   5580,    23,  20014680,  20920560 },
    {   5580,    22,  20920560,4294967295 },
    {   7571,    22,         0,  11132760 },
    {   7571,    22,  11465520,  18100980 },
    {   7571,    23,  18100980,  18118320 },
    {   7571,    22,  18118320,4294967295 },
    {   7597,    22,         0,  18098100 },
    {   7597,    22,  18128400,4294967295 },
    {   5640,    26,         0,4294967295 },
    {   4252,   142,         0,4294967295 },
    {    237,    14,         0,4294967295 },
    {   3679,    39,         0,4294967295 },
    {   7623,    15,         0,  21172860 },
    {   7623,    13,  21172860,4294967295 },
    {   5654,    26,         0,4294967295 },
    {   7646,    39,         0,4294967295 },
    {   5407,    13,         0,4294967295 },
    {   7660,    26,         0,4294967295 },
    {   7681,    12,         0,4294967295 },
    {   4361,    50,         0,4294967295 },
    {   7699,    15,         0,4294967295 },
    {   7713,    15,         0,  15689280 },
    {   7713,    13,  15689280,  16213380 },
    {   7713,    14,  16213380,  16223340 },
    {   7713,    13,  16223340,  16435260 },
    {   7713,    15,  16435260,4294967295 },
    {   7735,    12,         0,4294967295 },
    {   3868,    13,         0,   6303360 },
    {   3868,    14,   6303360,   6841740 },
    {   3868,    13,   6841740,  14630820 },
    {   3868,    14,  14630820,  15033960 },
    {   3868,    13,  15033960,  23712960 },
    {   3868,    14,  23712960,4294967295 },
    {   4694,   181,         0,4294967295 },
    {   4345,    68,         0,4294967295 },
    {   5496,    14,         0,4294967295 },
    {   2260,    13,         0,4294967295 },
    {   7756,    13,         0,  14862780 },
    {   7756,   123,  14862780,  27785280 },
    {   7756,    13,  27785280,4294967295 },
    {   7774,    13,         0,  14862780 },
    {   7774,   123,  14862780,  27785280 },
    {   7774,    13,  27785280,  27829800 },
    {   7774,    15,  27829800,4294967295 },
    {   5422,    13,         0,4294967295 },
    {   7796,    15,         0,4294967295 },
    {   3791,    12,         0,4294967295 },
    {   5723,    26,         0,4294967295 },
    {   7812,    80,         0,  13674420 },
    {   7812,    75,  13674420,4294967295 },
    {   7833,    16,   2010780,  26736900 },
    {   7833,   190,  26736900,4294967295 },
    {   7848,    16,         0,   1400220 },
    {   7848,    15,   1400220,4294967295 },
    {    805,    15,         0,4294967295 },
    {   3465,    14,         0,4294967295 },
    {   5739,    26,         0,4294967295 },
    {    893,    15,         0,4294967295 },
    {   7869,     1,         0,  20238060 },
    {   7869,    12,  20238060,  23067600 },
    {   7869,     1,  23067600,4294967295 },
    {   5441,    13,         0,4294967295 },
    {   7886,    16,         0,  23763480 },
    {   7886,    15,  23763480,4294967295 },
    {   7906,    39,         0,4294967295 },
    {   7924,    26,         0,4294967295 },
    {   6881,    26,         0,   9600721 },
    {   6881,    76,   9600721,   9903001 },
    {   6881,    26,   9903001,4294967295 },
    {   4612,    14,         0,  23763300 },
    {   4612,    26,  23763300,  25345860 },
    {   4612,    14,  25345860,4294967295 },
    {   5770,    26,         0,4294967295 },
    {   5786,    26,         0,4294967295 },
    {   3760,    13,         0,4294967295 },
    {   4841,    61,         0,4294967295 },
    {   6932,    83,         0,4294967295 },
    {   1939,    26,         0,4294967295 },
    {   2281,    52,         0,4294967295 },
    {   7942,   123,         0,4294967295 },
    {    348,    14,         0,4294967295 },
    {    481,    13,         0,  11475780 },
    {    481,    14,  11475780,  19066020 },
    {    481,    13,  19066020,4294967295 },
    {   7961,    14,         0,   2111460 },
    {   7961,    13,   2111460,   2534820 },
    {   7961,    14,   2534820,4294967295 },
    {   7985,    13,         0,   4117380 },
    {   7985,    14,   4117380,  19066020 },
    {   7985,    13,  19066020,  19902660 },
    {   7985,    14,  19902660,4294967295 },
    {   8012,    14,         0,  19066020 },
    {   8012,    13,  19066020,4294967295 },
    {   8038,    14,         0,4294967295 },
    {   8060,    14,         0,  19066020 },
    {   8060,    13,  19066020,  19902660 },
    {   8060,    14,  19902660,4294967295 },
    {   8086,    14,         0,  19066020 },
    {   8086,    13,  19066020,  19560000 },
    {   8086,    14,  19560000,4294967295 },
    {   8110,    16,         0,   4903800 },
    {   8110,    15,   4903800,4294967295 },
    {    660,    14,         0,  15689160 },
    {    660,    13,  15689160,  16213380 },
    {    660,    14,  16213380,4294967295 },
    {   2799,    14,         0,4294967295 },
    {   5380,    16,         0,   5427960 },
    {   5380,    16,   5690040,   7272540 },
    {   5380,     9,   7317180,4294967295 },
    {    521,    14,         0,   2111460 },
    {    521,    13,   2111460,   2534820 },
    {    521,    14,   2534820,4294967295 },
    {   8125,    13,         0,  16213380 },
    {   8125,    14,  16213380,4294967295 },
    {   5704,    26,         0,4294967295 },
    {   4376,    37,         0,4294967295 },
    {   7169,   143,         0,4294967295 },
    {   3239,    16,         0,4294967295 },
    {   5932,    26,         0,4294967295 },
    {   8153,    39,         0,4294967295 },
    {   8168,    13,         0,   1751400 },
    {   8168,    14,   1751400,   2695980 },
    {   8168,    13,   2695980,  11571000 },
    {   8168,    14,  11571000,  11955180 },
    {   8168,    13,  11955180,  12097800 },
    {   8168,    14,  12097800,  14201580 },
    {   8168,    13,  14201580,4294967295 },
    {   1908,    12,         0,4294967295 },
    {   5897,    26,         0,4294967295 },
    {   5805,    26,         0,4294967295 },
    {   8184,    13,         0,4294967295 },
    {   2917,   123,         0,4294967295 },
    {   8202,    14,         0,   1748580 },
    {   8202,    13,   1748580,4294967295 },
    {   8220,    13,         0,   6303360 },
    {   8220,    14,   6303360,   6751140 },
    {   8220,    13,   6751140,4294967295 },
    {   8235,    16,         0,  24106200 },
    {   8235,     9,  24106200,  25688760 },
    {   8235,    16,  25688760,  25799640 },
    {   8235,     9,  25799640,4294967295 },
    {   2949,    13,         0,4294967295 },
    {   4391,    26,         0,   5433360 },
    {   4391,   146,   5433360,4294967295 },
    {   8254,    26,         0,4294967295 },
    {   8270,    13,         0,4294967295 },
    {   4098,   177,         0,4294967295 },
    {   5824,    26,         0,4294967295 },
    {   5481,    14,         0,4294967295 },
    {   2379,    14,         0,4294967295 },
    {   8288,    35,         0,   7272720 },
    {   8288,     9,   7317180,4294967295 },
    {   1850,   134,         0,4294967295 },
    {   8301,    15,         0,  21485280 },
    {   8301,    13,  21485280,4294967295 },
    {   8329,    15,         0,  12000000 },
    {   8329,    13,  12000000,4294967295 },
    {   8357,    15,         0,  17785920 },
    {   8357,    13,  17785920,4294967295 },
    {    393,    80,         0,  28523580 },
    {    393,    77,  28523580,4294967295 },
    {   8388,    13,         0,  14862780 },
    {   8388,    15,  14862780,  27785280 },
    {   8388,    13,  27785280,4294967295 },
    {   2356,    14,         0,4294967295 },
    {   6173,    58,         0,   3094770 },
    {   6173,   162,   3094770,4294967295 },
    {   2973,    15,         0,4294967295 },
    {   3945,    14,         0,4294967295 },
    {   5954,    26,         0,4294967295 },
    {   8404,    12,         0,4294967295 },
    {   5843,    26,         0,4294967295 },
    {   4030,    46,         0,  24680100 },
    {   8424,    13,         0,  16213380 },
    {   8424,    14,  16213380,  16435200 },
    {   8424,    13,  16435200,4294967295 },
    {   8445,    39,         0,4294967295 },
    {   1995,    13,         0,4294967295 },
    {   8460,    13,         0,  16213380 },
    {   8460,    14,  16213380,  16435200 },
    {   8460,    13,  16435200,  19368420 },
    {   8460,    14,  19368420,  19560000 },
    {   8460,    13,  19560000,4294967295 },
    {    695,     1,         0,  20238060 },
    {    695,    12,  20238060,  23067600 },
    {    695,     1,  23067600,4294967295 },
    {   8477,    12,         0,  20238000 },
    {   8477,    39,  20238000,4294967295 },
    {   2201,    46,         0,4294967295 },
    {   6111,    55,         0,   2534700 },
    {   6111,    26,   2534700,  16213320 },
    {   6111,    14,  16213320,  16263720 },
    {   6111,    26,  16263720,4294967295 },
    {   1878,    39,         0,4294967295 },
    {   6899,    78,         0,   5911320 },
    {   6899,    79,   5911320,  28530780 },
    {   6899,    77,  28530780,4294967295 },
    {   8494,    16,         0,   7272540 },
    {   8494,     9,   7317180,4294967295 },
    {   8508,    26,         0,4294967295 },
    {   2061,   131,         0,4294967295 },
    {   5863,    26,         0,4294967295 },
    {   5880,    26,         0,4294967295 },
    {    855,    26,         0,4294967295 },
    {   5913,    26,         0,4294967295 },
    {   8530,    15,         0,   1224540 },
    {   8530,    13,   1224540,4294967295 },
    {   5461,    13,         0,4294967295 },
    {   5756,    26,         0,4294967295 },
    {    313,    16,         0,4294967295 },
    {    608,    14,         0,4294967295 },
    {   5688,    26,         0,4294967295 },
    {   2093,    16,         0,4294967295 },
    {   2144,    16,         0,  26736900 },
    {   2144,   190,  26736900,4294967295 },
    {    734,    13,         0,4294967295 },
    {   8552,     9,   7317180,4294967295 },
    {   6001,    30,         0,  20930040 },
    {   6001,    43,  20930040,  21128580 },
    {   6001,    30,  21128580,  21995640 },
    {   6001,    43,  21995640,  22164060 },
    {   6001,    30,  22164060,  24617760 },
    {   6001,    43,  24617760,  25345020 },
    {   6001,    30,  25345020,  25647600 },
    {   6001,    43,  25647600,  25879200 },
    {   6001,    30,  25879200,  26168820 },
    {   6001,    43,  26168820,  26393280 },
    {   6001,    30,  26393280,  26695681 },
    {   6001,    43,  26695681,  26927340 },
    {   6001,    30,  26927340,  27219841 },
    {   6001,    43,  27219841,  27451500 },
    {   6001,    30,  27451500,  27744001 },
    {   6001,    43,  27744001,  27971520 },
    {   6001,    30,  27971520,4294967295 },
    {   6094,    54,         0,4294967295 },
    {   6133,    56,         0,4294967295 },
    {   8568,    29,         0,4294967295 },
    {   7066,   122,         0,4294967295 },
    {   7113,   130,         0,4294967295 },
    {   8589,    22,         0,   6484500 },
    {   8589,    46,   6484500,  24680100 },
    {   7226,   152,         0,4294967295 },
    {   7295,   164,         0,4294967295 },
    {   8607,    75,         0,4294967295 },
    {   7374,   184,         0,4294967295 },
    {   1429,    62,         0,4294967295 },
    {   5570,    21,         0,4294967295 },
    {   5395,    11,         0,  18319440 },
    {   5395,   101,  18319440,  28487160 },
    {   5395,   100,  28487160,4294967295 },
    {   3968,    63,         0,  27781800 },
    {   5511,    17,         0,  21161640 },
    {   5511,   115,  21161640,  23570760 },
    {   5511,    17,  23570760,4294967295 },
    {   5523,   158,         0,  11547060 },
    {   5523,    19,  11547060,  18319560 },
    {   5523,   102,  18319560,  28487160 },
    {   5523,   100,  28487160,4294967295 },
    {   5368,     8,         0,  11547060 },
    {   5368,    20,  11547060,  18319500 },
    {   5368,   102,  18319500,  28487160 },
    {   5368,   100,  28487160,4294967295 },
    {    964,    25,         0,  11475120 },
    {    964,   174,  11475120,4294967295 },
    {   8624,   102,  18319560,  28487160 },
    {   8624,   100,  28487160,4294967295 },
    {   3607,    21,         0,4294967295 },
    {   5534,    82,         0,   1269840 },
    {   5534,    21,   1269840,4294967295 },
    {   3653,    33,         0,  11391600 },
    {   3653,    31,  11391600,4294967295 },
    {   4466,    89,         0,4294967295 },
    {   4086,    63,         0,4294967295 },
    {   3778,    70,         0,  11393040 },
    {   3778,   109,  11393040,4294967295 },
    {   5989,    41,         0,4294967295 },
    {   4601,   187,         0,  23570880 },
    {   4601,    95,  23570880,  24316920 },
    {   4601,   187,  24316920,4294967295 },
    {   4511,    87,         0,  13882710 },
    {   4511,   110,  13882710,  19083990 },
    {   4511,    87,  19083990,4294967295 },
    {   4540,    63,         0,  27781740 },
    {   1093,    53,         0,    646200 },
    {   1093,    34,    646200,4294967295 },
    {   6192,    59,         0,   3331620 },
    {   6192,    90,   3331620,  16152000 },
    {   6192,    59,  16152000,4294967295 },
    {   3596,    82,         0,4294967295 },
    {   6159,    57,         0,  11406060 },
    {   6159,   167,  11406060,4294967295 },
    {   8636,    63,         0,  24554700 },
    {   8651,    96,         0,  13674120 },
    {   8651,    63,  13674120,4294967295 },
    {   4795,    96,         0,  13674120 },
    {   4795,    63,  13674120,4294967295 },
    {   1259,    89,   2863680,4294967295 },
    {   2704,    85,         0,4294967295 },
    {   4771,    86,         0,4294967295 },
    {   4184,    95,         0,4294967295 },
    {   7006,    92,         0,4294967295 },
    {   6992,    91,         0,4294967295 },
    {   1290,    96,         0,4294967295 },
    {   3560,     2,         0,4294967295 },
    {   4316,    98,         0,4294967295 },
    {   4239,    99,         0,    646260 },
    {   4239,   139,    646260,4294967295 },
    {   1185,   128,         0,4294967295 },
    {   8661,   187,         0,  17881380 },
    {   8661,   182,  17881380,  21930540 },
    {   8661,   187,  21930540,4294967295 },
    {    992,    87,         0,4294967295 },
    {   4197,   106,         0,4294967295 },
    {   4859,   116,         0,   6311040 },
    {   4859,   117,   6311040,4294967295 },
    {   5976,    38,         0,   6311040 },
    {   5976,   117,   6311040,4294967295 },
    {   5547,    21,         0,4294967295 },
    {   1211,   114,         0,  15760320 },
    {   1211,    47,  15760320,4294967295 },
    {   4017,   115,         0,4294967295 },
    {   1349,    90,         0,4294967295 },
    {   7182,   144,         0,4294967295 },
    {   6920,    82,         0,4294967295 },
    {   2554,    63,         0,4294967295 },
    {   8675,   106,         0,  21161940 },
    {   8675,   136,  21161940,  23571060 },
    {   8675,   106,  23571060,4294967295 },
    {   4135,   136,         0,4294967295 },
    {   4229,   137,         0,4294967295 },
    {   7145,   176,         0,  11547120 },
    {   7145,   138,  11547120,  18319560 },
    {   7145,   102,  18319560,  28487160 },
    {   7145,   100,  28487160,4294967295 },
    {   6961,    89,         0,4294967295 },
    {   8693,    90,         0,   9466080 },
    {   8693,    92,   9466080,4294967295 },
    {   4214,   104,         0,  23992740 },
    {   4214,   149,  23992740,  25424100 },
    {   4214,   104,  25424100,4294967295 },
    {   5559,    82,         0,   1269840 },
    {   5559,    21,   1269840,4294967295 },
    {   8708,   101,  18319500,  28487160 },
    {   8708,   100,  28487160,4294967295 },
    {   4269,   103,         0,  11547060 },
    {   4269,   150,  11547060,  18319500 },
    {   4269,   101,  18319500,  25755480 },
    {   4269,   102,  25755480,  28487160 },
    {   4269,   100,  28487160,4294967295 },
    {   3584,    21,         0,4294967295 },
    {   4408,   153,         0,4294967295 },
    {   7245,   155,         0,   6178680 },
    {   7245,   168,   6178680,   6440760 },
    {   7245,   155,   6440760,  11394360 },
    {   7245,   179,  11394360,4294967295 },
    {   3279,   104,         0,4294967295 },
    {   1053,    47,         0,4294967295 },
    {   3300,   159,         0,4294967295 },
    {   4297,   115,         0,  23570760 },
    {   3263,   166,         0,4294967295 },
    {   4781,   168,         0,  11394360 },
    {   4781,   179,  11394360,4294967295 },
    {   3915,   169,         0,  11185680 },
    {   3915,    73,  11185680,4294967295 },
    {   2772,    93,         0,4294967295 },
    {   1317,    87,         0,   9333750 },
    {   1317,    36,   9333750,4294967295 },
    {   2821,    97,         0,4294967295 },
    {   1021,   124,         0,4294967295 },
    {   1159,   178,         0,4294967295 },
    {   8722,   187,         0,   5914980 },
    {   8722,   115,   5914980,  21930480 },
    {   8722,   182,  21930480,4294967295 },
    {   6977,    89,         0,4294967295 },
    {   4710,   182,         0,4294967295 },
    {   4828,   187,         0,4294967295 },
    {   1235,   126,         0,4294967295 },
    {   3883,   163,         0,  11596200 },
    {   3883,   188,  11596200,4294967295 },
    {   3747,   189,         0,  11426160 },
    {   3747,    24,  11426160,4294967295 },
    {   3663,    32,         0,  12090360 },
    {   3663,    65,  12090360,  12337980 },
    {   3663,    32,  12337980,4294967295 },
    {   5671,    26,         0,4294967295 },
    {   6628,    65,         0,4294967295 },
    {   3727,    42,         0,4294967295 },
    {   1395,    65,         0,4294967295 },
    {   8736,    65,         0,4294967295 },
    {   2748,    75,         0,4294967295 },
    {   7272,   161,         0,4294967295 },
    {   6821,    75,         0,4294967295 },
    {   6644,    66,         0,4294967295 },
    {   1670,    27,         0,4294967295 },
    {   1635,    29,         0,4294967295 },
    {   1799,    27,         0,4294967295 },
    {   1583,    27,         0,4294967295 },
    {   3637,    28,         0,4294967295 },
    {   1516,    29,         0,4294967295 },
    {   8753,    29,         0,4294967295 },
    {   1547,    29,         0,   5870280 },
    {   1547,   113,   5870280,4294967295 },
    {   1727,    29,         0,4294967295 },
    {   1762,    30,         0,4294967295 },
    {   1463,    29,         0,4294967295 },
    {   2406,    75,         0,4294967295 },
    {   6441,    62,         0,4294967295 },
    {   6216,    62,         0,4294967295 },
    {   3620,   125,  11697000,  24317220 },
    {   2300,    63,         0,4294967295 },
    {   6336,    62,         0,4294967295 },
    {   4757,    62,         0,4294967295 },
    {   6488,    62,         0,4294967295 },
    {   2167,    62,         0,4294967295 },
    {   3928,    63,         0,4294967295 },
    {   3806,    62,         0,4294967295 },
    {   8772,    62,         0,4294967295 },
    {   2583,   125,         0,  10699020 },
    {   2583,    63,  10699020,4294967295 },
    {   6289,    62,         0,4294967295 },
    {   2338,    94,         0,    962040 },
    {   2338,    75,    962040,4294967295 },
    {   6307,    62,         0,4294967295 },
    {   8788,    40,         0,    962040 },
    {   8788,    75,    962040,4294967295 },
    {   6612,    63,         0,4294967295 },
    {   8804,    40,         0,    962040 },
    {   8804,    75,    962040,4294967295 },
    {   1130,    63,         0,   4465260 },
    {   1130,   173,   4465260,   8046540 },
    {   1130,    63,   8046540,  24553260 },
    {   1130,   173,  24553260,4294967295 },
    {   8823,    40,         0,    962040 },
    {   8823,    75,    962040,4294967295 },
    {   3979,   125,         0,  10114500 },
    {   3979,    63,  10114500,  21686400 },
    {   3979,    64,  21686400,  23571300 },
    {   3979,    63,  23571300,4294967295 },
    {   2527,   125,         0,  10779720 },
    {   2527,    63,  10779720,4294967295 },
    {   3213,    62,         0,   3542400 },
    {   3213,    65,   3542400,  11959260 },
    {   3213,    62,  11959260,  13803900 },
    {   3213,    65,  13803900,4294967295 },
    {   6506,    62,         0,4294967295 },
    {   2501,    40,         0,    962040 },
    {   2501,    75,    962040,4294967295 },
    {   6365,    62,         0,4294967295 },
    {   4892,    62,         0,4294967295 },
    {   6397,    62,         0,4294967295 },
    {   6582,    63,         0,4294967295 },
    {   3693,   125,         0,  11172900 },
    {   3693,    63,  11172900,  21686400 },
    {   3693,    64,  21686400,  23572680 },
    {   3693,   125,  23572680,4294967295 },
    {   6410,    62,         0,4294967295 },
    {   3537,   125,         0,  11172900 },
    {   3537,    63,  11172900,  11596320 },
    {   3537,   125,  11596320,4294967295 },
    {   6458,    62,         0,4294967295 },
    {   4284,    62,         0,4294967295 },
    {   6424,    62,         0,4294967295 },
    {   6275,    62,         0,4294967295 },
    {   8837,   125,         0,  10114500 },
    {   8837,    63,  10114500,4294967295 },
    {   6324,    62,         0,4294967295 },
    {   4331,   107,         0,  10114440 },
    {   4331,   125,  10114440,  11172900 },
    {   4331,    63,  11172900,  11435040 },
    {   4331,   154,  11435040,4294967295 },
    {   6470,    62,         0,4294967295 },
    {   6245,    62,         0,4294967295 },
    {   4451,   125,  11697000,  24680100 },
    {   8849,   125,         0,  10779780 },
    {   8849,    63,  10779780,  12795660 },
    {   8849,   125,  12795660,  14328060 },
    {   8849,    63,  14328060,  23268960 },
    {   8849,   125,  23268960,4294967295 },
    {   6383,    62,         0,4294967295 },
    {   6599,    63,         0,4294967295 },
    {   6523,    62,         0,4294967295 },
    {   8867,   125,         0,  10114500 },
    {   8867,    63,  10114500,4294967295 },
    {   6202,    62,         0,4294967295 },
    {   8882,   125,  11596320,  24317220 },
    {   6352,    62,         0,4294967295 },
    {   6567,    62,         0,4294967295 },
    {   6231,    62,         0,4294967295 },
    {   8899,   125,         0,  10114500 },
    {   8899,    63,  10114500,  14852220 },
    {   8899,    62,  14852220,  15688860 },
    {   8899,    63,  15688860,4294967295 },
    {   4727,   183,         0,4294967295 },
    {   3190,    62,         0,4294967295 },
    {   6261,    62,         0,4294967295 },
    {   6540,    62,         0,4294967295 },
    {   5078,     4,         0,4294967295 },
    {   6947,    88,         0,4294967295 },
    {   6046,    48,         0,4294967295 },
    {   6063,    49,         0,4294967295 },
    {   5029,     4,         0,4294967295 },
    {   6661,    69,         0,4294967295 },
    {   7260,   157,         0,4294967295 },
    {   7050,   118,         0,4294967295 },
    {   4069,   121,         0,4294967295 },
    {   5098,     4,         0,4294967295 },
    {   7211,   151,         0,4294967295 },
    {   4422,    18,         0,4294967295 },
    {    932,   130,         0,4294967295 },
    {   3706,   141,         0,  23661600 },
    {   3015,    45,         0,4294967295 },
    {   3157,   172,         0,4294967295 },
    {   2237,    60,         0,4294967295 },
    {   7360,   180,         0,4294967295 },
    {   7327,   170,         0,4294967295 },
    {   3902,    67,         0,4294967295 },
    {   7343,   175,         0,4294967295 },
    {   6678,    61,         0,   8415660 },
    {   6678,    71,   8415660,4294967295 },
    {   6696,    72,         0,4294967295 },
    {   3822,   160,         0,4294967295 },
    {   6018,    81,         0,  16291560 },
    {   6018,    44,  16291560,4294967295 },
    {   2723,    10,         0,   7272660 },
    {   2723,    84,   7272660,4294967295 },
    {   3049,   145,         0,4294967295 },
    {   3998,   112,         0,4294967295 },
    {   7019,   105,         0,4294967295 },
    {   2842,   108,         0,  12432240 },
    {   2842,   120,  12432240,4294967295 },
    {   4877,   120,         0,4294967295 },
    {   4051,   119,         0,4294967295 },
    {   8914,    35,         0,   7272720 },
    {   8914,   156,   7272720,4294967295 },
    {   7084,   127,         0,4294967295 },
    {   7132,   132,         0,4294967295 },
    {   4168,   133,         0,4294967295 },
    {   7098,   129,         0,4294967295 },
    {   3126,    35,         0,   7272720 },
    {   3126,   156,   7272720,4294967295 },
    {   7155,   140,         0,4294967295 },
    {   7194,   147,         0,4294967295 },
    {   3096,   148,         0,4294967295 },
    {   4807,   141,         0,4294967295 },
    {   6076,    51,         0,4294967295 },
    {   6031,   135,         0,  16291560 },
    {   6031,    44,  16291560,4294967295 },
    {   7312,   165,         0,4294967295 },
    {   6712,    74,         0,4294967295 },
    {   4583,   171,         0,4294967295 },
    {   7392,   185,         0,4294967295 },
    {   7405,   186,         0,4294967295 },
};

#endif // timezone_locale but not ICU

static inline constexpr char windowsIdData[] = {
0x41, 0x66, 0x67, 0x68, 0x61, 0x6e, 0x69, 0x73, 0x74, 0x61, 0x6e, 0x20, 0x53, 0x74, 0x61, 0x6e,
0x64, 0x61, 0x72, 0x64, 0x20, 0x54, 0x69, 0x6d, 0x65, 0x0, 0x41, 0x6c, 0x61, 0x73, 0x6b, 0x61,
0x6e, 0x20, 0x53, 0x74, 0x61, 0x6e, 0x64, 0x61, 0x72, 0x64, 0x20, 0x54, 0x69, 0x6d, 0x65, 0x0,
0x41, 0x6c, 0x65, 0x75, 0x74, 0x69, 0x61, 0x6e, 0x20, 0x53, 0x74, 0x61, 0x6e, 0x64, 0x61, 0x72,
0x64, 0x20, 0x54, 0x69, 0x6d, 0x65, 0x0, 0x41, 0x6c, 0x74, 0x61, 0x69, 0x20, 0x53, 0x74, 0x61,
0x6e, 0x64, 0x61, 0x72, 0x64, 0x20, 0x54, 0x69, 0x6d, 0x65, 0x0, 0x41, 0x72, 0x61, 0x62, 0x20,
0x53, 0x74, 0x61, 0x6e, 0x64, 0x61, 0x72, 0x64, 0x20, 0x54, 0x69, 0x6d, 0x65, 0x0, 0x41, 0x72,
0x61, 0x62, 0x69, 0x61, 0x6e, 0x20, 0x53, 0x74, 0x61, 0x6e, 0x64, 0x61, 0x72, 0x64, 0x20, 0x54,
0x69, 0x6d, 0x65, 0x0, 0x41, 0x72, 0x61, 0x62, 0x69, 0x63, 0x20, 0x53, 0x74, 0x61, 0x6e, 0x64,
0x61, 0x72, 0x64, 0x20, 0x54, 0x69, 0x6d, 0x65, 0x0, 0x41, 0x72, 0x67, 0x65, 0x6e, 0x74, 0x69,
0x6e, 0x61, 0x20, 0x53, 0x74, 0x61, 0x6e, 0x64, 0x61, 0x72, 0x64, 0x20, 0x54, 0x69, 0x6d, 0x65,
0x0, 0x41, 0x73, 0x74, 0x72, 0x61, 0x6b, 0x68, 0x61, 0x6e, 0x20, 0x53, 0x74, 0x61, 0x6e, 0x64,
0x61, 0x72, 0x64, 0x20, 0x54, 0x69, 0x6d, 0x65, 0x0, 0x41, 0x74, 0x6c, 0x61, 0x6e, 0x74, 0x69,
0x63, 0x20, 0x53, 0x74, 0x61, 0x6e, 0x64, 0x61, 0x72, 0x64, 0x20, 0x54, 0x69, 0x6d, 0x65, 0x0,
0x41, 0x55, 0x53, 0x20, 0x43, 0x65, 0x6e, 0x74, 0x72, 0x61, 0x6c, 0x20, 0x53, 0x74, 0x61, 0x6e,
0x64, 0x61, 0x72, 0x64, 0x20, 0x54, 0x69, 0x6d, 0x65, 0x0, 0x41, 0x75, 0x73, 0x20, 0x43, 0x65,
0x6e, 0x74, 0x72, 0x61, 0x6c, 0x20, 0x57, 0x2e, 0x20, 0x53, 0x74, 0x61, 0x6e, 0x64, 0x61, 0x72,
0x64, 0x20, 0x54, 0x69, 0x6d, 0x65, 0x0, 0x41, 0x55, 0x53, 0x20, 0x45, 0x61, 0x73, 0x74, 0x65,
0x72, 0x6e, 0x20, 0x53, 0x74, 0x61, 0x6e, 0x64, 0x61, 0x72, 0x64, 0x20, 0x54, 0x69, 0x6d, 0x65,
0x0, 0x41, 0x7a, 0x65, 0x72, 0x62, 0x61, 0x69, 0x6a, 0x61, 0x6e, 0x20, 0x53, 0x74, 0x61, 0x6e,
0x64, 0x61, 0x72, 0x64, 0x20, 0x54, 0x69, 0x6d, 0x65, 0x0, 0x41, 0x7a, 0x6f, 0x72, 0x65, 0x73,
0x20, 0x53, 0x74, 0x61, 0x6e, 0x64, 0x61, 0x72, 0x64, 0x20, 0x54, 0x69, 0x6d, 0x65, 0x0, 0x42,
0x61, 0x68, 0x69, 0x61, 0x20, 0x53, 0x74, 0x61, 0x6e, 0x64, 0x61, 0x72, 0x64, 0x20, 0x54, 0x69,
0x6d, 0x65, 0x0, 0x42, 0x61, 0x6e, 0x67, 0x6c, 0x61, 0x64, 0x65, 0x73, 0x68, 0x20, 0x53, 0x74,
0x61, 0x6e, 0x64, 0x61, 0x72, 0x64, 0x20, 0x54, 0x69, 0x6d, 0x65, 0x0, 0x42, 0x65, 0x6c, 0x61,
0x72, 0x75, 0x73, 0x20, 0x53, 0x74, 0x61, 0x6e, 0x64, 0x61, 0x72, 0x64, 0x20, 0x54, 0x69, 0x6d,
0x65, 0x0, 0x42, 0x6f, 0x75, 0x67, 0x61, 0x69, 0x6e, 0x76, 0x69, 0x6c, 0x6c, 0x65, 0x20, 0x53,
0x74, 0x61, 0x6e, 0x64, 0x61, 0x72, 0x64, 0x20, 0x54, 0x69, 0x6d, 0x65, 0x0, 0x43, 0x61, 0x6e,
0x61, 0x64, 0x61, 0x20, 0x43, 0x65, 0x6e, 0x74, 0x72, 0x61, 0x6c, 0x20, 0x53, 0x74, 0x61, 0x6e,
0x64, 0x61, 0x72, 0x64, 0x20, 0x54, 0x69, 0x6d, 0x65, 0x0, 0x43, 0x61, 0x70, 0x65, 0x20, 0x56,
0x65, 0x72, 0x64, 0x65, 0x20, 0x53, 0x74, 0x61, 0x6e, 0x64, 0x61, 0x72, 0x64, 0x20, 0x54, 0x69,
0x6d, 0x65, 0x0, 0x43, 0x61, 0x75, 0x63, 0x61, 0x73, 0x75, 0x73, 0x20, 0x53, 0x74, 0x61, 0x6e,
0x64, 0x61, 0x72, 0x64, 0x20, 0x54, 0x69, 0x6d, 0x65, 0x0, 0x43, 0x65, 0x6e, 0x2e, 0x20, 0x41,
0x75, 0x73, 0x74, 0x72, 0x61, 0x6c, 0x69, 0x61, 0x20, 0x53, 0x74, 0x61, 0x6e, 0x64, 0x61, 0x72,
0x64, 0x20, 0x54, 0x69, 0x6d, 0x65, 0x0, 0x43, 0x65, 0x6e, 0x74, 0x72, 0x61, 0x6c, 0x20, 0x41,
0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x20, 0x53, 0x74, 0x61, 0x6e, 0x64, 0x61, 0x72, 0x64, 0x20,
0x54, 0x69, 0x6d, 0x65, 0x0, 0x43, 0x65, 0x6e, 0x74, 0x72, 0x61, 0x6c, 0x20, 0x41, 0x73, 0x69,
0x61, 0x20, 0x53, 0x74, 0x61, 0x6e, 0x64, 0x61, 0x72, 0x64, 0x20, 0x54, 0x69, 0x6d, 0x65, 0x0,
0x43, 0x65, 0x6e, 0x74, 0x72, 0x61, 0x6c, 0x20, 0x42, 0x72, 0x61, 0x7a, 0x69, 0x6c, 0x69, 0x61,
0x6e, 0x20, 0x53, 0x74, 0x61, 0x6e, 0x64, 0x61, 0x72, 0x64, 0x20, 0x54, 0x69, 0x6d, 0x65, 0x0,
0x43, 0x65, 0x6e, 0x74, 0x72, 0x61, 0x6c, 0x20, 0x45, 0x75, 0x72, 0x6f, 0x70, 0x65, 0x20, 0x53,
0x74, 0x61, 0x6e, 0x64, 0x61, 0x72, 0x64, 0x20, 0x54, 0x69, 0x6d, 0x65, 0x0, 0x43, 0x65, 0x6e,
0x74, 0x72, 0x61, 0x6c, 0x20, 0x45, 0x75, 0x72, 0x6f, 0x70, 0x65, 0x61, 0x6e, 0x20, 0x53, 0x74,
0x61, 0x6e, 0x64, 0x61, 0x72, 0x64, 0x20, 0x54, 0x69, 0x6d, 0x65, 0x0, 0x43, 0x65, 0x6e, 0x74,
0x72, 0x61, 0x6c, 0x20, 0x50, 0x61, 0x63, 0x69, 0x66, 0x69, 0x63, 0x20, 0x53, 0x74, 0x61, 0x6e,
0x64, 0x61, 0x72, 0x64, 0x20, 0x54, 0x69, 0x6d, 0x65, 0x0, 0x43, 0x65, 0x6e, 0x74, 0x72, 0x61,
0x6c, 0x20, 0x53, 0x74, 0x61, 0x6e, 0x64, 0x61, 0x72, 0x64, 0x20, 0x54, 0x69, 0x6d, 0x65, 0x0,
0x43, 0x65, 0x6e, 0x74, 0x72, 0x61, 0x6c, 0x20, 0x53, 0x74, 0x61, 0x6e, 0x64, 0x61, 0x72, 0x64,
0x20, 0x54, 0x69, 0x6d, 0x65, 0x20, 0x28, 0x4d, 0x65, 0x78, 0x69, 0x63, 0x6f, 0x29, 0x0, 0x43,
0x68, 0x61, 0x74, 0x68, 0x61, 0x6d, 0x20, 0x49, 0x73, 0x6c, 0x61, 0x6e, 0x64, 0x73, 0x20, 0x53,
0x74, 0x61, 0x6e, 0x64, 0x61, 0x72, 0x64, 0x20, 0x54, 0x69, 0x6d, 0x65, 0x0, 0x43, 0x68, 0x69,
0x6e, 0x61, 0x20, 0x53, 0x74, 0x61, 0x6e, 0x64, 0x61, 0x72, 0x64, 0x20, 0x54, 0x69, 0x6d, 0x65,
0x0, 0x43, 0x75, 0x62, 0x61, 0x20, 0x53, 0x74, 0x61, 0x6e, 0x64, 0x61, 0x72, 0x64, 0x20, 0x54,
0x69, 0x6d, 0x65, 0x0, 0x44, 0x61, 0x74, 0x65, 0x6c, 0x69, 0x6e, 0x65, 0x20, 0x53, 0x74, 0x61,
0x6e, 0x64, 0x61, 0x72, 0x64, 0x20, 0x54, 0x69, 0x6d, 0x65, 0x0, 0x45, 0x2e, 0x20, 0x41, 0x66,
0x72, 0x69, 0x63, 0x61, 0x20, 0x53, 0x74, 0x61, 0x6e, 0x64, 0x61, 0x72, 0x64, 0x20, 0x54, 0x69,
0x6d, 0x65, 0x0, 0x45, 0x2e, 0x20, 0x41, 0x75, 0x73, 0x74, 0x72, 0x61, 0x6c, 0x69, 0x61, 0x20,
0x53, 0x74, 0x61, 0x6e, 0x64, 0x61, 0x72, 0x64, 0x20, 0x54, 0x69, 0x6d, 0x65, 0x0, 0x45, 0x2e,
0x20, 0x45, 0x75, 0x72, 0x6f, 0x70, 0x65, 0x20, 0x53, 0x74, 0x61, 0x6e, 0x64, 0x61, 0x72, 0x64,
0x20, 0x54, 0x69, 0x6d, 0x65, 0x0, 0x45, 0x2e, 0x20, 0x53, 0x6f, 0x75, 0x74, 0x68, 0x20, 0x41,
0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x20, 0x53, 0x74, 0x61, 0x6e, 0x64, 0x61, 0x72, 0x64, 0x20,
0x54, 0x69, 0x6d, 0x65, 0x0, 0x45, 0x61, 0x73, 0x74, 0x65, 0x72, 0x20, 0x49, 0x73, 0x6c, 0x61,
0x6e, 0x64, 0x20, 0x53, 0x74, 0x61, 0x6e, 0x64, 0x61, 0x72, 0x64, 0x20, 0x54, 0x69, 0x6d, 0x65,
0x0, 0x45, 0x61, 0x73, 0x74, 0x65, 0x72, 0x6e, 0x20, 0x53, 0x74, 0x61, 0x6e, 0x64, 0x61, 0x72,
0x64, 0x20, 0x54, 0x69, 0x6d, 0x65, 0x0, 0x45, 0x61, 0x73, 0x74, 0x65, 0x72, 0x6e, 0x20, 0x53,
0x74, 0x61, 0x6e, 0x64, 0x61, 0x72, 0x64, 0x20, 0x54, 0x69, 0x6d, 0x65, 0x20, 0x28, 0x4d, 0x65,
0x78, 0x69, 0x63, 0x6f, 0x29, 0x0, 0x45, 0x67, 0x79, 0x70, 0x74, 0x20, 0x53, 0x74, 0x61, 0x6e,
0x64, 0x61, 0x72, 0x64, 0x20, 0x54, 0x69, 0x6d, 0x65, 0x0, 0x45, 0x6b, 0x61, 0x74, 0x65, 0x72,
0x69, 0x6e, 0x62, 0x75, 0x72, 0x67, 0x20, 0x53, 0x74, 0x61, 0x6e, 0x64, 0x61, 0x72, 0x64, 0x20,
0x54, 0x69, 0x6d, 0x65, 0x0, 0x46, 0x69, 0x6a, 0x69, 0x20, 0x53, 0x74, 0x61, 0x6e, 0x64, 0x61,
0x72, 0x64, 0x20, 0x54, 0x69, 0x6d, 0x65, 0x0, 0x46, 0x4c, 0x45, 0x20, 0x53, 0x74, 0x61, 0x6e,
0x64, 0x61, 0x72, 0x64, 0x20, 0x54, 0x69, 0x6d, 0x65, 0x0, 0x47, 0x65, 0x6f, 0x72, 0x67, 0x69,
0x61, 0x6e, 0x20, 0x53, 0x74, 0x61, 0x6e, 0x64, 0x61, 0x72, 0x64, 0x20, 0x54, 0x69, 0x6d, 0x65,
0x0, 0x47, 0x4d, 0x54, 0x20, 0x53, 0x74, 0x61, 0x6e, 0x64, 0x61, 0x72, 0x64, 0x20, 0x54, 0x69,
0x6d, 0x65, 0x0, 0x47, 0x72, 0x65, 0x65, 0x6e, 0x6c, 0x61, 0x6e, 0x64, 0x20, 0x53, 0x74, 0x61,
0x6e, 0x64, 0x61, 0x72, 0x64, 0x20, 0x54, 0x69, 0x6d, 0x65, 0x0, 0x47, 0x72, 0x65, 0x65, 0x6e,
0x77, 0x69, 0x63, 0x68, 0x20, 0x53, 0x74, 0x61, 0x6e, 0x64, 0x61, 0x72, 0x64, 0x20, 0x54, 0x69,
0x6d, 0x65, 0x0, 0x47, 0x54, 0x42, 0x20, 0x53, 0x74, 0x61, 0x6e, 0x64, 0x61, 0x72, 0x64, 0x20,
0x54, 0x69, 0x6d, 0x65, 0x0, 0x48, 0x61, 0x69, 0x74, 0x69, 0x20, 0x53, 0x74, 0x61, 0x6e, 0x64,
0x61, 0x72, 0x64, 0x20, 0x54, 0x69, 0x6d, 0x65, 0x0, 0x48, 0x61, 0x77, 0x61, 0x69, 0x69, 0x61,
0x6e, 0x20, 0x53, 0x74, 0x61, 0x6e, 0x64, 0x61, 0x72, 0x64, 0x20, 0x54, 0x69, 0x6d, 0x65, 0x0,
0x49, 0x6e, 0x64, 0x69, 0x61, 0x20, 0x53, 0x74, 0x61, 0x6e, 0x64, 0x61, 0x72, 0x64, 0x20, 0x54,
0x69, 0x6d, 0x65, 0x0, 0x49, 0x72, 0x61, 0x6e, 0x20, 0x53, 0x74, 0x61, 0x6e, 0x64, 0x61, 0x72,
0x64, 0x20, 0x54, 0x69, 0x6d, 0x65, 0x0, 0x49, 0x73, 0x72, 0x61, 0x65, 0x6c, 0x20, 0x53, 0x74,
0x61, 0x6e, 0x64, 0x61, 0x72, 0x64, 0x20, 0x54, 0x69, 0x6d, 0x65, 0x0, 0x4a, 0x6f, 0x72, 0x64,
0x61, 0x6e, 0x20, 0x53, 0x74, 0x61, 0x6e, 0x64, 0x61, 0x72, 0x64, 0x20, 0x54, 0x69, 0x6d, 0x65,
0x0, 0x4b, 0x61, 0x6c, 0x69, 0x6e, 0x69, 0x6e, 0x67, 0x72, 0x61, 0x64, 0x20, 0x53, 0x74, 0x61,
0x6e, 0x64, 0x61, 0x72, 0x64, 0x20, 0x54, 0x69, 0x6d, 0x65, 0x0, 0x4b, 0x6f, 0x72, 0x65, 0x61,
0x20, 0x53, 0x74, 0x61, 0x6e, 0x64, 0x61, 0x72, 0x64, 0x20, 0x54, 0x69, 0x6d, 0x65, 0x0, 0x4c,
0x69, 0x62, 0x79, 0x61, 0x20, 0x53, 0x74, 0x61, 0x6e, 0x64, 0x61, 0x72, 0x64, 0x20, 0x54, 0x69,
0x6d, 0x65, 0x0, 0x4c, 0x69, 0x6e, 0x65, 0x20, 0x49, 0x73, 0x6c, 0x61, 0x6e, 0x64, 0x73, 0x20,
0x53, 0x74, 0x61, 0x6e, 0x64, 0x61, 0x72, 0x64, 0x20, 0x54, 0x69, 0x6d, 0x65, 0x0, 0x4c, 0x6f,
0x72, 0x64, 0x20, 0x48, 0x6f, 0x77, 0x65, 0x20, 0x53, 0x74, 0x61, 0x6e, 0x64, 0x61, 0x72, 0x64,
0x20, 0x54, 0x69, 0x6d, 0x65, 0x0, 0x4d, 0x61, 0x67, 0x61, 0x64, 0x61, 0x6e, 0x20, 0x53, 0x74,
0x61, 0x6e, 0x64, 0x61, 0x72, 0x64, 0x20, 0x54, 0x69, 0x6d, 0x65, 0x0, 0x4d, 0x61, 0x67, 0x61,
0x6c, 0x6c, 0x61, 0x6e, 0x65, 0x73, 0x20, 0x53, 0x74, 0x61, 0x6e, 0x64, 0x61, 0x72, 0x64, 0x20,
0x54, 0x69, 0x6d, 0x65, 0x0, 0x4d, 0x61, 0x72, 0x71, 0x75, 0x65, 0x73, 0x61, 0x73, 0x20, 0x53,
0x74, 0x61, 0x6e, 0x64, 0x61, 0x72, 0x64, 0x20, 0x54, 0x69, 0x6d, 0x65, 0x0, 0x4d, 0x61, 0x75,
0x72, 0x69, 0x74, 0x69, 0x75, 0x73, 0x20, 0x53, 0x74, 0x61, 0x6e, 0x64, 0x61, 0x72, 0x64, 0x20,
0x54, 0x69, 0x6d, 0x65, 0x0, 0x4d, 0x69, 0x64, 0x64, 0x6c, 0x65, 0x20, 0x45, 0x61, 0x73, 0x74,
0x20, 0x53, 0x74, 0x61, 0x6e, 0x64, 0x61, 0x72, 0x64, 0x20, 0x54, 0x69, 0x6d, 0x65, 0x0, 0x4d,
0x6f, 0x6e, 0x74, 0x65, 0x76, 0x69, 0x64, 0x65, 0x6f, 0x20, 0x53, 0x74, 0x61, 0x6e, 0x64, 0x61,
0x72, 0x64, 0x20, 0x54, 0x69, 0x6d, 0x65, 0x0, 0x4d, 0x6f, 0x72, 0x6f, 0x63, 0x63, 0x6f, 0x20,
0x53, 0x74, 0x61, 0x6e, 0x64, 0x61, 0x72, 0x64, 0x20, 0x54, 0x69, 0x6d, 0x65, 0x0, 0x4d, 0x6f,
0x75, 0x6e, 0x74, 0x61, 0x69, 0x6e, 0x20, 0x53, 0x74, 0x61, 0x6e, 0x64, 0x61, 0x72, 0x64, 0x20,
0x54, 0x69, 0x6d, 0x65, 0x0, 0x4d, 0x6f, 0x75, 0x6e, 0x74, 0x61, 0x69, 0x6e, 0x20, 0x53, 0x74,
0x61, 0x6e, 0x64, 0x61, 0x72, 0x64, 0x20, 0x54, 0x69, 0x6d, 0x65, 0x20, 0x28, 0x4d, 0x65, 0x78,
0x69, 0x63, 0x6f, 0x29, 0x0, 0x4d, 0x79, 0x61, 0x6e, 0x6d, 0x61, 0x72, 0x20, 0x53, 0x74, 0x61,
0x6e, 0x64, 0x61, 0x72, 0x64, 0x20, 0x54, 0x69, 0x6d, 0x65, 0x0, 0x4e, 0x2e, 0x20, 0x43, 0x65,
0x6e, 0x74, 0x72, 0x61, 0x6c, 0x20, 0x41, 0x73, 0x69, 0x61, 0x20, 0x53, 0x74, 0x61, 0x6e, 0x64,
0x61, 0x72, 0x64, 0x20, 0x54, 0x69, 0x6d, 0x65, 0x0, 0x4e, 0x61, 0x6d, 0x69, 0x62, 0x69, 0x61,
0x20, 0x53, 0x74, 0x61, 0x6e, 0x64, 0x61, 0x72, 0x64, 0x20, 0x54, 0x69, 0x6d, 0x65, 0x0, 0x4e,
0x65, 0x70, 0x61, 0x6c, 0x20, 0x53, 0x74, 0x61, 0x6e, 0x64, 0x61, 0x72, 0x64, 0x20, 0x54, 0x69,
0x6d, 0x65, 0x0, 0x4e, 0x65, 0x77, 0x20, 0x5a, 0x65, 0x61, 0x6c, 0x61, 0x6e, 0x64, 0x20, 0x53,
0x74, 0x61, 0x6e, 0x64, 0x61, 0x72, 0x64, 0x20, 0x54, 0x69, 0x6d, 0x65, 0x0, 0x4e, 0x65, 0x77,
0x66, 0x6f, 0x75, 0x6e, 0x64, 0x6c, 0x61, 0x6e, 0x64, 0x20, 0x53, 0x74, 0x61, 0x6e, 0x64, 0x61,
0x72, 0x64, 0x20, 0x54, 0x69, 0x6d, 0x65, 0x0, 0x4e, 0x6f, 0x72, 0x66, 0x6f, 0x6c, 0x6b, 0x20,
0x53, 0x74, 0x61, 0x6e, 0x64, 0x61, 0x72, 0x64, 0x20, 0x54, 0x69, 0x6d, 0x65, 0x0, 0x4e, 0x6f,
0x72, 0x74, 0x68, 0x20, 0x41, 0x73, 0x69, 0x61, 0x20, 0x45, 0x61, 0x73, 0x74, 0x20, 0x53, 0x74,
0x61, 0x6e, 0x64, 0x61, 0x72, 0x64, 0x20, 0x54, 0x69, 0x6d, 0x65, 0x0, 0x4e, 0x6f, 0x72, 0x74,
0x68, 0x20, 0x41, 0x73, 0x69, 0x61, 0x20, 0x53, 0x74, 0x61, 0x6e, 0x64, 0x61, 0x72, 0x64, 0x20,
0x54, 0x69, 0x6d, 0x65, 0x0, 0x4e, 0x6f, 0x72, 0x74, 0x68, 0x20, 0x4b, 0x6f, 0x72, 0x65, 0x61,
0x20, 0x53, 0x74, 0x61, 0x6e, 0x64, 0x61, 0x72, 0x64, 0x20, 0x54, 0x69, 0x6d, 0x65, 0x0, 0x4f,
0x6d, 0x73, 0x6b, 0x20, 0x53, 0x74, 0x61, 0x6e, 0x64, 0x61, 0x72, 0x64, 0x20, 0x54, 0x69, 0x6d,
0x65, 0x0, 0x50, 0x61, 0x63, 0x69, 0x66, 0x69, 0x63, 0x20, 0x53, 0x41, 0x20, 0x53, 0x74, 0x61,
0x6e, 0x64, 0x61, 0x72, 0x64, 0x20, 0x54, 0x69, 0x6d, 0x65, 0x0, 0x50, 0x61, 0x63, 0x69, 0x66,
0x69, 0x63, 0x20, 0x53, 0x74, 0x61, 0x6e, 0x64, 0x61, 0x72, 0x64, 0x20, 0x54, 0x69, 0x6d, 0x65,
0x0, 0x50, 0x61, 0x63, 0x69, 0x66, 0x69, 0x63, 0x20, 0x53, 0x74, 0x61, 0x6e, 0x64, 0x61, 0x72,
0x64, 0x20, 0x54, 0x69, 0x6d, 0x65, 0x20, 0x28, 0x4d, 0x65, 0x78, 0x69, 0x63, 0x6f, 0x29, 0x0,
0x50, 0x61, 0x6b, 0x69, 0x73, 0x74, 0x61, 0x6e, 0x20, 0x53, 0x74, 0x61, 0x6e, 0x64, 0x61, 0x72,
0x64, 0x20, 0x54, 0x69, 0x6d, 0x65, 0x0, 0x50, 0x61, 0x72, 0x61, 0x67, 0x75, 0x61, 0x79, 0x20,
0x53, 0x74, 0x61, 0x6e, 0x64, 0x61, 0x72, 0x64, 0x20, 0x54, 0x69, 0x6d, 0x65, 0x0, 0x51, 0x79,
0x7a, 0x79, 0x6c, 0x6f, 0x72, 0x64, 0x61, 0x20, 0x53, 0x74, 0x61, 0x6e, 0x64, 0x61, 0x72, 0x64,
0x20, 0x54, 0x69, 0x6d, 0x65, 0x0, 0x52, 0x6f, 0x6d, 0x61, 0x6e, 0x63, 0x65, 0x20, 0x53, 0x74,
0x61, 0x6e, 0x64, 0x61, 0x72, 0x64, 0x20, 0x54, 0x69, 0x6d, 0x65, 0x0, 0x52, 0x75, 0x73, 0x73,
0x69, 0x61, 0x20, 0x54, 0x69, 0x6d, 0x65, 0x20, 0x5a, 0x6f, 0x6e, 0x65, 0x20, 0x31, 0x30, 0x0,
0x52, 0x75, 0x73, 0x73, 0x69, 0x61, 0x20, 0x54, 0x69, 0x6d, 0x65, 0x20, 0x5a, 0x6f, 0x6e, 0x65,
0x20, 0x31, 0x31, 0x0, 0x52, 0x75, 0x73, 0x73, 0x69, 0x61, 0x20, 0x54, 0x69, 0x6d, 0x65, 0x20,
0x5a, 0x6f, 0x6e, 0x65, 0x20, 0x33, 0x0, 0x52, 0x75, 0x73, 0x73, 0x69, 0x61, 0x6e, 0x20, 0x53,
0x74, 0x61, 0x6e, 0x64, 0x61, 0x72, 0x64, 0x20, 0x54, 0x69, 0x6d, 0x65, 0x0, 0x53, 0x41, 0x20,
0x45, 0x61, 0x73, 0x74, 0x65, 0x72, 0x6e, 0x20, 0x53, 0x74, 0x61, 0x6e, 0x64, 0x61, 0x72, 0x64,
0x20, 0x54, 0x69, 0x6d, 0x65, 0x0, 0x53, 0x41, 0x20, 0x50, 0x61, 0x63, 0x69, 0x66, 0x69, 0x63,
0x20, 0x53, 0x74, 0x61, 0x6e, 0x64, 0x61, 0x72, 0x64, 0x20, 0x54, 0x69, 0x6d, 0x65, 0x0, 0x53,
0x41, 0x20, 0x57, 0x65, 0x73, 0x74, 0x65, 0x72, 0x6e, 0x20, 0x53, 0x74, 0x61, 0x6e, 0x64, 0x61,
0x72, 0x64, 0x20, 0x54, 0x69, 0x6d, 0x65, 0x0, 0x53, 0x61, 0x69, 0x6e, 0x74, 0x20, 0x50, 0x69,
0x65, 0x72, 0x72, 0x65, 0x20, 0x53, 0x74, 0x61, 0x6e, 0x64, 0x61, 0x72, 0x64, 0x20, 0x54, 0x69,
0x6d, 0x65, 0x0, 0x53, 0x61, 0x6b, 0x68, 0x61, 0x6c, 0x69, 0x6e, 0x20, 0x53, 0x74, 0x61, 0x6e,
0x64, 0x61, 0x72, 0x64, 0x20, 0x54, 0x69, 0x6d, 0x65, 0x0, 0x53, 0x61, 0x6d, 0x6f, 0x61, 0x20,
0x53, 0x74, 0x61, 0x6e, 0x64, 0x61, 0x72, 0x64, 0x20, 0x54, 0x69, 0x6d, 0x65, 0x0, 0x53, 0x61,
0x6f, 0x20, 0x54, 0x6f, 0x6d, 0x65, 0x20, 0x53, 0x74, 0x61, 0x6e, 0x64, 0x61, 0x72, 0x64, 0x20,
0x54, 0x69, 0x6d, 0x65, 0x0, 0x53, 0x61, 0x72, 0x61, 0x74, 0x6f, 0x76, 0x20, 0x53, 0x74, 0x61,
0x6e, 0x64, 0x61, 0x72, 0x64, 0x20, 0x54, 0x69, 0x6d, 0x65, 0x0, 0x53, 0x45, 0x20, 0x41, 0x73,
0x69, 0x61, 0x20, 0x53, 0x74, 0x61, 0x6e, 0x64, 0x61, 0x72, 0x64, 0x20, 0x54, 0x69, 0x6d, 0x65,
0x0, 0x53, 0x69, 0x6e, 0x67, 0x61, 0x70, 0x6f, 0x72, 0x65, 0x20, 0x53, 0x74, 0x61, 0x6e, 0x64,
0x61, 0x72, 0x64, 0x20, 0x54, 0x69, 0x6d, 0x65, 0x0, 0x53, 0x6f, 0x75, 0x74, 0x68, 0x20, 0x41,
0x66, 0x72, 0x69, 0x63, 0x61, 0x20, 0x53, 0x74, 0x61, 0x6e, 0x64, 0x61, 0x72, 0x64, 0x20, 0x54,
0x69, 0x6d, 0x65, 0x0, 0x53, 0x6f, 0x75, 0x74, 0x68, 0x20, 0x53, 0x75, 0x64, 0x61, 0x6e, 0x20,
0x53, 0x74, 0x61, 0x6e, 0x64, 0x61, 0x72, 0x64, 0x20, 0x54, 0x69, 0x6d, 0x65, 0x0, 0x53, 0x72,
0x69, 0x20, 0x4c, 0x61, 0x6e, 0x6b, 0x61, 0x20, 0x53, 0x74, 0x61, 0x6e, 0x64, 0x61, 0x72, 0x64,
0x20, 0x54, 0x69, 0x6d, 0x65, 0x0, 0x53, 0x75, 0x64, 0x61, 0x6e, 0x20, 0x53, 0x74, 0x61, 0x6e,
0x64, 0x61, 0x72, 0x64, 0x20, 0x54, 0x69, 0x6d, 0x65, 0x0, 0x53, 0x79, 0x72, 0x69, 0x61, 0x20,
0x53, 0x74, 0x61, 0x6e, 0x64, 0x61, 0x72, 0x64, 0x20, 0x54, 0x69, 0x6d, 0x65, 0x0, 0x54, 0x61,
0x69, 0x70, 0x65, 0x69, 0x20, 0x53, 0x74, 0x61, 0x6e, 0x64, 0x61, 0x72, 0x64, 0x20, 0x54, 0x69,
0x6d, 0x65, 0x0, 0x54, 0x61, 0x73, 0x6d, 0x61, 0x6e, 0x69, 0x61, 0x20, 0x53, 0x74, 0x61, 0x6e,
0x64, 0x61, 0x72, 0x64, 0x20, 0x54, 0x69, 0x6d, 0x65, 0x0, 0x54, 0x6f, 0x63, 0x61, 0x6e, 0x74,
0x69, 0x6e, 0x73, 0x20, 0x53, 0x74, 0x61, 0x6e, 0x64, 0x61, 0x72, 0x64, 0x20, 0x54, 0x69, 0x6d,
0x65, 0x0, 0x54, 0x6f, 0x6b, 0x79, 0x6f, 0x20, 0x53, 0x74, 0x61, 0x6e, 0x64, 0x61, 0x72, 0x64,
0x20, 0x54, 0x69, 0x6d, 0x65, 0x0, 0x54, 0x6f, 0x6d, 0x73, 0x6b, 0x20, 0x53, 0x74, 0x61, 0x6e,
0x64, 0x61, 0x72, 0x64, 0x20, 0x54, 0x69, 0x6d, 0x65, 0x0, 0x54, 0x6f, 0x6e, 0x67, 0x61, 0x20,
0x53, 0x74, 0x61, 0x6e, 0x64, 0x61, 0x72, 0x64, 0x20, 0x54, 0x69, 0x6d, 0x65, 0x0, 0x54, 0x72,
0x61, 0x6e, 0x73, 0x62, 0x61, 0x69, 0x6b, 0x61, 0x6c, 0x20, 0x53, 0x74, 0x61, 0x6e, 0x64, 0x61,
0x72, 0x64, 0x20, 0x54, 0x69, 0x6d, 0x65, 0x0, 0x54, 0x75, 0x72, 0x6b, 0x65, 0x79, 0x20, 0x53,
0x74, 0x61, 0x6e, 0x64, 0x61, 0x72, 0x64, 0x20, 0x54, 0x69, 0x6d, 0x65, 0x0, 0x54, 0x75, 0x72,
0x6b, 0x73, 0x20, 0x41, 0x6e, 0x64, 0x20, 0x43, 0x61, 0x69, 0x63, 0x6f, 0x73, 0x20, 0x53, 0x74,
0x61, 0x6e, 0x64, 0x61, 0x72, 0x64, 0x20, 0x54, 0x69, 0x6d, 0x65, 0x0, 0x55, 0x6c, 0x61, 0x61,
0x6e, 0x62, 0x61, 0x61, 0x74, 0x61, 0x72, 0x20, 0x53, 0x74, 0x61, 0x6e, 0x64, 0x61, 0x72, 0x64,
0x20, 0x54, 0x69, 0x6d, 0x65, 0x0, 0x55, 0x53, 0x20, 0x45, 0x61, 0x73, 0x74, 0x65, 0x72, 0x6e,
0x20, 0x53, 0x74, 0x61, 0x6e, 0x64, 0x61, 0x72, 0x64, 0x20, 0x54, 0x69, 0x6d, 0x65, 0x0, 0x55,
0x53, 0x20, 0x4d, 0x6f, 0x75, 0x6e, 0x74, 0x61, 0x69, 0x6e, 0x20, 0x53, 0x74, 0x61, 0x6e, 0x64,
0x61, 0x72, 0x64, 0x20, 0x54, 0x69, 0x6d, 0x65, 0x0, 0x55, 0x54, 0x43, 0x0, 0x55, 0x54, 0x43,
0x2b, 0x31, 0x32, 0x0, 0x55, 0x54, 0x43, 0x2b, 0x31, 0x33, 0x0, 0x55, 0x54, 0x43, 0x2d, 0x30,
0x32, 0x0, 0x55, 0x54, 0x43, 0x2d, 0x30, 0x38, 0x0, 0x55, 0x54, 0x43, 0x2d, 0x30, 0x39, 0x0,
0x55, 0x54, 0x43, 0x2d, 0x31, 0x31, 0x0, 0x56, 0x65, 0x6e, 0x65, 0x7a, 0x75, 0x65, 0x6c, 0x61,
0x20, 0x53, 0x74, 0x61, 0x6e, 0x64, 0x61, 0x72, 0x64, 0x20, 0x54, 0x69, 0x6d, 0x65, 0x0, 0x56,
0x6c, 0x61, 0x64, 0x69, 0x76, 0x6f, 0x73, 0x74, 0x6f, 0x6b, 0x20, 0x53, 0x74, 0x61, 0x6e, 0x64,
0x61, 0x72, 0x64, 0x20, 0x54, 0x69, 0x6d, 0x65, 0x0, 0x56, 0x6f, 0x6c, 0x67, 0x6f, 0x67, 0x72,
0x61, 0x64, 0x20, 0x53, 0x74, 0x61, 0x6e, 0x64, 0x61, 0x72, 0x64, 0x20, 0x54, 0x69, 0x6d, 0x65,
0x0, 0x57, 0x2e, 0x20, 0x41, 0x75, 0x73, 0x74, 0x72, 0x61, 0x6c, 0x69, 0x61, 0x20, 0x53, 0x74,
0x61, 0x6e, 0x64, 0x61, 0x72, 0x64, 0x20, 0x54, 0x69, 0x6d, 0x65, 0x0, 0x57, 0x2e, 0x20, 0x43,
0x65, 0x6e, 0x74, 0x72, 0x61, 0x6c, 0x20, 0x41, 0x66, 0x72, 0x69, 0x63, 0x61, 0x20, 0x53, 0x74,
0x61, 0x6e, 0x64, 0x61, 0x72, 0x64, 0x20, 0x54, 0x69, 0x6d, 0x65, 0x0, 0x57, 0x2e, 0x20, 0x45,
0x75, 0x72, 0x6f, 0x70, 0x65, 0x20, 0x53, 0x74, 0x61, 0x6e, 0x64, 0x61, 0x72, 0x64, 0x20, 0x54,
0x69, 0x6d, 0x65, 0x0, 0x57, 0x2e, 0x20, 0x4d, 0x6f, 0x6e, 0x67, 0x6f, 0x6c, 0x69, 0x61, 0x20,
0x53, 0x74, 0x61, 0x6e, 0x64, 0x61, 0x72, 0x64, 0x20, 0x54, 0x69, 0x6d, 0x65, 0x0, 0x57, 0x65,
0x73, 0x74, 0x20, 0x41, 0x73, 0x69, 0x61, 0x20, 0x53, 0x74, 0x61, 0x6e, 0x64, 0x61, 0x72, 0x64,
0x20, 0x54, 0x69, 0x6d, 0x65, 0x0, 0x57, 0x65, 0x73, 0x74, 0x20, 0x42, 0x61, 0x6e, 0x6b, 0x20,
0x53, 0x74, 0x61, 0x6e, 0x64, 0x61, 0x72, 0x64, 0x20, 0x54, 0x69, 0x6d, 0x65, 0x0, 0x57, 0x65,
0x73, 0x74, 0x20, 0x50, 0x61, 0x63, 0x69, 0x66, 0x69, 0x63, 0x20, 0x53, 0x74, 0x61, 0x6e, 0x64,
0x61, 0x72, 0x64, 0x20, 0x54, 0x69, 0x6d, 0x65, 0x0, 0x59, 0x61, 0x6b, 0x75, 0x74, 0x73, 0x6b,
0x20, 0x53, 0x74, 0x61, 0x6e, 0x64, 0x61, 0x72, 0x64, 0x20, 0x54, 0x69, 0x6d, 0x65, 0x0, 0x59,
0x75, 0x6b, 0x6f, 0x6e, 0x20, 0x53, 0x74, 0x61, 0x6e, 0x64, 0x61, 0x72, 0x64, 0x20, 0x54, 0x69,
0x6d, 0x65, 0x0
};

static inline constexpr char ianaListData[] = {
0x41, 0x73, 0x69, 0x61, 0x2f, 0x4b, 0x61, 0x62, 0x75, 0x6c, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69,
0x63, 0x61, 0x2f, 0x41, 0x6e, 0x63, 0x68, 0x6f, 0x72, 0x61, 0x67, 0x65, 0x20, 0x41, 0x6d, 0x65,
0x72, 0x69, 0x63, 0x61, 0x2f, 0x4a, 0x75, 0x6e, 0x65, 0x61, 0x75, 0x20, 0x41, 0x6d, 0x65, 0x72,
0x69, 0x63, 0x61, 0x2f, 0x4d, 0x65, 0x74, 0x6c, 0x61, 0x6b, 0x61, 0x74, 0x6c, 0x61, 0x20, 0x41,
0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x4e, 0x6f, 0x6d, 0x65, 0x20, 0x41, 0x6d, 0x65, 0x72,
0x69, 0x63, 0x61, 0x2f, 0x53, 0x69, 0x74, 0x6b, 0x61, 0x20, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63,
0x61, 0x2f, 0x59, 0x61, 0x6b, 0x75, 0x74, 0x61, 0x74, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63,
0x61, 0x2f, 0x41, 0x64, 0x61, 0x6b, 0x0, 0x41, 0x73, 0x69, 0x61, 0x2f, 0x42, 0x61, 0x72, 0x6e,
0x61, 0x75, 0x6c, 0x0, 0x41, 0x73, 0x69, 0x61, 0x2f, 0x42, 0x61, 0x68, 0x72, 0x61, 0x69, 0x6e,
0x0, 0x41, 0x73, 0x69, 0x61, 0x2f, 0x4b, 0x75, 0x77, 0x61, 0x69, 0x74, 0x0, 0x41, 0x73, 0x69,
0x61, 0x2f, 0x51, 0x61, 0x74, 0x61, 0x72, 0x0, 0x41, 0x73, 0x69, 0x61, 0x2f, 0x52, 0x69, 0x79,
0x61, 0x64, 0x68, 0x0, 0x41, 0x73, 0x69, 0x61, 0x2f, 0x41, 0x64, 0x65, 0x6e, 0x0, 0x45, 0x74,
0x63, 0x2f, 0x47, 0x4d, 0x54, 0x2d, 0x34, 0x0, 0x41, 0x73, 0x69, 0x61, 0x2f, 0x4d, 0x75, 0x73,
0x63, 0x61, 0x74, 0x0, 0x41, 0x73, 0x69, 0x61, 0x2f, 0x44, 0x75, 0x62, 0x61, 0x69, 0x0, 0x41,
0x73, 0x69, 0x61, 0x2f, 0x42, 0x61, 0x67, 0x68, 0x64, 0x61, 0x64, 0x0, 0x41, 0x6d, 0x65, 0x72,
0x69, 0x63, 0x61, 0x2f, 0x41, 0x72, 0x67, 0x65, 0x6e, 0x74, 0x69, 0x6e, 0x61, 0x2f, 0x42, 0x75,
0x65, 0x6e, 0x6f, 0x73, 0x5f, 0x41, 0x69, 0x72, 0x65, 0x73, 0x20, 0x41, 0x6d, 0x65, 0x72, 0x69,
0x63, 0x61, 0x2f, 0x41, 0x72, 0x67, 0x65, 0x6e, 0x74, 0x69, 0x6e, 0x61, 0x2f, 0x4c, 0x61, 0x5f,
0x52, 0x69, 0x6f, 0x6a, 0x61, 0x20, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x41, 0x72,
0x67, 0x65, 0x6e, 0x74, 0x69, 0x6e, 0x61, 0x2f, 0x52, 0x69, 0x6f, 0x5f, 0x47, 0x61, 0x6c, 0x6c,
0x65, 0x67, 0x6f, 0x73, 0x20, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x41, 0x72, 0x67,
0x65, 0x6e, 0x74, 0x69, 0x6e, 0x61, 0x2f, 0x53, 0x61, 0x6c, 0x74, 0x61, 0x20, 0x41, 0x6d, 0x65,
0x72, 0x69, 0x63, 0x61, 0x2f, 0x41, 0x72, 0x67, 0x65, 0x6e, 0x74, 0x69, 0x6e, 0x61, 0x2f, 0x53,
0x61, 0x6e, 0x5f, 0x4a, 0x75, 0x61, 0x6e, 0x20, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f,
0x41, 0x72, 0x67, 0x65, 0x6e, 0x74, 0x69, 0x6e, 0x61, 0x2f, 0x53, 0x61, 0x6e, 0x5f, 0x4c, 0x75,
0x69, 0x73, 0x20, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x41, 0x72, 0x67, 0x65, 0x6e,
0x74, 0x69, 0x6e, 0x61, 0x2f, 0x54, 0x75, 0x63, 0x75, 0x6d, 0x61, 0x6e, 0x20, 0x41, 0x6d, 0x65,
0x72, 0x69, 0x63, 0x61, 0x2f, 0x41, 0x72, 0x67, 0x65, 0x6e, 0x74, 0x69, 0x6e, 0x61, 0x2f, 0x55,
0x73, 0x68, 0x75, 0x61, 0x69, 0x61, 0x20, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x41,
0x72, 0x67, 0x65, 0x6e, 0x74, 0x69, 0x6e, 0x61, 0x2f, 0x43, 0x61, 0x74, 0x61, 0x6d, 0x61, 0x72,
0x63, 0x61, 0x20, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x41, 0x72, 0x67, 0x65, 0x6e,
0x74, 0x69, 0x6e, 0x61, 0x2f, 0x43, 0x6f, 0x72, 0x64, 0x6f, 0x62, 0x61, 0x20, 0x41, 0x6d, 0x65,
0x72, 0x69, 0x63, 0x61, 0x2f, 0x41, 0x72, 0x67, 0x65, 0x6e, 0x74, 0x69, 0x6e, 0x61, 0x2f, 0x4a,
0x75, 0x6a, 0x75, 0x79, 0x20, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x41, 0x72, 0x67,
0x65, 0x6e, 0x74, 0x69, 0x6e, 0x61, 0x2f, 0x4d, 0x65, 0x6e, 0x64, 0x6f, 0x7a, 0x61, 0x0, 0x45,
0x75, 0x72, 0x6f, 0x70, 0x65, 0x2f, 0x41, 0x73, 0x74, 0x72, 0x61, 0x6b, 0x68, 0x61, 0x6e, 0x20,
0x45, 0x75, 0x72, 0x6f, 0x70, 0x65, 0x2f, 0x55, 0x6c, 0x79, 0x61, 0x6e, 0x6f, 0x76, 0x73, 0x6b,
0x0, 0x41, 0x74, 0x6c, 0x61, 0x6e, 0x74, 0x69, 0x63, 0x2f, 0x42, 0x65, 0x72, 0x6d, 0x75, 0x64,
0x61, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x48, 0x61, 0x6c, 0x69, 0x66, 0x61,
0x78, 0x20, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x47, 0x6c, 0x61, 0x63, 0x65, 0x5f,
0x42, 0x61, 0x79, 0x20, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x47, 0x6f, 0x6f, 0x73,
0x65, 0x5f, 0x42, 0x61, 0x79, 0x20, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x4d, 0x6f,
0x6e, 0x63, 0x74, 0x6f, 0x6e, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x54, 0x68,
0x75, 0x6c, 0x65, 0x0, 0x41, 0x75, 0x73, 0x74, 0x72, 0x61, 0x6c, 0x69, 0x61, 0x2f, 0x44, 0x61,
0x72, 0x77, 0x69, 0x6e, 0x0, 0x41, 0x75, 0x73, 0x74, 0x72, 0x61, 0x6c, 0x69, 0x61, 0x2f, 0x45,
0x75, 0x63, 0x6c, 0x61, 0x0, 0x41, 0x75, 0x73, 0x74, 0x72, 0x61, 0x6c, 0x69, 0x61, 0x2f, 0x53,
0x79, 0x64, 0x6e, 0x65, 0x79, 0x20, 0x41, 0x75, 0x73, 0x74, 0x72, 0x61, 0x6c, 0x69, 0x61, 0x2f,
0x4d, 0x65, 0x6c, 0x62, 0x6f, 0x75, 0x72, 0x6e, 0x65, 0x0, 0x41, 0x73, 0x69, 0x61, 0x2f, 0x42,
0x61, 0x6b, 0x75, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x53, 0x63, 0x6f, 0x72,
0x65, 0x73, 0x62, 0x79, 0x73, 0x75, 0x6e, 0x64, 0x0, 0x41, 0x74, 0x6c, 0x61, 0x6e, 0x74, 0x69,
0x63, 0x2f, 0x41, 0x7a, 0x6f, 0x72, 0x65, 0x73, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61,
0x2f, 0x42, 0x61, 0x68, 0x69, 0x61, 0x0, 0x41, 0x73, 0x69, 0x61, 0x2f, 0x44, 0x68, 0x61, 0x6b,
0x61, 0x0, 0x41, 0x73, 0x69, 0x61, 0x2f, 0x54, 0x68, 0x69, 0x6d, 0x70, 0x68, 0x75, 0x0, 0x45,
0x75, 0x72, 0x6f, 0x70, 0x65, 0x2f, 0x4d, 0x69, 0x6e, 0x73, 0x6b, 0x0, 0x50, 0x61, 0x63, 0x69,
0x66, 0x69, 0x63, 0x2f, 0x42, 0x6f, 0x75, 0x67, 0x61, 0x69, 0x6e, 0x76, 0x69, 0x6c, 0x6c, 0x65,
0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x52, 0x65, 0x67, 0x69, 0x6e, 0x61, 0x20,
0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x53, 0x77, 0x69, 0x66, 0x74, 0x5f, 0x43, 0x75,
0x72, 0x72, 0x65, 0x6e, 0x74, 0x0, 0x45, 0x74, 0x63, 0x2f, 0x47, 0x4d, 0x54, 0x2b, 0x31, 0x0,
0x41, 0x74, 0x6c, 0x61, 0x6e, 0x74, 0x69, 0x63, 0x2f, 0x43, 0x61, 0x70, 0x65, 0x5f, 0x56, 0x65,
0x72, 0x64, 0x65, 0x0, 0x41, 0x73, 0x69, 0x61, 0x2f, 0x59, 0x65, 0x72, 0x65, 0x76, 0x61, 0x6e,
0x0, 0x41, 0x75, 0x73, 0x74, 0x72, 0x61, 0x6c, 0x69, 0x61, 0x2f, 0x41, 0x64, 0x65, 0x6c, 0x61,
0x69, 0x64, 0x65, 0x20, 0x41, 0x75, 0x73, 0x74, 0x72, 0x61, 0x6c, 0x69, 0x61, 0x2f, 0x42, 0x72,
0x6f, 0x6b, 0x65, 0x6e, 0x5f, 0x48, 0x69, 0x6c, 0x6c, 0x0, 0x45, 0x74, 0x63, 0x2f, 0x47, 0x4d,
0x54, 0x2b, 0x36, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x42, 0x65, 0x6c, 0x69,
0x7a, 0x65, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x43, 0x6f, 0x73, 0x74, 0x61,
0x5f, 0x52, 0x69, 0x63, 0x61, 0x0, 0x50, 0x61, 0x63, 0x69, 0x66, 0x69, 0x63, 0x2f, 0x47, 0x61,
0x6c, 0x61, 0x70, 0x61, 0x67, 0x6f, 0x73, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f,
0x45, 0x6c, 0x5f, 0x53, 0x61, 0x6c, 0x76, 0x61, 0x64, 0x6f, 0x72, 0x0, 0x41, 0x6d, 0x65, 0x72,
0x69, 0x63, 0x61, 0x2f, 0x47, 0x75, 0x61, 0x74, 0x65, 0x6d, 0x61, 0x6c, 0x61, 0x0, 0x41, 0x6d,
0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x54, 0x65, 0x67, 0x75, 0x63, 0x69, 0x67, 0x61, 0x6c, 0x70,
0x61, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x4d, 0x61, 0x6e, 0x61, 0x67, 0x75,
0x61, 0x0, 0x45, 0x74, 0x63, 0x2f, 0x47, 0x4d, 0x54, 0x2d, 0x36, 0x0, 0x41, 0x6e, 0x74, 0x61,
0x72, 0x63, 0x74, 0x69, 0x63, 0x61, 0x2f, 0x56, 0x6f, 0x73, 0x74, 0x6f, 0x6b, 0x0, 0x49, 0x6e,
0x64, 0x69, 0x61, 0x6e, 0x2f, 0x43, 0x68, 0x61, 0x67, 0x6f, 0x73, 0x0, 0x41, 0x73, 0x69, 0x61,
0x2f, 0x55, 0x72, 0x75, 0x6d, 0x71, 0x69, 0x0, 0x41, 0x73, 0x69, 0x61, 0x2f, 0x42, 0x69, 0x73,
0x68, 0x6b, 0x65, 0x6b, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x43, 0x75, 0x69,
0x61, 0x62, 0x61, 0x20, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x43, 0x61, 0x6d, 0x70,
0x6f, 0x5f, 0x47, 0x72, 0x61, 0x6e, 0x64, 0x65, 0x0, 0x45, 0x75, 0x72, 0x6f, 0x70, 0x65, 0x2f,
0x54, 0x69, 0x72, 0x61, 0x6e, 0x65, 0x0, 0x45, 0x75, 0x72, 0x6f, 0x70, 0x65, 0x2f, 0x50, 0x72,
0x61, 0x67, 0x75, 0x65, 0x0, 0x45, 0x75, 0x72, 0x6f, 0x70, 0x65, 0x2f, 0x42, 0x75, 0x64, 0x61,
0x70, 0x65, 0x73, 0x74, 0x0, 0x45, 0x75, 0x72, 0x6f, 0x70, 0x65, 0x2f, 0x50, 0x6f, 0x64, 0x67,
0x6f, 0x72, 0x69, 0x63, 0x61, 0x0, 0x45, 0x75, 0x72, 0x6f, 0x70, 0x65, 0x2f, 0x42, 0x65, 0x6c,
0x67, 0x72, 0x61, 0x64, 0x65, 0x0, 0x45, 0x75, 0x72, 0x6f, 0x70, 0x65, 0x2f, 0x42, 0x72, 0x61,
0x74, 0x69, 0x73, 0x6c, 0x61, 0x76, 0x61, 0x0, 0x45, 0x75, 0x72, 0x6f, 0x70, 0x65, 0x2f, 0x4c,
0x6a, 0x75, 0x62, 0x6c, 0x6a, 0x61, 0x6e, 0x61, 0x0, 0x45, 0x75, 0x72, 0x6f, 0x70, 0x65, 0x2f,
0x53, 0x61, 0x72, 0x61, 0x6a, 0x65, 0x76, 0x6f, 0x0, 0x45, 0x75, 0x72, 0x6f, 0x70, 0x65, 0x2f,
0x5a, 0x61, 0x67, 0x72, 0x65, 0x62, 0x0, 0x45, 0x75, 0x72, 0x6f, 0x70, 0x65, 0x2f, 0x53, 0x6b,
0x6f, 0x70, 0x6a, 0x65, 0x0, 0x45, 0x75, 0x72, 0x6f, 0x70, 0x65, 0x2f, 0x57, 0x61, 0x72, 0x73,
0x61, 0x77, 0x0, 0x45, 0x74, 0x63, 0x2f, 0x47, 0x4d, 0x54, 0x2d, 0x31, 0x31, 0x0, 0x41, 0x6e,
0x74, 0x61, 0x72, 0x63, 0x74, 0x69, 0x63, 0x61, 0x2f, 0x43, 0x61, 0x73, 0x65, 0x79, 0x0, 0x50,
0x61, 0x63, 0x69, 0x66, 0x69, 0x63, 0x2f, 0x50, 0x6f, 0x68, 0x6e, 0x70, 0x65, 0x69, 0x20, 0x50,
0x61, 0x63, 0x69, 0x66, 0x69, 0x63, 0x2f, 0x4b, 0x6f, 0x73, 0x72, 0x61, 0x65, 0x0, 0x50, 0x61,
0x63, 0x69, 0x66, 0x69, 0x63, 0x2f, 0x4e, 0x6f, 0x75, 0x6d, 0x65, 0x61, 0x0, 0x50, 0x61, 0x63,
0x69, 0x66, 0x69, 0x63, 0x2f, 0x47, 0x75, 0x61, 0x64, 0x61, 0x6c, 0x63, 0x61, 0x6e, 0x61, 0x6c,
0x0, 0x50, 0x61, 0x63, 0x69, 0x66, 0x69, 0x63, 0x2f, 0x45, 0x66, 0x61, 0x74, 0x65, 0x0, 0x41,
0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x57, 0x69, 0x6e, 0x6e, 0x69, 0x70, 0x65, 0x67, 0x20,
0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x52, 0x61, 0x6e, 0x6b, 0x69, 0x6e, 0x5f, 0x49,
0x6e, 0x6c, 0x65, 0x74, 0x20, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x52, 0x65, 0x73,
0x6f, 0x6c, 0x75, 0x74, 0x65, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x4d, 0x61,
0x74, 0x61, 0x6d, 0x6f, 0x72, 0x6f, 0x73, 0x20, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f,
0x4f, 0x6a, 0x69, 0x6e, 0x61, 0x67, 0x61, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f,
0x43, 0x68, 0x69, 0x63, 0x61, 0x67, 0x6f, 0x20, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f,
0x49, 0x6e, 0x64, 0x69, 0x61, 0x6e, 0x61, 0x2f, 0x4b, 0x6e, 0x6f, 0x78, 0x20, 0x41, 0x6d, 0x65,
0x72, 0x69, 0x63, 0x61, 0x2f, 0x49, 0x6e, 0x64, 0x69, 0x61, 0x6e, 0x61, 0x2f, 0x54, 0x65, 0x6c,
0x6c, 0x5f, 0x43, 0x69, 0x74, 0x79, 0x20, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x4d,
0x65, 0x6e, 0x6f, 0x6d, 0x69, 0x6e, 0x65, 0x65, 0x20, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61,
0x2f, 0x4e, 0x6f, 0x72, 0x74, 0x68, 0x5f, 0x44, 0x61, 0x6b, 0x6f, 0x74, 0x61, 0x2f, 0x42, 0x65,
0x75, 0x6c, 0x61, 0x68, 0x20, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x4e, 0x6f, 0x72,
0x74, 0x68, 0x5f, 0x44, 0x61, 0x6b, 0x6f, 0x74, 0x61, 0x2f, 0x43, 0x65, 0x6e, 0x74, 0x65, 0x72,
0x20, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x4e, 0x6f, 0x72, 0x74, 0x68, 0x5f, 0x44,
0x61, 0x6b, 0x6f, 0x74, 0x61, 0x2f, 0x4e, 0x65, 0x77, 0x5f, 0x53, 0x61, 0x6c, 0x65, 0x6d, 0x0,
0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x4d, 0x65, 0x78, 0x69, 0x63, 0x6f, 0x5f, 0x43,
0x69, 0x74, 0x79, 0x20, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x42, 0x61, 0x68, 0x69,
0x61, 0x5f, 0x42, 0x61, 0x6e, 0x64, 0x65, 0x72, 0x61, 0x73, 0x20, 0x41, 0x6d, 0x65, 0x72, 0x69,
0x63, 0x61, 0x2f, 0x4d, 0x65, 0x72, 0x69, 0x64, 0x61, 0x20, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63,
0x61, 0x2f, 0x4d, 0x6f, 0x6e, 0x74, 0x65, 0x72, 0x72, 0x65, 0x79, 0x20, 0x41, 0x6d, 0x65, 0x72,
0x69, 0x63, 0x61, 0x2f, 0x43, 0x68, 0x69, 0x68, 0x75, 0x61, 0x68, 0x75, 0x61, 0x0, 0x50, 0x61,
0x63, 0x69, 0x66, 0x69, 0x63, 0x2f, 0x43, 0x68, 0x61, 0x74, 0x68, 0x61, 0x6d, 0x0, 0x41, 0x73,
0x69, 0x61, 0x2f, 0x53, 0x68, 0x61, 0x6e, 0x67, 0x68, 0x61, 0x69, 0x0, 0x41, 0x73, 0x69, 0x61,
0x2f, 0x48, 0x6f, 0x6e, 0x67, 0x5f, 0x4b, 0x6f, 0x6e, 0x67, 0x0, 0x41, 0x73, 0x69, 0x61, 0x2f,
0x4d, 0x61, 0x63, 0x61, 0x75, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x48, 0x61,
0x76, 0x61, 0x6e, 0x61, 0x0, 0x45, 0x74, 0x63, 0x2f, 0x47, 0x4d, 0x54, 0x2b, 0x31, 0x32, 0x0,
0x45, 0x74, 0x63, 0x2f, 0x47, 0x4d, 0x54, 0x2d, 0x33, 0x0, 0x41, 0x6e, 0x74, 0x61, 0x72, 0x63,
0x74, 0x69, 0x63, 0x61, 0x2f, 0x53, 0x79, 0x6f, 0x77, 0x61, 0x0, 0x49, 0x6e, 0x64, 0x69, 0x61,
0x6e, 0x2f, 0x43, 0x6f, 0x6d, 0x6f, 0x72, 0x6f, 0x0, 0x41, 0x66, 0x72, 0x69, 0x63, 0x61, 0x2f,
0x44, 0x6a, 0x69, 0x62, 0x6f, 0x75, 0x74, 0x69, 0x0, 0x41, 0x66, 0x72, 0x69, 0x63, 0x61, 0x2f,
0x41, 0x73, 0x6d, 0x61, 0x72, 0x61, 0x0, 0x41, 0x66, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x41, 0x64,
0x64, 0x69, 0x73, 0x5f, 0x41, 0x62, 0x61, 0x62, 0x61, 0x0, 0x41, 0x66, 0x72, 0x69, 0x63, 0x61,
0x2f, 0x4e, 0x61, 0x69, 0x72, 0x6f, 0x62, 0x69, 0x0, 0x49, 0x6e, 0x64, 0x69, 0x61, 0x6e, 0x2f,
0x41, 0x6e, 0x74, 0x61, 0x6e, 0x61, 0x6e, 0x61, 0x72, 0x69, 0x76, 0x6f, 0x0, 0x49, 0x6e, 0x64,
0x69, 0x61, 0x6e, 0x2f, 0x4d, 0x61, 0x79, 0x6f, 0x74, 0x74, 0x65, 0x0, 0x41, 0x66, 0x72, 0x69,
0x63, 0x61, 0x2f, 0x4d, 0x6f, 0x67, 0x61, 0x64, 0x69, 0x73, 0x68, 0x75, 0x0, 0x41, 0x66, 0x72,
0x69, 0x63, 0x61, 0x2f, 0x44, 0x61, 0x72, 0x5f, 0x65, 0x73, 0x5f, 0x53, 0x61, 0x6c, 0x61, 0x61,
0x6d, 0x0, 0x41, 0x66, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x4b, 0x61, 0x6d, 0x70, 0x61, 0x6c, 0x61,
0x0, 0x41, 0x75, 0x73, 0x74, 0x72, 0x61, 0x6c, 0x69, 0x61, 0x2f, 0x42, 0x72, 0x69, 0x73, 0x62,
0x61, 0x6e, 0x65, 0x20, 0x41, 0x75, 0x73, 0x74, 0x72, 0x61, 0x6c, 0x69, 0x61, 0x2f, 0x4c, 0x69,
0x6e, 0x64, 0x65, 0x6d, 0x61, 0x6e, 0x0, 0x45, 0x75, 0x72, 0x6f, 0x70, 0x65, 0x2f, 0x43, 0x68,
0x69, 0x73, 0x69, 0x6e, 0x61, 0x75, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x53,
0x61, 0x6f, 0x5f, 0x50, 0x61, 0x75, 0x6c, 0x6f, 0x0, 0x50, 0x61, 0x63, 0x69, 0x66, 0x69, 0x63,
0x2f, 0x45, 0x61, 0x73, 0x74, 0x65, 0x72, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f,
0x4e, 0x61, 0x73, 0x73, 0x61, 0x75, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x54,
0x6f, 0x72, 0x6f, 0x6e, 0x74, 0x6f, 0x20, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x49,
0x71, 0x61, 0x6c, 0x75, 0x69, 0x74, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x4e,
0x65, 0x77, 0x5f, 0x59, 0x6f, 0x72, 0x6b, 0x20, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f,
0x44, 0x65, 0x74, 0x72, 0x6f, 0x69, 0x74, 0x20, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f,
0x49, 0x6e, 0x64, 0x69, 0x61, 0x6e, 0x61, 0x2f, 0x50, 0x65, 0x74, 0x65, 0x72, 0x73, 0x62, 0x75,
0x72, 0x67, 0x20, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x49, 0x6e, 0x64, 0x69, 0x61,
0x6e, 0x61, 0x2f, 0x56, 0x69, 0x6e, 0x63, 0x65, 0x6e, 0x6e, 0x65, 0x73, 0x20, 0x41, 0x6d, 0x65,
0x72, 0x69, 0x63, 0x61, 0x2f, 0x49, 0x6e, 0x64, 0x69, 0x61, 0x6e, 0x61, 0x2f, 0x57, 0x69, 0x6e,
0x61, 0x6d, 0x61, 0x63, 0x20, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x4b, 0x65, 0x6e,
0x74, 0x75, 0x63, 0x6b, 0x79, 0x2f, 0x4d, 0x6f, 0x6e, 0x74, 0x69, 0x63, 0x65, 0x6c, 0x6c, 0x6f,
0x20, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x4b, 0x65, 0x6e, 0x74, 0x75, 0x63, 0x6b,
0x79, 0x2f, 0x4c, 0x6f, 0x75, 0x69, 0x73, 0x76, 0x69, 0x6c, 0x6c, 0x65, 0x0, 0x41, 0x6d, 0x65,
0x72, 0x69, 0x63, 0x61, 0x2f, 0x43, 0x61, 0x6e, 0x63, 0x75, 0x6e, 0x0, 0x41, 0x66, 0x72, 0x69,
0x63, 0x61, 0x2f, 0x43, 0x61, 0x69, 0x72, 0x6f, 0x0, 0x41, 0x73, 0x69, 0x61, 0x2f, 0x59, 0x65,
0x6b, 0x61, 0x74, 0x65, 0x72, 0x69, 0x6e, 0x62, 0x75, 0x72, 0x67, 0x0, 0x50, 0x61, 0x63, 0x69,
0x66, 0x69, 0x63, 0x2f, 0x46, 0x69, 0x6a, 0x69, 0x0, 0x45, 0x75, 0x72, 0x6f, 0x70, 0x65, 0x2f,
0x4d, 0x61, 0x72, 0x69, 0x65, 0x68, 0x61, 0x6d, 0x6e, 0x0, 0x45, 0x75, 0x72, 0x6f, 0x70, 0x65,
0x2f, 0x53, 0x6f, 0x66, 0x69, 0x61, 0x0, 0x45, 0x75, 0x72, 0x6f, 0x70, 0x65, 0x2f, 0x54, 0x61,
0x6c, 0x6c, 0x69, 0x6e, 0x6e, 0x0, 0x45, 0x75, 0x72, 0x6f, 0x70, 0x65, 0x2f, 0x48, 0x65, 0x6c,
0x73, 0x69, 0x6e, 0x6b, 0x69, 0x0, 0x45, 0x75, 0x72, 0x6f, 0x70, 0x65, 0x2f, 0x52, 0x69, 0x67,
0x61, 0x0, 0x45, 0x75, 0x72, 0x6f, 0x70, 0x65, 0x2f, 0x56, 0x69, 0x6c, 0x6e, 0x69, 0x75, 0x73,
0x0, 0x45, 0x75, 0x72, 0x6f, 0x70, 0x65, 0x2f, 0x4b, 0x79, 0x69, 0x76, 0x0, 0x41, 0x73, 0x69,
0x61, 0x2f, 0x54, 0x62, 0x69, 0x6c, 0x69, 0x73, 0x69, 0x0, 0x41, 0x74, 0x6c, 0x61, 0x6e, 0x74,
0x69, 0x63, 0x2f, 0x46, 0x61, 0x72, 0x6f, 0x65, 0x0, 0x45, 0x75, 0x72, 0x6f, 0x70, 0x65, 0x2f,
0x47, 0x75, 0x65, 0x72, 0x6e, 0x73, 0x65, 0x79, 0x0, 0x45, 0x75, 0x72, 0x6f, 0x70, 0x65, 0x2f,
0x44, 0x75, 0x62, 0x6c, 0x69, 0x6e, 0x0, 0x45, 0x75, 0x72, 0x6f, 0x70, 0x65, 0x2f, 0x49, 0x73,
0x6c, 0x65, 0x5f, 0x6f, 0x66, 0x5f, 0x4d, 0x61, 0x6e, 0x0, 0x45, 0x75, 0x72, 0x6f, 0x70, 0x65,
0x2f, 0x4a, 0x65, 0x72, 0x73, 0x65, 0x79, 0x0, 0x45, 0x75, 0x72, 0x6f, 0x70, 0x65, 0x2f, 0x4c,
0x69, 0x73, 0x62, 0x6f, 0x6e, 0x20, 0x41, 0x74, 0x6c, 0x61, 0x6e, 0x74, 0x69, 0x63, 0x2f, 0x4d,
0x61, 0x64, 0x65, 0x69, 0x72, 0x61, 0x0, 0x41, 0x74, 0x6c, 0x61, 0x6e, 0x74, 0x69, 0x63, 0x2f,
0x43, 0x61, 0x6e, 0x61, 0x72, 0x79, 0x0, 0x45, 0x75, 0x72, 0x6f, 0x70, 0x65, 0x2f, 0x4c, 0x6f,
0x6e, 0x64, 0x6f, 0x6e, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x4e, 0x75, 0x75,
0x6b, 0x0, 0x41, 0x66, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x4f, 0x75, 0x61, 0x67, 0x61, 0x64, 0x6f,
0x75, 0x67, 0x6f, 0x75, 0x0, 0x41, 0x66, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x42, 0x61, 0x6e, 0x6a,
0x75, 0x6c, 0x0, 0x41, 0x66, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x41, 0x63, 0x63, 0x72, 0x61, 0x0,
0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x44, 0x61, 0x6e, 0x6d, 0x61, 0x72, 0x6b, 0x73,
0x68, 0x61, 0x76, 0x6e, 0x0, 0x41, 0x66, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x42, 0x69, 0x73, 0x73,
0x61, 0x75, 0x0, 0x41, 0x66, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x43, 0x6f, 0x6e, 0x61, 0x6b, 0x72,
0x79, 0x0, 0x41, 0x74, 0x6c, 0x61, 0x6e, 0x74, 0x69, 0x63, 0x2f, 0x52, 0x65, 0x79, 0x6b, 0x6a,
0x61, 0x76, 0x69, 0x6b, 0x0, 0x41, 0x66, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x41, 0x62, 0x69, 0x64,
0x6a, 0x61, 0x6e, 0x0, 0x41, 0x66, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x4d, 0x6f, 0x6e, 0x72, 0x6f,
0x76, 0x69, 0x61, 0x0, 0x41, 0x66, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x42, 0x61, 0x6d, 0x61, 0x6b,
0x6f, 0x0, 0x41, 0x66, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x4e, 0x6f, 0x75, 0x61, 0x6b, 0x63, 0x68,
0x6f, 0x74, 0x74, 0x0, 0x41, 0x74, 0x6c, 0x61, 0x6e, 0x74, 0x69, 0x63, 0x2f, 0x53, 0x74, 0x5f,
0x48, 0x65, 0x6c, 0x65, 0x6e, 0x61, 0x0, 0x41, 0x66, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x44, 0x61,
0x6b, 0x61, 0x72, 0x0, 0x41, 0x66, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x46, 0x72, 0x65, 0x65, 0x74,
0x6f, 0x77, 0x6e, 0x0, 0x41, 0x66, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x4c, 0x6f, 0x6d, 0x65, 0x0,
0x41, 0x73, 0x69, 0x61, 0x2f, 0x4e, 0x69, 0x63, 0x6f, 0x73, 0x69, 0x61, 0x20, 0x41, 0x73, 0x69,
0x61, 0x2f, 0x46, 0x61, 0x6d, 0x61, 0x67, 0x75, 0x73, 0x74, 0x61, 0x0, 0x45, 0x75, 0x72, 0x6f,
0x70, 0x65, 0x2f, 0x41, 0x74, 0x68, 0x65, 0x6e, 0x73, 0x0, 0x45, 0x75, 0x72, 0x6f, 0x70, 0x65,
0x2f, 0x42, 0x75, 0x63, 0x68, 0x61, 0x72, 0x65, 0x73, 0x74, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69,
0x63, 0x61, 0x2f, 0x50, 0x6f, 0x72, 0x74, 0x2d, 0x61, 0x75, 0x2d, 0x50, 0x72, 0x69, 0x6e, 0x63,
0x65, 0x0, 0x45, 0x74, 0x63, 0x2f, 0x47, 0x4d, 0x54, 0x2b, 0x31, 0x30, 0x0, 0x50, 0x61, 0x63,
0x69, 0x66, 0x69, 0x63, 0x2f, 0x52, 0x61, 0x72, 0x6f, 0x74, 0x6f, 0x6e, 0x67, 0x61, 0x0, 0x50,
0x61, 0x63, 0x69, 0x66, 0x69, 0x63, 0x2f, 0x54, 0x61, 0x68, 0x69, 0x74, 0x69, 0x0, 0x50, 0x61,
0x63, 0x69, 0x66, 0x69, 0x63, 0x2f, 0x48, 0x6f, 0x6e, 0x6f, 0x6c, 0x75, 0x6c, 0x75, 0x0, 0x41,
0x73, 0x69, 0x61, 0x2f, 0x4b, 0x6f, 0x6c, 0x6b, 0x61, 0x74, 0x61, 0x0, 0x41, 0x73, 0x69, 0x61,
0x2f, 0x54, 0x65, 0x68, 0x72, 0x61, 0x6e, 0x0, 0x41, 0x73, 0x69, 0x61, 0x2f, 0x4a, 0x65, 0x72,
0x75, 0x73, 0x61, 0x6c, 0x65, 0x6d, 0x0, 0x41, 0x73, 0x69, 0x61, 0x2f, 0x41, 0x6d, 0x6d, 0x61,
0x6e, 0x0, 0x45, 0x75, 0x72, 0x6f, 0x70, 0x65, 0x2f, 0x4b, 0x61, 0x6c, 0x69, 0x6e, 0x69, 0x6e,
0x67, 0x72, 0x61, 0x64, 0x0, 0x41, 0x73, 0x69, 0x61, 0x2f, 0x53, 0x65, 0x6f, 0x75, 0x6c, 0x0,
0x41, 0x66, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x54, 0x72, 0x69, 0x70, 0x6f, 0x6c, 0x69, 0x0, 0x45,
0x74, 0x63, 0x2f, 0x47, 0x4d, 0x54, 0x2d, 0x31, 0x34, 0x0, 0x50, 0x61, 0x63, 0x69, 0x66, 0x69,
0x63, 0x2f, 0x4b, 0x69, 0x72, 0x69, 0x74, 0x69, 0x6d, 0x61, 0x74, 0x69, 0x0, 0x41, 0x75, 0x73,
0x74, 0x72, 0x61, 0x6c, 0x69, 0x61, 0x2f, 0x4c, 0x6f, 0x72, 0x64, 0x5f, 0x48, 0x6f, 0x77, 0x65,
0x0, 0x41, 0x73, 0x69, 0x61, 0x2f, 0x4d, 0x61, 0x67, 0x61, 0x64, 0x61, 0x6e, 0x0, 0x41, 0x6d,
0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x50, 0x75, 0x6e, 0x74, 0x61, 0x5f, 0x41, 0x72, 0x65, 0x6e,
0x61, 0x73, 0x0, 0x50, 0x61, 0x63, 0x69, 0x66, 0x69, 0x63, 0x2f, 0x4d, 0x61, 0x72, 0x71, 0x75,
0x65, 0x73, 0x61, 0x73, 0x0, 0x49, 0x6e, 0x64, 0x69, 0x61, 0x6e, 0x2f, 0x4d, 0x61, 0x75, 0x72,
0x69, 0x74, 0x69, 0x75, 0x73, 0x0, 0x49, 0x6e, 0x64, 0x69, 0x61, 0x6e, 0x2f, 0x52, 0x65, 0x75,
0x6e, 0x69, 0x6f, 0x6e, 0x0, 0x49, 0x6e, 0x64, 0x69, 0x61, 0x6e, 0x2f, 0x4d, 0x61, 0x68, 0x65,
0x0, 0x41, 0x73, 0x69, 0x61, 0x2f, 0x42, 0x65, 0x69, 0x72, 0x75, 0x74, 0x0, 0x41, 0x6d, 0x65,
0x72, 0x69, 0x63, 0x61, 0x2f, 0x4d, 0x6f, 0x6e, 0x74, 0x65, 0x76, 0x69, 0x64, 0x65, 0x6f, 0x0,
0x41, 0x66, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x43, 0x61, 0x73, 0x61, 0x62, 0x6c, 0x61, 0x6e, 0x63,
0x61, 0x0, 0x41, 0x66, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x45, 0x6c, 0x5f, 0x41, 0x61, 0x69, 0x75,
0x6e, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x45, 0x64, 0x6d, 0x6f, 0x6e, 0x74,
0x6f, 0x6e, 0x20, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x43, 0x61, 0x6d, 0x62, 0x72,
0x69, 0x64, 0x67, 0x65, 0x5f, 0x42, 0x61, 0x79, 0x20, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61,
0x2f, 0x49, 0x6e, 0x75, 0x76, 0x69, 0x6b, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f,
0x43, 0x69, 0x75, 0x64, 0x61, 0x64, 0x5f, 0x4a, 0x75, 0x61, 0x72, 0x65, 0x7a, 0x0, 0x41, 0x6d,
0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x44, 0x65, 0x6e, 0x76, 0x65, 0x72, 0x20, 0x41, 0x6d, 0x65,
0x72, 0x69, 0x63, 0x61, 0x2f, 0x42, 0x6f, 0x69, 0x73, 0x65, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69,
0x63, 0x61, 0x2f, 0x4d, 0x61, 0x7a, 0x61, 0x74, 0x6c, 0x61, 0x6e, 0x0, 0x49, 0x6e, 0x64, 0x69,
0x61, 0x6e, 0x2f, 0x43, 0x6f, 0x63, 0x6f, 0x73, 0x0, 0x41, 0x73, 0x69, 0x61, 0x2f, 0x59, 0x61,
0x6e, 0x67, 0x6f, 0x6e, 0x0, 0x41, 0x73, 0x69, 0x61, 0x2f, 0x4e, 0x6f, 0x76, 0x6f, 0x73, 0x69,
0x62, 0x69, 0x72, 0x73, 0x6b, 0x0, 0x41, 0x66, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x57, 0x69, 0x6e,
0x64, 0x68, 0x6f, 0x65, 0x6b, 0x0, 0x41, 0x73, 0x69, 0x61, 0x2f, 0x4b, 0x61, 0x74, 0x68, 0x6d,
0x61, 0x6e, 0x64, 0x75, 0x0, 0x41, 0x6e, 0x74, 0x61, 0x72, 0x63, 0x74, 0x69, 0x63, 0x61, 0x2f,
0x4d, 0x63, 0x4d, 0x75, 0x72, 0x64, 0x6f, 0x0, 0x50, 0x61, 0x63, 0x69, 0x66, 0x69, 0x63, 0x2f,
0x41, 0x75, 0x63, 0x6b, 0x6c, 0x61, 0x6e, 0x64, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61,
0x2f, 0x53, 0x74, 0x5f, 0x4a, 0x6f, 0x68, 0x6e, 0x73, 0x0, 0x50, 0x61, 0x63, 0x69, 0x66, 0x69,
0x63, 0x2f, 0x4e, 0x6f, 0x72, 0x66, 0x6f, 0x6c, 0x6b, 0x0, 0x41, 0x73, 0x69, 0x61, 0x2f, 0x49,
0x72, 0x6b, 0x75, 0x74, 0x73, 0x6b, 0x0, 0x41, 0x73, 0x69, 0x61, 0x2f, 0x4b, 0x72, 0x61, 0x73,
0x6e, 0x6f, 0x79, 0x61, 0x72, 0x73, 0x6b, 0x20, 0x41, 0x73, 0x69, 0x61, 0x2f, 0x4e, 0x6f, 0x76,
0x6f, 0x6b, 0x75, 0x7a, 0x6e, 0x65, 0x74, 0x73, 0x6b, 0x0, 0x41, 0x73, 0x69, 0x61, 0x2f, 0x50,
0x79, 0x6f, 0x6e, 0x67, 0x79, 0x61, 0x6e, 0x67, 0x0, 0x41, 0x73, 0x69, 0x61, 0x2f, 0x4f, 0x6d,
0x73, 0x6b, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x53, 0x61, 0x6e, 0x74, 0x69,
0x61, 0x67, 0x6f, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x56, 0x61, 0x6e, 0x63,
0x6f, 0x75, 0x76, 0x65, 0x72, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x4c, 0x6f,
0x73, 0x5f, 0x41, 0x6e, 0x67, 0x65, 0x6c, 0x65, 0x73, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63,
0x61, 0x2f, 0x54, 0x69, 0x6a, 0x75, 0x61, 0x6e, 0x61, 0x0, 0x41, 0x73, 0x69, 0x61, 0x2f, 0x4b,
0x61, 0x72, 0x61, 0x63, 0x68, 0x69, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x41,
0x73, 0x75, 0x6e, 0x63, 0x69, 0x6f, 0x6e, 0x0, 0x41, 0x73, 0x69, 0x61, 0x2f, 0x51, 0x79, 0x7a,
0x79, 0x6c, 0x6f, 0x72, 0x64, 0x61, 0x0, 0x45, 0x75, 0x72, 0x6f, 0x70, 0x65, 0x2f, 0x42, 0x72,
0x75, 0x73, 0x73, 0x65, 0x6c, 0x73, 0x0, 0x45, 0x75, 0x72, 0x6f, 0x70, 0x65, 0x2f, 0x43, 0x6f,
0x70, 0x65, 0x6e, 0x68, 0x61, 0x67, 0x65, 0x6e, 0x0, 0x45, 0x75, 0x72, 0x6f, 0x70, 0x65, 0x2f,
0x50, 0x61, 0x72, 0x69, 0x73, 0x0, 0x45, 0x75, 0x72, 0x6f, 0x70, 0x65, 0x2f, 0x4d, 0x61, 0x64,
0x72, 0x69, 0x64, 0x20, 0x41, 0x66, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x43, 0x65, 0x75, 0x74, 0x61,
0x0, 0x41, 0x73, 0x69, 0x61, 0x2f, 0x53, 0x72, 0x65, 0x64, 0x6e, 0x65, 0x6b, 0x6f, 0x6c, 0x79,
0x6d, 0x73, 0x6b, 0x0, 0x41, 0x73, 0x69, 0x61, 0x2f, 0x4b, 0x61, 0x6d, 0x63, 0x68, 0x61, 0x74,
0x6b, 0x61, 0x20, 0x41, 0x73, 0x69, 0x61, 0x2f, 0x41, 0x6e, 0x61, 0x64, 0x79, 0x72, 0x0, 0x45,
0x75, 0x72, 0x6f, 0x70, 0x65, 0x2f, 0x53, 0x61, 0x6d, 0x61, 0x72, 0x61, 0x0, 0x45, 0x75, 0x72,
0x6f, 0x70, 0x65, 0x2f, 0x4d, 0x6f, 0x73, 0x63, 0x6f, 0x77, 0x20, 0x45, 0x75, 0x72, 0x6f, 0x70,
0x65, 0x2f, 0x4b, 0x69, 0x72, 0x6f, 0x76, 0x0, 0x45, 0x75, 0x72, 0x6f, 0x70, 0x65, 0x2f, 0x53,
0x69, 0x6d, 0x66, 0x65, 0x72, 0x6f, 0x70, 0x6f, 0x6c, 0x0, 0x45, 0x74, 0x63, 0x2f, 0x47, 0x4d,
0x54, 0x2b, 0x33, 0x0, 0x41, 0x6e, 0x74, 0x61, 0x72, 0x63, 0x74, 0x69, 0x63, 0x61, 0x2f, 0x52,
0x6f, 0x74, 0x68, 0x65, 0x72, 0x61, 0x20, 0x41, 0x6e, 0x74, 0x61, 0x72, 0x63, 0x74, 0x69, 0x63,
0x61, 0x2f, 0x50, 0x61, 0x6c, 0x6d, 0x65, 0x72, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61,
0x2f, 0x46, 0x6f, 0x72, 0x74, 0x61, 0x6c, 0x65, 0x7a, 0x61, 0x20, 0x41, 0x6d, 0x65, 0x72, 0x69,
0x63, 0x61, 0x2f, 0x42, 0x65, 0x6c, 0x65, 0x6d, 0x20, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61,
0x2f, 0x4d, 0x61, 0x63, 0x65, 0x69, 0x6f, 0x20, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f,
0x52, 0x65, 0x63, 0x69, 0x66, 0x65, 0x20, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x53,
0x61, 0x6e, 0x74, 0x61, 0x72, 0x65, 0x6d, 0x0, 0x41, 0x74, 0x6c, 0x61, 0x6e, 0x74, 0x69, 0x63,
0x2f, 0x53, 0x74, 0x61, 0x6e, 0x6c, 0x65, 0x79, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61,
0x2f, 0x43, 0x61, 0x79, 0x65, 0x6e, 0x6e, 0x65, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61,
0x2f, 0x50, 0x61, 0x72, 0x61, 0x6d, 0x61, 0x72, 0x69, 0x62, 0x6f, 0x0, 0x45, 0x74, 0x63, 0x2f,
0x47, 0x4d, 0x54, 0x2b, 0x35, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x52, 0x69,
0x6f, 0x5f, 0x42, 0x72, 0x61, 0x6e, 0x63, 0x6f, 0x20, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61,
0x2f, 0x45, 0x69, 0x72, 0x75, 0x6e, 0x65, 0x70, 0x65, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63,
0x61, 0x2f, 0x41, 0x74, 0x69, 0x6b, 0x6f, 0x6b, 0x61, 0x6e, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69,
0x63, 0x61, 0x2f, 0x43, 0x61, 0x79, 0x6d, 0x61, 0x6e, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63,
0x61, 0x2f, 0x42, 0x6f, 0x67, 0x6f, 0x74, 0x61, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61,
0x2f, 0x47, 0x75, 0x61, 0x79, 0x61, 0x71, 0x75, 0x69, 0x6c, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69,
0x63, 0x61, 0x2f, 0x4a, 0x61, 0x6d, 0x61, 0x69, 0x63, 0x61, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69,
0x63, 0x61, 0x2f, 0x50, 0x61, 0x6e, 0x61, 0x6d, 0x61, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63,
0x61, 0x2f, 0x4c, 0x69, 0x6d, 0x61, 0x0, 0x45, 0x74, 0x63, 0x2f, 0x47, 0x4d, 0x54, 0x2b, 0x34,
0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x41, 0x6e, 0x67, 0x75, 0x69, 0x6c, 0x6c,
0x61, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x41, 0x6e, 0x74, 0x69, 0x67, 0x75,
0x61, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x41, 0x72, 0x75, 0x62, 0x61, 0x0,
0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x42, 0x61, 0x72, 0x62, 0x61, 0x64, 0x6f, 0x73,
0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x4c, 0x61, 0x5f, 0x50, 0x61, 0x7a, 0x0,
0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x4d, 0x61, 0x6e, 0x61, 0x75, 0x73, 0x20, 0x41,
0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x42, 0x6f, 0x61, 0x5f, 0x56, 0x69, 0x73, 0x74, 0x61,
0x20, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x50, 0x6f, 0x72, 0x74, 0x6f, 0x5f, 0x56,
0x65, 0x6c, 0x68, 0x6f, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x54, 0x6f, 0x72,
0x74, 0x6f, 0x6c, 0x61, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x42, 0x6c, 0x61,
0x6e, 0x63, 0x2d, 0x53, 0x61, 0x62, 0x6c, 0x6f, 0x6e, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63,
0x61, 0x2f, 0x4b, 0x72, 0x61, 0x6c, 0x65, 0x6e, 0x64, 0x69, 0x6a, 0x6b, 0x0, 0x41, 0x6d, 0x65,
0x72, 0x69, 0x63, 0x61, 0x2f, 0x43, 0x75, 0x72, 0x61, 0x63, 0x61, 0x6f, 0x0, 0x41, 0x6d, 0x65,
0x72, 0x69, 0x63, 0x61, 0x2f, 0x44, 0x6f, 0x6d, 0x69, 0x6e, 0x69, 0x63, 0x61, 0x0, 0x41, 0x6d,
0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x53, 0x61, 0x6e, 0x74, 0x6f, 0x5f, 0x44, 0x6f, 0x6d, 0x69,
0x6e, 0x67, 0x6f, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x47, 0x72, 0x65, 0x6e,
0x61, 0x64, 0x61, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x47, 0x75, 0x61, 0x64,
0x65, 0x6c, 0x6f, 0x75, 0x70, 0x65, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x47,
0x75, 0x79, 0x61, 0x6e, 0x61, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x4d, 0x61,
0x72, 0x74, 0x69, 0x6e, 0x69, 0x71, 0x75, 0x65, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61,
0x2f, 0x4d, 0x6f, 0x6e, 0x74, 0x73, 0x65, 0x72, 0x72, 0x61, 0x74, 0x0, 0x41, 0x6d, 0x65, 0x72,
0x69, 0x63, 0x61, 0x2f, 0x50, 0x75, 0x65, 0x72, 0x74, 0x6f, 0x5f, 0x52, 0x69, 0x63, 0x6f, 0x0,
0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x53, 0x74, 0x5f, 0x42, 0x61, 0x72, 0x74, 0x68,
0x65, 0x6c, 0x65, 0x6d, 0x79, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x53, 0x74,
0x5f, 0x4b, 0x69, 0x74, 0x74, 0x73, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x53,
0x74, 0x5f, 0x4c, 0x75, 0x63, 0x69, 0x61, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f,
0x4d, 0x61, 0x72, 0x69, 0x67, 0x6f, 0x74, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f,
0x53, 0x74, 0x5f, 0x56, 0x69, 0x6e, 0x63, 0x65, 0x6e, 0x74, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69,
0x63, 0x61, 0x2f, 0x4c, 0x6f, 0x77, 0x65, 0x72, 0x5f, 0x50, 0x72, 0x69, 0x6e, 0x63, 0x65, 0x73,
0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x50, 0x6f, 0x72, 0x74, 0x5f, 0x6f, 0x66,
0x5f, 0x53, 0x70, 0x61, 0x69, 0x6e, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x53,
0x74, 0x5f, 0x54, 0x68, 0x6f, 0x6d, 0x61, 0x73, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61,
0x2f, 0x4d, 0x69, 0x71, 0x75, 0x65, 0x6c, 0x6f, 0x6e, 0x0, 0x41, 0x73, 0x69, 0x61, 0x2f, 0x53,
0x61, 0x6b, 0x68, 0x61, 0x6c, 0x69, 0x6e, 0x0, 0x50, 0x61, 0x63, 0x69, 0x66, 0x69, 0x63, 0x2f,
0x41, 0x70, 0x69, 0x61, 0x0, 0x41, 0x66, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x53, 0x61, 0x6f, 0x5f,
0x54, 0x6f, 0x6d, 0x65, 0x0, 0x45, 0x75, 0x72, 0x6f, 0x70, 0x65, 0x2f, 0x53, 0x61, 0x72, 0x61,
0x74, 0x6f, 0x76, 0x0, 0x45, 0x74, 0x63, 0x2f, 0x47, 0x4d, 0x54, 0x2d, 0x37, 0x0, 0x41, 0x6e,
0x74, 0x61, 0x72, 0x63, 0x74, 0x69, 0x63, 0x61, 0x2f, 0x44, 0x61, 0x76, 0x69, 0x73, 0x0, 0x41,
0x73, 0x69, 0x61, 0x2f, 0x50, 0x68, 0x6e, 0x6f, 0x6d, 0x5f, 0x50, 0x65, 0x6e, 0x68, 0x0, 0x49,
0x6e, 0x64, 0x69, 0x61, 0x6e, 0x2f, 0x43, 0x68, 0x72, 0x69, 0x73, 0x74, 0x6d, 0x61, 0x73, 0x0,
0x41, 0x73, 0x69, 0x61, 0x2f, 0x4a, 0x61, 0x6b, 0x61, 0x72, 0x74, 0x61, 0x20, 0x41, 0x73, 0x69,
0x61, 0x2f, 0x50, 0x6f, 0x6e, 0x74, 0x69, 0x61, 0x6e, 0x61, 0x6b, 0x0, 0x41, 0x73, 0x69, 0x61,
0x2f, 0x56, 0x69, 0x65, 0x6e, 0x74, 0x69, 0x61, 0x6e, 0x65, 0x0, 0x41, 0x73, 0x69, 0x61, 0x2f,
0x42, 0x61, 0x6e, 0x67, 0x6b, 0x6f, 0x6b, 0x0, 0x41, 0x73, 0x69, 0x61, 0x2f, 0x48, 0x6f, 0x5f,
0x43, 0x68, 0x69, 0x5f, 0x4d, 0x69, 0x6e, 0x68, 0x0, 0x45, 0x74, 0x63, 0x2f, 0x47, 0x4d, 0x54,
0x2d, 0x38, 0x0, 0x41, 0x73, 0x69, 0x61, 0x2f, 0x42, 0x72, 0x75, 0x6e, 0x65, 0x69, 0x0, 0x41,
0x73, 0x69, 0x61, 0x2f, 0x4d, 0x61, 0x6b, 0x61, 0x73, 0x73, 0x61, 0x72, 0x0, 0x41, 0x73, 0x69,
0x61, 0x2f, 0x4b, 0x75, 0x61, 0x6c, 0x61, 0x5f, 0x4c, 0x75, 0x6d, 0x70, 0x75, 0x72, 0x20, 0x41,
0x73, 0x69, 0x61, 0x2f, 0x4b, 0x75, 0x63, 0x68, 0x69, 0x6e, 0x67, 0x0, 0x41, 0x73, 0x69, 0x61,
0x2f, 0x4d, 0x61, 0x6e, 0x69, 0x6c, 0x61, 0x0, 0x41, 0x73, 0x69, 0x61, 0x2f, 0x53, 0x69, 0x6e,
0x67, 0x61, 0x70, 0x6f, 0x72, 0x65, 0x0, 0x45, 0x74, 0x63, 0x2f, 0x47, 0x4d, 0x54, 0x2d, 0x32,
0x0, 0x41, 0x66, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x47, 0x61, 0x62, 0x6f, 0x72, 0x6f, 0x6e, 0x65,
0x0, 0x41, 0x66, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x42, 0x75, 0x6a, 0x75, 0x6d, 0x62, 0x75, 0x72,
0x61, 0x0, 0x41, 0x66, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x4c, 0x75, 0x62, 0x75, 0x6d, 0x62, 0x61,
0x73, 0x68, 0x69, 0x0, 0x41, 0x66, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x4d, 0x62, 0x61, 0x62, 0x61,
0x6e, 0x65, 0x0, 0x41, 0x66, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x4d, 0x61, 0x73, 0x65, 0x72, 0x75,
0x0, 0x41, 0x66, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x42, 0x6c, 0x61, 0x6e, 0x74, 0x79, 0x72, 0x65,
0x0, 0x41, 0x66, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x4d, 0x61, 0x70, 0x75, 0x74, 0x6f, 0x0, 0x41,
0x66, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x4b, 0x69, 0x67, 0x61, 0x6c, 0x69, 0x0, 0x41, 0x66, 0x72,
0x69, 0x63, 0x61, 0x2f, 0x4a, 0x6f, 0x68, 0x61, 0x6e, 0x6e, 0x65, 0x73, 0x62, 0x75, 0x72, 0x67,
0x0, 0x41, 0x66, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x4c, 0x75, 0x73, 0x61, 0x6b, 0x61, 0x0, 0x41,
0x66, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x48, 0x61, 0x72, 0x61, 0x72, 0x65, 0x0, 0x41, 0x66, 0x72,
0x69, 0x63, 0x61, 0x2f, 0x4a, 0x75, 0x62, 0x61, 0x0, 0x41, 0x73, 0x69, 0x61, 0x2f, 0x43, 0x6f,
0x6c, 0x6f, 0x6d, 0x62, 0x6f, 0x0, 0x41, 0x66, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x4b, 0x68, 0x61,
0x72, 0x74, 0x6f, 0x75, 0x6d, 0x0, 0x41, 0x73, 0x69, 0x61, 0x2f, 0x44, 0x61, 0x6d, 0x61, 0x73,
0x63, 0x75, 0x73, 0x0, 0x41, 0x73, 0x69, 0x61, 0x2f, 0x54, 0x61, 0x69, 0x70, 0x65, 0x69, 0x0,
0x41, 0x75, 0x73, 0x74, 0x72, 0x61, 0x6c, 0x69, 0x61, 0x2f, 0x48, 0x6f, 0x62, 0x61, 0x72, 0x74,
0x20, 0x41, 0x6e, 0x74, 0x61, 0x72, 0x63, 0x74, 0x69, 0x63, 0x61, 0x2f, 0x4d, 0x61, 0x63, 0x71,
0x75, 0x61, 0x72, 0x69, 0x65, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x41, 0x72,
0x61, 0x67, 0x75, 0x61, 0x69, 0x6e, 0x61, 0x0, 0x45, 0x74, 0x63, 0x2f, 0x47, 0x4d, 0x54, 0x2d,
0x39, 0x0, 0x41, 0x73, 0x69, 0x61, 0x2f, 0x4a, 0x61, 0x79, 0x61, 0x70, 0x75, 0x72, 0x61, 0x0,
0x41, 0x73, 0x69, 0x61, 0x2f, 0x54, 0x6f, 0x6b, 0x79, 0x6f, 0x0, 0x50, 0x61, 0x63, 0x69, 0x66,
0x69, 0x63, 0x2f, 0x50, 0x61, 0x6c, 0x61, 0x75, 0x0, 0x41, 0x73, 0x69, 0x61, 0x2f, 0x44, 0x69,
0x6c, 0x69, 0x0, 0x41, 0x73, 0x69, 0x61, 0x2f, 0x54, 0x6f, 0x6d, 0x73, 0x6b, 0x0, 0x50, 0x61,
0x63, 0x69, 0x66, 0x69, 0x63, 0x2f, 0x54, 0x6f, 0x6e, 0x67, 0x61, 0x74, 0x61, 0x70, 0x75, 0x0,
0x41, 0x73, 0x69, 0x61, 0x2f, 0x43, 0x68, 0x69, 0x74, 0x61, 0x0, 0x45, 0x75, 0x72, 0x6f, 0x70,
0x65, 0x2f, 0x49, 0x73, 0x74, 0x61, 0x6e, 0x62, 0x75, 0x6c, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69,
0x63, 0x61, 0x2f, 0x47, 0x72, 0x61, 0x6e, 0x64, 0x5f, 0x54, 0x75, 0x72, 0x6b, 0x0, 0x41, 0x73,
0x69, 0x61, 0x2f, 0x55, 0x6c, 0x61, 0x61, 0x6e, 0x62, 0x61, 0x61, 0x74, 0x61, 0x72, 0x0, 0x41,
0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x49, 0x6e, 0x64, 0x69, 0x61, 0x6e, 0x61, 0x2f, 0x49,
0x6e, 0x64, 0x69, 0x61, 0x6e, 0x61, 0x70, 0x6f, 0x6c, 0x69, 0x73, 0x20, 0x41, 0x6d, 0x65, 0x72,
0x69, 0x63, 0x61, 0x2f, 0x49, 0x6e, 0x64, 0x69, 0x61, 0x6e, 0x61, 0x2f, 0x4d, 0x61, 0x72, 0x65,
0x6e, 0x67, 0x6f, 0x20, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x49, 0x6e, 0x64, 0x69,
0x61, 0x6e, 0x61, 0x2f, 0x56, 0x65, 0x76, 0x61, 0x79, 0x0, 0x45, 0x74, 0x63, 0x2f, 0x47, 0x4d,
0x54, 0x2b, 0x37, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x43, 0x72, 0x65, 0x73,
0x74, 0x6f, 0x6e, 0x20, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x44, 0x61, 0x77, 0x73,
0x6f, 0x6e, 0x5f, 0x43, 0x72, 0x65, 0x65, 0x6b, 0x20, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61,
0x2f, 0x46, 0x6f, 0x72, 0x74, 0x5f, 0x4e, 0x65, 0x6c, 0x73, 0x6f, 0x6e, 0x0, 0x41, 0x6d, 0x65,
0x72, 0x69, 0x63, 0x61, 0x2f, 0x48, 0x65, 0x72, 0x6d, 0x6f, 0x73, 0x69, 0x6c, 0x6c, 0x6f, 0x0,
0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x50, 0x68, 0x6f, 0x65, 0x6e, 0x69, 0x78, 0x0,
0x45, 0x74, 0x63, 0x2f, 0x55, 0x54, 0x43, 0x20, 0x45, 0x74, 0x63, 0x2f, 0x47, 0x4d, 0x54, 0x0,
0x45, 0x74, 0x63, 0x2f, 0x47, 0x4d, 0x54, 0x2d, 0x31, 0x32, 0x0, 0x50, 0x61, 0x63, 0x69, 0x66,
0x69, 0x63, 0x2f, 0x54, 0x61, 0x72, 0x61, 0x77, 0x61, 0x0, 0x50, 0x61, 0x63, 0x69, 0x66, 0x69,
0x63, 0x2f, 0x4d, 0x61, 0x6a, 0x75, 0x72, 0x6f, 0x20, 0x50, 0x61, 0x63, 0x69, 0x66, 0x69, 0x63,
0x2f, 0x4b, 0x77, 0x61, 0x6a, 0x61, 0x6c, 0x65, 0x69, 0x6e, 0x0, 0x50, 0x61, 0x63, 0x69, 0x66,
0x69, 0x63, 0x2f, 0x4e, 0x61, 0x75, 0x72, 0x75, 0x0, 0x50, 0x61, 0x63, 0x69, 0x66, 0x69, 0x63,
0x2f, 0x46, 0x75, 0x6e, 0x61, 0x66, 0x75, 0x74, 0x69, 0x0, 0x50, 0x61, 0x63, 0x69, 0x66, 0x69,
0x63, 0x2f, 0x57, 0x61, 0x6b, 0x65, 0x0, 0x50, 0x61, 0x63, 0x69, 0x66, 0x69, 0x63, 0x2f, 0x57,
0x61, 0x6c, 0x6c, 0x69, 0x73, 0x0, 0x45, 0x74, 0x63, 0x2f, 0x47, 0x4d, 0x54, 0x2d, 0x31, 0x33,
0x0, 0x50, 0x61, 0x63, 0x69, 0x66, 0x69, 0x63, 0x2f, 0x4b, 0x61, 0x6e, 0x74, 0x6f, 0x6e, 0x0,
0x50, 0x61, 0x63, 0x69, 0x66, 0x69, 0x63, 0x2f, 0x46, 0x61, 0x6b, 0x61, 0x6f, 0x66, 0x6f, 0x0,
0x45, 0x74, 0x63, 0x2f, 0x47, 0x4d, 0x54, 0x2b, 0x32, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63,
0x61, 0x2f, 0x4e, 0x6f, 0x72, 0x6f, 0x6e, 0x68, 0x61, 0x0, 0x41, 0x74, 0x6c, 0x61, 0x6e, 0x74,
0x69, 0x63, 0x2f, 0x53, 0x6f, 0x75, 0x74, 0x68, 0x5f, 0x47, 0x65, 0x6f, 0x72, 0x67, 0x69, 0x61,
0x0, 0x45, 0x74, 0x63, 0x2f, 0x47, 0x4d, 0x54, 0x2b, 0x38, 0x0, 0x50, 0x61, 0x63, 0x69, 0x66,
0x69, 0x63, 0x2f, 0x50, 0x69, 0x74, 0x63, 0x61, 0x69, 0x72, 0x6e, 0x0, 0x45, 0x74, 0x63, 0x2f,
0x47, 0x4d, 0x54, 0x2b, 0x39, 0x0, 0x50, 0x61, 0x63, 0x69, 0x66, 0x69, 0x63, 0x2f, 0x47, 0x61,
0x6d, 0x62, 0x69, 0x65, 0x72, 0x0, 0x45, 0x74, 0x63, 0x2f, 0x47, 0x4d, 0x54, 0x2b, 0x31, 0x31,
0x0, 0x50, 0x61, 0x63, 0x69, 0x66, 0x69, 0x63, 0x2f, 0x50, 0x61, 0x67, 0x6f, 0x5f, 0x50, 0x61,
0x67, 0x6f, 0x0, 0x50, 0x61, 0x63, 0x69, 0x66, 0x69, 0x63, 0x2f, 0x4e, 0x69, 0x75, 0x65, 0x0,
0x50, 0x61, 0x63, 0x69, 0x66, 0x69, 0x63, 0x2f, 0x4d, 0x69, 0x64, 0x77, 0x61, 0x79, 0x0, 0x41,
0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x43, 0x61, 0x72, 0x61, 0x63, 0x61, 0x73, 0x0, 0x41,
0x73, 0x69, 0x61, 0x2f, 0x56, 0x6c, 0x61, 0x64, 0x69, 0x76, 0x6f, 0x73, 0x74, 0x6f, 0x6b, 0x20,
0x41, 0x73, 0x69, 0x61, 0x2f, 0x55, 0x73, 0x74, 0x2d, 0x4e, 0x65, 0x72, 0x61, 0x0, 0x45, 0x75,
0x72, 0x6f, 0x70, 0x65, 0x2f, 0x56, 0x6f, 0x6c, 0x67, 0x6f, 0x67, 0x72, 0x61, 0x64, 0x0, 0x41,
0x75, 0x73, 0x74, 0x72, 0x61, 0x6c, 0x69, 0x61, 0x2f, 0x50, 0x65, 0x72, 0x74, 0x68, 0x0, 0x45,
0x74, 0x63, 0x2f, 0x47, 0x4d, 0x54, 0x2d, 0x31, 0x0, 0x41, 0x66, 0x72, 0x69, 0x63, 0x61, 0x2f,
0x41, 0x6c, 0x67, 0x69, 0x65, 0x72, 0x73, 0x0, 0x41, 0x66, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x4c,
0x75, 0x61, 0x6e, 0x64, 0x61, 0x0, 0x41, 0x66, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x50, 0x6f, 0x72,
0x74, 0x6f, 0x2d, 0x4e, 0x6f, 0x76, 0x6f, 0x0, 0x41, 0x66, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x44,
0x6f, 0x75, 0x61, 0x6c, 0x61, 0x0, 0x41, 0x66, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x42, 0x61, 0x6e,
0x67, 0x75, 0x69, 0x0, 0x41, 0x66, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x4e, 0x64, 0x6a, 0x61, 0x6d,
0x65, 0x6e, 0x61, 0x0, 0x41, 0x66, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x42, 0x72, 0x61, 0x7a, 0x7a,
0x61, 0x76, 0x69, 0x6c, 0x6c, 0x65, 0x0, 0x41, 0x66, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x4b, 0x69,
0x6e, 0x73, 0x68, 0x61, 0x73, 0x61, 0x0, 0x41, 0x66, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x4d, 0x61,
0x6c, 0x61, 0x62, 0x6f, 0x0, 0x41, 0x66, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x4c, 0x69, 0x62, 0x72,
0x65, 0x76, 0x69, 0x6c, 0x6c, 0x65, 0x0, 0x41, 0x66, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x4c, 0x61,
0x67, 0x6f, 0x73, 0x0, 0x41, 0x66, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x4e, 0x69, 0x61, 0x6d, 0x65,
0x79, 0x0, 0x41, 0x66, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x54, 0x75, 0x6e, 0x69, 0x73, 0x0, 0x45,
0x75, 0x72, 0x6f, 0x70, 0x65, 0x2f, 0x41, 0x6e, 0x64, 0x6f, 0x72, 0x72, 0x61, 0x0, 0x45, 0x75,
0x72, 0x6f, 0x70, 0x65, 0x2f, 0x56, 0x69, 0x65, 0x6e, 0x6e, 0x61, 0x0, 0x45, 0x75, 0x72, 0x6f,
0x70, 0x65, 0x2f, 0x42, 0x65, 0x72, 0x6c, 0x69, 0x6e, 0x20, 0x45, 0x75, 0x72, 0x6f, 0x70, 0x65,
0x2f, 0x42, 0x75, 0x73, 0x69, 0x6e, 0x67, 0x65, 0x6e, 0x0, 0x45, 0x75, 0x72, 0x6f, 0x70, 0x65,
0x2f, 0x47, 0x69, 0x62, 0x72, 0x61, 0x6c, 0x74, 0x61, 0x72, 0x0, 0x45, 0x75, 0x72, 0x6f, 0x70,
0x65, 0x2f, 0x52, 0x6f, 0x6d, 0x65, 0x0, 0x45, 0x75, 0x72, 0x6f, 0x70, 0x65, 0x2f, 0x56, 0x61,
0x64, 0x75, 0x7a, 0x0, 0x45, 0x75, 0x72, 0x6f, 0x70, 0x65, 0x2f, 0x4c, 0x75, 0x78, 0x65, 0x6d,
0x62, 0x6f, 0x75, 0x72, 0x67, 0x0, 0x45, 0x75, 0x72, 0x6f, 0x70, 0x65, 0x2f, 0x4d, 0x61, 0x6c,
0x74, 0x61, 0x0, 0x45, 0x75, 0x72, 0x6f, 0x70, 0x65, 0x2f, 0x4d, 0x6f, 0x6e, 0x61, 0x63, 0x6f,
0x0, 0x45, 0x75, 0x72, 0x6f, 0x70, 0x65, 0x2f, 0x41, 0x6d, 0x73, 0x74, 0x65, 0x72, 0x64, 0x61,
0x6d, 0x0, 0x45, 0x75, 0x72, 0x6f, 0x70, 0x65, 0x2f, 0x4f, 0x73, 0x6c, 0x6f, 0x0, 0x45, 0x75,
0x72, 0x6f, 0x70, 0x65, 0x2f, 0x53, 0x61, 0x6e, 0x5f, 0x4d, 0x61, 0x72, 0x69, 0x6e, 0x6f, 0x0,
0x41, 0x72, 0x63, 0x74, 0x69, 0x63, 0x2f, 0x4c, 0x6f, 0x6e, 0x67, 0x79, 0x65, 0x61, 0x72, 0x62,
0x79, 0x65, 0x6e, 0x0, 0x45, 0x75, 0x72, 0x6f, 0x70, 0x65, 0x2f, 0x53, 0x74, 0x6f, 0x63, 0x6b,
0x68, 0x6f, 0x6c, 0x6d, 0x0, 0x45, 0x75, 0x72, 0x6f, 0x70, 0x65, 0x2f, 0x5a, 0x75, 0x72, 0x69,
0x63, 0x68, 0x0, 0x45, 0x75, 0x72, 0x6f, 0x70, 0x65, 0x2f, 0x56, 0x61, 0x74, 0x69, 0x63, 0x61,
0x6e, 0x0, 0x41, 0x73, 0x69, 0x61, 0x2f, 0x48, 0x6f, 0x76, 0x64, 0x0, 0x45, 0x74, 0x63, 0x2f,
0x47, 0x4d, 0x54, 0x2d, 0x35, 0x0, 0x41, 0x6e, 0x74, 0x61, 0x72, 0x63, 0x74, 0x69, 0x63, 0x61,
0x2f, 0x4d, 0x61, 0x77, 0x73, 0x6f, 0x6e, 0x0, 0x49, 0x6e, 0x64, 0x69, 0x61, 0x6e, 0x2f, 0x4b,
0x65, 0x72, 0x67, 0x75, 0x65, 0x6c, 0x65, 0x6e, 0x0, 0x41, 0x73, 0x69, 0x61, 0x2f, 0x4f, 0x72,
0x61, 0x6c, 0x20, 0x41, 0x73, 0x69, 0x61, 0x2f, 0x41, 0x6c, 0x6d, 0x61, 0x74, 0x79, 0x20, 0x41,
0x73, 0x69, 0x61, 0x2f, 0x41, 0x71, 0x74, 0x61, 0x75, 0x20, 0x41, 0x73, 0x69, 0x61, 0x2f, 0x41,
0x71, 0x74, 0x6f, 0x62, 0x65, 0x20, 0x41, 0x73, 0x69, 0x61, 0x2f, 0x41, 0x74, 0x79, 0x72, 0x61,
0x75, 0x20, 0x41, 0x73, 0x69, 0x61, 0x2f, 0x51, 0x6f, 0x73, 0x74, 0x61, 0x6e, 0x61, 0x79, 0x0,
0x49, 0x6e, 0x64, 0x69, 0x61, 0x6e, 0x2f, 0x4d, 0x61, 0x6c, 0x64, 0x69, 0x76, 0x65, 0x73, 0x0,
0x41, 0x73, 0x69, 0x61, 0x2f, 0x44, 0x75, 0x73, 0x68, 0x61, 0x6e, 0x62, 0x65, 0x0, 0x41, 0x73,
0x69, 0x61, 0x2f, 0x41, 0x73, 0x68, 0x67, 0x61, 0x62, 0x61, 0x74, 0x0, 0x41, 0x73, 0x69, 0x61,
0x2f, 0x54, 0x61, 0x73, 0x68, 0x6b, 0x65, 0x6e, 0x74, 0x20, 0x41, 0x73, 0x69, 0x61, 0x2f, 0x53,
0x61, 0x6d, 0x61, 0x72, 0x6b, 0x61, 0x6e, 0x64, 0x0, 0x41, 0x73, 0x69, 0x61, 0x2f, 0x48, 0x65,
0x62, 0x72, 0x6f, 0x6e, 0x20, 0x41, 0x73, 0x69, 0x61, 0x2f, 0x47, 0x61, 0x7a, 0x61, 0x0, 0x45,
0x74, 0x63, 0x2f, 0x47, 0x4d, 0x54, 0x2d, 0x31, 0x30, 0x0, 0x41, 0x6e, 0x74, 0x61, 0x72, 0x63,
0x74, 0x69, 0x63, 0x61, 0x2f, 0x44, 0x75, 0x6d, 0x6f, 0x6e, 0x74, 0x44, 0x55, 0x72, 0x76, 0x69,
0x6c, 0x6c, 0x65, 0x0, 0x50, 0x61, 0x63, 0x69, 0x66, 0x69, 0x63, 0x2f, 0x47, 0x75, 0x61, 0x6d,
0x0, 0x50, 0x61, 0x63, 0x69, 0x66, 0x69, 0x63, 0x2f, 0x43, 0x68, 0x75, 0x75, 0x6b, 0x0, 0x50,
0x61, 0x63, 0x69, 0x66, 0x69, 0x63, 0x2f, 0x53, 0x61, 0x69, 0x70, 0x61, 0x6e, 0x0, 0x50, 0x61,
0x63, 0x69, 0x66, 0x69, 0x63, 0x2f, 0x50, 0x6f, 0x72, 0x74, 0x5f, 0x4d, 0x6f, 0x72, 0x65, 0x73,
0x62, 0x79, 0x0, 0x41, 0x73, 0x69, 0x61, 0x2f, 0x59, 0x61, 0x6b, 0x75, 0x74, 0x73, 0x6b, 0x20,
0x41, 0x73, 0x69, 0x61, 0x2f, 0x4b, 0x68, 0x61, 0x6e, 0x64, 0x79, 0x67, 0x61, 0x0, 0x41, 0x6d,
0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x57, 0x68, 0x69, 0x74, 0x65, 0x68, 0x6f, 0x72, 0x73, 0x65,
0x20, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x44, 0x61, 0x77, 0x73, 0x6f, 0x6e, 0x0,
0x55, 0x54, 0x43, 0x2d, 0x31, 0x34, 0x3a, 0x30, 0x30, 0x0, 0x55, 0x54, 0x43, 0x2d, 0x31, 0x33,
0x3a, 0x30, 0x30, 0x0, 0x55, 0x54, 0x43, 0x2d, 0x31, 0x32, 0x3a, 0x30, 0x30, 0x0, 0x55, 0x54,
0x43, 0x2d, 0x31, 0x31, 0x3a, 0x30, 0x30, 0x0, 0x55, 0x54, 0x43, 0x2d, 0x31, 0x30, 0x3a, 0x30,
0x30, 0x0, 0x55, 0x54, 0x43, 0x2d, 0x30, 0x39, 0x3a, 0x33, 0x30, 0x0, 0x55, 0x54, 0x43, 0x2d,
0x30, 0x39, 0x3a, 0x30, 0x30, 0x0, 0x55, 0x54, 0x43, 0x2d, 0x30, 0x38, 0x3a, 0x30, 0x30, 0x0,
0x55, 0x54, 0x43, 0x2d, 0x30, 0x37, 0x3a, 0x30, 0x30, 0x0, 0x55, 0x54, 0x43, 0x2d, 0x30, 0x36,
0x3a, 0x30, 0x30, 0x0, 0x55, 0x54, 0x43, 0x2d, 0x30, 0x35, 0x3a, 0x30, 0x30, 0x0, 0x55, 0x54,
0x43, 0x2d, 0x30, 0x34, 0x3a, 0x33, 0x30, 0x0, 0x55, 0x54, 0x43, 0x2d, 0x30, 0x34, 0x3a, 0x30,
0x30, 0x0, 0x55, 0x54, 0x43, 0x2d, 0x30, 0x33, 0x3a, 0x33, 0x30, 0x0, 0x55, 0x54, 0x43, 0x2d,
0x30, 0x33, 0x3a, 0x30, 0x30, 0x0, 0x55, 0x54, 0x43, 0x2d, 0x30, 0x32, 0x3a, 0x30, 0x30, 0x0,
0x55, 0x54, 0x43, 0x2d, 0x30, 0x31, 0x3a, 0x30, 0x30, 0x0, 0x55, 0x54, 0x43, 0x20, 0x55, 0x54,
0x43, 0x2b, 0x30, 0x30, 0x3a, 0x30, 0x30, 0x20, 0x55, 0x54, 0x43, 0x2d, 0x30, 0x30, 0x3a, 0x30,
0x30, 0x0, 0x55, 0x54, 0x43, 0x2b, 0x30, 0x31, 0x3a, 0x30, 0x30, 0x0, 0x55, 0x54, 0x43, 0x2b,
0x30, 0x32, 0x3a, 0x30, 0x30, 0x0, 0x55, 0x54, 0x43, 0x2b, 0x30, 0x33, 0x3a, 0x30, 0x30, 0x0,
0x55, 0x54, 0x43, 0x2b, 0x30, 0x33, 0x3a, 0x33, 0x30, 0x0, 0x55, 0x54, 0x43, 0x2b, 0x30, 0x34,
0x3a, 0x30, 0x30, 0x0, 0x55, 0x54, 0x43, 0x2b, 0x30, 0x34, 0x3a, 0x33, 0x30, 0x0, 0x55, 0x54,
0x43, 0x2b, 0x30, 0x35, 0x3a, 0x30, 0x30, 0x0, 0x55, 0x54, 0x43, 0x2b, 0x30, 0x35, 0x3a, 0x33,
0x30, 0x0, 0x55, 0x54, 0x43, 0x2b, 0x30, 0x35, 0x3a, 0x34, 0x35, 0x0, 0x55, 0x54, 0x43, 0x2b,
0x30, 0x36, 0x3a, 0x30, 0x30, 0x0, 0x55, 0x54, 0x43, 0x2b, 0x30, 0x36, 0x3a, 0x33, 0x30, 0x0,
0x55, 0x54, 0x43, 0x2b, 0x30, 0x37, 0x3a, 0x30, 0x30, 0x0, 0x55, 0x54, 0x43, 0x2b, 0x30, 0x38,
0x3a, 0x30, 0x30, 0x0, 0x55, 0x54, 0x43, 0x2b, 0x30, 0x38, 0x3a, 0x33, 0x30, 0x0, 0x55, 0x54,
0x43, 0x2b, 0x30, 0x38, 0x3a, 0x34, 0x35, 0x0, 0x55, 0x54, 0x43, 0x2b, 0x30, 0x39, 0x3a, 0x30,
0x30, 0x0, 0x55, 0x54, 0x43, 0x2b, 0x30, 0x39, 0x3a, 0x33, 0x30, 0x0, 0x55, 0x54, 0x43, 0x2b,
0x31, 0x30, 0x3a, 0x30, 0x30, 0x0, 0x55, 0x54, 0x43, 0x2b, 0x31, 0x30, 0x3a, 0x33, 0x30, 0x0,
0x55, 0x54, 0x43, 0x2b, 0x31, 0x31, 0x3a, 0x30, 0x30, 0x0, 0x55, 0x54, 0x43, 0x2b, 0x31, 0x32,
0x3a, 0x30, 0x30, 0x0, 0x55, 0x54, 0x43, 0x2b, 0x31, 0x32, 0x3a, 0x34, 0x35, 0x0, 0x55, 0x54,
0x43, 0x2b, 0x31, 0x33, 0x3a, 0x30, 0x30, 0x0, 0x55, 0x54, 0x43, 0x2b, 0x31, 0x34, 0x3a, 0x30,
0x30, 0x0
};

static inline constexpr char ianaIdData[] = {
0x41, 0x66, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x41, 0x73, 0x6d, 0x65, 0x72, 0x61, 0x0, 0x41, 0x66,
0x72, 0x69, 0x63, 0x61, 0x2f, 0x41, 0x73, 0x6d, 0x61, 0x72, 0x61, 0x0, 0x41, 0x66, 0x72, 0x69,
0x63, 0x61, 0x2f, 0x54, 0x69, 0x6d, 0x62, 0x75, 0x6b, 0x74, 0x75, 0x0, 0x41, 0x66, 0x72, 0x69,
0x63, 0x61, 0x2f, 0x42, 0x61, 0x6d, 0x61, 0x6b, 0x6f, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63,
0x61, 0x2f, 0x41, 0x72, 0x67, 0x65, 0x6e, 0x74, 0x69, 0x6e, 0x61, 0x2f, 0x43, 0x6f, 0x6d, 0x6f,
0x64, 0x52, 0x69, 0x76, 0x61, 0x64, 0x61, 0x76, 0x69, 0x61, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69,
0x63, 0x61, 0x2f, 0x41, 0x72, 0x67, 0x65, 0x6e, 0x74, 0x69, 0x6e, 0x61, 0x2f, 0x43, 0x61, 0x74,
0x61, 0x6d, 0x61, 0x72, 0x63, 0x61, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x41,
0x74, 0x6b, 0x61, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x41, 0x64, 0x61, 0x6b,
0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x42, 0x75, 0x65, 0x6e, 0x6f, 0x73, 0x5f,
0x41, 0x69, 0x72, 0x65, 0x73, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x41, 0x72,
0x67, 0x65, 0x6e, 0x74, 0x69, 0x6e, 0x61, 0x2f, 0x42, 0x75, 0x65, 0x6e, 0x6f, 0x73, 0x5f, 0x41,
0x69, 0x72, 0x65, 0x73, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x43, 0x61, 0x74,
0x61, 0x6d, 0x61, 0x72, 0x63, 0x61, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x43,
0x6f, 0x72, 0x61, 0x6c, 0x5f, 0x48, 0x61, 0x72, 0x62, 0x6f, 0x75, 0x72, 0x0, 0x41, 0x6d, 0x65,
0x72, 0x69, 0x63, 0x61, 0x2f, 0x41, 0x74, 0x69, 0x6b, 0x6f, 0x6b, 0x61, 0x6e, 0x0, 0x41, 0x6d,
0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x43, 0x6f, 0x72, 0x64, 0x6f, 0x62, 0x61, 0x0, 0x41, 0x6d,
0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x41, 0x72, 0x67, 0x65, 0x6e, 0x74, 0x69, 0x6e, 0x61, 0x2f,
0x43, 0x6f, 0x72, 0x64, 0x6f, 0x62, 0x61, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f,
0x45, 0x6e, 0x73, 0x65, 0x6e, 0x61, 0x64, 0x61, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61,
0x2f, 0x54, 0x69, 0x6a, 0x75, 0x61, 0x6e, 0x61, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61,
0x2f, 0x46, 0x6f, 0x72, 0x74, 0x5f, 0x57, 0x61, 0x79, 0x6e, 0x65, 0x0, 0x41, 0x6d, 0x65, 0x72,
0x69, 0x63, 0x61, 0x2f, 0x49, 0x6e, 0x64, 0x69, 0x61, 0x6e, 0x61, 0x2f, 0x49, 0x6e, 0x64, 0x69,
0x61, 0x6e, 0x61, 0x70, 0x6f, 0x6c, 0x69, 0x73, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61,
0x2f, 0x47, 0x6f, 0x64, 0x74, 0x68, 0x61, 0x62, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61,
0x2f, 0x4e, 0x75, 0x75, 0x6b, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x49, 0x6e,
0x64, 0x69, 0x61, 0x6e, 0x61, 0x70, 0x6f, 0x6c, 0x69, 0x73, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69,
0x63, 0x61, 0x2f, 0x4a, 0x75, 0x6a, 0x75, 0x79, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61,
0x2f, 0x41, 0x72, 0x67, 0x65, 0x6e, 0x74, 0x69, 0x6e, 0x61, 0x2f, 0x4a, 0x75, 0x6a, 0x75, 0x79,
0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x4b, 0x6e, 0x6f, 0x78, 0x5f, 0x49, 0x4e,
0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x49, 0x6e, 0x64, 0x69, 0x61, 0x6e, 0x61,
0x2f, 0x4b, 0x6e, 0x6f, 0x78, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x4c, 0x6f,
0x75, 0x69, 0x73, 0x76, 0x69, 0x6c, 0x6c, 0x65, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61,
0x2f, 0x4b, 0x65, 0x6e, 0x74, 0x75, 0x63, 0x6b, 0x79, 0x2f, 0x4c, 0x6f, 0x75, 0x69, 0x73, 0x76,
0x69, 0x6c, 0x6c, 0x65, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x4d, 0x65, 0x6e,
0x64, 0x6f, 0x7a, 0x61, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x41, 0x72, 0x67,
0x65, 0x6e, 0x74, 0x69, 0x6e, 0x61, 0x2f, 0x4d, 0x65, 0x6e, 0x64, 0x6f, 0x7a, 0x61, 0x0, 0x41,
0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x4d, 0x6f, 0x6e, 0x74, 0x72, 0x65, 0x61, 0x6c, 0x0,
0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x54, 0x6f, 0x72, 0x6f, 0x6e, 0x74, 0x6f, 0x0,
0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x4e, 0x69, 0x70, 0x69, 0x67, 0x6f, 0x6e, 0x0,
0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x50, 0x61, 0x6e, 0x67, 0x6e, 0x69, 0x72, 0x74,
0x75, 0x6e, 0x67, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x49, 0x71, 0x61, 0x6c,
0x75, 0x69, 0x74, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x50, 0x6f, 0x72, 0x74,
0x6f, 0x5f, 0x41, 0x63, 0x72, 0x65, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x52,
0x69, 0x6f, 0x5f, 0x42, 0x72, 0x61, 0x6e, 0x63, 0x6f, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63,
0x61, 0x2f, 0x52, 0x61, 0x69, 0x6e, 0x79, 0x5f, 0x52, 0x69, 0x76, 0x65, 0x72, 0x0, 0x41, 0x6d,
0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x57, 0x69, 0x6e, 0x6e, 0x69, 0x70, 0x65, 0x67, 0x0, 0x41,
0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x52, 0x6f, 0x73, 0x61, 0x72, 0x69, 0x6f, 0x0, 0x41,
0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x53, 0x61, 0x6e, 0x74, 0x61, 0x5f, 0x49, 0x73, 0x61,
0x62, 0x65, 0x6c, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x53, 0x68, 0x69, 0x70,
0x72, 0x6f, 0x63, 0x6b, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x44, 0x65, 0x6e,
0x76, 0x65, 0x72, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x54, 0x68, 0x75, 0x6e,
0x64, 0x65, 0x72, 0x5f, 0x42, 0x61, 0x79, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f,
0x56, 0x69, 0x72, 0x67, 0x69, 0x6e, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x53,
0x74, 0x5f, 0x54, 0x68, 0x6f, 0x6d, 0x61, 0x73, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61,
0x2f, 0x59, 0x65, 0x6c, 0x6c, 0x6f, 0x77, 0x6b, 0x6e, 0x69, 0x66, 0x65, 0x0, 0x41, 0x6d, 0x65,
0x72, 0x69, 0x63, 0x61, 0x2f, 0x45, 0x64, 0x6d, 0x6f, 0x6e, 0x74, 0x6f, 0x6e, 0x0, 0x41, 0x6e,
0x74, 0x61, 0x72, 0x63, 0x74, 0x69, 0x63, 0x61, 0x2f, 0x53, 0x6f, 0x75, 0x74, 0x68, 0x5f, 0x50,
0x6f, 0x6c, 0x65, 0x0, 0x50, 0x61, 0x63, 0x69, 0x66, 0x69, 0x63, 0x2f, 0x41, 0x75, 0x63, 0x6b,
0x6c, 0x61, 0x6e, 0x64, 0x0, 0x41, 0x73, 0x69, 0x61, 0x2f, 0x41, 0x73, 0x68, 0x6b, 0x68, 0x61,
0x62, 0x61, 0x64, 0x0, 0x41, 0x73, 0x69, 0x61, 0x2f, 0x41, 0x73, 0x68, 0x67, 0x61, 0x62, 0x61,
0x74, 0x0, 0x41, 0x73, 0x69, 0x61, 0x2f, 0x43, 0x61, 0x6c, 0x63, 0x75, 0x74, 0x74, 0x61, 0x0,
0x41, 0x73, 0x69, 0x61, 0x2f, 0x4b, 0x6f, 0x6c, 0x6b, 0x61, 0x74, 0x61, 0x0, 0x41, 0x73, 0x69,
0x61, 0x2f, 0x43, 0x68, 0x6f, 0x69, 0x62, 0x61, 0x6c, 0x73, 0x61, 0x6e, 0x0, 0x41, 0x73, 0x69,
0x61, 0x2f, 0x55, 0x6c, 0x61, 0x61, 0x6e, 0x62, 0x61, 0x61, 0x74, 0x61, 0x72, 0x0, 0x41, 0x73,
0x69, 0x61, 0x2f, 0x43, 0x68, 0x6f, 0x6e, 0x67, 0x71, 0x69, 0x6e, 0x67, 0x0, 0x41, 0x73, 0x69,
0x61, 0x2f, 0x53, 0x68, 0x61, 0x6e, 0x67, 0x68, 0x61, 0x69, 0x0, 0x41, 0x73, 0x69, 0x61, 0x2f,
0x43, 0x68, 0x75, 0x6e, 0x67, 0x6b, 0x69, 0x6e, 0x67, 0x0, 0x41, 0x73, 0x69, 0x61, 0x2f, 0x44,
0x61, 0x63, 0x63, 0x61, 0x0, 0x41, 0x73, 0x69, 0x61, 0x2f, 0x44, 0x68, 0x61, 0x6b, 0x61, 0x0,
0x41, 0x73, 0x69, 0x61, 0x2f, 0x48, 0x61, 0x72, 0x62, 0x69, 0x6e, 0x0, 0x41, 0x73, 0x69, 0x61,
0x2f, 0x49, 0x73, 0x74, 0x61, 0x6e, 0x62, 0x75, 0x6c, 0x0, 0x45, 0x75, 0x72, 0x6f, 0x70, 0x65,
0x2f, 0x49, 0x73, 0x74, 0x61, 0x6e, 0x62, 0x75, 0x6c, 0x0, 0x41, 0x73, 0x69, 0x61, 0x2f, 0x4b,
0x61, 0x73, 0x68, 0x67, 0x61, 0x72, 0x0, 0x41, 0x73, 0x69, 0x61, 0x2f, 0x55, 0x72, 0x75, 0x6d,
0x71, 0x69, 0x0, 0x41, 0x73, 0x69, 0x61, 0x2f, 0x4b, 0x61, 0x74, 0x6d, 0x61, 0x6e, 0x64, 0x75,
0x0, 0x41, 0x73, 0x69, 0x61, 0x2f, 0x4b, 0x61, 0x74, 0x68, 0x6d, 0x61, 0x6e, 0x64, 0x75, 0x0,
0x41, 0x73, 0x69, 0x61, 0x2f, 0x4d, 0x61, 0x63, 0x61, 0x6f, 0x0, 0x41, 0x73, 0x69, 0x61, 0x2f,
0x4d, 0x61, 0x63, 0x61, 0x75, 0x0, 0x41, 0x73, 0x69, 0x61, 0x2f, 0x52, 0x61, 0x6e, 0x67, 0x6f,
0x6f, 0x6e, 0x0, 0x41, 0x73, 0x69, 0x61, 0x2f, 0x59, 0x61, 0x6e, 0x67, 0x6f, 0x6e, 0x0, 0x41,
0x73, 0x69, 0x61, 0x2f, 0x53, 0x61, 0x69, 0x67, 0x6f, 0x6e, 0x0, 0x41, 0x73, 0x69, 0x61, 0x2f,
0x48, 0x6f, 0x5f, 0x43, 0x68, 0x69, 0x5f, 0x4d, 0x69, 0x6e, 0x68, 0x0, 0x41, 0x73, 0x69, 0x61,
0x2f, 0x54, 0x65, 0x6c, 0x5f, 0x41, 0x76, 0x69, 0x76, 0x0, 0x41, 0x73, 0x69, 0x61, 0x2f, 0x4a,
0x65, 0x72, 0x75, 0x73, 0x61, 0x6c, 0x65, 0x6d, 0x0, 0x41, 0x73, 0x69, 0x61, 0x2f, 0x54, 0x68,
0x69, 0x6d, 0x62, 0x75, 0x0, 0x41, 0x73, 0x69, 0x61, 0x2f, 0x54, 0x68, 0x69, 0x6d, 0x70, 0x68,
0x75, 0x0, 0x41, 0x73, 0x69, 0x61, 0x2f, 0x55, 0x6a, 0x75, 0x6e, 0x67, 0x5f, 0x50, 0x61, 0x6e,
0x64, 0x61, 0x6e, 0x67, 0x0, 0x41, 0x73, 0x69, 0x61, 0x2f, 0x4d, 0x61, 0x6b, 0x61, 0x73, 0x73,
0x61, 0x72, 0x0, 0x41, 0x73, 0x69, 0x61, 0x2f, 0x55, 0x6c, 0x61, 0x6e, 0x5f, 0x42, 0x61, 0x74,
0x6f, 0x72, 0x0, 0x41, 0x74, 0x6c, 0x61, 0x6e, 0x74, 0x69, 0x63, 0x2f, 0x46, 0x61, 0x65, 0x72,
0x6f, 0x65, 0x0, 0x41, 0x74, 0x6c, 0x61, 0x6e, 0x74, 0x69, 0x63, 0x2f, 0x46, 0x61, 0x72, 0x6f,
0x65, 0x0, 0x41, 0x74, 0x6c, 0x61, 0x6e, 0x74, 0x69, 0x63, 0x2f, 0x4a, 0x61, 0x6e, 0x5f, 0x4d,
0x61, 0x79, 0x65, 0x6e, 0x0, 0x41, 0x72, 0x63, 0x74, 0x69, 0x63, 0x2f, 0x4c, 0x6f, 0x6e, 0x67,
0x79, 0x65, 0x61, 0x72, 0x62, 0x79, 0x65, 0x6e, 0x0, 0x41, 0x75, 0x73, 0x74, 0x72, 0x61, 0x6c,
0x69, 0x61, 0x2f, 0x41, 0x43, 0x54, 0x0, 0x41, 0x75, 0x73, 0x74, 0x72, 0x61, 0x6c, 0x69, 0x61,
0x2f, 0x53, 0x79, 0x64, 0x6e, 0x65, 0x79, 0x0, 0x41, 0x75, 0x73, 0x74, 0x72, 0x61, 0x6c, 0x69,
0x61, 0x2f, 0x43, 0x61, 0x6e, 0x62, 0x65, 0x72, 0x72, 0x61, 0x0, 0x41, 0x75, 0x73, 0x74, 0x72,
0x61, 0x6c, 0x69, 0x61, 0x2f, 0x43, 0x75, 0x72, 0x72, 0x69, 0x65, 0x0, 0x41, 0x75, 0x73, 0x74,
0x72, 0x61, 0x6c, 0x69, 0x61, 0x2f, 0x48, 0x6f, 0x62, 0x61, 0x72, 0x74, 0x0, 0x41, 0x75, 0x73,
0x74, 0x72, 0x61, 0x6c, 0x69, 0x61, 0x2f, 0x4c, 0x48, 0x49, 0x0, 0x41, 0x75, 0x73, 0x74, 0x72,
0x61, 0x6c, 0x69, 0x61, 0x2f, 0x4c, 0x6f, 0x72, 0x64, 0x5f, 0x48, 0x6f, 0x77, 0x65, 0x0, 0x41,
0x75, 0x73, 0x74, 0x72, 0x61, 0x6c, 0x69, 0x61, 0x2f, 0x4e, 0x6f, 0x72, 0x74, 0x68, 0x0, 0x41,
0x75, 0x73, 0x74, 0x72, 0x61, 0x6c, 0x69, 0x61, 0x2f, 0x44, 0x61, 0x72, 0x77, 0x69, 0x6e, 0x0,
0x41, 0x75, 0x73, 0x74, 0x72, 0x61, 0x6c, 0x69, 0x61, 0x2f, 0x4e, 0x53, 0x57, 0x0, 0x41, 0x75,
0x73, 0x74, 0x72, 0x61, 0x6c, 0x69, 0x61, 0x2f, 0x51, 0x75, 0x65, 0x65, 0x6e, 0x73, 0x6c, 0x61,
0x6e, 0x64, 0x0, 0x41, 0x75, 0x73, 0x74, 0x72, 0x61, 0x6c, 0x69, 0x61, 0x2f, 0x42, 0x72, 0x69,
0x73, 0x62, 0x61, 0x6e, 0x65, 0x0, 0x41, 0x75, 0x73, 0x74, 0x72, 0x61, 0x6c, 0x69, 0x61, 0x2f,
0x53, 0x6f, 0x75, 0x74, 0x68, 0x0, 0x41, 0x75, 0x73, 0x74, 0x72, 0x61, 0x6c, 0x69, 0x61, 0x2f,
0x41, 0x64, 0x65, 0x6c, 0x61, 0x69, 0x64, 0x65, 0x0, 0x41, 0x75, 0x73, 0x74, 0x72, 0x61, 0x6c,
0x69, 0x61, 0x2f, 0x54, 0x61, 0x73, 0x6d, 0x61, 0x6e, 0x69, 0x61, 0x0, 0x41, 0x75, 0x73, 0x74,
0x72, 0x61, 0x6c, 0x69, 0x61, 0x2f, 0x56, 0x69, 0x63, 0x74, 0x6f, 0x72, 0x69, 0x61, 0x0, 0x41,
0x75, 0x73, 0x74, 0x72, 0x61, 0x6c, 0x69, 0x61, 0x2f, 0x4d, 0x65, 0x6c, 0x62, 0x6f, 0x75, 0x72,
0x6e, 0x65, 0x0, 0x41, 0x75, 0x73, 0x74, 0x72, 0x61, 0x6c, 0x69, 0x61, 0x2f, 0x57, 0x65, 0x73,
0x74, 0x0, 0x41, 0x75, 0x73, 0x74, 0x72, 0x61, 0x6c, 0x69, 0x61, 0x2f, 0x50, 0x65, 0x72, 0x74,
0x68, 0x0, 0x41, 0x75, 0x73, 0x74, 0x72, 0x61, 0x6c, 0x69, 0x61, 0x2f, 0x59, 0x61, 0x6e, 0x63,
0x6f, 0x77, 0x69, 0x6e, 0x6e, 0x61, 0x0, 0x41, 0x75, 0x73, 0x74, 0x72, 0x61, 0x6c, 0x69, 0x61,
0x2f, 0x42, 0x72, 0x6f, 0x6b, 0x65, 0x6e, 0x5f, 0x48, 0x69, 0x6c, 0x6c, 0x0, 0x42, 0x72, 0x61,
0x7a, 0x69, 0x6c, 0x2f, 0x41, 0x63, 0x72, 0x65, 0x0, 0x42, 0x72, 0x61, 0x7a, 0x69, 0x6c, 0x2f,
0x44, 0x65, 0x4e, 0x6f, 0x72, 0x6f, 0x6e, 0x68, 0x61, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63,
0x61, 0x2f, 0x4e, 0x6f, 0x72, 0x6f, 0x6e, 0x68, 0x61, 0x0, 0x42, 0x72, 0x61, 0x7a, 0x69, 0x6c,
0x2f, 0x45, 0x61, 0x73, 0x74, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x53, 0x61,
0x6f, 0x5f, 0x50, 0x61, 0x75, 0x6c, 0x6f, 0x0, 0x42, 0x72, 0x61, 0x7a, 0x69, 0x6c, 0x2f, 0x57,
0x65, 0x73, 0x74, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x4d, 0x61, 0x6e, 0x61,
0x75, 0x73, 0x0, 0x43, 0x61, 0x6e, 0x61, 0x64, 0x61, 0x2f, 0x41, 0x74, 0x6c, 0x61, 0x6e, 0x74,
0x69, 0x63, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x48, 0x61, 0x6c, 0x69, 0x66,
0x61, 0x78, 0x0, 0x43, 0x61, 0x6e, 0x61, 0x64, 0x61, 0x2f, 0x43, 0x65, 0x6e, 0x74, 0x72, 0x61,
0x6c, 0x0, 0x43, 0x61, 0x6e, 0x61, 0x64, 0x61, 0x2f, 0x45, 0x61, 0x73, 0x74, 0x2d, 0x53, 0x61,
0x73, 0x6b, 0x61, 0x74, 0x63, 0x68, 0x65, 0x77, 0x61, 0x6e, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69,
0x63, 0x61, 0x2f, 0x52, 0x65, 0x67, 0x69, 0x6e, 0x61, 0x0, 0x43, 0x61, 0x6e, 0x61, 0x64, 0x61,
0x2f, 0x45, 0x61, 0x73, 0x74, 0x65, 0x72, 0x6e, 0x0, 0x43, 0x61, 0x6e, 0x61, 0x64, 0x61, 0x2f,
0x4d, 0x6f, 0x75, 0x6e, 0x74, 0x61, 0x69, 0x6e, 0x0, 0x43, 0x61, 0x6e, 0x61, 0x64, 0x61, 0x2f,
0x4e, 0x65, 0x77, 0x66, 0x6f, 0x75, 0x6e, 0x64, 0x6c, 0x61, 0x6e, 0x64, 0x0, 0x41, 0x6d, 0x65,
0x72, 0x69, 0x63, 0x61, 0x2f, 0x53, 0x74, 0x5f, 0x4a, 0x6f, 0x68, 0x6e, 0x73, 0x0, 0x43, 0x61,
0x6e, 0x61, 0x64, 0x61, 0x2f, 0x50, 0x61, 0x63, 0x69, 0x66, 0x69, 0x63, 0x0, 0x41, 0x6d, 0x65,
0x72, 0x69, 0x63, 0x61, 0x2f, 0x56, 0x61, 0x6e, 0x63, 0x6f, 0x75, 0x76, 0x65, 0x72, 0x0, 0x43,
0x61, 0x6e, 0x61, 0x64, 0x61, 0x2f, 0x53, 0x61, 0x73, 0x6b, 0x61, 0x74, 0x63, 0x68, 0x65, 0x77,
0x61, 0x6e, 0x0, 0x43, 0x61, 0x6e, 0x61, 0x64, 0x61, 0x2f, 0x59, 0x75, 0x6b, 0x6f, 0x6e, 0x0,
0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x57, 0x68, 0x69, 0x74, 0x65, 0x68, 0x6f, 0x72,
0x73, 0x65, 0x0, 0x43, 0x45, 0x54, 0x0, 0x45, 0x75, 0x72, 0x6f, 0x70, 0x65, 0x2f, 0x42, 0x72,
0x75, 0x73, 0x73, 0x65, 0x6c, 0x73, 0x0, 0x43, 0x68, 0x69, 0x6c, 0x65, 0x2f, 0x43, 0x6f, 0x6e,
0x74, 0x69, 0x6e, 0x65, 0x6e, 0x74, 0x61, 0x6c, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61,
0x2f, 0x53, 0x61, 0x6e, 0x74, 0x69, 0x61, 0x67, 0x6f, 0x0, 0x43, 0x68, 0x69, 0x6c, 0x65, 0x2f,
0x45, 0x61, 0x73, 0x74, 0x65, 0x72, 0x49, 0x73, 0x6c, 0x61, 0x6e, 0x64, 0x0, 0x50, 0x61, 0x63,
0x69, 0x66, 0x69, 0x63, 0x2f, 0x45, 0x61, 0x73, 0x74, 0x65, 0x72, 0x0, 0x43, 0x53, 0x54, 0x36,
0x43, 0x44, 0x54, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x43, 0x68, 0x69, 0x63,
0x61, 0x67, 0x6f, 0x0, 0x43, 0x75, 0x62, 0x61, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61,
0x2f, 0x48, 0x61, 0x76, 0x61, 0x6e, 0x61, 0x0, 0x45, 0x45, 0x54, 0x0, 0x45, 0x75, 0x72, 0x6f,
0x70, 0x65, 0x2f, 0x41, 0x74, 0x68, 0x65, 0x6e, 0x73, 0x0, 0x45, 0x67, 0x79, 0x70, 0x74, 0x0,
0x41, 0x66, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x43, 0x61, 0x69, 0x72, 0x6f, 0x0, 0x45, 0x69, 0x72,
0x65, 0x0, 0x45, 0x75, 0x72, 0x6f, 0x70, 0x65, 0x2f, 0x44, 0x75, 0x62, 0x6c, 0x69, 0x6e, 0x0,
0x45, 0x53, 0x54, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x50, 0x61, 0x6e, 0x61,
0x6d, 0x61, 0x0, 0x45, 0x53, 0x54, 0x35, 0x45, 0x44, 0x54, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69,
0x63, 0x61, 0x2f, 0x4e, 0x65, 0x77, 0x5f, 0x59, 0x6f, 0x72, 0x6b, 0x0, 0x45, 0x74, 0x63, 0x2f,
0x47, 0x4d, 0x54, 0x2b, 0x30, 0x0, 0x45, 0x74, 0x63, 0x2f, 0x47, 0x4d, 0x54, 0x0, 0x45, 0x74,
0x63, 0x2f, 0x47, 0x4d, 0x54, 0x2d, 0x30, 0x0, 0x45, 0x74, 0x63, 0x2f, 0x47, 0x4d, 0x54, 0x30,
0x0, 0x45, 0x74, 0x63, 0x2f, 0x47, 0x72, 0x65, 0x65, 0x6e, 0x77, 0x69, 0x63, 0x68, 0x0, 0x45,
0x74, 0x63, 0x2f, 0x55, 0x43, 0x54, 0x0, 0x45, 0x74, 0x63, 0x2f, 0x55, 0x54, 0x43, 0x0, 0x45,
0x74, 0x63, 0x2f, 0x55, 0x6e, 0x69, 0x76, 0x65, 0x72, 0x73, 0x61, 0x6c, 0x0, 0x45, 0x74, 0x63,
0x2f, 0x5a, 0x75, 0x6c, 0x75, 0x0, 0x45, 0x75, 0x72, 0x6f, 0x70, 0x65, 0x2f, 0x42, 0x65, 0x6c,
0x66, 0x61, 0x73, 0x74, 0x0, 0x45, 0x75, 0x72, 0x6f, 0x70, 0x65, 0x2f, 0x4c, 0x6f, 0x6e, 0x64,
0x6f, 0x6e, 0x0, 0x45, 0x75, 0x72, 0x6f, 0x70, 0x65, 0x2f, 0x4b, 0x69, 0x65, 0x76, 0x0, 0x45,
0x75, 0x72, 0x6f, 0x70, 0x65, 0x2f, 0x4b, 0x79, 0x69, 0x76, 0x0, 0x45, 0x75, 0x72, 0x6f, 0x70,
0x65, 0x2f, 0x4e, 0x69, 0x63, 0x6f, 0x73, 0x69, 0x61, 0x0, 0x41, 0x73, 0x69, 0x61, 0x2f, 0x4e,
0x69, 0x63, 0x6f, 0x73, 0x69, 0x61, 0x0, 0x45, 0x75, 0x72, 0x6f, 0x70, 0x65, 0x2f, 0x54, 0x69,
0x72, 0x61, 0x73, 0x70, 0x6f, 0x6c, 0x0, 0x45, 0x75, 0x72, 0x6f, 0x70, 0x65, 0x2f, 0x43, 0x68,
0x69, 0x73, 0x69, 0x6e, 0x61, 0x75, 0x0, 0x45, 0x75, 0x72, 0x6f, 0x70, 0x65, 0x2f, 0x55, 0x7a,
0x68, 0x67, 0x6f, 0x72, 0x6f, 0x64, 0x0, 0x45, 0x75, 0x72, 0x6f, 0x70, 0x65, 0x2f, 0x5a, 0x61,
0x70, 0x6f, 0x72, 0x6f, 0x7a, 0x68, 0x79, 0x65, 0x0, 0x46, 0x61, 0x63, 0x74, 0x6f, 0x72, 0x79,
0x0, 0x45, 0x74, 0x63, 0x2f, 0x55, 0x6e, 0x6b, 0x6e, 0x6f, 0x77, 0x6e, 0x0, 0x47, 0x42, 0x0,
0x47, 0x42, 0x2d, 0x45, 0x69, 0x72, 0x65, 0x0, 0x47, 0x4d, 0x54, 0x0, 0x47, 0x4d, 0x54, 0x2b,
0x30, 0x0, 0x47, 0x4d, 0x54, 0x2d, 0x30, 0x0, 0x47, 0x4d, 0x54, 0x30, 0x0, 0x47, 0x72, 0x65,
0x65, 0x6e, 0x77, 0x69, 0x63, 0x68, 0x0, 0x48, 0x6f, 0x6e, 0x67, 0x6b, 0x6f, 0x6e, 0x67, 0x0,
0x41, 0x73, 0x69, 0x61, 0x2f, 0x48, 0x6f, 0x6e, 0x67, 0x5f, 0x4b, 0x6f, 0x6e, 0x67, 0x0, 0x48,
0x53, 0x54, 0x0, 0x50, 0x61, 0x63, 0x69, 0x66, 0x69, 0x63, 0x2f, 0x48, 0x6f, 0x6e, 0x6f, 0x6c,
0x75, 0x6c, 0x75, 0x0, 0x49, 0x63, 0x65, 0x6c, 0x61, 0x6e, 0x64, 0x0, 0x41, 0x74, 0x6c, 0x61,
0x6e, 0x74, 0x69, 0x63, 0x2f, 0x52, 0x65, 0x79, 0x6b, 0x6a, 0x61, 0x76, 0x69, 0x6b, 0x0, 0x49,
0x72, 0x61, 0x6e, 0x0, 0x41, 0x73, 0x69, 0x61, 0x2f, 0x54, 0x65, 0x68, 0x72, 0x61, 0x6e, 0x0,
0x49, 0x73, 0x72, 0x61, 0x65, 0x6c, 0x0, 0x4a, 0x61, 0x6d, 0x61, 0x69, 0x63, 0x61, 0x0, 0x41,
0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x4a, 0x61, 0x6d, 0x61, 0x69, 0x63, 0x61, 0x0, 0x4a,
0x61, 0x70, 0x61, 0x6e, 0x0, 0x41, 0x73, 0x69, 0x61, 0x2f, 0x54, 0x6f, 0x6b, 0x79, 0x6f, 0x0,
0x4b, 0x77, 0x61, 0x6a, 0x61, 0x6c, 0x65, 0x69, 0x6e, 0x0, 0x50, 0x61, 0x63, 0x69, 0x66, 0x69,
0x63, 0x2f, 0x4b, 0x77, 0x61, 0x6a, 0x61, 0x6c, 0x65, 0x69, 0x6e, 0x0, 0x4c, 0x69, 0x62, 0x79,
0x61, 0x0, 0x41, 0x66, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x54, 0x72, 0x69, 0x70, 0x6f, 0x6c, 0x69,
0x0, 0x4d, 0x45, 0x54, 0x0, 0x4d, 0x65, 0x78, 0x69, 0x63, 0x6f, 0x2f, 0x42, 0x61, 0x6a, 0x61,
0x4e, 0x6f, 0x72, 0x74, 0x65, 0x0, 0x4d, 0x65, 0x78, 0x69, 0x63, 0x6f, 0x2f, 0x42, 0x61, 0x6a,
0x61, 0x53, 0x75, 0x72, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x4d, 0x61, 0x7a,
0x61, 0x74, 0x6c, 0x61, 0x6e, 0x0, 0x4d, 0x65, 0x78, 0x69, 0x63, 0x6f, 0x2f, 0x47, 0x65, 0x6e,
0x65, 0x72, 0x61, 0x6c, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x4d, 0x65, 0x78,
0x69, 0x63, 0x6f, 0x5f, 0x43, 0x69, 0x74, 0x79, 0x0, 0x4d, 0x53, 0x54, 0x0, 0x41, 0x6d, 0x65,
0x72, 0x69, 0x63, 0x61, 0x2f, 0x50, 0x68, 0x6f, 0x65, 0x6e, 0x69, 0x78, 0x0, 0x4d, 0x53, 0x54,
0x37, 0x4d, 0x44, 0x54, 0x0, 0x4e, 0x61, 0x76, 0x61, 0x6a, 0x6f, 0x0, 0x4e, 0x5a, 0x0, 0x4e,
0x5a, 0x2d, 0x43, 0x48, 0x41, 0x54, 0x0, 0x50, 0x61, 0x63, 0x69, 0x66, 0x69, 0x63, 0x2f, 0x43,
0x68, 0x61, 0x74, 0x68, 0x61, 0x6d, 0x0, 0x50, 0x61, 0x63, 0x69, 0x66, 0x69, 0x63, 0x2f, 0x45,
0x6e, 0x64, 0x65, 0x72, 0x62, 0x75, 0x72, 0x79, 0x0, 0x50, 0x61, 0x63, 0x69, 0x66, 0x69, 0x63,
0x2f, 0x4b, 0x61, 0x6e, 0x74, 0x6f, 0x6e, 0x0, 0x50, 0x61, 0x63, 0x69, 0x66, 0x69, 0x63, 0x2f,
0x4a, 0x6f, 0x68, 0x6e, 0x73, 0x74, 0x6f, 0x6e, 0x0, 0x50, 0x61, 0x63, 0x69, 0x66, 0x69, 0x63,
0x2f, 0x50, 0x6f, 0x6e, 0x61, 0x70, 0x65, 0x0, 0x50, 0x61, 0x63, 0x69, 0x66, 0x69, 0x63, 0x2f,
0x50, 0x6f, 0x68, 0x6e, 0x70, 0x65, 0x69, 0x0, 0x50, 0x61, 0x63, 0x69, 0x66, 0x69, 0x63, 0x2f,
0x53, 0x61, 0x6d, 0x6f, 0x61, 0x0, 0x50, 0x61, 0x63, 0x69, 0x66, 0x69, 0x63, 0x2f, 0x50, 0x61,
0x67, 0x6f, 0x5f, 0x50, 0x61, 0x67, 0x6f, 0x0, 0x50, 0x61, 0x63, 0x69, 0x66, 0x69, 0x63, 0x2f,
0x54, 0x72, 0x75, 0x6b, 0x0, 0x50, 0x61, 0x63, 0x69, 0x66, 0x69, 0x63, 0x2f, 0x43, 0x68, 0x75,
0x75, 0x6b, 0x0, 0x50, 0x61, 0x63, 0x69, 0x66, 0x69, 0x63, 0x2f, 0x59, 0x61, 0x70, 0x0, 0x50,
0x6f, 0x6c, 0x61, 0x6e, 0x64, 0x0, 0x45, 0x75, 0x72, 0x6f, 0x70, 0x65, 0x2f, 0x57, 0x61, 0x72,
0x73, 0x61, 0x77, 0x0, 0x50, 0x6f, 0x72, 0x74, 0x75, 0x67, 0x61, 0x6c, 0x0, 0x45, 0x75, 0x72,
0x6f, 0x70, 0x65, 0x2f, 0x4c, 0x69, 0x73, 0x62, 0x6f, 0x6e, 0x0, 0x50, 0x52, 0x43, 0x0, 0x50,
0x53, 0x54, 0x38, 0x50, 0x44, 0x54, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x4c,
0x6f, 0x73, 0x5f, 0x41, 0x6e, 0x67, 0x65, 0x6c, 0x65, 0x73, 0x0, 0x52, 0x4f, 0x43, 0x0, 0x41,
0x73, 0x69, 0x61, 0x2f, 0x54, 0x61, 0x69, 0x70, 0x65, 0x69, 0x0, 0x52, 0x4f, 0x4b, 0x0, 0x41,
0x73, 0x69, 0x61, 0x2f, 0x53, 0x65, 0x6f, 0x75, 0x6c, 0x0, 0x53, 0x69, 0x6e, 0x67, 0x61, 0x70,
0x6f, 0x72, 0x65, 0x0, 0x41, 0x73, 0x69, 0x61, 0x2f, 0x53, 0x69, 0x6e, 0x67, 0x61, 0x70, 0x6f,
0x72, 0x65, 0x0, 0x54, 0x75, 0x72, 0x6b, 0x65, 0x79, 0x0, 0x55, 0x43, 0x54, 0x0, 0x55, 0x6e,
0x69, 0x76, 0x65, 0x72, 0x73, 0x61, 0x6c, 0x0, 0x55, 0x53, 0x2f, 0x41, 0x6c, 0x61, 0x73, 0x6b,
0x61, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x41, 0x6e, 0x63, 0x68, 0x6f, 0x72,
0x61, 0x67, 0x65, 0x0, 0x55, 0x53, 0x2f, 0x41, 0x6c, 0x65, 0x75, 0x74, 0x69, 0x61, 0x6e, 0x0,
0x55, 0x53, 0x2f, 0x41, 0x72, 0x69, 0x7a, 0x6f, 0x6e, 0x61, 0x0, 0x55, 0x53, 0x2f, 0x43, 0x65,
0x6e, 0x74, 0x72, 0x61, 0x6c, 0x0, 0x55, 0x53, 0x2f, 0x45, 0x61, 0x73, 0x74, 0x2d, 0x49, 0x6e,
0x64, 0x69, 0x61, 0x6e, 0x61, 0x0, 0x55, 0x53, 0x2f, 0x45, 0x61, 0x73, 0x74, 0x65, 0x72, 0x6e,
0x0, 0x55, 0x53, 0x2f, 0x48, 0x61, 0x77, 0x61, 0x69, 0x69, 0x0, 0x55, 0x53, 0x2f, 0x49, 0x6e,
0x64, 0x69, 0x61, 0x6e, 0x61, 0x2d, 0x53, 0x74, 0x61, 0x72, 0x6b, 0x65, 0x0, 0x55, 0x53, 0x2f,
0x4d, 0x69, 0x63, 0x68, 0x69, 0x67, 0x61, 0x6e, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61,
0x2f, 0x44, 0x65, 0x74, 0x72, 0x6f, 0x69, 0x74, 0x0, 0x55, 0x53, 0x2f, 0x4d, 0x6f, 0x75, 0x6e,
0x74, 0x61, 0x69, 0x6e, 0x0, 0x55, 0x53, 0x2f, 0x50, 0x61, 0x63, 0x69, 0x66, 0x69, 0x63, 0x0,
0x55, 0x53, 0x2f, 0x50, 0x61, 0x63, 0x69, 0x66, 0x69, 0x63, 0x2d, 0x4e, 0x65, 0x77, 0x0, 0x55,
0x53, 0x2f, 0x53, 0x61, 0x6d, 0x6f, 0x61, 0x0, 0x55, 0x54, 0x43, 0x0, 0x57, 0x2d, 0x53, 0x55,
0x0, 0x45, 0x75, 0x72, 0x6f, 0x70, 0x65, 0x2f, 0x4d, 0x6f, 0x73, 0x63, 0x6f, 0x77, 0x0, 0x57,
0x45, 0x54, 0x0, 0x5a, 0x75, 0x6c, 0x75, 0x0, 0x41, 0x73, 0x69, 0x61, 0x2f, 0x4b, 0x61, 0x62,
0x75, 0x6c, 0x0, 0x41, 0x73, 0x69, 0x61, 0x2f, 0x42, 0x61, 0x72, 0x6e, 0x61, 0x75, 0x6c, 0x0,
0x41, 0x73, 0x69, 0x61, 0x2f, 0x52, 0x69, 0x79, 0x61, 0x64, 0x68, 0x0, 0x41, 0x73, 0x69, 0x61,
0x2f, 0x44, 0x75, 0x62, 0x61, 0x69, 0x0, 0x41, 0x73, 0x69, 0x61, 0x2f, 0x42, 0x61, 0x67, 0x68,
0x64, 0x61, 0x64, 0x0, 0x45, 0x75, 0x72, 0x6f, 0x70, 0x65, 0x2f, 0x41, 0x73, 0x74, 0x72, 0x61,
0x6b, 0x68, 0x61, 0x6e, 0x0, 0x41, 0x75, 0x73, 0x74, 0x72, 0x61, 0x6c, 0x69, 0x61, 0x2f, 0x45,
0x75, 0x63, 0x6c, 0x61, 0x0, 0x41, 0x73, 0x69, 0x61, 0x2f, 0x42, 0x61, 0x6b, 0x75, 0x0, 0x41,
0x74, 0x6c, 0x61, 0x6e, 0x74, 0x69, 0x63, 0x2f, 0x41, 0x7a, 0x6f, 0x72, 0x65, 0x73, 0x0, 0x41,
0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x42, 0x61, 0x68, 0x69, 0x61, 0x0, 0x45, 0x75, 0x72,
0x6f, 0x70, 0x65, 0x2f, 0x4d, 0x69, 0x6e, 0x73, 0x6b, 0x0, 0x50, 0x61, 0x63, 0x69, 0x66, 0x69,
0x63, 0x2f, 0x42, 0x6f, 0x75, 0x67, 0x61, 0x69, 0x6e, 0x76, 0x69, 0x6c, 0x6c, 0x65, 0x0, 0x41,
0x74, 0x6c, 0x61, 0x6e, 0x74, 0x69, 0x63, 0x2f, 0x43, 0x61, 0x70, 0x65, 0x5f, 0x56, 0x65, 0x72,
0x64, 0x65, 0x0, 0x41, 0x73, 0x69, 0x61, 0x2f, 0x59, 0x65, 0x72, 0x65, 0x76, 0x61, 0x6e, 0x0,
0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x47, 0x75, 0x61, 0x74, 0x65, 0x6d, 0x61, 0x6c,
0x61, 0x0, 0x41, 0x73, 0x69, 0x61, 0x2f, 0x42, 0x69, 0x73, 0x68, 0x6b, 0x65, 0x6b, 0x0, 0x41,
0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x43, 0x75, 0x69, 0x61, 0x62, 0x61, 0x0, 0x45, 0x75,
0x72, 0x6f, 0x70, 0x65, 0x2f, 0x42, 0x75, 0x64, 0x61, 0x70, 0x65, 0x73, 0x74, 0x0, 0x50, 0x61,
0x63, 0x69, 0x66, 0x69, 0x63, 0x2f, 0x47, 0x75, 0x61, 0x64, 0x61, 0x6c, 0x63, 0x61, 0x6e, 0x61,
0x6c, 0x0, 0x45, 0x74, 0x63, 0x2f, 0x47, 0x4d, 0x54, 0x2b, 0x31, 0x32, 0x0, 0x41, 0x66, 0x72,
0x69, 0x63, 0x61, 0x2f, 0x4e, 0x61, 0x69, 0x72, 0x6f, 0x62, 0x69, 0x0, 0x41, 0x6d, 0x65, 0x72,
0x69, 0x63, 0x61, 0x2f, 0x43, 0x61, 0x6e, 0x63, 0x75, 0x6e, 0x0, 0x41, 0x73, 0x69, 0x61, 0x2f,
0x59, 0x65, 0x6b, 0x61, 0x74, 0x65, 0x72, 0x69, 0x6e, 0x62, 0x75, 0x72, 0x67, 0x0, 0x50, 0x61,
0x63, 0x69, 0x66, 0x69, 0x63, 0x2f, 0x46, 0x69, 0x6a, 0x69, 0x0, 0x41, 0x73, 0x69, 0x61, 0x2f,
0x54, 0x62, 0x69, 0x6c, 0x69, 0x73, 0x69, 0x0, 0x45, 0x75, 0x72, 0x6f, 0x70, 0x65, 0x2f, 0x42,
0x75, 0x63, 0x68, 0x61, 0x72, 0x65, 0x73, 0x74, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61,
0x2f, 0x50, 0x6f, 0x72, 0x74, 0x2d, 0x61, 0x75, 0x2d, 0x50, 0x72, 0x69, 0x6e, 0x63, 0x65, 0x0,
0x41, 0x73, 0x69, 0x61, 0x2f, 0x41, 0x6d, 0x6d, 0x61, 0x6e, 0x0, 0x45, 0x75, 0x72, 0x6f, 0x70,
0x65, 0x2f, 0x4b, 0x61, 0x6c, 0x69, 0x6e, 0x69, 0x6e, 0x67, 0x72, 0x61, 0x64, 0x0, 0x50, 0x61,
0x63, 0x69, 0x66, 0x69, 0x63, 0x2f, 0x4b, 0x69, 0x72, 0x69, 0x74, 0x69, 0x6d, 0x61, 0x74, 0x69,
0x0, 0x41, 0x73, 0x69, 0x61, 0x2f, 0x4d, 0x61, 0x67, 0x61, 0x64, 0x61, 0x6e, 0x0, 0x41, 0x6d,
0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x50, 0x75, 0x6e, 0x74, 0x61, 0x5f, 0x41, 0x72, 0x65, 0x6e,
0x61, 0x73, 0x0, 0x50, 0x61, 0x63, 0x69, 0x66, 0x69, 0x63, 0x2f, 0x4d, 0x61, 0x72, 0x71, 0x75,
0x65, 0x73, 0x61, 0x73, 0x0, 0x49, 0x6e, 0x64, 0x69, 0x61, 0x6e, 0x2f, 0x4d, 0x61, 0x75, 0x72,
0x69, 0x74, 0x69, 0x75, 0x73, 0x0, 0x41, 0x73, 0x69, 0x61, 0x2f, 0x42, 0x65, 0x69, 0x72, 0x75,
0x74, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x4d, 0x6f, 0x6e, 0x74, 0x65, 0x76,
0x69, 0x64, 0x65, 0x6f, 0x0, 0x41, 0x66, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x43, 0x61, 0x73, 0x61,
0x62, 0x6c, 0x61, 0x6e, 0x63, 0x61, 0x0, 0x41, 0x73, 0x69, 0x61, 0x2f, 0x4e, 0x6f, 0x76, 0x6f,
0x73, 0x69, 0x62, 0x69, 0x72, 0x73, 0x6b, 0x0, 0x41, 0x66, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x57,
0x69, 0x6e, 0x64, 0x68, 0x6f, 0x65, 0x6b, 0x0, 0x50, 0x61, 0x63, 0x69, 0x66, 0x69, 0x63, 0x2f,
0x4e, 0x6f, 0x72, 0x66, 0x6f, 0x6c, 0x6b, 0x0, 0x41, 0x73, 0x69, 0x61, 0x2f, 0x49, 0x72, 0x6b,
0x75, 0x74, 0x73, 0x6b, 0x0, 0x41, 0x73, 0x69, 0x61, 0x2f, 0x4b, 0x72, 0x61, 0x73, 0x6e, 0x6f,
0x79, 0x61, 0x72, 0x73, 0x6b, 0x0, 0x41, 0x73, 0x69, 0x61, 0x2f, 0x50, 0x79, 0x6f, 0x6e, 0x67,
0x79, 0x61, 0x6e, 0x67, 0x0, 0x41, 0x73, 0x69, 0x61, 0x2f, 0x4f, 0x6d, 0x73, 0x6b, 0x0, 0x41,
0x73, 0x69, 0x61, 0x2f, 0x4b, 0x61, 0x72, 0x61, 0x63, 0x68, 0x69, 0x0, 0x41, 0x6d, 0x65, 0x72,
0x69, 0x63, 0x61, 0x2f, 0x41, 0x73, 0x75, 0x6e, 0x63, 0x69, 0x6f, 0x6e, 0x0, 0x41, 0x73, 0x69,
0x61, 0x2f, 0x51, 0x79, 0x7a, 0x79, 0x6c, 0x6f, 0x72, 0x64, 0x61, 0x0, 0x45, 0x75, 0x72, 0x6f,
0x70, 0x65, 0x2f, 0x50, 0x61, 0x72, 0x69, 0x73, 0x0, 0x41, 0x73, 0x69, 0x61, 0x2f, 0x53, 0x72,
0x65, 0x64, 0x6e, 0x65, 0x6b, 0x6f, 0x6c, 0x79, 0x6d, 0x73, 0x6b, 0x0, 0x41, 0x73, 0x69, 0x61,
0x2f, 0x4b, 0x61, 0x6d, 0x63, 0x68, 0x61, 0x74, 0x6b, 0x61, 0x0, 0x45, 0x75, 0x72, 0x6f, 0x70,
0x65, 0x2f, 0x53, 0x61, 0x6d, 0x61, 0x72, 0x61, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61,
0x2f, 0x43, 0x61, 0x79, 0x65, 0x6e, 0x6e, 0x65, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61,
0x2f, 0x42, 0x6f, 0x67, 0x6f, 0x74, 0x61, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f,
0x4c, 0x61, 0x5f, 0x50, 0x61, 0x7a, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x4d,
0x69, 0x71, 0x75, 0x65, 0x6c, 0x6f, 0x6e, 0x0, 0x41, 0x73, 0x69, 0x61, 0x2f, 0x53, 0x61, 0x6b,
0x68, 0x61, 0x6c, 0x69, 0x6e, 0x0, 0x50, 0x61, 0x63, 0x69, 0x66, 0x69, 0x63, 0x2f, 0x41, 0x70,
0x69, 0x61, 0x0, 0x41, 0x66, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x53, 0x61, 0x6f, 0x5f, 0x54, 0x6f,
0x6d, 0x65, 0x0, 0x45, 0x75, 0x72, 0x6f, 0x70, 0x65, 0x2f, 0x53, 0x61, 0x72, 0x61, 0x74, 0x6f,
0x76, 0x0, 0x41, 0x73, 0x69, 0x61, 0x2f, 0x42, 0x61, 0x6e, 0x67, 0x6b, 0x6f, 0x6b, 0x0, 0x41,
0x66, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x4a, 0x6f, 0x68, 0x61, 0x6e, 0x6e, 0x65, 0x73, 0x62, 0x75,
0x72, 0x67, 0x0, 0x41, 0x66, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x4a, 0x75, 0x62, 0x61, 0x0, 0x41,
0x73, 0x69, 0x61, 0x2f, 0x43, 0x6f, 0x6c, 0x6f, 0x6d, 0x62, 0x6f, 0x0, 0x41, 0x66, 0x72, 0x69,
0x63, 0x61, 0x2f, 0x4b, 0x68, 0x61, 0x72, 0x74, 0x6f, 0x75, 0x6d, 0x0, 0x41, 0x73, 0x69, 0x61,
0x2f, 0x44, 0x61, 0x6d, 0x61, 0x73, 0x63, 0x75, 0x73, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63,
0x61, 0x2f, 0x41, 0x72, 0x61, 0x67, 0x75, 0x61, 0x69, 0x6e, 0x61, 0x0, 0x41, 0x73, 0x69, 0x61,
0x2f, 0x54, 0x6f, 0x6d, 0x73, 0x6b, 0x0, 0x50, 0x61, 0x63, 0x69, 0x66, 0x69, 0x63, 0x2f, 0x54,
0x6f, 0x6e, 0x67, 0x61, 0x74, 0x61, 0x70, 0x75, 0x0, 0x41, 0x73, 0x69, 0x61, 0x2f, 0x43, 0x68,
0x69, 0x74, 0x61, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x47, 0x72, 0x61, 0x6e,
0x64, 0x5f, 0x54, 0x75, 0x72, 0x6b, 0x0, 0x45, 0x74, 0x63, 0x2f, 0x47, 0x4d, 0x54, 0x2d, 0x31,
0x32, 0x0, 0x45, 0x74, 0x63, 0x2f, 0x47, 0x4d, 0x54, 0x2d, 0x31, 0x33, 0x0, 0x45, 0x74, 0x63,
0x2f, 0x47, 0x4d, 0x54, 0x2b, 0x32, 0x0, 0x45, 0x74, 0x63, 0x2f, 0x47, 0x4d, 0x54, 0x2b, 0x38,
0x0, 0x45, 0x74, 0x63, 0x2f, 0x47, 0x4d, 0x54, 0x2b, 0x39, 0x0, 0x45, 0x74, 0x63, 0x2f, 0x47,
0x4d, 0x54, 0x2b, 0x31, 0x31, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x43, 0x61,
0x72, 0x61, 0x63, 0x61, 0x73, 0x0, 0x41, 0x73, 0x69, 0x61, 0x2f, 0x56, 0x6c, 0x61, 0x64, 0x69,
0x76, 0x6f, 0x73, 0x74, 0x6f, 0x6b, 0x0, 0x45, 0x75, 0x72, 0x6f, 0x70, 0x65, 0x2f, 0x56, 0x6f,
0x6c, 0x67, 0x6f, 0x67, 0x72, 0x61, 0x64, 0x0, 0x41, 0x66, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x4c,
0x61, 0x67, 0x6f, 0x73, 0x0, 0x45, 0x75, 0x72, 0x6f, 0x70, 0x65, 0x2f, 0x42, 0x65, 0x72, 0x6c,
0x69, 0x6e, 0x0, 0x41, 0x73, 0x69, 0x61, 0x2f, 0x48, 0x6f, 0x76, 0x64, 0x0, 0x41, 0x73, 0x69,
0x61, 0x2f, 0x54, 0x61, 0x73, 0x68, 0x6b, 0x65, 0x6e, 0x74, 0x0, 0x41, 0x73, 0x69, 0x61, 0x2f,
0x48, 0x65, 0x62, 0x72, 0x6f, 0x6e, 0x0, 0x50, 0x61, 0x63, 0x69, 0x66, 0x69, 0x63, 0x2f, 0x50,
0x6f, 0x72, 0x74, 0x5f, 0x4d, 0x6f, 0x72, 0x65, 0x73, 0x62, 0x79, 0x0, 0x41, 0x73, 0x69, 0x61,
0x2f, 0x59, 0x61, 0x6b, 0x75, 0x74, 0x73, 0x6b, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61,
0x2f, 0x47, 0x75, 0x61, 0x79, 0x61, 0x71, 0x75, 0x69, 0x6c, 0x0, 0x41, 0x73, 0x69, 0x61, 0x2f,
0x4b, 0x75, 0x61, 0x6c, 0x61, 0x5f, 0x4c, 0x75, 0x6d, 0x70, 0x75, 0x72, 0x0, 0x50, 0x61, 0x63,
0x69, 0x66, 0x69, 0x63, 0x2f, 0x4d, 0x61, 0x6a, 0x75, 0x72, 0x6f, 0x0, 0x45, 0x75, 0x72, 0x6f,
0x70, 0x65, 0x2f, 0x4d, 0x61, 0x64, 0x72, 0x69, 0x64, 0x0, 0x41, 0x66, 0x72, 0x69, 0x63, 0x61,
0x2f, 0x47, 0x61, 0x62, 0x6f, 0x72, 0x6f, 0x6e, 0x65, 0x0, 0x41, 0x66, 0x72, 0x69, 0x63, 0x61,
0x2f, 0x42, 0x75, 0x6a, 0x75, 0x6d, 0x62, 0x75, 0x72, 0x61, 0x0, 0x41, 0x66, 0x72, 0x69, 0x63,
0x61, 0x2f, 0x4c, 0x75, 0x62, 0x75, 0x6d, 0x62, 0x61, 0x73, 0x68, 0x69, 0x0, 0x41, 0x66, 0x72,
0x69, 0x63, 0x61, 0x2f, 0x42, 0x6c, 0x61, 0x6e, 0x74, 0x79, 0x72, 0x65, 0x0, 0x41, 0x66, 0x72,
0x69, 0x63, 0x61, 0x2f, 0x4b, 0x69, 0x67, 0x61, 0x6c, 0x69, 0x0, 0x41, 0x66, 0x72, 0x69, 0x63,
0x61, 0x2f, 0x4d, 0x61, 0x70, 0x75, 0x74, 0x6f, 0x0, 0x41, 0x66, 0x72, 0x69, 0x63, 0x61, 0x2f,
0x4c, 0x75, 0x73, 0x61, 0x6b, 0x61, 0x0, 0x41, 0x66, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x48, 0x61,
0x72, 0x61, 0x72, 0x65, 0x0, 0x49, 0x6e, 0x64, 0x69, 0x61, 0x6e, 0x2f, 0x43, 0x6f, 0x6d, 0x6f,
0x72, 0x6f, 0x0, 0x41, 0x66, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x44, 0x6a, 0x69, 0x62, 0x6f, 0x75,
0x74, 0x69, 0x0, 0x41, 0x66, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x41, 0x64, 0x64, 0x69, 0x73, 0x5f,
0x41, 0x62, 0x61, 0x62, 0x61, 0x0, 0x49, 0x6e, 0x64, 0x69, 0x61, 0x6e, 0x2f, 0x41, 0x6e, 0x74,
0x61, 0x6e, 0x61, 0x6e, 0x61, 0x72, 0x69, 0x76, 0x6f, 0x0, 0x49, 0x6e, 0x64, 0x69, 0x61, 0x6e,
0x2f, 0x4d, 0x61, 0x79, 0x6f, 0x74, 0x74, 0x65, 0x0, 0x41, 0x66, 0x72, 0x69, 0x63, 0x61, 0x2f,
0x4d, 0x6f, 0x67, 0x61, 0x64, 0x69, 0x73, 0x68, 0x75, 0x0, 0x41, 0x66, 0x72, 0x69, 0x63, 0x61,
0x2f, 0x44, 0x61, 0x72, 0x5f, 0x65, 0x73, 0x5f, 0x53, 0x61, 0x6c, 0x61, 0x61, 0x6d, 0x0, 0x41,
0x66, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x4b, 0x61, 0x6d, 0x70, 0x61, 0x6c, 0x61, 0x0, 0x41, 0x66,
0x72, 0x69, 0x63, 0x61, 0x2f, 0x45, 0x6c, 0x5f, 0x41, 0x61, 0x69, 0x75, 0x6e, 0x0, 0x41, 0x66,
0x72, 0x69, 0x63, 0x61, 0x2f, 0x4d, 0x62, 0x61, 0x62, 0x61, 0x6e, 0x65, 0x0, 0x41, 0x66, 0x72,
0x69, 0x63, 0x61, 0x2f, 0x4d, 0x61, 0x73, 0x65, 0x72, 0x75, 0x0, 0x41, 0x66, 0x72, 0x69, 0x63,
0x61, 0x2f, 0x4c, 0x75, 0x61, 0x6e, 0x64, 0x61, 0x0, 0x41, 0x66, 0x72, 0x69, 0x63, 0x61, 0x2f,
0x50, 0x6f, 0x72, 0x74, 0x6f, 0x2d, 0x4e, 0x6f, 0x76, 0x6f, 0x0, 0x41, 0x66, 0x72, 0x69, 0x63,
0x61, 0x2f, 0x44, 0x6f, 0x75, 0x61, 0x6c, 0x61, 0x0, 0x41, 0x66, 0x72, 0x69, 0x63, 0x61, 0x2f,
0x42, 0x61, 0x6e, 0x67, 0x75, 0x69, 0x0, 0x41, 0x66, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x4e, 0x64,
0x6a, 0x61, 0x6d, 0x65, 0x6e, 0x61, 0x0, 0x41, 0x66, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x42, 0x72,
0x61, 0x7a, 0x7a, 0x61, 0x76, 0x69, 0x6c, 0x6c, 0x65, 0x0, 0x41, 0x66, 0x72, 0x69, 0x63, 0x61,
0x2f, 0x4b, 0x69, 0x6e, 0x73, 0x68, 0x61, 0x73, 0x61, 0x0, 0x41, 0x66, 0x72, 0x69, 0x63, 0x61,
0x2f, 0x4d, 0x61, 0x6c, 0x61, 0x62, 0x6f, 0x0, 0x41, 0x66, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x4c,
0x69, 0x62, 0x72, 0x65, 0x76, 0x69, 0x6c, 0x6c, 0x65, 0x0, 0x41, 0x66, 0x72, 0x69, 0x63, 0x61,
0x2f, 0x4e, 0x69, 0x61, 0x6d, 0x65, 0x79, 0x0, 0x41, 0x73, 0x69, 0x61, 0x2f, 0x41, 0x71, 0x74,
0x6f, 0x62, 0x65, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x4a, 0x75, 0x6e, 0x65,
0x61, 0x75, 0x0, 0x41, 0x73, 0x69, 0x61, 0x2f, 0x41, 0x6c, 0x6d, 0x61, 0x74, 0x79, 0x0, 0x41,
0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x42, 0x65, 0x6c, 0x69, 0x7a, 0x65, 0x0, 0x41, 0x6d,
0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x43, 0x6f, 0x73, 0x74, 0x61, 0x5f, 0x52, 0x69, 0x63, 0x61,
0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x45, 0x6c, 0x5f, 0x53, 0x61, 0x6c, 0x76,
0x61, 0x64, 0x6f, 0x72, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x54, 0x65, 0x67,
0x75, 0x63, 0x69, 0x67, 0x61, 0x6c, 0x70, 0x61, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61,
0x2f, 0x4e, 0x61, 0x73, 0x73, 0x61, 0x75, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f,
0x43, 0x61, 0x79, 0x6d, 0x61, 0x6e, 0x0, 0x41, 0x73, 0x69, 0x61, 0x2f, 0x41, 0x6e, 0x61, 0x64,
0x79, 0x72, 0x0, 0x41, 0x73, 0x69, 0x61, 0x2f, 0x41, 0x71, 0x74, 0x61, 0x75, 0x0, 0x41, 0x73,
0x69, 0x61, 0x2f, 0x42, 0x61, 0x68, 0x72, 0x61, 0x69, 0x6e, 0x0, 0x41, 0x73, 0x69, 0x61, 0x2f,
0x4b, 0x75, 0x77, 0x61, 0x69, 0x74, 0x0, 0x41, 0x73, 0x69, 0x61, 0x2f, 0x51, 0x61, 0x74, 0x61,
0x72, 0x0, 0x41, 0x73, 0x69, 0x61, 0x2f, 0x41, 0x64, 0x65, 0x6e, 0x0, 0x41, 0x6d, 0x65, 0x72,
0x69, 0x63, 0x61, 0x2f, 0x41, 0x72, 0x67, 0x65, 0x6e, 0x74, 0x69, 0x6e, 0x61, 0x2f, 0x53, 0x61,
0x6e, 0x5f, 0x4c, 0x75, 0x69, 0x73, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x41,
0x6e, 0x67, 0x75, 0x69, 0x6c, 0x6c, 0x61, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f,
0x41, 0x6e, 0x74, 0x69, 0x67, 0x75, 0x61, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f,
0x41, 0x72, 0x75, 0x62, 0x61, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x42, 0x61,
0x72, 0x62, 0x61, 0x64, 0x6f, 0x73, 0x0, 0x41, 0x74, 0x6c, 0x61, 0x6e, 0x74, 0x69, 0x63, 0x2f,
0x42, 0x65, 0x72, 0x6d, 0x75, 0x64, 0x61, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f,
0x54, 0x6f, 0x72, 0x74, 0x6f, 0x6c, 0x61, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f,
0x4b, 0x72, 0x61, 0x6c, 0x65, 0x6e, 0x64, 0x69, 0x6a, 0x6b, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69,
0x63, 0x61, 0x2f, 0x43, 0x75, 0x72, 0x61, 0x63, 0x61, 0x6f, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69,
0x63, 0x61, 0x2f, 0x44, 0x6f, 0x6d, 0x69, 0x6e, 0x69, 0x63, 0x61, 0x0, 0x41, 0x6d, 0x65, 0x72,
0x69, 0x63, 0x61, 0x2f, 0x54, 0x68, 0x75, 0x6c, 0x65, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63,
0x61, 0x2f, 0x47, 0x72, 0x65, 0x6e, 0x61, 0x64, 0x61, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63,
0x61, 0x2f, 0x47, 0x75, 0x61, 0x64, 0x65, 0x6c, 0x6f, 0x75, 0x70, 0x65, 0x0, 0x41, 0x6d, 0x65,
0x72, 0x69, 0x63, 0x61, 0x2f, 0x4d, 0x61, 0x72, 0x74, 0x69, 0x6e, 0x69, 0x71, 0x75, 0x65, 0x0,
0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x4d, 0x6f, 0x6e, 0x74, 0x73, 0x65, 0x72, 0x72,
0x61, 0x74, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x50, 0x75, 0x65, 0x72, 0x74,
0x6f, 0x5f, 0x52, 0x69, 0x63, 0x6f, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x53,
0x74, 0x5f, 0x4b, 0x69, 0x74, 0x74, 0x73, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f,
0x53, 0x74, 0x5f, 0x4c, 0x75, 0x63, 0x69, 0x61, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61,
0x2f, 0x4d, 0x61, 0x72, 0x69, 0x67, 0x6f, 0x74, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61,
0x2f, 0x53, 0x74, 0x5f, 0x56, 0x69, 0x6e, 0x63, 0x65, 0x6e, 0x74, 0x0, 0x41, 0x6d, 0x65, 0x72,
0x69, 0x63, 0x61, 0x2f, 0x4c, 0x6f, 0x77, 0x65, 0x72, 0x5f, 0x50, 0x72, 0x69, 0x6e, 0x63, 0x65,
0x73, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x50, 0x6f, 0x72, 0x74, 0x5f, 0x6f,
0x66, 0x5f, 0x53, 0x70, 0x61, 0x69, 0x6e, 0x0, 0x41, 0x73, 0x69, 0x61, 0x2f, 0x4b, 0x75, 0x63,
0x68, 0x69, 0x6e, 0x67, 0x0, 0x41, 0x73, 0x69, 0x61, 0x2f, 0x42, 0x72, 0x75, 0x6e, 0x65, 0x69,
0x0, 0x41, 0x6e, 0x74, 0x61, 0x72, 0x63, 0x74, 0x69, 0x63, 0x61, 0x2f, 0x43, 0x61, 0x73, 0x65,
0x79, 0x0, 0x50, 0x61, 0x63, 0x69, 0x66, 0x69, 0x63, 0x2f, 0x47, 0x75, 0x61, 0x6d, 0x0, 0x50,
0x61, 0x63, 0x69, 0x66, 0x69, 0x63, 0x2f, 0x53, 0x61, 0x69, 0x70, 0x61, 0x6e, 0x0, 0x49, 0x6e,
0x64, 0x69, 0x61, 0x6e, 0x2f, 0x43, 0x68, 0x72, 0x69, 0x73, 0x74, 0x6d, 0x61, 0x73, 0x0, 0x49,
0x6e, 0x64, 0x69, 0x61, 0x6e, 0x2f, 0x43, 0x6f, 0x63, 0x6f, 0x73, 0x0, 0x50, 0x61, 0x63, 0x69,
0x66, 0x69, 0x63, 0x2f, 0x52, 0x61, 0x72, 0x6f, 0x74, 0x6f, 0x6e, 0x67, 0x61, 0x0, 0x41, 0x6e,
0x74, 0x61, 0x72, 0x63, 0x74, 0x69, 0x63, 0x61, 0x2f, 0x44, 0x61, 0x76, 0x69, 0x73, 0x0, 0x41,
0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x53, 0x61, 0x6e, 0x74, 0x6f, 0x5f, 0x44, 0x6f, 0x6d,
0x69, 0x6e, 0x67, 0x6f, 0x0, 0x41, 0x6e, 0x74, 0x61, 0x72, 0x63, 0x74, 0x69, 0x63, 0x61, 0x2f,
0x44, 0x75, 0x6d, 0x6f, 0x6e, 0x74, 0x44, 0x55, 0x72, 0x76, 0x69, 0x6c, 0x6c, 0x65, 0x0, 0x41,
0x73, 0x69, 0x61, 0x2f, 0x44, 0x75, 0x73, 0x68, 0x61, 0x6e, 0x62, 0x65, 0x0, 0x41, 0x6d, 0x65,
0x72, 0x69, 0x63, 0x61, 0x2f, 0x50, 0x61, 0x72, 0x61, 0x6d, 0x61, 0x72, 0x69, 0x62, 0x6f, 0x0,
0x41, 0x73, 0x69, 0x61, 0x2f, 0x44, 0x69, 0x6c, 0x69, 0x0, 0x45, 0x75, 0x72, 0x6f, 0x70, 0x65,
0x2f, 0x54, 0x69, 0x72, 0x61, 0x6e, 0x65, 0x0, 0x45, 0x75, 0x72, 0x6f, 0x70, 0x65, 0x2f, 0x41,
0x6e, 0x64, 0x6f, 0x72, 0x72, 0x61, 0x0, 0x45, 0x75, 0x72, 0x6f, 0x70, 0x65, 0x2f, 0x56, 0x69,
0x65, 0x6e, 0x6e, 0x61, 0x0, 0x45, 0x75, 0x72, 0x6f, 0x70, 0x65, 0x2f, 0x53, 0x61, 0x72, 0x61,
0x6a, 0x65, 0x76, 0x6f, 0x0, 0x45, 0x75, 0x72, 0x6f, 0x70, 0x65, 0x2f, 0x5a, 0x61, 0x67, 0x72,
0x65, 0x62, 0x0, 0x45, 0x75, 0x72, 0x6f, 0x70, 0x65, 0x2f, 0x50, 0x72, 0x61, 0x67, 0x75, 0x65,
0x0, 0x45, 0x75, 0x72, 0x6f, 0x70, 0x65, 0x2f, 0x43, 0x6f, 0x70, 0x65, 0x6e, 0x68, 0x61, 0x67,
0x65, 0x6e, 0x0, 0x45, 0x75, 0x72, 0x6f, 0x70, 0x65, 0x2f, 0x47, 0x69, 0x62, 0x72, 0x61, 0x6c,
0x74, 0x61, 0x72, 0x0, 0x45, 0x75, 0x72, 0x6f, 0x70, 0x65, 0x2f, 0x52, 0x6f, 0x6d, 0x65, 0x0,
0x45, 0x75, 0x72, 0x6f, 0x70, 0x65, 0x2f, 0x42, 0x65, 0x6c, 0x67, 0x72, 0x61, 0x64, 0x65, 0x0,
0x45, 0x75, 0x72, 0x6f, 0x70, 0x65, 0x2f, 0x56, 0x61, 0x64, 0x75, 0x7a, 0x0, 0x45, 0x75, 0x72,
0x6f, 0x70, 0x65, 0x2f, 0x4c, 0x75, 0x78, 0x65, 0x6d, 0x62, 0x6f, 0x75, 0x72, 0x67, 0x0, 0x45,
0x75, 0x72, 0x6f, 0x70, 0x65, 0x2f, 0x53, 0x6b, 0x6f, 0x70, 0x6a, 0x65, 0x0, 0x45, 0x75, 0x72,
0x6f, 0x70, 0x65, 0x2f, 0x4d, 0x61, 0x6c, 0x74, 0x61, 0x0, 0x45, 0x75, 0x72, 0x6f, 0x70, 0x65,
0x2f, 0x4d, 0x6f, 0x6e, 0x61, 0x63, 0x6f, 0x0, 0x45, 0x75, 0x72, 0x6f, 0x70, 0x65, 0x2f, 0x50,
0x6f, 0x64, 0x67, 0x6f, 0x72, 0x69, 0x63, 0x61, 0x0, 0x45, 0x75, 0x72, 0x6f, 0x70, 0x65, 0x2f,
0x41, 0x6d, 0x73, 0x74, 0x65, 0x72, 0x64, 0x61, 0x6d, 0x0, 0x45, 0x75, 0x72, 0x6f, 0x70, 0x65,
0x2f, 0x4f, 0x73, 0x6c, 0x6f, 0x0, 0x45, 0x75, 0x72, 0x6f, 0x70, 0x65, 0x2f, 0x53, 0x61, 0x6e,
0x5f, 0x4d, 0x61, 0x72, 0x69, 0x6e, 0x6f, 0x0, 0x45, 0x75, 0x72, 0x6f, 0x70, 0x65, 0x2f, 0x42,
0x72, 0x61, 0x74, 0x69, 0x73, 0x6c, 0x61, 0x76, 0x61, 0x0, 0x45, 0x75, 0x72, 0x6f, 0x70, 0x65,
0x2f, 0x4c, 0x6a, 0x75, 0x62, 0x6c, 0x6a, 0x61, 0x6e, 0x61, 0x0, 0x45, 0x75, 0x72, 0x6f, 0x70,
0x65, 0x2f, 0x53, 0x74, 0x6f, 0x63, 0x6b, 0x68, 0x6f, 0x6c, 0x6d, 0x0, 0x45, 0x75, 0x72, 0x6f,
0x70, 0x65, 0x2f, 0x5a, 0x75, 0x72, 0x69, 0x63, 0x68, 0x0, 0x41, 0x66, 0x72, 0x69, 0x63, 0x61,
0x2f, 0x54, 0x75, 0x6e, 0x69, 0x73, 0x0, 0x45, 0x75, 0x72, 0x6f, 0x70, 0x65, 0x2f, 0x56, 0x61,
0x74, 0x69, 0x63, 0x61, 0x6e, 0x0, 0x45, 0x75, 0x72, 0x6f, 0x70, 0x65, 0x2f, 0x4d, 0x61, 0x72,
0x69, 0x65, 0x68, 0x61, 0x6d, 0x6e, 0x0, 0x45, 0x75, 0x72, 0x6f, 0x70, 0x65, 0x2f, 0x53, 0x6f,
0x66, 0x69, 0x61, 0x0, 0x45, 0x75, 0x72, 0x6f, 0x70, 0x65, 0x2f, 0x48, 0x65, 0x6c, 0x73, 0x69,
0x6e, 0x6b, 0x69, 0x0, 0x41, 0x74, 0x6c, 0x61, 0x6e, 0x74, 0x69, 0x63, 0x2f, 0x43, 0x61, 0x6e,
0x61, 0x72, 0x79, 0x0, 0x41, 0x74, 0x6c, 0x61, 0x6e, 0x74, 0x69, 0x63, 0x2f, 0x53, 0x74, 0x61,
0x6e, 0x6c, 0x65, 0x79, 0x0, 0x49, 0x6e, 0x64, 0x69, 0x61, 0x6e, 0x2f, 0x4b, 0x65, 0x72, 0x67,
0x75, 0x65, 0x6c, 0x65, 0x6e, 0x0, 0x50, 0x61, 0x63, 0x69, 0x66, 0x69, 0x63, 0x2f, 0x47, 0x61,
0x6c, 0x61, 0x70, 0x61, 0x67, 0x6f, 0x73, 0x0, 0x50, 0x61, 0x63, 0x69, 0x66, 0x69, 0x63, 0x2f,
0x47, 0x61, 0x6d, 0x62, 0x69, 0x65, 0x72, 0x0, 0x50, 0x61, 0x63, 0x69, 0x66, 0x69, 0x63, 0x2f,
0x54, 0x61, 0x72, 0x61, 0x77, 0x61, 0x0, 0x41, 0x66, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x4f, 0x75,
0x61, 0x67, 0x61, 0x64, 0x6f, 0x75, 0x67, 0x6f, 0x75, 0x0, 0x41, 0x66, 0x72, 0x69, 0x63, 0x61,
0x2f, 0x42, 0x61, 0x6e, 0x6a, 0x75, 0x6c, 0x0, 0x41, 0x66, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x41,
0x63, 0x63, 0x72, 0x61, 0x0, 0x41, 0x66, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x43, 0x6f, 0x6e, 0x61,
0x6b, 0x72, 0x79, 0x0, 0x41, 0x66, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x41, 0x62, 0x69, 0x64, 0x6a,
0x61, 0x6e, 0x0, 0x41, 0x66, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x4e, 0x6f, 0x75, 0x61, 0x6b, 0x63,
0x68, 0x6f, 0x74, 0x74, 0x0, 0x41, 0x74, 0x6c, 0x61, 0x6e, 0x74, 0x69, 0x63, 0x2f, 0x53, 0x74,
0x5f, 0x48, 0x65, 0x6c, 0x65, 0x6e, 0x61, 0x0, 0x41, 0x66, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x44,
0x61, 0x6b, 0x61, 0x72, 0x0, 0x41, 0x66, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x46, 0x72, 0x65, 0x65,
0x74, 0x6f, 0x77, 0x6e, 0x0, 0x41, 0x66, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x4c, 0x6f, 0x6d, 0x65,
0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x47, 0x6f, 0x6f, 0x73, 0x65, 0x5f, 0x42,
0x61, 0x79, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x53, 0x63, 0x6f, 0x72, 0x65,
0x73, 0x62, 0x79, 0x73, 0x75, 0x6e, 0x64, 0x0, 0x41, 0x73, 0x69, 0x61, 0x2f, 0x4d, 0x75, 0x73,
0x63, 0x61, 0x74, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x47, 0x75, 0x79, 0x61,
0x6e, 0x61, 0x0, 0x49, 0x6e, 0x64, 0x69, 0x61, 0x6e, 0x2f, 0x43, 0x68, 0x61, 0x67, 0x6f, 0x73,
0x0, 0x41, 0x73, 0x69, 0x61, 0x2f, 0x50, 0x68, 0x6e, 0x6f, 0x6d, 0x5f, 0x50, 0x65, 0x6e, 0x68,
0x0, 0x41, 0x73, 0x69, 0x61, 0x2f, 0x56, 0x69, 0x65, 0x6e, 0x74, 0x69, 0x61, 0x6e, 0x65, 0x0,
0x41, 0x73, 0x69, 0x61, 0x2f, 0x4a, 0x61, 0x79, 0x61, 0x70, 0x75, 0x72, 0x61, 0x0, 0x41, 0x73,
0x69, 0x61, 0x2f, 0x4a, 0x61, 0x6b, 0x61, 0x72, 0x74, 0x61, 0x0, 0x50, 0x61, 0x63, 0x69, 0x66,
0x69, 0x63, 0x2f, 0x4b, 0x6f, 0x73, 0x72, 0x61, 0x65, 0x0, 0x41, 0x66, 0x72, 0x69, 0x63, 0x61,
0x2f, 0x4d, 0x6f, 0x6e, 0x72, 0x6f, 0x76, 0x69, 0x61, 0x0, 0x49, 0x6e, 0x64, 0x69, 0x61, 0x6e,
0x2f, 0x4d, 0x61, 0x6c, 0x64, 0x69, 0x76, 0x65, 0x73, 0x0, 0x41, 0x6e, 0x74, 0x61, 0x72, 0x63,
0x74, 0x69, 0x63, 0x61, 0x2f, 0x4d, 0x61, 0x77, 0x73, 0x6f, 0x6e, 0x0, 0x50, 0x61, 0x63, 0x69,
0x66, 0x69, 0x63, 0x2f, 0x4e, 0x61, 0x75, 0x72, 0x75, 0x0, 0x50, 0x61, 0x63, 0x69, 0x66, 0x69,
0x63, 0x2f, 0x4e, 0x6f, 0x75, 0x6d, 0x65, 0x61, 0x0, 0x41, 0x6e, 0x74, 0x61, 0x72, 0x63, 0x74,
0x69, 0x63, 0x61, 0x2f, 0x4d, 0x63, 0x4d, 0x75, 0x72, 0x64, 0x6f, 0x0, 0x50, 0x61, 0x63, 0x69,
0x66, 0x69, 0x63, 0x2f, 0x4e, 0x69, 0x75, 0x65, 0x0, 0x41, 0x73, 0x69, 0x61, 0x2f, 0x4f, 0x72,
0x61, 0x6c, 0x0, 0x50, 0x61, 0x63, 0x69, 0x66, 0x69, 0x63, 0x2f, 0x50, 0x61, 0x6c, 0x61, 0x75,
0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x4c, 0x69, 0x6d, 0x61, 0x0, 0x41, 0x73,
0x69, 0x61, 0x2f, 0x4d, 0x61, 0x6e, 0x69, 0x6c, 0x61, 0x0, 0x50, 0x61, 0x63, 0x69, 0x66, 0x69,
0x63, 0x2f, 0x50, 0x69, 0x74, 0x63, 0x61, 0x69, 0x72, 0x6e, 0x0, 0x49, 0x6e, 0x64, 0x69, 0x61,
0x6e, 0x2f, 0x52, 0x65, 0x75, 0x6e, 0x69, 0x6f, 0x6e, 0x0, 0x41, 0x6e, 0x74, 0x61, 0x72, 0x63,
0x74, 0x69, 0x63, 0x61, 0x2f, 0x52, 0x6f, 0x74, 0x68, 0x65, 0x72, 0x61, 0x0, 0x41, 0x73, 0x69,
0x61, 0x2f, 0x53, 0x61, 0x6d, 0x61, 0x72, 0x6b, 0x61, 0x6e, 0x64, 0x0, 0x49, 0x6e, 0x64, 0x69,
0x61, 0x6e, 0x2f, 0x4d, 0x61, 0x68, 0x65, 0x0, 0x41, 0x74, 0x6c, 0x61, 0x6e, 0x74, 0x69, 0x63,
0x2f, 0x53, 0x6f, 0x75, 0x74, 0x68, 0x5f, 0x47, 0x65, 0x6f, 0x72, 0x67, 0x69, 0x61, 0x0, 0x41,
0x6e, 0x74, 0x61, 0x72, 0x63, 0x74, 0x69, 0x63, 0x61, 0x2f, 0x53, 0x79, 0x6f, 0x77, 0x61, 0x0,
0x50, 0x61, 0x63, 0x69, 0x66, 0x69, 0x63, 0x2f, 0x54, 0x61, 0x68, 0x69, 0x74, 0x69, 0x0, 0x50,
0x61, 0x63, 0x69, 0x66, 0x69, 0x63, 0x2f, 0x46, 0x61, 0x6b, 0x61, 0x6f, 0x66, 0x6f, 0x0, 0x50,
0x61, 0x63, 0x69, 0x66, 0x69, 0x63, 0x2f, 0x46, 0x75, 0x6e, 0x61, 0x66, 0x75, 0x74, 0x69, 0x0,
0x50, 0x61, 0x63, 0x69, 0x66, 0x69, 0x63, 0x2f, 0x45, 0x66, 0x61, 0x74, 0x65, 0x0, 0x41, 0x6e,
0x74, 0x61, 0x72, 0x63, 0x74, 0x69, 0x63, 0x61, 0x2f, 0x56, 0x6f, 0x73, 0x74, 0x6f, 0x6b, 0x0,
0x50, 0x61, 0x63, 0x69, 0x66, 0x69, 0x63, 0x2f, 0x57, 0x61, 0x6b, 0x65, 0x0, 0x50, 0x61, 0x63,
0x69, 0x66, 0x69, 0x63, 0x2f, 0x57, 0x61, 0x6c, 0x6c, 0x69, 0x73, 0x0, 0x41, 0x66, 0x72, 0x69,
0x63, 0x61, 0x2f, 0x41, 0x6c, 0x67, 0x69, 0x65, 0x72, 0x73, 0x0, 0x41, 0x66, 0x72, 0x69, 0x63,
0x61, 0x2f, 0x42, 0x69, 0x73, 0x73, 0x61, 0x75, 0x0, 0x41, 0x66, 0x72, 0x69, 0x63, 0x61, 0x2f,
0x43, 0x65, 0x75, 0x74, 0x61, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x41, 0x72,
0x67, 0x65, 0x6e, 0x74, 0x69, 0x6e, 0x61, 0x2f, 0x4c, 0x61, 0x5f, 0x52, 0x69, 0x6f, 0x6a, 0x61,
0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x41, 0x72, 0x67, 0x65, 0x6e, 0x74, 0x69,
0x6e, 0x61, 0x2f, 0x52, 0x69, 0x6f, 0x5f, 0x47, 0x61, 0x6c, 0x6c, 0x65, 0x67, 0x6f, 0x73, 0x0,
0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x41, 0x72, 0x67, 0x65, 0x6e, 0x74, 0x69, 0x6e,
0x61, 0x2f, 0x53, 0x61, 0x6c, 0x74, 0x61, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f,
0x41, 0x72, 0x67, 0x65, 0x6e, 0x74, 0x69, 0x6e, 0x61, 0x2f, 0x53, 0x61, 0x6e, 0x5f, 0x4a, 0x75,
0x61, 0x6e, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x41, 0x72, 0x67, 0x65, 0x6e,
0x74, 0x69, 0x6e, 0x61, 0x2f, 0x54, 0x75, 0x63, 0x75, 0x6d, 0x61, 0x6e, 0x0, 0x41, 0x6d, 0x65,
0x72, 0x69, 0x63, 0x61, 0x2f, 0x41, 0x72, 0x67, 0x65, 0x6e, 0x74, 0x69, 0x6e, 0x61, 0x2f, 0x55,
0x73, 0x68, 0x75, 0x61, 0x69, 0x61, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x42,
0x61, 0x68, 0x69, 0x61, 0x5f, 0x42, 0x61, 0x6e, 0x64, 0x65, 0x72, 0x61, 0x73, 0x0, 0x41, 0x6d,
0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x42, 0x65, 0x6c, 0x65, 0x6d, 0x0, 0x41, 0x6d, 0x65, 0x72,
0x69, 0x63, 0x61, 0x2f, 0x42, 0x6c, 0x61, 0x6e, 0x63, 0x2d, 0x53, 0x61, 0x62, 0x6c, 0x6f, 0x6e,
0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x42, 0x6f, 0x61, 0x5f, 0x56, 0x69, 0x73,
0x74, 0x61, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x42, 0x6f, 0x69, 0x73, 0x65,
0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x43, 0x61, 0x6d, 0x62, 0x72, 0x69, 0x64,
0x67, 0x65, 0x5f, 0x42, 0x61, 0x79, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x43,
0x61, 0x6d, 0x70, 0x6f, 0x5f, 0x47, 0x72, 0x61, 0x6e, 0x64, 0x65, 0x0, 0x41, 0x6d, 0x65, 0x72,
0x69, 0x63, 0x61, 0x2f, 0x43, 0x68, 0x69, 0x68, 0x75, 0x61, 0x68, 0x75, 0x61, 0x0, 0x41, 0x6d,
0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x43, 0x69, 0x75, 0x64, 0x61, 0x64, 0x5f, 0x4a, 0x75, 0x61,
0x72, 0x65, 0x7a, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x43, 0x72, 0x65, 0x73,
0x74, 0x6f, 0x6e, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x44, 0x61, 0x6e, 0x6d,
0x61, 0x72, 0x6b, 0x73, 0x68, 0x61, 0x76, 0x6e, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61,
0x2f, 0x44, 0x61, 0x77, 0x73, 0x6f, 0x6e, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f,
0x44, 0x61, 0x77, 0x73, 0x6f, 0x6e, 0x5f, 0x43, 0x72, 0x65, 0x65, 0x6b, 0x0, 0x41, 0x6d, 0x65,
0x72, 0x69, 0x63, 0x61, 0x2f, 0x45, 0x69, 0x72, 0x75, 0x6e, 0x65, 0x70, 0x65, 0x0, 0x41, 0x6d,
0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x46, 0x6f, 0x72, 0x74, 0x5f, 0x4e, 0x65, 0x6c, 0x73, 0x6f,
0x6e, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x46, 0x6f, 0x72, 0x74, 0x61, 0x6c,
0x65, 0x7a, 0x61, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x47, 0x6c, 0x61, 0x63,
0x65, 0x5f, 0x42, 0x61, 0x79, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x48, 0x65,
0x72, 0x6d, 0x6f, 0x73, 0x69, 0x6c, 0x6c, 0x6f, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61,
0x2f, 0x49, 0x6e, 0x64, 0x69, 0x61, 0x6e, 0x61, 0x2f, 0x4d, 0x61, 0x72, 0x65, 0x6e, 0x67, 0x6f,
0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x49, 0x6e, 0x64, 0x69, 0x61, 0x6e, 0x61,
0x2f, 0x50, 0x65, 0x74, 0x65, 0x72, 0x73, 0x62, 0x75, 0x72, 0x67, 0x0, 0x41, 0x6d, 0x65, 0x72,
0x69, 0x63, 0x61, 0x2f, 0x49, 0x6e, 0x64, 0x69, 0x61, 0x6e, 0x61, 0x2f, 0x54, 0x65, 0x6c, 0x6c,
0x5f, 0x43, 0x69, 0x74, 0x79, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x49, 0x6e,
0x64, 0x69, 0x61, 0x6e, 0x61, 0x2f, 0x56, 0x65, 0x76, 0x61, 0x79, 0x0, 0x41, 0x6d, 0x65, 0x72,
0x69, 0x63, 0x61, 0x2f, 0x49, 0x6e, 0x64, 0x69, 0x61, 0x6e, 0x61, 0x2f, 0x56, 0x69, 0x6e, 0x63,
0x65, 0x6e, 0x6e, 0x65, 0x73, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x49, 0x6e,
0x64, 0x69, 0x61, 0x6e, 0x61, 0x2f, 0x57, 0x69, 0x6e, 0x61, 0x6d, 0x61, 0x63, 0x0, 0x41, 0x6d,
0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x49, 0x6e, 0x75, 0x76, 0x69, 0x6b, 0x0, 0x41, 0x6d, 0x65,
0x72, 0x69, 0x63, 0x61, 0x2f, 0x4b, 0x65, 0x6e, 0x74, 0x75, 0x63, 0x6b, 0x79, 0x2f, 0x4d, 0x6f,
0x6e, 0x74, 0x69, 0x63, 0x65, 0x6c, 0x6c, 0x6f, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61,
0x2f, 0x4d, 0x61, 0x63, 0x65, 0x69, 0x6f, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f,
0x4d, 0x61, 0x6e, 0x61, 0x67, 0x75, 0x61, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f,
0x4d, 0x61, 0x74, 0x61, 0x6d, 0x6f, 0x72, 0x6f, 0x73, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63,
0x61, 0x2f, 0x4d, 0x65, 0x6e, 0x6f, 0x6d, 0x69, 0x6e, 0x65, 0x65, 0x0, 0x41, 0x6d, 0x65, 0x72,
0x69, 0x63, 0x61, 0x2f, 0x4d, 0x65, 0x72, 0x69, 0x64, 0x61, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69,
0x63, 0x61, 0x2f, 0x4d, 0x65, 0x74, 0x6c, 0x61, 0x6b, 0x61, 0x74, 0x6c, 0x61, 0x0, 0x41, 0x6d,
0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x4d, 0x6f, 0x6e, 0x63, 0x74, 0x6f, 0x6e, 0x0, 0x41, 0x6d,
0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x4d, 0x6f, 0x6e, 0x74, 0x65, 0x72, 0x72, 0x65, 0x79, 0x0,
0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x4e, 0x6f, 0x6d, 0x65, 0x0, 0x41, 0x6d, 0x65,
0x72, 0x69, 0x63, 0x61, 0x2f, 0x4e, 0x6f, 0x72, 0x74, 0x68, 0x5f, 0x44, 0x61, 0x6b, 0x6f, 0x74,
0x61, 0x2f, 0x42, 0x65, 0x75, 0x6c, 0x61, 0x68, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61,
0x2f, 0x4e, 0x6f, 0x72, 0x74, 0x68, 0x5f, 0x44, 0x61, 0x6b, 0x6f, 0x74, 0x61, 0x2f, 0x43, 0x65,
0x6e, 0x74, 0x65, 0x72, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x4e, 0x6f, 0x72,
0x74, 0x68, 0x5f, 0x44, 0x61, 0x6b, 0x6f, 0x74, 0x61, 0x2f, 0x4e, 0x65, 0x77, 0x5f, 0x53, 0x61,
0x6c, 0x65, 0x6d, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x4f, 0x6a, 0x69, 0x6e,
0x61, 0x67, 0x61, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x50, 0x6f, 0x72, 0x74,
0x6f, 0x5f, 0x56, 0x65, 0x6c, 0x68, 0x6f, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f,
0x52, 0x61, 0x6e, 0x6b, 0x69, 0x6e, 0x5f, 0x49, 0x6e, 0x6c, 0x65, 0x74, 0x0, 0x41, 0x6d, 0x65,
0x72, 0x69, 0x63, 0x61, 0x2f, 0x52, 0x65, 0x63, 0x69, 0x66, 0x65, 0x0, 0x41, 0x6d, 0x65, 0x72,
0x69, 0x63, 0x61, 0x2f, 0x52, 0x65, 0x73, 0x6f, 0x6c, 0x75, 0x74, 0x65, 0x0, 0x41, 0x6d, 0x65,
0x72, 0x69, 0x63, 0x61, 0x2f, 0x53, 0x61, 0x6e, 0x74, 0x61, 0x72, 0x65, 0x6d, 0x0, 0x41, 0x6d,
0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x53, 0x69, 0x74, 0x6b, 0x61, 0x0, 0x41, 0x6d, 0x65, 0x72,
0x69, 0x63, 0x61, 0x2f, 0x53, 0x74, 0x5f, 0x42, 0x61, 0x72, 0x74, 0x68, 0x65, 0x6c, 0x65, 0x6d,
0x79, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f, 0x53, 0x77, 0x69, 0x66, 0x74, 0x5f,
0x43, 0x75, 0x72, 0x72, 0x65, 0x6e, 0x74, 0x0, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x2f,
0x59, 0x61, 0x6b, 0x75, 0x74, 0x61, 0x74, 0x0, 0x41, 0x6e, 0x74, 0x61, 0x72, 0x63, 0x74, 0x69,
0x63, 0x61, 0x2f, 0x4d, 0x61, 0x63, 0x71, 0x75, 0x61, 0x72, 0x69, 0x65, 0x0, 0x41, 0x6e, 0x74,
0x61, 0x72, 0x63, 0x74, 0x69, 0x63, 0x61, 0x2f, 0x50, 0x61, 0x6c, 0x6d, 0x65, 0x72, 0x0, 0x41,
0x6e, 0x74, 0x61, 0x72, 0x63, 0x74, 0x69, 0x63, 0x61, 0x2f, 0x54, 0x72, 0x6f, 0x6c, 0x6c, 0x0,
0x41, 0x73, 0x69, 0x61, 0x2f, 0x41, 0x74, 0x79, 0x72, 0x61, 0x75, 0x0, 0x41, 0x73, 0x69, 0x61,
0x2f, 0x46, 0x61, 0x6d, 0x61, 0x67, 0x75, 0x73, 0x74, 0x61, 0x0, 0x41, 0x73, 0x69, 0x61, 0x2f,
0x47, 0x61, 0x7a, 0x61, 0x0, 0x41, 0x73, 0x69, 0x61, 0x2f, 0x4b, 0x68, 0x61, 0x6e, 0x64, 0x79,
0x67, 0x61, 0x0, 0x41, 0x73, 0x69, 0x61, 0x2f, 0x4e, 0x6f, 0x76, 0x6f, 0x6b, 0x75, 0x7a, 0x6e,
0x65, 0x74, 0x73, 0x6b, 0x0, 0x41, 0x73, 0x69, 0x61, 0x2f, 0x50, 0x6f, 0x6e, 0x74, 0x69, 0x61,
0x6e, 0x61, 0x6b, 0x0, 0x41, 0x73, 0x69, 0x61, 0x2f, 0x51, 0x6f, 0x73, 0x74, 0x61, 0x6e, 0x61,
0x79, 0x0, 0x41, 0x73, 0x69, 0x61, 0x2f, 0x55, 0x73, 0x74, 0x2d, 0x4e, 0x65, 0x72, 0x61, 0x0,
0x41, 0x74, 0x6c, 0x61, 0x6e, 0x74, 0x69, 0x63, 0x2f, 0x4d, 0x61, 0x64, 0x65, 0x69, 0x72, 0x61,
0x0, 0x41, 0x75, 0x73, 0x74, 0x72, 0x61, 0x6c, 0x69, 0x61, 0x2f, 0x4c, 0x69, 0x6e, 0x64, 0x65,
0x6d, 0x61, 0x6e, 0x0, 0x45, 0x75, 0x72, 0x6f, 0x70, 0x65, 0x2f, 0x42, 0x75, 0x73, 0x69, 0x6e,
0x67, 0x65, 0x6e, 0x0, 0x45, 0x75, 0x72, 0x6f, 0x70, 0x65, 0x2f, 0x47, 0x75, 0x65, 0x72, 0x6e,
0x73, 0x65, 0x79, 0x0, 0x45, 0x75, 0x72, 0x6f, 0x70, 0x65, 0x2f, 0x49, 0x73, 0x6c, 0x65, 0x5f,
0x6f, 0x66, 0x5f, 0x4d, 0x61, 0x6e, 0x0, 0x45, 0x75, 0x72, 0x6f, 0x70, 0x65, 0x2f, 0x4a, 0x65,
0x72, 0x73, 0x65, 0x79, 0x0, 0x45, 0x75, 0x72, 0x6f, 0x70, 0x65, 0x2f, 0x52, 0x69, 0x67, 0x61,
0x0, 0x45, 0x75, 0x72, 0x6f, 0x70, 0x65, 0x2f, 0x53, 0x69, 0x6d, 0x66, 0x65, 0x72, 0x6f, 0x70,
0x6f, 0x6c, 0x0, 0x45, 0x75, 0x72, 0x6f, 0x70, 0x65, 0x2f, 0x54, 0x61, 0x6c, 0x6c, 0x69, 0x6e,
0x6e, 0x0, 0x45, 0x75, 0x72, 0x6f, 0x70, 0x65, 0x2f, 0x55, 0x6c, 0x79, 0x61, 0x6e, 0x6f, 0x76,
0x73, 0x6b, 0x0, 0x45, 0x75, 0x72, 0x6f, 0x70, 0x65, 0x2f, 0x56, 0x69, 0x6c, 0x6e, 0x69, 0x75,
0x73, 0x0, 0x50, 0x61, 0x63, 0x69, 0x66, 0x69, 0x63, 0x2f, 0x4d, 0x69, 0x64, 0x77, 0x61, 0x79,
0x0, 0x45, 0x75, 0x72, 0x6f, 0x70, 0x65, 0x2f, 0x4b, 0x69, 0x72, 0x6f, 0x76, 0x0
};

#if QT_CONFIG(timezone_locale) && !QT_CONFIG(icu)

static inline constexpr char metaIdData[] = {
0x41, 0x63, 0x72, 0x65, 0x0, 0x41, 0x66, 0x67, 0x68, 0x61, 0x6e, 0x69, 0x73, 0x74, 0x61, 0x6e,
0x0, 0x41, 0x66, 0x72, 0x69, 0x63, 0x61, 0x5f, 0x43, 0x65, 0x6e, 0x74, 0x72, 0x61, 0x6c, 0x0,
0x41, 0x66, 0x72, 0x69, 0x63, 0x61, 0x5f, 0x45, 0x61, 0x73, 0x74, 0x65, 0x72, 0x6e, 0x0, 0x41,
0x66, 0x72, 0x69, 0x63, 0x61, 0x5f, 0x46, 0x61, 0x72, 0x57, 0x65, 0x73, 0x74, 0x65, 0x72, 0x6e,
0x0, 0x41, 0x66, 0x72, 0x69, 0x63, 0x61, 0x5f, 0x53, 0x6f, 0x75, 0x74, 0x68, 0x65, 0x72, 0x6e,
0x0, 0x41, 0x66, 0x72, 0x69, 0x63, 0x61, 0x5f, 0x57, 0x65, 0x73, 0x74, 0x65, 0x72, 0x6e, 0x0,
0x41, 0x6b, 0x74, 0x79, 0x75, 0x62, 0x69, 0x6e, 0x73, 0x6b, 0x0, 0x41, 0x6c, 0x61, 0x73, 0x6b,
0x61, 0x0, 0x41, 0x6c, 0x61, 0x73, 0x6b, 0x61, 0x5f, 0x48, 0x61, 0x77, 0x61, 0x69, 0x69, 0x0,
0x41, 0x6c, 0x6d, 0x61, 0x74, 0x79, 0x0, 0x41, 0x6d, 0x61, 0x7a, 0x6f, 0x6e, 0x0, 0x41, 0x6d,
0x65, 0x72, 0x69, 0x63, 0x61, 0x5f, 0x43, 0x65, 0x6e, 0x74, 0x72, 0x61, 0x6c, 0x0, 0x41, 0x6d,
0x65, 0x72, 0x69, 0x63, 0x61, 0x5f, 0x45, 0x61, 0x73, 0x74, 0x65, 0x72, 0x6e, 0x0, 0x41, 0x6d,
0x65, 0x72, 0x69, 0x63, 0x61, 0x5f, 0x4d, 0x6f, 0x75, 0x6e, 0x74, 0x61, 0x69, 0x6e, 0x0, 0x41,
0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x5f, 0x50, 0x61, 0x63, 0x69, 0x66, 0x69, 0x63, 0x0, 0x41,
0x6e, 0x61, 0x64, 0x79, 0x72, 0x0, 0x41, 0x70, 0x69, 0x61, 0x0, 0x41, 0x71, 0x74, 0x61, 0x75,
0x0, 0x41, 0x71, 0x74, 0x6f, 0x62, 0x65, 0x0, 0x41, 0x72, 0x61, 0x62, 0x69, 0x61, 0x6e, 0x0,
0x41, 0x72, 0x67, 0x65, 0x6e, 0x74, 0x69, 0x6e, 0x61, 0x0, 0x41, 0x72, 0x67, 0x65, 0x6e, 0x74,
0x69, 0x6e, 0x61, 0x5f, 0x57, 0x65, 0x73, 0x74, 0x65, 0x72, 0x6e, 0x0, 0x41, 0x72, 0x6d, 0x65,
0x6e, 0x69, 0x61, 0x0, 0x41, 0x73, 0x68, 0x6b, 0x68, 0x61, 0x62, 0x61, 0x64, 0x0, 0x41, 0x74,
0x6c, 0x61, 0x6e, 0x74, 0x69, 0x63, 0x0, 0x41, 0x75, 0x73, 0x74, 0x72, 0x61, 0x6c, 0x69, 0x61,
0x5f, 0x43, 0x65, 0x6e, 0x74, 0x72, 0x61, 0x6c, 0x0, 0x41, 0x75, 0x73, 0x74, 0x72, 0x61, 0x6c,
0x69, 0x61, 0x5f, 0x43, 0x65, 0x6e, 0x74, 0x72, 0x61, 0x6c, 0x57, 0x65, 0x73, 0x74, 0x65, 0x72,
0x6e, 0x0, 0x41, 0x75, 0x73, 0x74, 0x72, 0x61, 0x6c, 0x69, 0x61, 0x5f, 0x45, 0x61, 0x73, 0x74,
0x65, 0x72, 0x6e, 0x0, 0x41, 0x75, 0x73, 0x74, 0x72, 0x61, 0x6c, 0x69, 0x61, 0x5f, 0x57, 0x65,
0x73, 0x74, 0x65, 0x72, 0x6e, 0x0, 0x41, 0x7a, 0x65, 0x72, 0x62, 0x61, 0x69, 0x6a, 0x61, 0x6e,
0x0, 0x41, 0x7a, 0x6f, 0x72, 0x65, 0x73, 0x0, 0x42, 0x61, 0x6b, 0x75, 0x0, 0x42, 0x61, 0x6e,
0x67, 0x6c, 0x61, 0x64, 0x65, 0x73, 0x68, 0x0, 0x42, 0x65, 0x72, 0x69, 0x6e, 0x67, 0x0, 0x42,
0x68, 0x75, 0x74, 0x61, 0x6e, 0x0, 0x42, 0x6f, 0x6c, 0x69, 0x76, 0x69, 0x61, 0x0, 0x42, 0x6f,
0x72, 0x6e, 0x65, 0x6f, 0x0, 0x42, 0x72, 0x61, 0x73, 0x69, 0x6c, 0x69, 0x61, 0x0, 0x42, 0x72,
0x69, 0x74, 0x69, 0x73, 0x68, 0x0, 0x42, 0x72, 0x75, 0x6e, 0x65, 0x69, 0x0, 0x43, 0x61, 0x70,
0x65, 0x5f, 0x56, 0x65, 0x72, 0x64, 0x65, 0x0, 0x43, 0x61, 0x73, 0x65, 0x79, 0x0, 0x43, 0x68,
0x61, 0x6d, 0x6f, 0x72, 0x72, 0x6f, 0x0, 0x43, 0x68, 0x61, 0x74, 0x68, 0x61, 0x6d, 0x0, 0x43,
0x68, 0x69, 0x6c, 0x65, 0x0, 0x43, 0x68, 0x69, 0x6e, 0x61, 0x0, 0x43, 0x68, 0x72, 0x69, 0x73,
0x74, 0x6d, 0x61, 0x73, 0x0, 0x43, 0x6f, 0x63, 0x6f, 0x73, 0x0, 0x43, 0x6f, 0x6c, 0x6f, 0x6d,
0x62, 0x69, 0x61, 0x0, 0x43, 0x6f, 0x6f, 0x6b, 0x0, 0x43, 0x75, 0x62, 0x61, 0x0, 0x44, 0x61,
0x63, 0x63, 0x61, 0x0, 0x44, 0x61, 0x76, 0x69, 0x73, 0x0, 0x44, 0x6f, 0x6d, 0x69, 0x6e, 0x69,
0x63, 0x61, 0x6e, 0x0, 0x44, 0x75, 0x6d, 0x6f, 0x6e, 0x74, 0x44, 0x55, 0x72, 0x76, 0x69, 0x6c,
0x6c, 0x65, 0x0, 0x44, 0x75, 0x73, 0x68, 0x61, 0x6e, 0x62, 0x65, 0x0, 0x44, 0x75, 0x74, 0x63,
0x68, 0x5f, 0x47, 0x75, 0x69, 0x61, 0x6e, 0x61, 0x0, 0x45, 0x61, 0x73, 0x74, 0x5f, 0x54, 0x69,
0x6d, 0x6f, 0x72, 0x0, 0x45, 0x61, 0x73, 0x74, 0x65, 0x72, 0x0, 0x45, 0x63, 0x75, 0x61, 0x64,
0x6f, 0x72, 0x0, 0x45, 0x75, 0x72, 0x6f, 0x70, 0x65, 0x5f, 0x43, 0x65, 0x6e, 0x74, 0x72, 0x61,
0x6c, 0x0, 0x45, 0x75, 0x72, 0x6f, 0x70, 0x65, 0x5f, 0x45, 0x61, 0x73, 0x74, 0x65, 0x72, 0x6e,
0x0, 0x45, 0x75, 0x72, 0x6f, 0x70, 0x65, 0x5f, 0x46, 0x75, 0x72, 0x74, 0x68, 0x65, 0x72, 0x5f,
0x45, 0x61, 0x73, 0x74, 0x65, 0x72, 0x6e, 0x0, 0x45, 0x75, 0x72, 0x6f, 0x70, 0x65, 0x5f, 0x57,
0x65, 0x73, 0x74, 0x65, 0x72, 0x6e, 0x0, 0x46, 0x61, 0x6c, 0x6b, 0x6c, 0x61, 0x6e, 0x64, 0x0,
0x46, 0x69, 0x6a, 0x69, 0x0, 0x46, 0x72, 0x65, 0x6e, 0x63, 0x68, 0x5f, 0x47, 0x75, 0x69, 0x61,
0x6e, 0x61, 0x0, 0x46, 0x72, 0x65, 0x6e, 0x63, 0x68, 0x5f, 0x53, 0x6f, 0x75, 0x74, 0x68, 0x65,
0x72, 0x6e, 0x0, 0x46, 0x72, 0x75, 0x6e, 0x7a, 0x65, 0x0, 0x47, 0x61, 0x6c, 0x61, 0x70, 0x61,
0x67, 0x6f, 0x73, 0x0, 0x47, 0x61, 0x6d, 0x62, 0x69, 0x65, 0x72, 0x0, 0x47, 0x65, 0x6f, 0x72,
0x67, 0x69, 0x61, 0x0, 0x47, 0x69, 0x6c, 0x62, 0x65, 0x72, 0x74, 0x5f, 0x49, 0x73, 0x6c, 0x61,
0x6e, 0x64, 0x73, 0x0, 0x47, 0x4d, 0x54, 0x0, 0x47, 0x6f, 0x6f, 0x73, 0x65, 0x5f, 0x42, 0x61,
0x79, 0x0, 0x47, 0x72, 0x65, 0x65, 0x6e, 0x6c, 0x61, 0x6e, 0x64, 0x0, 0x47, 0x72, 0x65, 0x65,
0x6e, 0x6c, 0x61, 0x6e, 0x64, 0x5f, 0x43, 0x65, 0x6e, 0x74, 0x72, 0x61, 0x6c, 0x0, 0x47, 0x72,
0x65, 0x65, 0x6e, 0x6c, 0x61, 0x6e, 0x64, 0x5f, 0x45, 0x61, 0x73, 0x74, 0x65, 0x72, 0x6e, 0x0,
0x47, 0x72, 0x65, 0x65, 0x6e, 0x6c, 0x61, 0x6e, 0x64, 0x5f, 0x57, 0x65, 0x73, 0x74, 0x65, 0x72,
0x6e, 0x0, 0x47, 0x75, 0x61, 0x6d, 0x0, 0x47, 0x75, 0x6c, 0x66, 0x0, 0x47, 0x75, 0x79, 0x61,
0x6e, 0x61, 0x0, 0x48, 0x61, 0x77, 0x61, 0x69, 0x69, 0x5f, 0x41, 0x6c, 0x65, 0x75, 0x74, 0x69,
0x61, 0x6e, 0x0, 0x48, 0x6f, 0x6e, 0x67, 0x5f, 0x4b, 0x6f, 0x6e, 0x67, 0x0, 0x48, 0x6f, 0x76,
0x64, 0x0, 0x49, 0x6e, 0x64, 0x69, 0x61, 0x0, 0x49, 0x6e, 0x64, 0x69, 0x61, 0x6e, 0x5f, 0x4f,
0x63, 0x65, 0x61, 0x6e, 0x0, 0x49, 0x6e, 0x64, 0x6f, 0x63, 0x68, 0x69, 0x6e, 0x61, 0x0, 0x49,
0x6e, 0x64, 0x6f, 0x6e, 0x65, 0x73, 0x69, 0x61, 0x5f, 0x43, 0x65, 0x6e, 0x74, 0x72, 0x61, 0x6c,
0x0, 0x49, 0x6e, 0x64, 0x6f, 0x6e, 0x65, 0x73, 0x69, 0x61, 0x5f, 0x45, 0x61, 0x73, 0x74, 0x65,
0x72, 0x6e, 0x0, 0x49, 0x6e, 0x64, 0x6f, 0x6e, 0x65, 0x73, 0x69, 0x61, 0x5f, 0x57, 0x65, 0x73,
0x74, 0x65, 0x72, 0x6e, 0x0, 0x49, 0x72, 0x61, 0x6e, 0x0, 0x49, 0x72, 0x69, 0x73, 0x68, 0x0,
0x49, 0x72, 0x6b, 0x75, 0x74, 0x73, 0x6b, 0x0, 0x49, 0x73, 0x72, 0x61, 0x65, 0x6c, 0x0, 0x4a,
0x61, 0x70, 0x61, 0x6e, 0x0, 0x4b, 0x61, 0x6d, 0x63, 0x68, 0x61, 0x74, 0x6b, 0x61, 0x0, 0x4b,
0x61, 0x72, 0x61, 0x63, 0x68, 0x69, 0x0, 0x4b, 0x61, 0x7a, 0x61, 0x6b, 0x68, 0x73, 0x74, 0x61,
0x6e, 0x0, 0x4b, 0x61, 0x7a, 0x61, 0x6b, 0x68, 0x73, 0x74, 0x61, 0x6e, 0x5f, 0x45, 0x61, 0x73,
0x74, 0x65, 0x72, 0x6e, 0x0, 0x4b, 0x61, 0x7a, 0x61, 0x6b, 0x68, 0x73, 0x74, 0x61, 0x6e, 0x5f,
0x57, 0x65, 0x73, 0x74, 0x65, 0x72, 0x6e, 0x0, 0x4b, 0x69, 0x7a, 0x69, 0x6c, 0x6f, 0x72, 0x64,
0x61, 0x0, 0x4b, 0x6f, 0x72, 0x65, 0x61, 0x0, 0x4b, 0x6f, 0x73, 0x72, 0x61, 0x65, 0x0, 0x4b,
0x72, 0x61, 0x73, 0x6e, 0x6f, 0x79, 0x61, 0x72, 0x73, 0x6b, 0x0, 0x4b, 0x75, 0x79, 0x62, 0x79,
0x73, 0x68, 0x65, 0x76, 0x0, 0x4b, 0x77, 0x61, 0x6a, 0x61, 0x6c, 0x65, 0x69, 0x6e, 0x0, 0x4b,
0x79, 0x72, 0x67, 0x79, 0x73, 0x74, 0x61, 0x6e, 0x0, 0x4c, 0x61, 0x6e, 0x6b, 0x61, 0x0, 0x4c,
0x69, 0x62, 0x65, 0x72, 0x69, 0x61, 0x0, 0x4c, 0x69, 0x6e, 0x65, 0x5f, 0x49, 0x73, 0x6c, 0x61,
0x6e, 0x64, 0x73, 0x0, 0x4c, 0x6f, 0x72, 0x64, 0x5f, 0x48, 0x6f, 0x77, 0x65, 0x0, 0x4d, 0x61,
0x63, 0x61, 0x75, 0x0, 0x4d, 0x61, 0x67, 0x61, 0x64, 0x61, 0x6e, 0x0, 0x4d, 0x61, 0x6c, 0x61,
0x79, 0x61, 0x0, 0x4d, 0x61, 0x6c, 0x61, 0x79, 0x73, 0x69, 0x61, 0x0, 0x4d, 0x61, 0x6c, 0x64,
0x69, 0x76, 0x65, 0x73, 0x0, 0x4d, 0x61, 0x72, 0x71, 0x75, 0x65, 0x73, 0x61, 0x73, 0x0, 0x4d,
0x61, 0x72, 0x73, 0x68, 0x61, 0x6c, 0x6c, 0x5f, 0x49, 0x73, 0x6c, 0x61, 0x6e, 0x64, 0x73, 0x0,
0x4d, 0x61, 0x75, 0x72, 0x69, 0x74, 0x69, 0x75, 0x73, 0x0, 0x4d, 0x61, 0x77, 0x73, 0x6f, 0x6e,
0x0, 0x4d, 0x65, 0x78, 0x69, 0x63, 0x6f, 0x5f, 0x50, 0x61, 0x63, 0x69, 0x66, 0x69, 0x63, 0x0,
0x4d, 0x6f, 0x6e, 0x67, 0x6f, 0x6c, 0x69, 0x61, 0x0, 0x4d, 0x6f, 0x73, 0x63, 0x6f, 0x77, 0x0,
0x4d, 0x79, 0x61, 0x6e, 0x6d, 0x61, 0x72, 0x0, 0x4e, 0x61, 0x75, 0x72, 0x75, 0x0, 0x4e, 0x65,
0x70, 0x61, 0x6c, 0x0, 0x4e, 0x65, 0x77, 0x5f, 0x43, 0x61, 0x6c, 0x65, 0x64, 0x6f, 0x6e, 0x69,
0x61, 0x0, 0x4e, 0x65, 0x77, 0x5f, 0x5a, 0x65, 0x61, 0x6c, 0x61, 0x6e, 0x64, 0x0, 0x4e, 0x65,
0x77, 0x66, 0x6f, 0x75, 0x6e, 0x64, 0x6c, 0x61, 0x6e, 0x64, 0x0, 0x4e, 0x69, 0x75, 0x65, 0x0,
0x4e, 0x6f, 0x72, 0x66, 0x6f, 0x6c, 0x6b, 0x0, 0x4e, 0x6f, 0x72, 0x6f, 0x6e, 0x68, 0x61, 0x0,
0x4e, 0x6f, 0x72, 0x74, 0x68, 0x5f, 0x4d, 0x61, 0x72, 0x69, 0x61, 0x6e, 0x61, 0x0, 0x4e, 0x6f,
0x76, 0x6f, 0x73, 0x69, 0x62, 0x69, 0x72, 0x73, 0x6b, 0x0, 0x4f, 0x6d, 0x73, 0x6b, 0x0, 0x4f,
0x72, 0x61, 0x6c, 0x0, 0x50, 0x61, 0x6b, 0x69, 0x73, 0x74, 0x61, 0x6e, 0x0, 0x50, 0x61, 0x6c,
0x61, 0x75, 0x0, 0x50, 0x61, 0x70, 0x75, 0x61, 0x5f, 0x4e, 0x65, 0x77, 0x5f, 0x47, 0x75, 0x69,
0x6e, 0x65, 0x61, 0x0, 0x50, 0x61, 0x72, 0x61, 0x67, 0x75, 0x61, 0x79, 0x0, 0x50, 0x65, 0x72,
0x75, 0x0, 0x50, 0x68, 0x69, 0x6c, 0x69, 0x70, 0x70, 0x69, 0x6e, 0x65, 0x73, 0x0, 0x50, 0x68,
0x6f, 0x65, 0x6e, 0x69, 0x78, 0x5f, 0x49, 0x73, 0x6c, 0x61, 0x6e, 0x64, 0x73, 0x0, 0x50, 0x69,
0x65, 0x72, 0x72, 0x65, 0x5f, 0x4d, 0x69, 0x71, 0x75, 0x65, 0x6c, 0x6f, 0x6e, 0x0, 0x50, 0x69,
0x74, 0x63, 0x61, 0x69, 0x72, 0x6e, 0x0, 0x50, 0x6f, 0x6e, 0x61, 0x70, 0x65, 0x0, 0x50, 0x79,
0x6f, 0x6e, 0x67, 0x79, 0x61, 0x6e, 0x67, 0x0, 0x51, 0x79, 0x7a, 0x79, 0x6c, 0x6f, 0x72, 0x64,
0x61, 0x0, 0x52, 0x65, 0x75, 0x6e, 0x69, 0x6f, 0x6e, 0x0, 0x52, 0x6f, 0x74, 0x68, 0x65, 0x72,
0x61, 0x0, 0x53, 0x61, 0x6b, 0x68, 0x61, 0x6c, 0x69, 0x6e, 0x0, 0x53, 0x61, 0x6d, 0x61, 0x72,
0x61, 0x0, 0x53, 0x61, 0x6d, 0x61, 0x72, 0x6b, 0x61, 0x6e, 0x64, 0x0, 0x53, 0x61, 0x6d, 0x6f,
0x61, 0x0, 0x53, 0x65, 0x79, 0x63, 0x68, 0x65, 0x6c, 0x6c, 0x65, 0x73, 0x0, 0x53, 0x68, 0x65,
0x76, 0x63, 0x68, 0x65, 0x6e, 0x6b, 0x6f, 0x0, 0x53, 0x69, 0x6e, 0x67, 0x61, 0x70, 0x6f, 0x72,
0x65, 0x0, 0x53, 0x6f, 0x6c, 0x6f, 0x6d, 0x6f, 0x6e, 0x0, 0x53, 0x6f, 0x75, 0x74, 0x68, 0x5f,
0x47, 0x65, 0x6f, 0x72, 0x67, 0x69, 0x61, 0x0, 0x53, 0x75, 0x72, 0x69, 0x6e, 0x61, 0x6d, 0x65,
0x0, 0x53, 0x76, 0x65, 0x72, 0x64, 0x6c, 0x6f, 0x76, 0x73, 0x6b, 0x0, 0x53, 0x79, 0x6f, 0x77,
0x61, 0x0, 0x54, 0x61, 0x68, 0x69, 0x74, 0x69, 0x0, 0x54, 0x61, 0x69, 0x70, 0x65, 0x69, 0x0,
0x54, 0x61, 0x6a, 0x69, 0x6b, 0x69, 0x73, 0x74, 0x61, 0x6e, 0x0, 0x54, 0x61, 0x73, 0x68, 0x6b,
0x65, 0x6e, 0x74, 0x0, 0x54, 0x62, 0x69, 0x6c, 0x69, 0x73, 0x69, 0x0, 0x54, 0x6f, 0x6b, 0x65,
0x6c, 0x61, 0x75, 0x0, 0x54, 0x6f, 0x6e, 0x67, 0x61, 0x0, 0x54, 0x72, 0x75, 0x6b, 0x0, 0x54,
0x75, 0x72, 0x6b, 0x65, 0x79, 0x0, 0x54, 0x75, 0x72, 0x6b, 0x6d, 0x65, 0x6e, 0x69, 0x73, 0x74,
0x61, 0x6e, 0x0, 0x54, 0x75, 0x76, 0x61, 0x6c, 0x75, 0x0, 0x55, 0x72, 0x61, 0x6c, 0x73, 0x6b,
0x0, 0x55, 0x72, 0x75, 0x67, 0x75, 0x61, 0x79, 0x0, 0x55, 0x72, 0x75, 0x6d, 0x71, 0x69, 0x0,
0x55, 0x7a, 0x62, 0x65, 0x6b, 0x69, 0x73, 0x74, 0x61, 0x6e, 0x0, 0x56, 0x61, 0x6e, 0x75, 0x61,
0x74, 0x75, 0x0, 0x56, 0x65, 0x6e, 0x65, 0x7a, 0x75, 0x65, 0x6c, 0x61, 0x0, 0x56, 0x6c, 0x61,
0x64, 0x69, 0x76, 0x6f, 0x73, 0x74, 0x6f, 0x6b, 0x0, 0x56, 0x6f, 0x6c, 0x67, 0x6f, 0x67, 0x72,
0x61, 0x64, 0x0, 0x56, 0x6f, 0x73, 0x74, 0x6f, 0x6b, 0x0, 0x57, 0x61, 0x6b, 0x65, 0x0, 0x57,
0x61, 0x6c, 0x6c, 0x69, 0x73, 0x0, 0x59, 0x61, 0x6b, 0x75, 0x74, 0x73, 0x6b, 0x0, 0x59, 0x65,
0x6b, 0x61, 0x74, 0x65, 0x72, 0x69, 0x6e, 0x62, 0x75, 0x72, 0x67, 0x0, 0x59, 0x65, 0x72, 0x65,
0x76, 0x61, 0x6e, 0x0, 0x59, 0x75, 0x6b, 0x6f, 0x6e, 0x0
};

#endif // timezone_locale but not ICU

// GENERATED PART ENDS HERE

constexpr QByteArrayView AliasData::aliasId() const { return ianaIdData + aliasIdIndex; }
constexpr QByteArrayView AliasData::ianaId() const { return ianaIdData + ianaIdIndex; }
#if QT_CONFIG(timezone_locale) && !QT_CONFIG(icu)
constexpr QByteArrayView TerritoryZone::ianaId() const { return ianaIdData + ianaIdIndex; }
constexpr QByteArrayView MetaZoneData::ianaId() const { return ianaIdData + ianaIdIndex; }
constexpr QByteArrayView MetaZoneData::metaZoneId() const { return metaIdData + metaIdIndex; }
constexpr QByteArrayView ZoneMetaHistory::ianaId() const { return ianaIdData + ianaIdIndex; }
#endif // timezone_locale but not ICU
constexpr QByteArrayView WindowsData::windowsId() const { return windowsIdData + windowsIdIndex; }
constexpr QByteArrayView WindowsData::ianaId() const { return ianaIdData + ianaIdIndex; }
// Each of the following returns a space-joined sequence of IANA IDs:
constexpr QByteArrayView UtcData::id() const { return ianaListData + ianaIdIndex; }
constexpr QLatin1StringView ZoneData::id() const
{ return QLatin1StringView(ianaListData + ianaIdIndex); }

} // namespace QtTimeZoneCldr

QT_END_NAMESPACE

#endif // QTIMEZONEPRIVATE_DATA_P_H
