/// @file parseMicrostripStep_QucsS.cpp
/// @brief Converts a microstrip impedance step to Qucs-S format
/// @author Andrés Martínez Mera - andresmmera@protonmail.com
/// @date Jan 4, 2026
/// @copyright Copyright (C) 2026 Andrés Martínez Mera
/// @license GPL-3.0-or-later

#include "./../../QucsS/QucsSExporter.h"

QString QucsSExporter::parseMicrostripStep_QucsS(ComponentInfo Comp) {
  // Format: <MSTEP ID status x y text_x text_y vertical_mirror rotation
  // <properties>> Temp_Visibility> ID: e.g. Lx, where x is a positive integer
  // status:
  //         0 -> opened
  //         1 -> active
  //         2 -> shorted
  // x: x-axis position
  // y: y-axis position
  // text_x: x-axis position of the texts showing the properties
  // text_y: y-axis position of the texts showing the properties
  // vertical_mirror: {0, 1}
  // rotation: Component's rotation {0, 1, 2, 3}

  // Example: <MSTEP MS1 1 690 500 -26 17 0 0 "Subst1" 1 "2 mm" 1 "1 mm" 1
  // "Hammerstad" 0 "Kirschning" 0>

  int status = 1;
  int x_pos = Comp.Coordinates.at(0) * scale_x + x_offset;
  int y_pos = Comp.Coordinates.at(1) * scale_y + y_offset;
  int x_text = 25;
  int y_text = 0;
  int rotation = static_cast<int>(Comp.Rotation / 90);

  // Parameters
  QString W1 = Comp.val["W1"];
  int W1_visibility = 1;

  QString W2 = Comp.val["W2"];
  int W2_visibility = 1;

  // Adjust text position depending on orientation
  switch (rotation) {
  case 2:
  case 0: // Horizontal orientation
    x_text = -30;
    y_text = -70;

    // Save pin position. This is needed for wiring later
    ComponentPinMap[Comp.ID][0] = QPoint(x_pos - 30, y_pos); // Pin 1
    ComponentPinMap[Comp.ID][1] = QPoint(x_pos + 30, y_pos); // Pin 2
    break;
  case 1: // Vertical orientation
    x_text = 20;
    y_text = -10;

    // Save pin position. This is needed for wiring later
    ComponentPinMap[Comp.ID][0] = QPoint(x_pos, y_pos + 30); // Pin 1
    ComponentPinMap[Comp.ID][1] = QPoint(x_pos, y_pos - 30); // Pin 2
    break;
  }

  QString componentLine =
      QString("<MSTEP %1 %2 %3 %4 %5 %6 1 %7 \"Subst1\" 0 \"%8\" %9 "
              "\"%10\" %11 \"Hammerstad\" 0 \"Kirschning\" "
              "0 \"26.85\" 0 \"DC\" 0>\n")
          .arg(Comp.ID)
          .arg(status)
          .arg(x_pos)
          .arg(y_pos)
          .arg(x_text)
          .arg(y_text)
          .arg(rotation)
          .arg(W1)
          .arg(W1_visibility)
          .arg(W2)
          .arg(W2_visibility);

  return componentLine;
}
