/*  RetroArch - A frontend for libretro.
 *  Copyright (C) 2010-2017 - Hans-Kristian Arntzen
 *
 *  RetroArch is free software: you can redistribute it and/or modify it under the terms
 *  of the GNU General Public License as published by the Free Software Found-
 *  ation, either version 3 of the License, or (at your option) any later version.
 *
 *  RetroArch is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 *  without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
 *  PURPOSE.  See the GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License along with RetroArch.
 *  If not, see <http://www.gnu.org/licenses/>.
 */

#ifndef SLANG_REFLECTION_H_
#define SLANG_REFLECTION_H_

/* Textures with built-in meaning. */
enum slang_texture_semantic
{
   /* The input texture to the filter chain.
    * Canonical name: "Original". */
   SLANG_TEXTURE_SEMANTIC_ORIGINAL         = 0,

   /* The output from pass N - 1 if executing pass N, or ORIGINAL
    * if pass #0 is executed.
    * Canonical name: "Source".
    */
   SLANG_TEXTURE_SEMANTIC_SOURCE           = 1,

   /* The original inputs with a history back in time.
    * Canonical name: "OriginalHistory#", e.g. "OriginalHistory2" <- Two frames back.
    * "OriginalHistory0" is an alias for SEMANTIC_ORIGINAL.
    * Size name: "OriginalHistorySize#".
    */
   SLANG_TEXTURE_SEMANTIC_ORIGINAL_HISTORY = 2,

   /* The output from pass #N, where pass #0 is the first pass.
    * Canonical name: "PassOutput#", e.g. "PassOutput3".
    * Size name: "PassOutputSize#".
    */
   SLANG_TEXTURE_SEMANTIC_PASS_OUTPUT      = 3,

   /* The output from pass #N, one frame ago where pass #0 is the first pass.
    * It is not valid to use the pass feedback from a pass which is not offscreen.
    * Canonical name: "PassFeedback#", e.g. "PassFeedback2".
    */
   SLANG_TEXTURE_SEMANTIC_PASS_FEEDBACK    = 4,

   /* Inputs from static textures, defined by the user.
    * There is no canonical name, and the only way to use these semantics are by
    * remapping.
    */
   SLANG_TEXTURE_SEMANTIC_USER             = 5,

   SLANG_NUM_TEXTURE_SEMANTICS,
   SLANG_INVALID_TEXTURE_SEMANTIC       = -1
};

enum slang_semantic
{
   /* mat4, MVP */
   SLANG_SEMANTIC_MVP              = 0,
   /* vec4, viewport size of current pass */
   SLANG_SEMANTIC_OUTPUT           = 1,
   /* vec4, viewport size of final pass */
   SLANG_SEMANTIC_FINAL_VIEWPORT   = 2,
   /* uint, frame count with modulo */
   SLANG_SEMANTIC_FRAME_COUNT      = 3,
   /* int, frame direction */
   SLANG_SEMANTIC_FRAME_DIRECTION  = 4,
   /* uint, FrameTimeDelta */
   SLANG_SEMANTIC_FRAME_TIME_DELTA = 5,
   /* uint, OriginalFPS */
   SLANG_SEMANTIC_ORIGINAL_FPS     = 6,
   /* uint, rotation */
   SLANG_SEMANTIC_ROTATION         = 7,
   /* float, OriginalAspect */
   SLANG_SEMANTIC_CORE_ASPECT      = 8,
   /* float, OriginalAspectRotated */
   SLANG_SEMANTIC_CORE_ASPECT_ROT  = 9,
   /* uint, sub frames per content frame */
   SLANG_SEMANTIC_TOTAL_SUBFRAMES  = 10,
   /* uint, current sub frame */
   SLANG_SEMANTIC_CURRENT_SUBFRAME = 11,
   /* float, Enable HDR */
   SLANG_SEMANTIC_HDR              = 12,
   /* float, HDR Paper white luminace */
   SLANG_SEMANTIC_PAPER_WHITE_NITS = 13,
   /* float, HDR Peak luminace */
   SLANG_SEMANTIC_MAX_NITS         = 14,
   /* float, Enable HDR scanlines */
   SLANG_SEMANTIC_SCANLINES        = 15,
   /* uint, HDR Scanline Subpixel Layout */
   SLANG_SEMANTIC_SUBPIXEL_LAYOUT  = 16,
   /* uint, Enable HDR colour boost */
   SLANG_SEMANTIC_EXPAND_GAMUT     = 17,
   /* float, Enable HDR Inverse Tonemapper */
   SLANG_SEMANTIC_INVERSE_TONEMAP  = 18,
   /* float, Enable HDR10 conversion */
   SLANG_SEMANTIC_HDR10            = 19,
   /* float, OriginalAspectRotated */
   SLANG_NUM_BASE_SEMANTICS        = 20,

   /* float, user defined parameter, arrayed */
   SLANG_SEMANTIC_FLOAT_PARAMETER  = 21,

   SLANG_NUM_SEMANTICS,
   SLANG_INVALID_SEMANTIC          = -1
};

enum slang_stage
{
   SLANG_STAGE_VERTEX_MASK   = 1 << 0,
   SLANG_STAGE_FRAGMENT_MASK = 1 << 1
};

enum slang_constant_buffer
{
   SLANG_CBUFFER_UBO = 0,
   SLANG_CBUFFER_PC,
   SLANG_CBUFFER_MAX
};

/* Vulkan maximum texture bindings inside shader. D3D11 has hard limit of 16 */
#define SLANG_NUM_BINDINGS 16

struct slang_texture_semantic_map
{
   enum slang_texture_semantic semantic;
   unsigned index;
};

struct slang_semantic_map
{
   enum slang_semantic semantic;
   unsigned index;
};


#endif
