import {
  SearchBuilder
} from "./chunk-HOBLKBZ6.js";
import {
  Fields,
  RankingType
} from "./chunk-5UDKQCPJ.js";
import {
  addDays,
  formatDate
} from "./chunk-RNHRR56W.js";

// src/ranking.ts
var RankingBuilder = class {
  /**
   * constructor
   * @param params - 初期クエリパラメータ
   * @param api - API実行クラスのインスタンス
   * @private
   */
  constructor(params = {}, api) {
    this.params = params;
    this.api = api;
    this.date$ = addDays(/* @__PURE__ */ new Date(), -1);
    this.type$ = RankingType.Daily;
  }
  /**
   * ランキング集計対象の日付を指定します。
   *
   * - 日間: 任意の日付
   * - 週間: 火曜日の日付
   * - 月間・四半期: 1日の日付
   *
   * @param date 集計対象の日付
   * @returns {RankingBuilder} this
   * @see https://dev.syosetu.com/man/rankapi/
   */
  date(date) {
    this.date$ = date;
    return this;
  }
  /**
   * ランキング種別を指定します。
   * @param type ランキング種別
   * @returns {RankingBuilder} this
   * @see https://dev.syosetu.com/man/rankapi/
   */
  type(type) {
    this.type$ = type;
    return this;
  }
  /**
   * gzip圧縮する。
   *
   * 転送量上限を減らすためにも推奨
   * @param {GzipLevel} level gzip圧縮レベル(1～5)
   * @return {RankingBuilder} this
   */
  gzip(level) {
    this.set({ gzip: level });
    return this;
  }
  /**
   * クエリパラメータを内部的にセットします。
   * @param obj - セットするパラメータオブジェクト
   * @returns {RankingBuilder} this
   * @private
   */
  set(obj) {
    Object.assign(this.params, obj);
    return this;
  }
  /**
   * 設定されたパラメータに基づき、なろう小説ランキングAPIへのリクエストを実行します。
   *
   * 返される結果には、Nコード、ポイント、順位が含まれます。
   * @returns {Promise<NarouRankingResult[]>} ランキング結果の配列
   * @see https://dev.syosetu.com/man/rankapi/#output
   */
  execute() {
    const date = formatDate(this.date$);
    this.set({ rtype: `${date}-${this.type$}` });
    return this.api.executeRanking(this.params);
  }
  /**
   * ランキングAPIを実行し、取得したNコードを元になろう小説APIで詳細情報を取得して結合します。
   *
   * @template TFields - 取得する小説情報のフィールド型
   * @template TOpt - オプショナルな取得フィールドの型
   * @param fields - 取得するフィールドの配列 (省略時はデフォルトフィールド)
   * @param opt - オプショナルな取得フィールド (`weekly` など)
   * @returns {Promise<RankingResult<SearchResultFields<TFields>>[]>} 詳細情報を含むランキング結果の配列
   */
  async executeWithFields(fields = [], opt) {
    const ranking = await this.execute();
    const fields$ = Array.isArray(fields) ? fields.length == 0 ? [] : [...fields, Fields.ncode] : [fields, Fields.ncode];
    const rankingNcodes = ranking.map(({ ncode }) => ncode);
    const builder = new SearchBuilder({}, this.api);
    builder.fields(fields$);
    if (opt) {
      builder.opt(opt);
    }
    builder.ncode(rankingNcodes);
    builder.limit(ranking.length);
    const result = await builder.execute();
    return ranking.map((r) => ({
      ...r,
      // eslint-disable-next-line @typescript-eslint/no-explicit-any
      ...result.values.find((novel) => novel.ncode == r.ncode)
    }));
  }
};

export {
  RankingBuilder
};
//# sourceMappingURL=chunk-BQLSW236.js.map