'use strict';

var tsCaseConvert = require('ts-case-convert');
var client_js = require('../client.js');

function buildRequest(path, method) {
  const _request = async (client, init, mapResult) => {
    const { data, error } = await client[method.toUpperCase()](path, init);
    if (error || !data) {
      throw new Error(
        [
          "Request failed:",
          `path: ${path}`,
          `error: ${error}`,
          `data: ${data}`
        ].join("\n  ")
      );
    }
    const result = tsCaseConvert.objectToCamel(data);
    return mapResult?.(result) ?? result;
  };
  return {
    withParams(getInit, mapResult) {
      return async (params, init, client = client_js.getDefaultClient()) => _request(client, { ...getInit(params), ...init }, mapResult);
    },
    withoutParams(mapResult) {
      return async (init, client = client_js.getDefaultClient()) => _request(client, init, mapResult);
    }
  };
}
const getActivity = buildRequest(
  "/decentralized/tx/{id}",
  "get"
).withParams(({ id, ...query }) => ({
  params: tsCaseConvert.objectToSnake({ path: { id }, query })
}));
const getActivities = buildRequest(
  "/decentralized/{account}",
  "get"
).withParams(
  ({ account, ...query }) => ({
    params: tsCaseConvert.objectToSnake({ path: { account }, query })
  }),
  ({ data, meta }) => ({
    data,
    cursor: meta?.cursor
  })
);
const getActivitiesByAccounts = buildRequest(
  "/decentralized/accounts",
  "post"
).withParams(
  (body) => ({
    body: tsCaseConvert.objectToSnake({
      limit: 20,
      actionLimit: 10,
      ...body
    })
  }),
  ({ data, meta }) => ({
    data,
    cursor: meta?.cursor
  })
);
const getRSSActivity = buildRequest("/rss/{path}", "get").withParams(
  (path) => ({
    params: tsCaseConvert.objectToSnake({ path })
  }),
  ({ data, meta }) => ({
    data,
    cursor: meta?.cursor
  })
);
const getNetworkActivities = buildRequest(
  "/decentralized/network/{network}",
  "get"
).withParams(
  (path) => ({
    params: tsCaseConvert.objectToSnake({ path })
  }),
  ({ data, meta }) => ({
    data,
    cursor: meta?.cursor
  })
);
const getPlatformActivities = buildRequest(
  "/decentralized/platform/{platform}",
  "get"
).withParams(
  (path) => ({
    params: tsCaseConvert.objectToSnake({ path })
  }),
  ({ data, meta }) => ({
    data,
    cursor: meta?.cursor
  })
);
const getFederatedActivitiesByAccount = buildRequest(
  "/federated/{account}",
  "get"
).withParams(({ account, ...query }) => ({
  params: tsCaseConvert.objectToSnake({ path: { account }, query })
}));
const getFederatedActivity = buildRequest(
  "/federated/tx/{id}",
  "get"
).withParams((path) => ({
  params: tsCaseConvert.objectToSnake({ path })
}));
const getFederatedActivitiesByAccounts = buildRequest(
  "/federated/accounts",
  "post"
).withParams((body) => ({
  body: tsCaseConvert.objectToSnake(body)
}));
const getFederatedNetworkActivities = buildRequest(
  "/federated/network/{network}",
  "get"
).withParams((path) => ({
  params: tsCaseConvert.objectToSnake({ path })
}));
const getFederatedPlatformActivities = buildRequest(
  "/federated/platform/{platform}",
  "get"
).withParams((path) => ({
  params: tsCaseConvert.objectToSnake({ path })
}));
const getBridgingTransactions = buildRequest(
  "/nta/bridgings/transactions",
  "get"
).withParams((query) => ({
  params: tsCaseConvert.objectToSnake({ query })
}));
const getBridgingTransaction = buildRequest(
  "/nta/bridgings/transactions/{transaction_hash}",
  "get"
).withParams((path) => ({
  params: tsCaseConvert.objectToSnake({ path })
}));
const getStakingTransactions = buildRequest(
  "/nta/stakings/transactions",
  "get"
).withParams((query) => ({
  params: tsCaseConvert.objectToSnake({ query })
}));
const getStakingTransaction = buildRequest(
  "/nta/stakings/transactions/{transaction_hash}",
  "get"
).withParams((path) => ({
  params: tsCaseConvert.objectToSnake({ path })
}));
const getStakings = buildRequest(
  "/nta/stakings/stakings",
  "get"
).withParams((query) => ({
  params: tsCaseConvert.objectToSnake({ query })
}));
const getStakingProfit = buildRequest(
  "/nta/stakings/{staker_address}/profit",
  "get"
).withParams((path) => ({
  params: tsCaseConvert.objectToSnake({ path })
}));
const getChips = buildRequest("/nta/chips", "get").withParams(
  (query) => ({
    params: tsCaseConvert.objectToSnake({ query })
  })
);
const getChip = buildRequest("/nta/chips/{chip_id}", "get").withParams(
  (path) => ({
    params: tsCaseConvert.objectToSnake({ path })
  })
);
const getChipSvgURL = (params, client = client_js.getDefaultClient()) => new URL(`/nta/chips/${params.chipId}/image.svg`, client.baseUrl);
const getChipSvgSrc = (params, client = client_js.getDefaultClient()) => getChipSvgURL(params, client).href;
const getChipSvg = buildRequest(
  "/nta/chips/{chip_id}/image.svg",
  "get"
).withParams((path) => ({
  params: tsCaseConvert.objectToSnake({ path })
}));
const getNodeCountSnapshot = buildRequest(
  "/nta/snapshots/nodes/count",
  "get"
).withoutParams();
const getStakerCountSnapshot = buildRequest(
  "/nta/snapshots/stakers/count",
  "get"
).withoutParams();
const getStakerProfitSnapshot = buildRequest(
  "/nta/snapshots/stakers/profit",
  "get"
).withParams((query) => ({
  params: tsCaseConvert.objectToSnake({ query })
}));
const getStakingStat = buildRequest(
  "/nta/stakings/{staker_address}/stat",
  "get"
).withParams((path) => ({
  params: tsCaseConvert.objectToSnake({ path })
}));
const getOperationProfitSnapshot = buildRequest(
  "/nta/snapshots/nodes/operation/profit",
  "get"
).withParams((query) => ({
  params: tsCaseConvert.objectToSnake({ query })
}));
const getEpochsApySnapshot = buildRequest(
  "/nta/snapshots/epochs/apy",
  "get"
).withoutParams();
const getAllNodes = buildRequest("/nta/nodes", "get").withParams(
  (query) => ({
    params: tsCaseConvert.objectToSnake({ query })
  })
);
const getNode = buildRequest("/nta/nodes/{address}", "get").withParams(
  (path) => ({
    params: tsCaseConvert.objectToSnake({ path })
  })
);
const getNodeAvatarSvgURL = (params, client = client_js.getDefaultClient()) => new URL(`/nta/nodes/${params.address}/avatar.svg`, client.baseUrl);
const getNodeAvatarSvgSrc = (params, client = client_js.getDefaultClient()) => getNodeAvatarSvgURL(params, client).href;
const getNodeAvatarSvg = buildRequest(
  "/nta/nodes/{address}/avatar.svg",
  "get"
).withParams((path) => ({
  params: tsCaseConvert.objectToSnake({ path })
}));
const getNodeEvents = buildRequest(
  "/nta/nodes/{address}/events",
  "get"
).withParams(({ address, ...query }) => ({
  params: tsCaseConvert.objectToSnake({ path: { address }, query })
}));
const getNodeOperationProfit = buildRequest(
  "/nta/nodes/{address}/operation/profit",
  "get"
).withParams((path) => ({
  params: tsCaseConvert.objectToSnake({ path })
}));
const getEpochs = buildRequest("/nta/epochs", "get").withParams(
  (query) => ({
    params: tsCaseConvert.objectToSnake({ query })
  })
);
const getEpoch = buildRequest(
  "/nta/epochs/{epoch_id}",
  "get"
).withParams((path) => ({
  params: tsCaseConvert.objectToSnake({ path })
}));
const getEpochTransaction = buildRequest(
  "/nta/epochs/distributions/{transaction_hash}",
  "get"
).withParams((path) => ({
  params: tsCaseConvert.objectToSnake({ path })
}));
const getNodeRewards = buildRequest(
  "/nta/epochs/{address}/rewards",
  "get"
).withParams(({ address, ...query }) => ({
  params: tsCaseConvert.objectToSnake({ path: { address }, query })
}));
const getAverageEpochsApy = buildRequest(
  "/nta/epochs/apy",
  "get"
).withoutParams();
const getNetworkAssets = buildRequest(
  "/nta/networks/assets",
  "get"
).withoutParams();
const getNetworkConfig = buildRequest(
  "/nta/networks/config",
  "get"
).withoutParams();
const getTotalRequests = buildRequest(
  "/nta/dsl/total_requests",
  "get"
).withoutParams();
const getTokenSupply = buildRequest(
  "/nta/token/supply",
  "get"
).withoutParams();
const getTokenTvl = buildRequest(
  "/nta/token/tvl",
  "get"
).withoutParams();

exports.getActivities = getActivities;
exports.getActivitiesByAccounts = getActivitiesByAccounts;
exports.getActivity = getActivity;
exports.getAllNodes = getAllNodes;
exports.getAverageEpochsApy = getAverageEpochsApy;
exports.getBridgingTransaction = getBridgingTransaction;
exports.getBridgingTransactions = getBridgingTransactions;
exports.getChip = getChip;
exports.getChipSvg = getChipSvg;
exports.getChipSvgSrc = getChipSvgSrc;
exports.getChipSvgURL = getChipSvgURL;
exports.getChips = getChips;
exports.getEpoch = getEpoch;
exports.getEpochTransaction = getEpochTransaction;
exports.getEpochs = getEpochs;
exports.getEpochsApySnapshot = getEpochsApySnapshot;
exports.getFederatedActivitiesByAccount = getFederatedActivitiesByAccount;
exports.getFederatedActivitiesByAccounts = getFederatedActivitiesByAccounts;
exports.getFederatedActivity = getFederatedActivity;
exports.getFederatedNetworkActivities = getFederatedNetworkActivities;
exports.getFederatedPlatformActivities = getFederatedPlatformActivities;
exports.getNetworkActivities = getNetworkActivities;
exports.getNetworkAssets = getNetworkAssets;
exports.getNetworkConfig = getNetworkConfig;
exports.getNode = getNode;
exports.getNodeAvatarSvg = getNodeAvatarSvg;
exports.getNodeAvatarSvgSrc = getNodeAvatarSvgSrc;
exports.getNodeAvatarSvgURL = getNodeAvatarSvgURL;
exports.getNodeCountSnapshot = getNodeCountSnapshot;
exports.getNodeEvents = getNodeEvents;
exports.getNodeOperationProfit = getNodeOperationProfit;
exports.getNodeRewards = getNodeRewards;
exports.getOperationProfitSnapshot = getOperationProfitSnapshot;
exports.getPlatformActivities = getPlatformActivities;
exports.getRSSActivity = getRSSActivity;
exports.getStakerCountSnapshot = getStakerCountSnapshot;
exports.getStakerProfitSnapshot = getStakerProfitSnapshot;
exports.getStakingProfit = getStakingProfit;
exports.getStakingStat = getStakingStat;
exports.getStakingTransaction = getStakingTransaction;
exports.getStakingTransactions = getStakingTransactions;
exports.getStakings = getStakings;
exports.getTokenSupply = getTokenSupply;
exports.getTokenTvl = getTokenTvl;
exports.getTotalRequests = getTotalRequests;
//# sourceMappingURL=index.cjs.map
//# sourceMappingURL=index.cjs.map