import { Join } from './util/type.cjs';

declare const RankingType: {
    readonly Daily: "d";
    readonly Weekly: "w";
    readonly Monthly: "m";
    readonly Quarterly: "q";
};
type RankingType = (typeof RankingType)[keyof typeof RankingType];
/**
 * すべてのAPIで共通のクエリパラメータ
 */
interface ParamsBase {
    /**
     * gzip圧縮してgzipファイルとして返します。
     * gzip圧縮レベルを1～5で指定できます。
     * 転送量上限を減らすためにも推奨
     */
    gzip?: GzipLevel;
    /**
     * 出力形式を指定
     * 本ライブラリはJSONとJSONPのみ対応
     */
    out?: "json" | "jsonp";
}
/**
 * 検索APIで共通のクエリパラメータ
 */
interface ParamsBaseWithOrder<TOrder extends string> extends ParamsBase {
    /**
     * 出力する項目を個別に指定できます。未指定時は全項目出力されます。
     * 転送量軽減のため、このパラメータの使用が推奨されます。
     */
    of?: string;
    /**
     * 最大出力数を指定できます。指定しない場合は20件になります。
     */
    lim?: number;
    /**	表示開始位置の指定です。 */
    st?: number;
    /** 出力順序を指定できます。 */
    order?: TOrder;
}
/**
 * メソッドにパラメータを指定する際のヘルパー。
 * @see https://dev.syosetu.com/man/api/
 * @see https://dev.syosetu.com/xman/atom/
 */
interface SearchParams extends ParamsBaseWithOrder<Order> {
    word?: string;
    notword?: string;
    title?: BooleanNumber;
    ex?: BooleanNumber;
    keyword?: BooleanNumber;
    wname?: BooleanNumber;
    biggenre?: Join<BigGenre> | BigGenre;
    notbiggenre?: Join<BigGenre> | BigGenre;
    genre?: Join<Genre> | Genre;
    notgenre?: Join<Genre> | Genre;
    userid?: Join<number> | number;
    nocgenre?: Join<R18Site> | R18Site;
    notnocgenre?: Join<R18Site> | R18Site;
    xid?: Join<number> | number;
    isr15?: BooleanNumber;
    isbl?: BooleanNumber;
    isgl?: BooleanNumber;
    iszankoku?: BooleanNumber;
    istensei?: BooleanNumber;
    istenni?: BooleanNumber;
    istt?: BooleanNumber;
    notr15?: BooleanNumber;
    notbl?: BooleanNumber;
    notgl?: BooleanNumber;
    notzankoku?: BooleanNumber;
    nottensei?: BooleanNumber;
    nottenni?: BooleanNumber;
    minlen?: number;
    maxlen?: number;
    length?: number | Join<number | "">;
    kaiwaritu?: number | string;
    sasie?: number | string;
    mintime?: number;
    maxtime?: number;
    time?: number | string;
    ncode?: string | Join<string>;
    type?: NovelTypeParam;
    buntai?: BuntaiParam | Join<BuntaiParam>;
    stop?: StopParam;
    ispickup?: typeof BooleanNumber.True;
    lastup?: string;
    lastupdate?: string;
    opt?: Join<OptionalFields>;
}
interface RankingParams extends ParamsBase {
    rtype: `${string}-${RankingType}`;
}
interface RankingHistoryParams extends ParamsBase {
    ncode: string;
}
/**
 * ユーザー検索パラメータ
 */
interface UserSearchParams extends ParamsBaseWithOrder<UserOrder> {
    /** 単語を指定できます。半角または全角スペースで区切るとAND抽出になります。部分一致でHITします。検索の対象はユーザ名とユーザ名のフリガナです。 */
    word?: string;
    /** 含みたくない単語を指定できます。スペースで区切ることにより含ませない単語を増やせます。部分一致で除外されます。除外の対象はユーザ名とユーザ名のフリガナです。 */
    notword?: string;
    /** ユーザIDで抽出可能。 */
    userid?: number;
    /** 抽出するユーザのユーザ名のフリガナの頭文字を指定できます。頭文字はユーザ名のフリガナをひらがなに変換し、最初の1文字が「ぁ」～「ん」の場合に対象となります。 */
    name1st?: string;
    /** 抽出するユーザの小説投稿数の下限を指定できます。小説投稿件数が指定された数値以上のユーザを抽出します。 */
    minnovel?: number;
    /** 抽出するユーザの小説投稿数の上限を指定できます。小説投稿件数が指定された数値以下のユーザを抽出します。 */
    maxnovel?: number;
    /** 抽出するユーザのレビュー投稿数の下限を指定できます。レビュー投稿件数が指定された数値以上のユーザを抽出します。 */
    minreview?: number;
    /** 抽出するユーザのレビュー投稿数の上限を指定できます。レビュー投稿件数が指定された数値以下のユーザを抽出します。 */
    maxreview?: number;
}
declare const BooleanNumber: {
    readonly True: 1;
    readonly False: 0;
};
type BooleanNumber = (typeof BooleanNumber)[keyof typeof BooleanNumber];
type SearchResultFieldNames = keyof NarouSearchResult;
/**
 * なろう小説APIのofパラメータに指定できる出力する項目
 * @see https://dev.syosetu.com/man/api/#output
 */
declare const Fields: {
    /** 小説名 */
    readonly title: "t";
    /** Nコード */
    readonly ncode: "n";
    /** 作者のユーザID(数値) */
    readonly userid: "u";
    /** 作者名 */
    readonly writer: "w";
    /** 小説のあらすじ */
    readonly story: "s";
    /** 大ジャンル */
    readonly biggenre: "bg";
    /** ジャンル */
    readonly genre: "g";
    /** キーワード */
    readonly keyword: "k";
    /** 初回掲載日 */
    readonly general_firstup: "gf";
    /** 最終掲載日 */
    readonly general_lastup: "gl";
    /** 連載の場合は1、短編の場合は2 */
    readonly noveltype: "nt";
    /** 短編小説と完結済小説は0となっています。連載中は1です。 */
    readonly end: "e";
    /** 全掲載部分数 */
    readonly general_all_no: "ga";
    /** 小説文字数 */
    readonly length: "l";
    /** 読了時間(分単位) */
    readonly time: "ti";
    /** 長期連載停止中 */
    readonly isstop: "i";
    /** 登録必須キーワードに「R15」が含まれる場合は1、それ以外は0です。 */
    readonly isr15: "isr";
    /** 登録必須キーワードに「ボーイズラブ」が含まれる場合は1、それ以外は0です。 */
    readonly isbl: "ibl";
    /** 登録必須キーワードに「ガールズラブ」が含まれる場合は1、それ以外は0です。 */
    readonly isgl: "igl";
    /** 登録必須キーワードに「残酷な描写あり」が含まれる場合は1、それ以外は0です。 */
    readonly iszankoku: "izk";
    /** 登録必須キーワードに「異世界転生」が含まれる場合は1、それ以外は0です。 */
    readonly istensei: "its";
    /** 登録必須キーワードに「異世界転移」が含まれる場合は1、それ以外は0です。 */
    readonly istenni: "iti";
    /** 総合評価ポイント */
    readonly global_point: "gp";
    /** 日間ポイント */
    readonly daily_point: "dp";
    /** 週間ポイント */
    readonly weekly_point: "wp";
    /** 月間ポイント */
    readonly monthly_point: "mp";
    /** 四半期ポイント */
    readonly quarter_point: "qp";
    /** 年間ポイント */
    readonly yearly_point: "yp";
    /** ブックマーク数 */
    readonly fav_novel_cnt: "f";
    /** 感想数 */
    readonly impression_cnt: "imp";
    /** レビュー数 */
    readonly review_cnt: "r";
    /** 評価ポイント */
    readonly all_point: "a";
    /** 評価者数 */
    readonly all_hyoka_cnt: "ah";
    /** 挿絵の数 */
    readonly sasie_cnt: "sa";
    /** 会話率 */
    readonly kaiwaritu: "ka";
    /** 小説の更新日時 */
    readonly novelupdated_at: "nu";
    /**
     * 最終更新日時
     * システム用で小説更新時とは関係ありません
     */
    readonly updated_at: "ua";
};
type Fields = (typeof Fields)[keyof Omit<NarouSearchResult, "novel_type" | "weekly_unique" | "nocgenre">];
/**
 * なろうR18小説APIのofパラメータに指定できる出力する項目
 * @see https://dev.syosetu.com/xman/api/#output
 */
declare const R18Fields: {
    /** 小説名 */
    readonly title: "t";
    /** Nコード */
    readonly ncode: "n";
    /** 作者のユーザID(数値) */
    readonly userid: "u";
    /** 作者名 */
    readonly writer: "w";
    /** 小説のあらすじ */
    readonly story: "s";
    /** 掲載サイト */
    readonly nocgenre: "ng";
    /** キーワード */
    readonly keyword: "k";
    /** 初回掲載日 */
    readonly general_firstup: "gf";
    /** 最終掲載日 */
    readonly general_lastup: "gl";
    /** 連載の場合は1、短編の場合は2 */
    readonly noveltype: "nt";
    /** 短編小説と完結済小説は0となっています。連載中は1です。 */
    readonly end: "e";
    /** 全掲載部分数 */
    readonly general_all_no: "ga";
    /** 小説文字数 */
    readonly length: "l";
    /** 読了時間(分単位) */
    readonly time: "ti";
    /** 長期連載停止中 */
    readonly isstop: "i";
    /** 登録必須キーワードに「ボーイズラブ」が含まれる場合は1、それ以外は0です。 */
    readonly isbl: "ibl";
    /** 登録必須キーワードに「ガールズラブ」が含まれる場合は1、それ以外は0です。 */
    readonly isgl: "igl";
    /** 登録必須キーワードに「残酷な描写あり」が含まれる場合は1、それ以外は0です。 */
    readonly iszankoku: "izk";
    /** 登録必須キーワードに「異世界転生」が含まれる場合は1、それ以外は0です。 */
    readonly istensei: "its";
    /** 登録必須キーワードに「異世界転移」が含まれる場合は1、それ以外は0です。 */
    readonly istenni: "iti";
    /** 総合評価ポイント */
    readonly global_point: "gp";
    /** 日間ポイント */
    readonly daily_point: "dp";
    /** 週間ポイント */
    readonly weekly_point: "wp";
    /** 月間ポイント */
    readonly monthly_point: "mp";
    /** 四半期ポイント */
    readonly quarter_point: "qp";
    /** 年間ポイント */
    readonly yearly_point: "yp";
    /** R18ブックマーク数 */
    readonly fav_novel_cnt: "f";
    /** 感想数 */
    readonly impression_cnt: "imp";
    /** レビュー数 */
    readonly review_cnt: "r";
    /** 評価ポイント */
    readonly all_point: "a";
    /** 評価者数 */
    readonly all_hyoka_cnt: "ah";
    /** 挿絵の数 */
    readonly sasie_cnt: "sa";
    /** 会話率 */
    readonly kaiwaritu: "ka";
    /** 小説の更新日時 */
    readonly novelupdated_at: "nu";
    /**
     * 最終更新日時
     * システム用で小説更新時とは関係ありません
     */
    readonly updated_at: "ua";
};
type R18Fields = (typeof R18Fields)[keyof Omit<NarouSearchResult, "novel_type" | "weekly_unique" | "biggenre" | "genre" | "isr15">];
/**
 * オプション項目
 */
declare const OptionalFields: {
    /**
     * 週間ユニークユーザ[項目名:weekly_unique]が追加されます。
     * 週間ユニークユーザは前週の日曜日から土曜日分のユニークの合計です。
     * 毎週火曜日早朝に更新されます。
     */
    readonly weekly_unique: "weekly";
};
type OptionalFields = (typeof OptionalFields)[keyof Pick<NarouSearchResult, "weekly_unique">];
/**
 * ユーザ検索APIのofパラメータに指定できる出力する項目
 * @see https://dev.syosetu.com/man/userapi/#output
 */
declare const UserFields: {
    /** ユーザID */
    readonly userid: "u";
    /** ユーザ名 */
    readonly name: "n";
    /** ユーザ名のフリガナ */
    readonly yomikata: "y";
    /** ユーザ名のフリガナの頭文字 */
    readonly name1st: "1";
    /** 小説投稿数 */
    readonly novel_cnt: "nc";
    /** レビュー投稿数 */
    readonly review_cnt: "rc";
    /** 小説累計文字数 */
    readonly novel_length: "nl";
    /** 総合評価ポイントの合計 */
    readonly sum_global_point: "sg";
};
type UserFields = (typeof UserFields)[keyof UserSearchResult];
/**
 * 出力順序
 */
declare const Order: {
    /** ブックマーク数の多い順 */
    readonly FavoriteNovelCount: "favnovelcnt";
    /** レビュー数の多い順 */
    readonly ReviewCount: "favnovelcnt";
    /** 総合ポイントの高い順 */
    readonly HyokaDesc: "hyoka";
    /** 総合ポイントの低い順 */
    readonly HyokaAsc: "hyokaasc";
    /** 感想の多い順 */
    readonly ImpressionCount: "impressioncnt";
    /** 評価者数の多い順 */
    readonly HyokaCountDesc: "hyokacnt";
    /** 評価者数の少ない順 */
    readonly HyokaCountAsc: "hyokacntasc";
    /** 週間ユニークユーザの多い順 */
    readonly Weekly: "weekly";
    /** 小説本文の文字数が多い順 */
    readonly LengthDesc: "lengthdesc";
    /** 小説本文の文字数が少ない順 */
    readonly LengthAsc: "lengthasc";
    /** Nコードが新しい順 */
    readonly NCodeDesc: "ncodedesc";
    /** 新着更新順 */
    readonly New: "new";
    /** 古い順 */
    readonly Old: "old";
    /** 日間ポイントの高い順 */
    readonly DailyPoint: "dailypoint";
    /** 週間ポイントの高い順 */
    readonly WeeklyPoint: "weeklypoint";
    /** 月間ポイントの高い順 */
    readonly MonthlyPoint: "monthlypoint";
    /** 四半期ポイントの高い順 */
    readonly QuarterPoint: "quarterpoint";
    /** 年間ポイントの高い順 */
    readonly YearlyPoint: "yearlypoint";
    /** 初回掲載順 */
    readonly GeneralFirstUp: "generalfirstup";
};
type Order = (typeof Order)[keyof typeof Order];
/** R18掲載サイト */
declare const R18Site: {
    /** ノクターンノベルズ(男性向け) */
    readonly Nocturne: 1;
    /** ムーンライトノベルズ(女性向け) */
    readonly MoonLight: 2;
    /** ムーンライトノベルズ(BL) */
    readonly MoonLightBL: 3;
    /** ミッドナイトノベルズ(大人向け) */
    readonly Midnight: 4;
};
type R18Site = (typeof R18Site)[keyof typeof R18Site];
/** R18掲載サイト表記ヘルパー */
declare const R18SiteNotation: {
    readonly [K in R18Site]: string;
};
/** 大ジャンル */
declare const BigGenre: {
    /** 恋愛 */
    readonly Renai: 1;
    /** ファンタジー */
    readonly Fantasy: 2;
    /** 文芸 */
    readonly Bungei: 3;
    /** SF */
    readonly Sf: 4;
    /** その他 */
    readonly Sonota: 99;
    /** ノンジャンル */
    readonly NonGenre: 98;
};
type BigGenre = (typeof BigGenre)[keyof typeof BigGenre];
/** 大ジャンル表記ヘルパー */
declare const BigGenreNotation: {
    readonly [K in BigGenre]: string;
};
/** ジャンル */
declare const Genre: {
    /** 異世界〔恋愛〕*/
    readonly RenaiIsekai: 101;
    /** 現実世界〔恋愛〕*/
    readonly RenaiGenjitsusekai: 102;
    /** ハイファンタジー〔ファンタジー〕*/
    readonly FantasyHigh: 201;
    /** ローファンタジー〔ファンタジー〕*/
    readonly FantasyLow: 202;
    /** 純文学〔文芸〕*/
    readonly BungeiJyunbungei: 301;
    /** ヒューマンドラマ〔文芸〕*/
    readonly BungeiHumanDrama: 302;
    /** 歴史〔文芸〕*/
    readonly BungeiHistory: 303;
    /** 推理〔文芸〕*/
    readonly BungeiSuiri: 304;
    /** ホラー〔文芸〕*/
    readonly BungeiHorror: 305;
    /** アクション〔文芸〕*/
    readonly BungeiAction: 306;
    /** コメディー〔文芸〕*/
    readonly BungeiComedy: 307;
    /** VRゲーム〔SF〕*/
    readonly SfVrgame: 401;
    /** 宇宙〔SF〕*/
    readonly SfSpace: 402;
    /** 空想科学〔SF〕*/
    readonly SfKuusoukagaku: 403;
    /** パニック〔SF〕*/
    readonly SfPanic: 404;
    /** 童話〔その他〕*/
    readonly SonotaDouwa: 9901;
    /** 詩〔その他〕*/
    readonly SonotaShi: 9902;
    /** エッセイ〔その他〕*/
    readonly SonotaEssei: 9903;
    /** リプレイ〔その他〕*/
    readonly SonotaReplay: 9904;
    /** その他〔その他〕 */
    readonly SonotaSonota: 9999;
    /** ノンジャンル〔ノンジャンル〕*/
    readonly NonGenre: 9801;
};
type Genre = (typeof Genre)[keyof typeof Genre];
/** ジャンル表記ヘルパー */
declare const GenreNotation: {
    readonly [K in Genre]: string;
};
/** 文体指定 */
declare const BuntaiParam: {
    /** 字下げされておらず、連続改行が多い作品 */
    readonly NoJisageKaigyouOoi: 1;
    /** 字下げされていないが、改行数は平均な作品 */
    readonly NoJisageKaigyoHutsuu: 2;
    /** 字下げが適切だが、連続改行が多い作品 */
    readonly JisageKaigyoOoi: 4;
    /** 字下げが適切でかつ改行数も平均な作品 */
    readonly JisageKaigyoHutsuu: 6;
};
type BuntaiParam = (typeof BuntaiParam)[keyof typeof BuntaiParam];
/** 連載停止中指定 */
declare const StopParam: {
    /** 長期連載停止中を除きます */
    readonly NoStopping: 1;
    /** 長期連載停止中のみ取得します */
    readonly Stopping: 2;
};
type StopParam = (typeof StopParam)[keyof typeof StopParam];
/** 小説タイプ指定 */
declare const NovelTypeParam: {
    /** 短編 */
    readonly Short: "t";
    /** 連載中 */
    readonly RensaiNow: "r";
    /** 完結済連載小説 */
    readonly RensaiEnd: "er";
    /** すべての連載小説(連載中および完結済) */
    readonly Rensai: "re";
    /** 短編と完結済連載小説 */
    readonly ShortAndRensai: "ter";
};
type NovelTypeParam = (typeof NovelTypeParam)[keyof typeof NovelTypeParam];
/** 日付指定パラメータ */
declare const DateParam: {
    ThisWeek: string;
    LastWeek: string;
    SevenDays: string;
    ThisMonth: string;
    LastMonth: string;
};
type DateParam = (typeof DateParam)[keyof typeof DateParam];
declare const UserOrder: {
    /** ユーザIDの新しい順 */
    readonly New: "new";
    /** 小説投稿数の多い順 */
    readonly NovelCount: "novelcnt";
    /** レビュー投稿数の多い順 */
    readonly ReviewCount: "reviewcnt";
    /** 小説累計文字数の多い順 */
    readonly NovelLength: "novellength";
    /** 総合評価ポイントの合計の多い順 */
    readonly SumGlobalPoint: "sumglobalpoint";
    /** ユーザIDの古い順 */
    readonly Old: "old";
};
type UserOrder = (typeof UserOrder)[keyof typeof UserOrder];
type GzipLevel = 0 | 1 | 2 | 3 | 4 | 5;

/**
 * なろう小説API検索結果
 */
declare class NarouSearchResults<T, TKey extends keyof T> {
    /**
     * 検索結果数
     */
    allcount: number;
    /**
     * 結果表示上限数
     */
    limit: number;
    /**
     * 結果表示開始数
     */
    start: number;
    /**
     * 結果表示の現在ページ(=start/limit)
     */
    page: number;
    /**
     * 今回取得できた検索結果の数
     */
    length: number;
    /**
     * 検索結果
     */
    values: readonly Pick<T, TKey>[];
    /**
     * @constractor
     * @private
     */
    constructor([header, ...result]: [{
        allcount: number;
    }, ...Pick<T, TKey>[]], params: SearchParams | UserSearchParams);
}
/**
 * 小説情報
 * @see https://dev.syosetu.com/man/api/#output
 * @see https://dev.syosetu.com/xman/api/#output
 */
interface NarouSearchResult {
    /** 小説名 */
    title: string;
    /** Nコード */
    ncode: string;
    /** 作者のユーザID(数値) */
    userid: number;
    /** 作者名 */
    writer: string;
    /** 小説のあらすじ */
    story: string;
    /** 掲載サイト */
    nocgenre: R18Site;
    /** 大ジャンル */
    biggenre: BigGenre;
    /** ジャンル */
    genre: Genre;
    /** キーワード */
    keyword: string;
    /** 初回掲載日 YYYY-MM-DD HH:MM:SSの形式 */
    general_firstup: string;
    /** 最終掲載日 YYYY-MM-DD HH:MM:SSの形式 */
    general_lastup: string;
    /** 連載の場合は1、短編の場合は2 */
    novel_type: NovelType;
    /** 連載の場合は1、短編の場合は2 */
    noveltype: NovelType;
    /** 短編小説と完結済小説は0となっています。連載中は1です。 */
    end: End;
    /** 全掲載話数です。短編の場合は1です。 */
    general_all_no: number;
    /** 小説文字数です。スペースや改行は文字数としてカウントしません。 */
    length: number;
    /** 読了時間(分単位)です。読了時間は小説文字数÷500を切り上げした数値です。 */
    time: number;
    /** 長期連載中は1、それ以外は0です。 */
    isstop: BooleanNumber;
    /** 登録必須キーワードに「R15」が含まれる場合は1、それ以外は0です。 */
    isr15: BooleanNumber;
    /** 登録必須キーワードに「ボーイズラブ」が含まれる場合は1、それ以外は0です。 */
    isbl: BooleanNumber;
    /** 登録必須キーワードに「ガールズラブ」が含まれる場合は1、それ以外は0です。 */
    isgl: BooleanNumber;
    /** 登録必須キーワードに「残酷な描写あり」が含まれる場合は1、それ以外は0です。 */
    iszankoku: BooleanNumber;
    /** 登録必須キーワードに「異世界転生」が含まれる場合は1、それ以外は0です。 */
    istensei: BooleanNumber;
    /** 登録必須キーワードに「異世界転移」が含まれる場合は1、それ以外は0です。 */
    istenni: BooleanNumber;
    /** 総合得点(=(ブックマーク数×2)+評価点) */
    global_point: number;
    /**
     * 日間ポイント
     * ランキング集計時点から過去24時間以内で新たに登録されたブックマークや評価が対象
     */
    daily_point: number;
    /**
     * 週間ポイント
     * ランキング集計時点から過去7日以内で新たに登録されたブックマークや評価が対象
     */
    weekly_point: number;
    /**
     * 月間ポイント
     * ランキング集計時点から過去30日以内で新たに登録されたブックマークや評価が対象
     */
    monthly_point: number;
    /**
     * 四半期ポイント
     * ランキング集計時点から過去90日以内で新たに登録されたブックマークや評価が対象
     */
    quarter_point: number;
    /**
     * 年間ポイント
     * ランキング集計時点から過去365日以内で新たに登録されたブックマークや評価が対象
     */
    yearly_point: number;
    /** ブックマーク数 */
    fav_novel_cnt: number;
    /** 感想数 */
    impression_cnt: number;
    /** レビュー数 */
    review_cnt: number;
    /** 評価ポイント */
    all_point: number;
    /** 評価者数 */
    all_hyoka_cnt: number;
    /** 挿絵の数 */
    sasie_cnt: number;
    /**
     * 会話率
     * @see https://dev.syosetu.com/man/kaiwa/
     */
    kaiwaritu: number;
    /**
     * 小説の更新日時
     */
    novelupdated_at: string;
    /**
     * 最終更新日時
     * システム用で小説更新時とは関係ありません
     */
    updated_at: string;
    /** 週間ユニークユーザー数 */
    weekly_unique: number;
}
/**
 * ユーザ情報
 * @see https://dev.syosetu.com/man/userapi/#output
 */
interface UserSearchResult {
    /** ユーザID */
    userid: number;
    /** ユーザ名 */
    name: string;
    /** ユーザ名のフリガナ */
    yomikata: string;
    /**
     * ユーザ名のフリガナの頭文字
     * ひらがな以外の場合はnullまたは空文字となります。
     */
    name1st: string;
    /** 小説投稿数 */
    novel_cnt: number;
    /** レビュー投稿数 */
    review_cnt: number;
    /**
     * 小説累計文字数
     * スペースや改行は文字数としてカウントしません。
     */
    novel_length: number;
    /**
     * 総合評価ポイントの合計
     * 投稿済小説でそれぞれ獲得した総合評価ポイントの合計です。
     */
    sum_global_point: number;
}
/**
 * noveltype/novel_typeの値ヘルパー
 */
declare const NovelType: {
    /** 連載 */
    readonly Rensai: 1;
    /** 短編 */
    readonly Tanpen: 2;
};
type NovelType = typeof NovelType[keyof typeof NovelType];
/**
 * endの値ヘルパー
 */
declare const End: {
    /** 短編小説と完結済小説 */
    readonly KanketsuOrTanpen: 0;
    /** 連載中 */
    readonly Rensai: 1;
};
type End = typeof End[keyof typeof End];
type SearchResultFields<T extends Fields> = {
    [K in keyof typeof Fields]: typeof Fields[K] extends T ? K : never;
}[keyof typeof Fields];
type SearchResultOptionalFields<T extends OptionalFields> = {
    [K in keyof typeof OptionalFields]: typeof OptionalFields[K] extends T ? K : never;
}[keyof typeof OptionalFields];
type SearchResultR18Fields<T extends R18Fields> = {
    [K in keyof typeof R18Fields]: typeof R18Fields[K] extends T ? K : never;
}[keyof typeof R18Fields];
type UserSearchResultFields<T extends UserFields> = {
    [K in keyof typeof UserFields]: typeof UserFields[K] extends T ? K : never;
}[keyof typeof UserFields];
type PickedNarouSearchResult<T extends keyof NarouSearchResult> = Pick<NarouSearchResult, T>;

export { BooleanNumber as B, DateParam as D, End as E, Fields as F, Genre as G, NarouSearchResults as N, OptionalFields as O, type ParamsBase as P, RankingType as R, type SearchParams as S, type UserSearchParams as U, type ParamsBaseWithOrder as a, type RankingParams as b, type RankingHistoryParams as c, type SearchResultFieldNames as d, R18Fields as e, UserFields as f, Order as g, R18Site as h, R18SiteNotation as i, BigGenre as j, BigGenreNotation as k, GenreNotation as l, BuntaiParam as m, StopParam as n, NovelTypeParam as o, UserOrder as p, type GzipLevel as q, type NarouSearchResult as r, type UserSearchResult as s, NovelType as t, type SearchResultFields as u, type SearchResultOptionalFields as v, type SearchResultR18Fields as w, type UserSearchResultFields as x, type PickedNarouSearchResult as y };
