Object.defineProperty(exports, '__esModule', { value: true });
const require_index_common = require('./index.common-CYj0n0aU.cjs');
let zlib = require("zlib");
let util = require("util");

//#region src/util/unzipp.ts
const gunzipAsync = (0, util.promisify)(zlib.gunzip);
const decoder = new TextDecoder();
/**
* 圧縮されたJSONデータを解凍して解析します。
* 
* @param data - ArrayBuffer形式の圧縮データ
* @returns 解凍されたデータからパースされたJSONオブジェクト
* @throws {string} データが解凍できない、または解凍されたデータが有効なJSONでない場合、
*                  解凍されたデータの文字列表現をスローします。
* @throws {string} 解凍中にエラーが発生した場合、元のデータの文字列表現をスローします。
*/
async function unzipp(data) {
	try {
		const buffer = await gunzipAsync(data);
		try {
			return JSON.parse(decoder.decode(buffer));
		} catch {
			throw decoder.decode(buffer);
		}
	} catch (e) {
		if (typeof e === "string") throw e;
		throw decoder.decode(data);
	}
}

//#endregion
//#region src/narou-fetch.ts
/**
* なろう小説APIへのリクエストを実行する
*/
var NarouNovelFetch = class extends require_index_common.NarouNovel {
	/**
	* コンストラクタ
	* @param fetch fetch関数（デフォルトはネイティブのfetch）
	*/
	constructor(fetch$1) {
		super();
		this.fetch = fetch$1;
	}
	async execute(params, endpoint, options) {
		const query = {
			...params,
			out: "json"
		};
		if (query.gzip === void 0) query.gzip = 5;
		if (query.gzip === 0) delete query.gzip;
		const url = new URL(endpoint);
		Object.entries(query).forEach(([key, value]) => {
			if (value !== void 0) url.searchParams.append(key, value.toString());
		});
		const res = await (this.fetch ?? fetch)(url, options?.fetchOptions);
		if (!query.gzip) return await res.json();
		return await unzipp(await res.arrayBuffer());
	}
};

//#endregion
//#region src/index.ts
const narouNovelFetch = new NarouNovelFetch();
/**
* なろう小説 API で小説を検索する
* @param {string} [word] 検索ワード
* @returns {SearchBuilder}
* @see https://dev.syosetu.com/man/api/
*/
function search(word = "", api = narouNovelFetch) {
	const builder = new require_index_common.SearchBuilder({}, api);
	if (word != "") builder.word(word);
	return builder;
}
/**
* 18禁小説 API で小説を検索する
* @param {string} [word] 検索ワード
* @returns {SearchBuilder}
* @see https://dev.syosetu.com/xman/api/
*/
function searchR18(word = "", api = narouNovelFetch) {
	const builder = new require_index_common.SearchBuilderR18({}, api);
	if (word != "") builder.word(word);
	return builder;
}
/**
* なろうユーザ検索 API でユーザを検索する
* @param {string} [word] - 検索ワード
* @returns {UserSearchBuilder}
* @see https://dev.syosetu.com/man/userapi/
*/
function searchUser(word = "", api = narouNovelFetch) {
	const builder = new require_index_common.UserSearchBuilder({}, api);
	if (word != "") builder.word(word);
	return builder;
}
/**
* なろう小説ランキング API でランキングを取得する
* @returns {RankingBuilder}
* @see https://dev.syosetu.com/man/rankapi/
*/
function ranking(api = narouNovelFetch) {
	return new require_index_common.RankingBuilder({}, api);
}
/**
* なろう殿堂入り API でランキング履歴を取得する
* @param {string} ncode 小説のNコード
* @param {ExecuteOptions} [options] 実行オプション
* @param {NarouNovel} [api] API実行クラスのインスタンス
* @see https://dev.syosetu.com/man/rankinapi/
*/
async function rankingHistory(ncode, options, api = narouNovelFetch) {
	const result = await api.executeRankingHistory({ ncode }, options);
	if (Array.isArray(result)) return result.map(require_index_common.formatRankingHistory);
	else throw new Error(result);
}
var src_default = {
	search,
	searchR18,
	searchUser,
	ranking,
	rankingHistory
};

//#endregion
exports.BigGenre = require_index_common.BigGenre;
exports.BigGenreNotation = require_index_common.BigGenreNotation;
exports.BooleanNumber = require_index_common.BooleanNumber;
exports.BuntaiParam = require_index_common.BuntaiParam;
exports.DateParam = require_index_common.DateParam;
exports.End = require_index_common.End;
exports.Fields = require_index_common.Fields;
exports.Genre = require_index_common.Genre;
exports.GenreNotation = require_index_common.GenreNotation;
exports.NarouNovel = require_index_common.NarouNovel;
exports.NarouNovelFetch = NarouNovelFetch;
exports.NarouNovelJsonp = require_index_common.NarouNovelJsonp;
exports.NarouSearchResults = require_index_common.NarouSearchResults;
exports.NovelSearchBuilderBase = require_index_common.NovelSearchBuilderBase;
exports.NovelType = require_index_common.NovelType;
exports.NovelTypeParam = require_index_common.NovelTypeParam;
exports.OptionalFields = require_index_common.OptionalFields;
exports.Order = require_index_common.Order;
exports.R18Fields = require_index_common.R18Fields;
exports.R18Site = require_index_common.R18Site;
exports.R18SiteNotation = require_index_common.R18SiteNotation;
exports.RankingBuilder = require_index_common.RankingBuilder;
exports.RankingType = require_index_common.RankingType;
exports.SearchBuilder = require_index_common.SearchBuilder;
exports.SearchBuilderBase = require_index_common.SearchBuilderBase;
exports.SearchBuilderR18 = require_index_common.SearchBuilderR18;
exports.StopParam = require_index_common.StopParam;
exports.UserFields = require_index_common.UserFields;
exports.UserOrder = require_index_common.UserOrder;
exports.default = src_default;
exports.formatRankingHistory = require_index_common.formatRankingHistory;
exports.ranking = ranking;
exports.rankingHistory = rankingHistory;
exports.search = search;
exports.searchR18 = searchR18;
exports.searchUser = searchUser;
//# sourceMappingURL=index.cjs.map