import { A as NovelType, C as StopParam, D as NarouNovel, E as NarouNovelJsonp, O as End, S as RankingType, T as UserOrder, _ as OptionalFields, a as NovelSearchBuilderBase, b as R18Site, c as BigGenre, d as BuntaiParam, f as DateParam, g as NovelTypeParam, h as GenreNotation, i as RankingBuilder, k as NarouSearchResults, l as BigGenreNotation, m as Genre, n as SearchBuilderR18, o as SearchBuilder, p as Fields, r as formatRankingHistory, s as SearchBuilderBase, t as UserSearchBuilder, u as BooleanNumber, v as Order, w as UserFields, x as R18SiteNotation, y as R18Fields } from "./index.common-8lS2K_1Y.mjs";
import { gunzip } from "zlib";
import { promisify } from "util";

//#region src/util/unzipp.ts
const gunzipAsync = promisify(gunzip);
const decoder = new TextDecoder();
/**
* 圧縮されたJSONデータを解凍して解析します。
* 
* @param data - ArrayBuffer形式の圧縮データ
* @returns 解凍されたデータからパースされたJSONオブジェクト
* @throws {string} データが解凍できない、または解凍されたデータが有効なJSONでない場合、
*                  解凍されたデータの文字列表現をスローします。
* @throws {string} 解凍中にエラーが発生した場合、元のデータの文字列表現をスローします。
*/
async function unzipp(data) {
	try {
		const buffer = await gunzipAsync(data);
		try {
			return JSON.parse(decoder.decode(buffer));
		} catch {
			throw decoder.decode(buffer);
		}
	} catch (e) {
		if (typeof e === "string") throw e;
		throw decoder.decode(data);
	}
}

//#endregion
//#region src/narou-fetch.ts
/**
* なろう小説APIへのリクエストを実行する
*/
var NarouNovelFetch = class extends NarouNovel {
	/**
	* コンストラクタ
	* @param fetch fetch関数（デフォルトはネイティブのfetch）
	*/
	constructor(fetch$1) {
		super();
		this.fetch = fetch$1;
	}
	async execute(params, endpoint, options) {
		const query = {
			...params,
			out: "json"
		};
		if (query.gzip === void 0) query.gzip = 5;
		if (query.gzip === 0) delete query.gzip;
		const url = new URL(endpoint);
		Object.entries(query).forEach(([key, value]) => {
			if (value !== void 0) url.searchParams.append(key, value.toString());
		});
		const res = await (this.fetch ?? fetch)(url, options?.fetchOptions);
		if (!query.gzip) return await res.json();
		return await unzipp(await res.arrayBuffer());
	}
};

//#endregion
//#region src/index.ts
const narouNovelFetch = new NarouNovelFetch();
/**
* なろう小説 API で小説を検索する
* @param {string} [word] 検索ワード
* @returns {SearchBuilder}
* @see https://dev.syosetu.com/man/api/
*/
function search(word = "", api = narouNovelFetch) {
	const builder = new SearchBuilder({}, api);
	if (word != "") builder.word(word);
	return builder;
}
/**
* 18禁小説 API で小説を検索する
* @param {string} [word] 検索ワード
* @returns {SearchBuilder}
* @see https://dev.syosetu.com/xman/api/
*/
function searchR18(word = "", api = narouNovelFetch) {
	const builder = new SearchBuilderR18({}, api);
	if (word != "") builder.word(word);
	return builder;
}
/**
* なろうユーザ検索 API でユーザを検索する
* @param {string} [word] - 検索ワード
* @returns {UserSearchBuilder}
* @see https://dev.syosetu.com/man/userapi/
*/
function searchUser(word = "", api = narouNovelFetch) {
	const builder = new UserSearchBuilder({}, api);
	if (word != "") builder.word(word);
	return builder;
}
/**
* なろう小説ランキング API でランキングを取得する
* @returns {RankingBuilder}
* @see https://dev.syosetu.com/man/rankapi/
*/
function ranking(api = narouNovelFetch) {
	return new RankingBuilder({}, api);
}
/**
* なろう殿堂入り API でランキング履歴を取得する
* @param {string} ncode 小説のNコード
* @param {ExecuteOptions} [options] 実行オプション
* @param {NarouNovel} [api] API実行クラスのインスタンス
* @see https://dev.syosetu.com/man/rankinapi/
*/
async function rankingHistory(ncode, options, api = narouNovelFetch) {
	const result = await api.executeRankingHistory({ ncode }, options);
	if (Array.isArray(result)) return result.map(formatRankingHistory);
	else throw new Error(result);
}
var src_default = {
	search,
	searchR18,
	searchUser,
	ranking,
	rankingHistory
};

//#endregion
export { BigGenre, BigGenreNotation, BooleanNumber, BuntaiParam, DateParam, End, Fields, Genre, GenreNotation, NarouNovel, NarouNovelFetch, NarouNovelJsonp, NarouSearchResults, NovelSearchBuilderBase, NovelType, NovelTypeParam, OptionalFields, Order, R18Fields, R18Site, R18SiteNotation, RankingBuilder, RankingType, SearchBuilder, SearchBuilderBase, SearchBuilderR18, StopParam, UserFields, UserOrder, src_default as default, formatRankingHistory, ranking, rankingHistory, search, searchR18, searchUser };
//# sourceMappingURL=index.mjs.map