import type { Route } from '@/types';
import ofetch from '@/utils/ofetch';

import { getData } from './utils';

const ENDPOINT = 'https://api.aeonmedia.co/graphql';
const LIST_BY_TYPE = /* GraphQL */ `
query getAeonArticlesByType($type: [ArticleTypeEnum!], $sortField: ArticleSortEnum = published_at, $afterCursor: String, $tag: String) {
  articles(
    site: aeon
    type: $type
    status: [published]
    tag: $tag
    sort: {field: $sortField, order: desc}
    after: $afterCursor
    first: 12
  ) {
    nodes {
      slug
      ...aeonArticleCardFragment
    }
  }
}

fragment aeonArticleCardFragment on Article {
  id
  title
  slug
  type
  standfirstLong
  authors { name }
  image { url }
  primaryTopic { title }
  section { slug }
}
`;

export const route: Route = {
    path: '/:type',
    categories: ['new-media'],
    example: '/aeon/essays',
    parameters: {
        type: {
            description: 'Type',
            options: [
                { value: 'essays', label: 'Essays' },
                { value: 'videos', label: 'Videos' },
            ],
        },
    },
    features: {
        requireConfig: false,
        requirePuppeteer: false,
        antiCrawler: false,
        supportBT: false,
        supportPodcast: false,
        supportScihub: false,
    },
    radar: [
        {
            source: ['aeon.co/:type'],
        },
    ],
    name: 'Types',
    maintainers: ['emdoe'],
    handler,
    description: `Supported types: Essays and Videos.

  Compared to the official one, the RSS feed generated by RSSHub not only has more fine-grained options, but also eliminates pull quotes, which can't be easily distinguished from other paragraphs by any RSS reader, but only disrupt the reading flow. This feed also provides users with a bio of the author at the top.`,
};

async function handler(ctx) {
    const type = ctx.req.param('type');
    const capitalizedType = type.charAt(0).toUpperCase() + type.slice(1);

    const url = `https://aeon.co/${type}`;
    const response = await ofetch(ENDPOINT, {
        method: 'POST',
        body: {
            query: LIST_BY_TYPE,
            variables: { type: [type.slice(0, -1)], sortField: 'published_at' },
            operationName: 'getAeonArticlesByType',
        },
    });

    const list = response.data.articles.nodes.map((node) => ({
        title: node.title,
        description: node.standfirstLong,
        author: node.authors.map((author) => author.name).join(', '),
        link: `https://aeon.co/${type}/${node.slug}`,
        category: node.primaryTopic.title,
        image: node.image.url,
        type: node.type,
        section: node.section.slug,
        slug: node.slug,
    }));

    const items = await getData(list);

    return {
        title: `AEON | ${capitalizedType}`,
        link: url,
        item: items,
    };
}
