import { OPEN_BRAKET, CLOSE_BRAKET, SLASH } from '@bbob/plugin-helper';
// type, value, line, row, start pos, end pos
export const TYPE_ID = 't'; // 0;
export const VALUE_ID = 'v'; // 1;
export const LINE_ID = 'l'; // 3;
export const COLUMN_ID = 'r'; // 2;
export const START_POS_ID = 's'; // 4;
export const END_POS_ID = 'e'; // 5;
export const TYPE_WORD = 1; // 'word';
export const TYPE_TAG = 2; // 'tag';
export const TYPE_ATTR_NAME = 3; // 'attr-name';
export const TYPE_ATTR_VALUE = 4; // 'attr-value';
export const TYPE_SPACE = 5; // 'space';
export const TYPE_NEW_LINE = 6; // 'new-line';
const getTokenValue = (token)=>{
    if (token && typeof token[VALUE_ID] !== 'undefined') {
        return token[VALUE_ID];
    }
    return '';
};
const getTokenLine = (token)=>token && token[LINE_ID] || 0;
const getTokenColumn = (token)=>token && token[COLUMN_ID] || 0;
const getStartPosition = (token)=>token && token[START_POS_ID] || 0;
const getEndPosition = (token)=>token && token[END_POS_ID] || 0;
const isTextToken = (token)=>{
    if (token && typeof token[TYPE_ID] !== 'undefined') {
        return token[TYPE_ID] === TYPE_SPACE || token[TYPE_ID] === TYPE_NEW_LINE || token[TYPE_ID] === TYPE_WORD;
    }
    return false;
};
const isTagToken = (token)=>{
    if (token && typeof token[TYPE_ID] !== 'undefined') {
        return token[TYPE_ID] === TYPE_TAG;
    }
    return false;
};
const isTagEnd = (token)=>getTokenValue(token).charCodeAt(0) === SLASH.charCodeAt(0);
const isTagStart = (token)=>!isTagEnd(token);
const isAttrNameToken = (token)=>{
    if (token && typeof token[TYPE_ID] !== 'undefined') {
        return token[TYPE_ID] === TYPE_ATTR_NAME;
    }
    return false;
};
const isAttrValueToken = (token)=>{
    if (token && typeof token[TYPE_ID] !== 'undefined') {
        return token[TYPE_ID] === TYPE_ATTR_VALUE;
    }
    return false;
};
const getTagName = (token)=>{
    const value = getTokenValue(token);
    return isTagEnd(token) ? value.slice(1) : value;
};
const tokenToText = (token, openTag = OPEN_BRAKET, closeTag = CLOSE_BRAKET)=>{
    let text = openTag;
    text += getTokenValue(token);
    text += closeTag;
    return text;
};
/**
 * @export
 * @class Token
 */ class Token {
    get type() {
        return this[TYPE_ID];
    }
    isEmpty() {
        return this[TYPE_ID] === 0 || isNaN(this[TYPE_ID]);
    }
    isText() {
        return isTextToken(this);
    }
    isTag() {
        return isTagToken(this);
    }
    isAttrName() {
        return isAttrNameToken(this);
    }
    isAttrValue() {
        return isAttrValueToken(this);
    }
    isStart() {
        return isTagStart(this);
    }
    isEnd() {
        return isTagEnd(this);
    }
    getName() {
        return getTagName(this);
    }
    getValue() {
        return getTokenValue(this);
    }
    getLine() {
        return getTokenLine(this);
    }
    getColumn() {
        return getTokenColumn(this);
    }
    getStart() {
        return getStartPosition(this);
    }
    getEnd() {
        return getEndPosition(this);
    }
    toString({ openTag = OPEN_BRAKET, closeTag = CLOSE_BRAKET } = {}) {
        return tokenToText(this, openTag, closeTag);
    }
    constructor(type, value, row = 0, col = 0, start = 0, end = 0){
        this[LINE_ID] = row;
        this[COLUMN_ID] = col;
        this[TYPE_ID] = type || 0;
        this[VALUE_ID] = String(value);
        this[START_POS_ID] = start;
        this[END_POS_ID] = end;
    }
}
export { Token };
export default Token;
