import type { Snowflake } from '../../globals';
import type { APIApplication } from './application';
import type { APIChannel, ChannelType } from './channel';
import type { APIPartialEmoji } from './emoji';
import type { APIInteractionDataResolved, APIMessageInteraction, APIMessageInteractionMetadata } from './interactions';
import type { APIRole } from './permissions';
import type { APIPoll } from './poll';
import type { APISticker, APIStickerItem } from './sticker';
import type { APIUser } from './user';
export interface APIMessageMentions {
    /**
     * Users specifically mentioned in the message
     *
     * The `member` field is only present in `MESSAGE_CREATE` and `MESSAGE_UPDATE` events
     * from text-based guild channels
     *
     * @see {@link https://discord.com/developers/docs/resources/user#user-object}
     * @see {@link https://discord.com/developers/docs/resources/guild#guild-member-object}
     */
    mentions: APIUser[];
}
/**
 * @see {@link https://discord.com/developers/docs/resources/message#message-object-message-structure}
 */
export interface APIBaseMessageNoChannel {
    /**
     * ID of the message
     */
    id: Snowflake;
    /**
     * The author of this message (only a valid user in the case where the message is generated by a user or bot user)
     *
     * If the message is generated by a webhook, the author object corresponds to the webhook's id,
     * username, and avatar. You can tell if a message is generated by a webhook by checking for the `webhook_id` property
     *
     * @see {@link https://discord.com/developers/docs/resources/user#user-object}
     */
    author: APIUser;
    /**
     * Contents of the message
     *
     * The `MESSAGE_CONTENT` privileged gateway intent is required for verified applications to receive a non-empty value from this field
     *
     * In the Discord Developers Portal, you need to enable the toggle of this intent of your application in **Bot \> Privileged Gateway Intents**.
     * You also need to specify the intent bit value (`1 << 15`) if you are connecting to the gateway
     *
     * @see {@link https://support-dev.discord.com/hc/articles/6207308062871}
     */
    content: string;
    /**
     * When this message was sent
     */
    timestamp: string;
    /**
     * When this message was edited (or null if never)
     */
    edited_timestamp: string | null;
    /**
     * Whether this was a TTS message
     */
    tts: boolean;
    /**
     * Whether this message mentions everyone
     */
    mention_everyone: boolean;
    /**
     * Roles specifically mentioned in this message
     *
     * @see {@link https://discord.com/developers/docs/topics/permissions#role-object}
     */
    mention_roles: APIRole['id'][];
    /**
     * Channels specifically mentioned in this message
     *
     * Not all channel mentions in a message will appear in `mention_channels`.
     * - Only textual channels that are visible to everyone in a lurkable guild will ever be included
     * - Only crossposted messages (via Channel Following) currently include `mention_channels` at all
     *
     * If no mentions in the message meet these requirements, this field will not be sent
     *
     * @see {@link https://discord.com/developers/docs/resources/message#channel-mention-object}
     */
    mention_channels?: APIChannelMention[];
    /**
     * Any attached files
     *
     * @see {@link https://discord.com/developers/docs/resources/message#attachment-object-attachment-structure}
     *
     * The `MESSAGE_CONTENT` privileged gateway intent is required for verified applications to receive a non-empty value from this field
     *
     * In the Discord Developers Portal, you need to enable the toggle of this intent of your application in **Bot \> Privileged Gateway Intents**.
     * You also need to specify the intent bit value (`1 << 15`) if you are connecting to the gateway
     * @see {@link https://support-dev.discord.com/hc/articles/6207308062871}
     */
    attachments: APIAttachment[];
    /**
     * Any embedded content
     *
     * @see {@link https://discord.com/developers/docs/resources/message#embed-object}
     *
     * The `MESSAGE_CONTENT` privileged gateway intent is required for verified applications to receive a non-empty value from this field
     *
     * In the Discord Developers Portal, you need to enable the toggle of this intent of your application in **Bot \> Privileged Gateway Intents**.
     * You also need to specify the intent bit value (`1 << 15`) if you are connecting to the gateway
     * @see {@link https://support-dev.discord.com/hc/articles/6207308062871}
     */
    embeds: APIEmbed[];
    /**
     * Reactions to the message
     *
     * @see {@link https://discord.com/developers/docs/resources/message#reaction-object}
     */
    reactions?: APIReaction[];
    /**
     * A nonce that can be used for optimistic message sending (up to 25 characters)
     *
     * **You will not receive this from further fetches. This is received only once from a `MESSAGE_CREATE`
     * event to ensure it got sent**
     */
    nonce?: number | string;
    /**
     * Whether this message is pinned
     */
    pinned: boolean;
    /**
     * If the message is generated by a webhook, this is the webhook's id
     */
    webhook_id?: Snowflake;
    /**
     * Type of message
     *
     * @see {@link https://discord.com/developers/docs/resources/message#message-object-message-types}
     */
    type: MessageType;
    /**
     * Sent with Rich Presence-related chat embeds
     *
     * @see {@link https://discord.com/developers/docs/resources/message#message-object-message-activity-structure}
     */
    activity?: APIMessageActivity;
    /**
     * Sent with Rich Presence-related chat embeds
     *
     * @see {@link https://discord.com/developers/docs/resources/application#application-object}
     */
    application?: Partial<APIApplication>;
    /**
     * If the message is a response to an Interaction, this is the id of the interaction's application
     */
    application_id?: Snowflake;
    /**
     * Reference data sent with crossposted messages, replies, pins, and thread starter messages
     *
     * @see {@link https://discord.com/developers/docs/resources/message#message-reference-structure}
     */
    message_reference?: APIMessageReference;
    /**
     * Message flags combined as a bitfield
     *
     * @see {@link https://discord.com/developers/docs/resources/message#message-object-message-flags}
     * @see {@link https://en.wikipedia.org/wiki/Bit_field}
     */
    flags?: MessageFlags;
    /**
     * The message associated with the `message_reference`
     *
     * This field is only returned for messages with a `type` of `19` (REPLY).
     *
     * If the message is a reply but the `referenced_message` field is not present,
     * the backend did not attempt to fetch the message that was being replied to,
     * so its state is unknown.
     *
     * If the field exists but is `null`, the referenced message was deleted
     *
     * @see {@link https://discord.com/developers/docs/resources/message#message-object}
     */
    referenced_message?: APIMessage | null;
    /**
     * Sent if the message is sent as a result of an interaction
     */
    interaction_metadata?: APIMessageInteractionMetadata;
    /**
     * Sent if the message is a response to an Interaction
     *
     * @deprecated In favor of `interaction_metadata`
     */
    interaction?: APIMessageInteraction;
    /**
     * Sent if a thread was started from this message
     */
    thread?: APIChannel;
    /**
     * Sent if the message contains components like buttons, action rows, or other interactive components
     *
     * The `MESSAGE_CONTENT` privileged gateway intent is required for verified applications to receive a non-empty value from this field
     *
     * In the Discord Developers Portal, you need to enable the toggle of this intent of your application in **Bot \> Privileged Gateway Intents**.
     * You also need to specify the intent bit value (`1 << 15`) if you are connecting to the gateway
     *
     * @see {@link https://support-dev.discord.com/hc/articles/6207308062871}
     */
    components?: APIMessageTopLevelComponent[];
    /**
     * Sent if the message contains stickers
     *
     * @see {@link https://discord.com/developers/docs/resources/sticker#sticker-item-object}
     */
    sticker_items?: APIStickerItem[];
    /**
     * The stickers sent with the message
     *
     * @see {@link https://discord.com/developers/docs/resources/sticker#sticker-object}
     * @deprecated Use {@link APIBaseMessageNoChannel.sticker_items} instead
     */
    stickers?: APISticker[];
    /**
     * A generally increasing integer (there may be gaps or duplicates) that represents the approximate position of the message in a thread
     *
     * It can be used to estimate the relative position of the message in a thread in company with `total_message_sent` on parent thread
     */
    position?: number;
    /**
     * Data of the role subscription purchase or renewal that prompted this `ROLE_SUBSCRIPTION_PURCHASE` message
     */
    role_subscription_data?: APIMessageRoleSubscriptionData;
    /**
     * Data for users, members, channels, and roles in the message's auto-populated select menus
     *
     * @see {@link https://discord.com/developers/docs/interactions/receiving-and-responding#interaction-object-resolved-data-structure}
     */
    resolved?: APIInteractionDataResolved;
    /**
     * A poll!
     *
     * The `MESSAGE_CONTENT` privileged gateway intent is required for verified applications to receive a non-empty value from this field
     *
     * In the Discord Developers Portal, you need to enable the toggle of this intent of your application in **Bot \> Privileged Gateway Intents**.
     * You also need to specify the intent bit value (`1 << 15`) if you are connecting to the gateway
     *
     * @see {@link https://support-dev.discord.com/hc/articles/6207308062871}
     */
    poll?: APIPoll;
    /**
     * The message associated with the message_reference. This is a minimal subset of fields in a message (e.g. author is excluded.)
     */
    message_snapshots?: APIMessageSnapshot[];
    /**
     * The call associated with the message
     */
    call?: APIMessageCall;
}
/**
 * @see {@link https://discord.com/developers/docs/resources/message#message-object-message-structure}
 */
export interface APIBaseMessage extends APIBaseMessageNoChannel {
    /**
     * ID of the channel the message was sent in
     */
    channel_id: Snowflake;
}
/**
 * @see {@link https://discord.com/developers/docs/resources/message#message-object-message-structure}
 */
export interface APIMessage extends APIBaseMessage, APIMessageMentions {
}
/**
 * @see {@link https://discord.com/developers/docs/resources/message#message-object-message-types}
 */
export declare enum MessageType {
    Default = 0,
    RecipientAdd = 1,
    RecipientRemove = 2,
    Call = 3,
    ChannelNameChange = 4,
    ChannelIconChange = 5,
    ChannelPinnedMessage = 6,
    UserJoin = 7,
    GuildBoost = 8,
    GuildBoostTier1 = 9,
    GuildBoostTier2 = 10,
    GuildBoostTier3 = 11,
    ChannelFollowAdd = 12,
    GuildDiscoveryDisqualified = 14,
    GuildDiscoveryRequalified = 15,
    GuildDiscoveryGracePeriodInitialWarning = 16,
    GuildDiscoveryGracePeriodFinalWarning = 17,
    ThreadCreated = 18,
    Reply = 19,
    ChatInputCommand = 20,
    ThreadStarterMessage = 21,
    GuildInviteReminder = 22,
    ContextMenuCommand = 23,
    AutoModerationAction = 24,
    RoleSubscriptionPurchase = 25,
    InteractionPremiumUpsell = 26,
    StageStart = 27,
    StageEnd = 28,
    StageSpeaker = 29,
    /**
     * @unstable https://github.com/discord/discord-api-docs/pull/5927#discussion_r1107678548
     */
    StageRaiseHand = 30,
    StageTopic = 31,
    GuildApplicationPremiumSubscription = 32,
    GuildIncidentAlertModeEnabled = 36,
    GuildIncidentAlertModeDisabled = 37,
    GuildIncidentReportRaid = 38,
    GuildIncidentReportFalseAlarm = 39,
    PurchaseNotification = 44,
    PollResult = 46
}
/**
 * @see {@link https://discord.com/developers/docs/resources/message#message-object-message-activity-structure}
 */
export interface APIMessageActivity {
    /**
     * Type of message activity
     *
     * @see {@link https://discord.com/developers/docs/resources/message#message-object-message-activity-types}
     */
    type: MessageActivityType;
    /**
     * `party_id` from a Rich Presence event
     *
     * @see {@link https://discord.com/developers/docs/rich-presence/how-to#updating-presence-update-presence-payload-fields}
     */
    party_id?: string;
}
/**
 * @see {@link https://discord.com/developers/docs/resources/message#message-reference-structure}
 */
export interface APIMessageReference {
    /**
     * Type of reference
     */
    type?: MessageReferenceType;
    /**
     * ID of the originating message
     */
    message_id?: Snowflake;
    /**
     * ID of the originating message's channel
     */
    channel_id: Snowflake;
    /**
     * ID of the originating message's guild
     */
    guild_id?: Snowflake;
}
/**
 * @see {@link https://discord.com/developers/docs/resources/message#message-object-message-activity-types}
 */
export declare enum MessageActivityType {
    Join = 1,
    Spectate = 2,
    Listen = 3,
    JoinRequest = 5
}
/**
 * @see {@link https://discord.com/developers/docs/resources/message#message-reference-types}
 */
export declare enum MessageReferenceType {
    /**
     * A standard reference used by replies
     */
    Default = 0,
    /**
     * Reference used to point to a message at a point in time
     */
    Forward = 1
}
/**
 * @see {@link https://discord.com/developers/docs/resources/message#message-object-message-flags}
 */
export declare enum MessageFlags {
    /**
     * This message has been published to subscribed channels (via Channel Following)
     */
    Crossposted = 1,
    /**
     * This message originated from a message in another channel (via Channel Following)
     */
    IsCrosspost = 2,
    /**
     * Do not include any embeds when serializing this message
     */
    SuppressEmbeds = 4,
    /**
     * The source message for this crosspost has been deleted (via Channel Following)
     */
    SourceMessageDeleted = 8,
    /**
     * This message came from the urgent message system
     */
    Urgent = 16,
    /**
     * This message has an associated thread, which shares its id
     */
    HasThread = 32,
    /**
     * This message is only visible to the user who invoked the Interaction
     */
    Ephemeral = 64,
    /**
     * This message is an Interaction Response and the bot is "thinking"
     */
    Loading = 128,
    /**
     * This message failed to mention some roles and add their members to the thread
     */
    FailedToMentionSomeRolesInThread = 256,
    /**
     * @unstable This message flag is currently not documented by Discord but has a known value which we will try to keep up to date.
     */
    ShouldShowLinkNotDiscordWarning = 1024,
    /**
     * This message will not trigger push and desktop notifications
     */
    SuppressNotifications = 4096,
    /**
     * This message is a voice message
     */
    IsVoiceMessage = 8192,
    /**
     * This message has a snapshot (via Message Forwarding)
     */
    HasSnapshot = 16384,
    /**
     * Allows you to create fully component-driven messages
     *
     * @see {@link https://discord.com/developers/docs/components/overview}
     */
    IsComponentsV2 = 32768
}
/**
 * @see {@link https://discord.com/developers/docs/resources/message#message-call-object-message-call-object-structure}
 */
export interface APIMessageCall {
    /**
     * Array of user ids that participated in the call
     */
    participants: Snowflake[];
    /**
     * ISO8601 timestamp when the call ended
     */
    ended_timestamp?: string | null;
}
/**
 * @see {@link https://discord.com/developers/docs/resources/message#role-subscription-data-object-role-subscription-data-object-structure}
 */
export interface APIMessageRoleSubscriptionData {
    /**
     * The id of the SKU and listing the user is subscribed to
     */
    role_subscription_listing_id: Snowflake;
    /**
     * The name of the tier the user is subscribed to
     */
    tier_name: string;
    /**
     * The number of months the user has been subscribed for
     */
    total_months_subscribed: number;
    /**
     * Whether this notification is for a renewal
     */
    is_renewal: boolean;
}
/**
 * @see {@link https://discord.com/developers/docs/resources/message#reaction-object-reaction-structure}
 */
export interface APIReaction {
    /**
     * Total number of times this emoji has been used to react (including super reacts)
     */
    count: number;
    /**
     * An object detailing the individual reaction counts for different types of reactions
     */
    count_details: APIReactionCountDetails;
    /**
     * Whether the current user reacted using this emoji
     */
    me: boolean;
    /**
     * Whether the current user super-reacted using this emoji
     */
    me_burst: boolean;
    /**
     * Emoji information
     *
     * @see {@link https://discord.com/developers/docs/resources/emoji#emoji-object}
     */
    emoji: APIPartialEmoji;
    /**
     * Hexadecimal colors used for this super reaction
     */
    burst_colors: string[];
}
/**
 * @see {@link https://discord.com/developers/docs/resources/message#reaction-count-details-object-reaction-count-details-structure}
 */
export interface APIReactionCountDetails {
    /**
     * Count of super reactions
     */
    burst: number;
    /**
     * Count of normal reactions
     */
    normal: number;
}
/**
 * @see {@link https://discord.com/developers/docs/resources/message#embed-object-embed-structure}
 *
 * Length limit: 6000 characters
 */
export interface APIEmbed {
    /**
     * Title of embed
     *
     * Length limit: 256 characters
     */
    title?: string;
    /**
     * Type of embed (always "rich" for webhook embeds)
     */
    type?: EmbedType;
    /**
     * Description of embed
     *
     * Length limit: 4096 characters
     */
    description?: string;
    /**
     * URL of embed
     */
    url?: string;
    /**
     * Timestamp of embed content
     */
    timestamp?: string;
    /**
     * Color code of the embed
     */
    color?: number;
    /**
     * Footer information
     *
     * @see {@link https://discord.com/developers/docs/resources/message#embed-object-embed-footer-structure}
     */
    footer?: APIEmbedFooter;
    /**
     * Image information
     *
     * @see {@link https://discord.com/developers/docs/resources/message#embed-object-embed-image-structure}
     */
    image?: APIEmbedImage;
    /**
     * Thumbnail information
     *
     * @see {@link https://discord.com/developers/docs/resources/message#embed-object-embed-thumbnail-structure}
     */
    thumbnail?: APIEmbedThumbnail;
    /**
     * Video information
     *
     * @see {@link https://discord.com/developers/docs/resources/message#embed-object-embed-video-structure}
     */
    video?: APIEmbedVideo;
    /**
     * Provider information
     *
     * @see {@link https://discord.com/developers/docs/resources/message#embed-object-embed-provider-structure}
     */
    provider?: APIEmbedProvider;
    /**
     * Author information
     *
     * @see {@link https://discord.com/developers/docs/resources/message#embed-object-embed-author-structure}
     */
    author?: APIEmbedAuthor;
    /**
     * Fields information
     *
     * Length limit: 25 field objects
     *
     * @see {@link https://discord.com/developers/docs/resources/message#embed-object-embed-field-structure}
     */
    fields?: APIEmbedField[];
}
/**
 * @see {@link https://discord.com/developers/docs/resources/message#embed-object-embed-types}
 */
export declare enum EmbedType {
    /**
     * Generic embed rendered from embed attributes
     */
    Rich = "rich",
    /**
     * Image embed
     */
    Image = "image",
    /**
     * Video embed
     */
    Video = "video",
    /**
     * Animated gif image embed rendered as a video embed
     */
    GIFV = "gifv",
    /**
     * Article embed
     */
    Article = "article",
    /**
     * Link embed
     */
    Link = "link",
    /**
     * Auto moderation alert embed
     *
     * @unstable This embed type is currently not documented by Discord, but it is returned in the auto moderation system messages.
     */
    AutoModerationMessage = "auto_moderation_message",
    /**
     * Poll result embed
     */
    PollResult = "poll_result"
}
/**
 * @see {@link https://discord.com/developers/docs/resources/message#embed-object-embed-thumbnail-structure}
 */
export interface APIEmbedThumbnail {
    /**
     * Source url of thumbnail (only supports http(s) and attachments)
     */
    url: string;
    /**
     * A proxied url of the thumbnail
     */
    proxy_url?: string;
    /**
     * Height of thumbnail
     */
    height?: number;
    /**
     * Width of thumbnail
     */
    width?: number;
}
/**
 * @see {@link https://discord.com/developers/docs/resources/message#embed-object-embed-video-structure}
 */
export interface APIEmbedVideo {
    /**
     * Source url of video
     */
    url?: string;
    /**
     * A proxied url of the video
     */
    proxy_url?: string;
    /**
     * Height of video
     */
    height?: number;
    /**
     * Width of video
     */
    width?: number;
}
/**
 * @see {@link https://discord.com/developers/docs/resources/message#embed-object-embed-image-structure}
 */
export interface APIEmbedImage {
    /**
     * Source url of image (only supports http(s) and attachments)
     */
    url: string;
    /**
     * A proxied url of the image
     */
    proxy_url?: string;
    /**
     * Height of image
     */
    height?: number;
    /**
     * Width of image
     */
    width?: number;
}
/**
 * @see {@link https://discord.com/developers/docs/resources/message#embed-object-embed-provider-structure}
 */
export interface APIEmbedProvider {
    /**
     * Name of provider
     */
    name?: string;
    /**
     * URL of provider
     */
    url?: string;
}
/**
 * @see {@link https://discord.com/developers/docs/resources/message#embed-object-embed-author-structure}
 */
export interface APIEmbedAuthor {
    /**
     * Name of author
     *
     * Length limit: 256 characters
     */
    name: string;
    /**
     * URL of author
     */
    url?: string;
    /**
     * URL of author icon (only supports http(s) and attachments)
     */
    icon_url?: string;
    /**
     * A proxied url of author icon
     */
    proxy_icon_url?: string;
}
/**
 * @see {@link https://discord.com/developers/docs/resources/message#embed-object-embed-footer-structure}
 */
export interface APIEmbedFooter {
    /**
     * Footer text
     *
     * Length limit: 2048 characters
     */
    text: string;
    /**
     * URL of footer icon (only supports http(s) and attachments)
     */
    icon_url?: string;
    /**
     * A proxied url of footer icon
     */
    proxy_icon_url?: string;
}
/**
 * @see {@link https://discord.com/developers/docs/resources/message#embed-object-embed-field-structure}
 */
export interface APIEmbedField {
    /**
     * Name of the field
     *
     * Length limit: 256 characters
     */
    name: string;
    /**
     * Value of the field
     *
     * Length limit: 1024 characters
     */
    value: string;
    /**
     * Whether or not this field should display inline
     */
    inline?: boolean;
}
/**
 * @see {@link https://discord.com/developers/docs/resources/message#attachment-object-attachment-structure}
 */
export interface APIAttachment {
    /**
     * Attachment id
     */
    id: Snowflake;
    /**
     * Name of file attached
     */
    filename: string;
    /**
     * The original filename of the upload with special characters preserved
     *
     * This will be present when the filename contains special characters (e.g. Cyrillic),
     * in which case the `filename` field will be a sanitized version without those characters
     */
    title?: string;
    /**
     * Description for the file
     */
    description?: string;
    /**
     * The attachment's media type
     *
     * @see {@link https://en.wikipedia.org/wiki/Media_type}
     */
    content_type?: string;
    /**
     * Size of file in bytes
     */
    size: number;
    /**
     * Source url of file
     */
    url: string;
    /**
     * A proxied url of file
     */
    proxy_url: string;
    /**
     * Height of file (if image)
     */
    height?: number | null;
    /**
     * Width of file (if image)
     */
    width?: number | null;
    /**
     * Whether this attachment is ephemeral
     */
    ephemeral?: boolean;
    /**
     * The duration of the audio file (currently for voice messages)
     */
    duration_secs?: number;
    /**
     * Base64 encoded bytearray representing a sampled waveform (currently for voice messages)
     */
    waveform?: string;
    /**
     * Attachment flags combined as a bitfield
     */
    flags?: AttachmentFlags;
}
/**
 * @see {@link https://discord.com/developers/docs/resources/message#attachment-object-attachment-structure-attachment-flags}
 */
export declare enum AttachmentFlags {
    /**
     * This attachment has been edited using the remix feature on mobile
     */
    IsRemix = 4
}
/**
 * @see {@link https://discord.com/developers/docs/resources/message#channel-mention-object-channel-mention-structure}
 */
export interface APIChannelMention {
    /**
     * ID of the channel
     */
    id: Snowflake;
    /**
     * ID of the guild containing the channel
     */
    guild_id: Snowflake;
    /**
     * The type of channel
     *
     * @see {@link https://discord.com/developers/docs/resources/message#channel-object-channel-types}
     */
    type: ChannelType;
    /**
     * The name of the channel
     */
    name: string;
}
/**
 * @see {@link https://discord.com/developers/docs/resources/message#allowed-mentions-object-allowed-mention-types}
 */
export declare enum AllowedMentionsTypes {
    /**
     * Controls `@everyone` and `@here` mentions
     */
    Everyone = "everyone",
    /**
     * Controls role mentions
     */
    Role = "roles",
    /**
     * Controls user mentions
     */
    User = "users"
}
/**
 * @see {@link https://discord.com/developers/docs/resources/message#allowed-mentions-object-allowed-mentions-structure}
 */
export interface APIAllowedMentions {
    /**
     * An array of allowed mention types to parse from the content
     *
     * @see {@link https://discord.com/developers/docs/resources/message#allowed-mentions-object-allowed-mention-types}
     */
    parse?: AllowedMentionsTypes[];
    /**
     * Array of role_ids to mention (Max size of 100)
     */
    roles?: Snowflake[];
    /**
     * Array of user_ids to mention (Max size of 100)
     */
    users?: Snowflake[];
    /**
     * For replies, whether to mention the author of the message being replied to
     *
     * @defaultValue `false`
     */
    replied_user?: boolean;
}
/**
 * @see {@link https://discord.com/developers/docs/components/reference#anatomy-of-a-component}
 */
export interface APIBaseComponent<T extends ComponentType> {
    /**
     * The type of the component
     */
    type: T;
    /**
     * 32 bit integer used as an optional identifier for component
     *
     * The id field is optional and is used to identify components in the response from an interaction that aren't interactive components. The id must be unique within the message and is generated sequentially if left empty. Generation of ids won't use another id that exists in the message if you have one defined for another component.
     */
    id?: number;
}
/**
 * @see {@link https://discord.com/developers/docs/components/reference#component-object-component-types}
 */
export declare enum ComponentType {
    /**
     * Container to display a row of interactive components
     */
    ActionRow = 1,
    /**
     * Button component
     */
    Button = 2,
    /**
     * Select menu for picking from defined text options
     */
    StringSelect = 3,
    /**
     * Text Input component
     */
    TextInput = 4,
    /**
     * Select menu for users
     */
    UserSelect = 5,
    /**
     * Select menu for roles
     */
    RoleSelect = 6,
    /**
     * Select menu for users and roles
     */
    MentionableSelect = 7,
    /**
     * Select menu for channels
     */
    ChannelSelect = 8,
    /**
     * Container to display text alongside an accessory component
     */
    Section = 9,
    /**
     * Markdown text
     */
    TextDisplay = 10,
    /**
     * Small image that can be used as an accessory
     */
    Thumbnail = 11,
    /**
     * Display images and other media
     */
    MediaGallery = 12,
    /**
     * Displays an attached file
     */
    File = 13,
    /**
     * Component to add vertical padding between other components
     */
    Separator = 14,
    /**
     * @unstable This component type is currently not documented by Discord but has a known value which we will try to keep up to date.
     */
    ContentInventoryEntry = 16,
    /**
     * Container that visually groups a set of components
     */
    Container = 17,
    /**
     * Container associating a label and description with a component
     */
    Label = 18,
    /**
     * Component for uploading files
     */
    FileUpload = 19,
    /**
     * Single-choice set of radio group option
     *
     * @unstable This component is not publicly released and is currently in preview.
     */
    RadioGroup = 21,
    /**
     * Multi-select group of checkboxes
     *
     * @unstable This component is not publicly released and is currently in preview.
     */
    CheckboxGroup = 22,
    /**
     * Single checkbox for binary choice
     *
     * @unstable This component is not publicly released and is currently in preview.
     */
    Checkbox = 23,
    /**
     * Select menu for picking from defined text options
     *
     * @deprecated This is the old name for {@link ComponentType.StringSelect}
     */
    SelectMenu = 3
}
/**
 * An Action Row is a top-level layout component used in messages. Use in modals is deprecated.
 *
 * @see {@link https://discord.com/developers/docs/components/reference#action-row}
 */
export interface APIActionRowComponent<T extends APIComponentInActionRow> extends APIBaseComponent<ComponentType.ActionRow> {
    /**
     * The components in the ActionRow
     */
    components: T[];
}
export interface APIButtonBase<Style extends ButtonStyle> extends APIBaseComponent<ComponentType.Button> {
    /**
     * The style of the button
     */
    style: Style;
    /**
     * The status of the button
     */
    disabled?: boolean;
}
/**
 * @see {@link https://discord.com/developers/docs/components/reference#button}
 */
export interface APIButtonComponentBase<Style extends ButtonStyle> extends APIButtonBase<Style> {
    /**
     * The label to be displayed on the button
     */
    label?: string;
    /**
     * The emoji to display to the left of the text
     */
    emoji?: APIMessageComponentEmoji;
}
export interface APIMessageComponentEmoji {
    /**
     * Emoji id
     */
    id?: Snowflake;
    /**
     * Emoji name
     */
    name?: string;
    /**
     * Whether this emoji is animated
     */
    animated?: boolean;
}
/**
 * @see {@link https://discord.com/developers/docs/components/reference#button}
 */
export interface APIButtonComponentWithCustomId extends APIButtonComponentBase<ButtonStyle.Danger | ButtonStyle.Primary | ButtonStyle.Secondary | ButtonStyle.Success> {
    /**
     * The custom_id to be sent in the interaction when clicked
     */
    custom_id: string;
}
/**
 * @see {@link https://discord.com/developers/docs/components/reference#button}
 */
export interface APIButtonComponentWithURL extends APIButtonComponentBase<ButtonStyle.Link> {
    /**
     * The URL to direct users to when clicked for Link buttons
     */
    url: string;
}
/**
 * @see {@link https://discord.com/developers/docs/components/reference#button}
 */
export interface APIButtonComponentWithSKUId extends APIButtonBase<ButtonStyle.Premium> {
    /**
     * The id for a purchasable SKU
     */
    sku_id: Snowflake;
}
/**
 * A Button is an interactive component that can only be used in messages. It creates clickable elements that users can interact with, sending an interaction to your app when clicked.
 *
 * Buttons must be placed inside an Action Row or a Section's accessory field.
 *
 * @see {@link https://discord.com/developers/docs/components/reference#button}
 */
export type APIButtonComponent = APIButtonComponentWithCustomId | APIButtonComponentWithSKUId | APIButtonComponentWithURL;
/**
 * @see {@link https://discord.com/developers/docs/components/reference#button-button-styles}
 */
export declare enum ButtonStyle {
    /**
     * The most important or recommended action in a group of options
     */
    Primary = 1,
    /**
     * Alternative or supporting actions
     */
    Secondary = 2,
    /**
     * Positive confirmation or completion actions
     */
    Success = 3,
    /**
     * An action with irreversible consequences
     */
    Danger = 4,
    /**
     * Navigates to a URL
     */
    Link = 5,
    /**
     * Purchase
     */
    Premium = 6
}
/**
 * @see {@link https://discord.com/developers/docs/components/reference#text-input-text-input-styles}
 */
export declare enum TextInputStyle {
    /**
     * Single-line input
     */
    Short = 1,
    /**
     * Multi-line input
     */
    Paragraph = 2
}
/**
 * @see {@link https://discord.com/developers/docs/components/reference}
 */
export interface APIBaseSelectMenuComponent<T extends ComponentType.ChannelSelect | ComponentType.MentionableSelect | ComponentType.RoleSelect | ComponentType.StringSelect | ComponentType.UserSelect> extends APIBaseComponent<T> {
    /**
     * A developer-defined identifier for the select menu, max 100 characters
     */
    custom_id: string;
    /**
     * Custom placeholder text if nothing is selected, max 150 characters
     */
    placeholder?: string;
    /**
     * The minimum number of items that must be chosen; min 0, max 25
     *
     * @defaultValue `1`
     */
    min_values?: number;
    /**
     * The maximum number of items that can be chosen; max 25
     *
     * @defaultValue `1`
     */
    max_values?: number;
    /**
     * Disable the select
     *
     * @defaultValue `false`
     */
    disabled?: boolean;
    /**
     * Whether the component is required to answer in a modal.
     *
     * @defaultValue `true`
     */
    required?: boolean;
}
/**
 * @see {@link https://discord.com/developers/docs/components/reference}
 */
export interface APIBaseAutoPopulatedSelectMenuComponent<T extends ComponentType.ChannelSelect | ComponentType.MentionableSelect | ComponentType.RoleSelect | ComponentType.UserSelect, D extends SelectMenuDefaultValueType> extends APIBaseSelectMenuComponent<T> {
    /**
     * List of default values for auto-populated select menu components
     */
    default_values?: APISelectMenuDefaultValue<D>[];
}
/**
 * A String Select is an interactive component that allows users to select one or more provided options in a message.
 *
 * String Selects can be configured for both single-select and multi-select behavior. When a user finishes making their choice(s) your app receives an interaction.
 *
 * An Action Row can contain only one select menu and cannot contain buttons if it has a select menu.
 *
 * @see {@link https://discord.com/developers/docs/components/reference#string-select}
 */
export interface APIStringSelectComponent extends APIBaseSelectMenuComponent<ComponentType.StringSelect> {
    /**
     * Specified choices in a select menu; max 25
     */
    options: APISelectMenuOption[];
}
/**
 * A User Select is an interactive component that allows users to select one or more users in a message. Options are automatically populated based on the server's available users.
 *
 * User Selects can be configured for both single-select and multi-select behavior. When a user finishes making their choice(s) your app receives an interaction.
 *
 * User Selects must be placed inside an Action Row and are only available in messages. An Action Row can contain only one select menu and cannot contain buttons if it has a select menu.
 *
 * @see {@link https://discord.com/developers/docs/components/reference#user-select}
 */
export type APIUserSelectComponent = APIBaseAutoPopulatedSelectMenuComponent<ComponentType.UserSelect, SelectMenuDefaultValueType.User>;
/**
 * A Role Select is an interactive component that allows users to select one or more roles in a message. Options are automatically populated based on the server's available roles.
 *
 * Role Selects can be configured for both single-select and multi-select behavior. When a user finishes making their choice(s) your app receives an interaction.
 *
 * Role Selects must be placed inside an Action Row and are only available in messages. An Action Row can contain only one select menu and cannot contain buttons if it has a select menu.
 *
 * @see {@link https://discord.com/developers/docs/components/reference#role-select}
 */
export type APIRoleSelectComponent = APIBaseAutoPopulatedSelectMenuComponent<ComponentType.RoleSelect, SelectMenuDefaultValueType.Role>;
/**
 * A Mentionable Select is an interactive component that allows users to select one or more mentionables in a message. Options are automatically populated based on available mentionables in the server.
 *
 * Mentionable Selects can be configured for both single-select and multi-select behavior. When a user finishes making their choice(s), your app receives an interaction.
 *
 * Mentionable Selects must be placed inside an Action Row and are only available in messages. An Action Row can contain only one select menu and cannot contain buttons if it has a select menu.
 *
 * @see {@link https://discord.com/developers/docs/components/reference#mentionable-select}
 */
export type APIMentionableSelectComponent = APIBaseAutoPopulatedSelectMenuComponent<ComponentType.MentionableSelect, SelectMenuDefaultValueType.Role | SelectMenuDefaultValueType.User>;
/**
 * A Channel Select is an interactive component that allows users to select one or more channels in a message. Options are automatically populated based on available channels in the server and can be filtered by channel types.
 *
 * Channel Selects can be configured for both single-select and multi-select behavior. When a user finishes making their choice(s) your app receives an interaction.
 *
 * Channel Selects must be placed inside an Action Row and are only available in messages. An Action Row can contain only one select menu and cannot contain buttons if it has a select menu.
 *
 * @see {@link https://discord.com/developers/docs/components/reference#channel-select}
 */
export interface APIChannelSelectComponent extends APIBaseAutoPopulatedSelectMenuComponent<ComponentType.ChannelSelect, SelectMenuDefaultValueType.Channel> {
    /**
     * List of channel types to include in the ChannelSelect component
     */
    channel_types?: ChannelType[];
}
/**
 * @see {@link https://discord.com/developers/docs/components/reference#user-select-select-default-value-structure}
 */
export declare enum SelectMenuDefaultValueType {
    Channel = "channel",
    Role = "role",
    User = "user"
}
/**
 * @see {@link https://discord.com/developers/docs/components/reference#user-select-select-default-value-structure}
 */
export interface APISelectMenuDefaultValue<T extends SelectMenuDefaultValueType> {
    type: T;
    id: Snowflake;
}
/**
 * @see {@link https://discord.com/developers/docs/components/reference}
 */
export type APIAutoPopulatedSelectMenuComponent = APIChannelSelectComponent | APIMentionableSelectComponent | APIRoleSelectComponent | APIUserSelectComponent;
/**
 * @see {@link https://discord.com/developers/docs/components/reference}
 */
export type APISelectMenuComponent = APIChannelSelectComponent | APIMentionableSelectComponent | APIRoleSelectComponent | APIStringSelectComponent | APIUserSelectComponent;
/**
 * @see {@link https://discord.com/developers/docs/components/reference#string-select-select-option-structure}
 */
export interface APISelectMenuOption {
    /**
     * The user-facing name of the option (max 100 chars)
     */
    label: string;
    /**
     * The dev-defined value of the option (max 100 chars)
     */
    value: string;
    /**
     * An additional description of the option (max 100 chars)
     */
    description?: string;
    /**
     * The emoji to display to the left of the option
     */
    emoji?: APIMessageComponentEmoji;
    /**
     * Whether this option should be already-selected by default
     */
    default?: boolean;
}
/**
 * Text input is an interactive component that allows users to enter free-form text responses in modals. It supports both short, single-line inputs and longer, multi-line paragraph inputs.
 *
 * Text inputs can only be used within modals.
 *
 * When defining a text input component, you can set attributes to customize the behavior and appearance of it. However, not all attributes will be returned in the text input interaction payload.
 *
 * @see {@link https://discord.com/developers/docs/components/reference#text-input}
 */
export interface APITextInputComponent extends APIBaseComponent<ComponentType.TextInput> {
    /**
     * One of text input styles
     */
    style: TextInputStyle;
    /**
     * The custom id for the text input
     */
    custom_id: string;
    /**
     * Text that appears on top of the text input field, max 45 characters.
     *
     * @remarks Cannot be used in a label component.
     */
    label?: string;
    /**
     * Placeholder for the text input
     */
    placeholder?: string;
    /**
     * The pre-filled text in the text input
     */
    value?: string;
    /**
     * Minimal length of text input
     */
    min_length?: number;
    /**
     * Maximal length of text input
     */
    max_length?: number;
    /**
     * Whether this text input is required
     */
    required?: boolean;
}
export declare enum UnfurledMediaItemLoadingState {
    Unknown = 0,
    Loading = 1,
    LoadedSuccess = 2,
    LoadedNotFound = 3
}
/**
 * @see {@link https://discord.com/developers/docs/components/reference#unfurled-media-item-structure}
 */
export interface APIUnfurledMediaItem {
    /**
     * Supports arbitrary urls and attachment://<filename> references
     */
    url: string;
    /**
     * The proxied url of the media item. This field is ignored and provided by the API as part of the response
     */
    proxy_url?: string;
    /**
     * The width of the media item. This field is ignored and provided by the API as part of the response
     */
    width?: number | null;
    /**
     * The height of the media item. This field is ignored and provided by the API as part of the response
     */
    height?: number | null;
    placeholder?: string | null;
    placeholder_version?: number | null;
    /**
     * The media type of the content. This field is ignored and provided by the API as part of the response
     *
     * @see {@link https://en.wikipedia.org/wiki/Media_type}
     */
    content_type?: string | null;
    loading_state?: UnfurledMediaItemLoadingState;
    flags?: number;
    /**
     * The id of the uploaded attachment. This field is ignored and provided by the API as part of the response
     */
    attachment_id?: Snowflake;
}
/**
 * A Section is a top-level layout component that allows you to join text contextually with an accessory.
 *
 * Sections are only available in messages.
 *
 * @see {@link https://discord.com/developers/docs/components/reference#section}
 */
export interface APISectionComponent extends APIBaseComponent<ComponentType.Section> {
    /**
     * One to three text components
     */
    components: APITextDisplayComponent[];
    /**
     * A thumbnail or a button component, with a future possibility of adding more compatible components
     */
    accessory: APISectionAccessoryComponent;
}
/**
 * A Text Display is a top-level content component that allows you to add text to your message formatted with markdown and mention users and roles. This is similar to the content field of a message, but allows you to add multiple text components, controlling the layout of your message.
 *
 * Text Displays are only available in messages.
 *
 * @see {@link https://discord.com/developers/docs/components/reference#text-display}
 */
export interface APITextDisplayComponent extends APIBaseComponent<ComponentType.TextDisplay> {
    /**
     * Text that will be displayed similar to a message
     */
    content: string;
}
/**
 * A Thumbnail is a content component that is a small image only usable as an accessory in a section. The preview comes from an url or attachment through the unfurled media item structure.
 *
 * Thumbnails are only available in messages as an accessory in a section.
 *
 * @see {@link https://discord.com/developers/docs/components/reference#thumbnail}
 */
export interface APIThumbnailComponent extends APIBaseComponent<ComponentType.Thumbnail> {
    /**
     * A url or attachment
     */
    media: APIUnfurledMediaItem;
    /**
     * Alt text for the media
     */
    description?: string | null;
    /**
     * Whether the thumbnail should be a spoiler (or blurred out)
     *
     * @defaultValue `false`
     */
    spoiler?: boolean;
}
/**
 * @see {@link https://discord.com/developers/docs/components/reference#media-gallery-media-gallery-item-structure}
 */
export interface APIMediaGalleryItem {
    /**
     * A url or attachment
     */
    media: APIUnfurledMediaItem;
    /**
     * Alt text for the media
     */
    description?: string | null;
    /**
     * Whether the media should be a spoiler (or blurred out)
     *
     * @defaultValue `false`
     */
    spoiler?: boolean;
}
/**
 * A Media Gallery is a top-level content component that allows you to display 1-10 media attachments in an organized gallery format. Each item can have optional descriptions and can be marked as spoilers.
 *
 * Media Galleries are only available in messages.
 *
 * @see {@link https://discord.com/developers/docs/components/reference#media-gallery}
 */
export interface APIMediaGalleryComponent extends APIBaseComponent<ComponentType.MediaGallery> {
    /**
     * 1 to 10 media gallery items
     */
    items: APIMediaGalleryItem[];
}
/**
 * A File is a top-level component that allows you to display an uploaded file as an attachment to the message and reference it in the component.
 *
 * Each file component can only display 1 attached file, but you can upload multiple files and add them to different file components within your payload.
 *
 * Files are only available in messages.
 *
 * @see {@link https://discord.com/developers/docs/components/reference#file}
 */
export interface APIFileComponent extends APIBaseComponent<ComponentType.File> {
    /**
     * This unfurled media item is unique in that it **only** support attachment references using the `attachment://<filename>` syntax
     */
    file: APIUnfurledMediaItem;
    /**
     * Whether the media should be a spoiler (or blurred out)
     *
     * @defaultValue `false`
     */
    spoiler?: boolean;
    /**
     * The name of the file. This field is ignored and provided by the API as part of the response
     */
    name?: string;
    /**
     * The size of the file in bytes. This field is ignored and provided by the API as part of the response
     */
    size?: number;
}
/**
 * @see {@link https://discord.com/developers/docs/components/reference#separator}
 */
export declare enum SeparatorSpacingSize {
    Small = 1,
    Large = 2
}
/**
 * A Separator is a top-level layout component that adds vertical padding and visual division between other components.
 *
 * Separators are only available in messages.
 *
 * @see {@link https://discord.com/developers/docs/components/reference#separator}
 */
export interface APISeparatorComponent extends APIBaseComponent<ComponentType.Separator> {
    /**
     * Whether a visual divider should be displayed in the component
     *
     * @defaultValue `true`
     */
    divider?: boolean;
    /**
     * Size of separator padding
     *
     * @defaultValue `SeparatorSpacingSize.Small`
     */
    spacing?: SeparatorSpacingSize;
}
/**
 * A Container is a top-level layout component that holds up to 10 components. Containers are visually distinct from surrounding components and has an optional customizable color bar.
 *
 * Containers are only available in messages.
 *
 * @see {@link https://discord.com/developers/docs/components/reference#container}
 */
export interface APIContainerComponent extends APIBaseComponent<ComponentType.Container> {
    /**
     * Color for the accent on the container as RGB from `0x000000` to `0xFFFFFF`
     */
    accent_color?: number | null;
    /**
     * Whether the container should be a spoiler (or blurred out)
     *
     * @defaultValue `false`
     */
    spoiler?: boolean;
    /**
     * Up to 10 components of the type action row, text display, section, media gallery, separator, or file
     */
    components: APIComponentInContainer[];
}
/**
 * A Label is a top-level layout component. Labels wrap modal components with text as a label and optional description.
 *
 * @see {@link https://discord.com/developers/docs/components/reference#label}
 */
export interface APILabelComponent extends APIBaseComponent<ComponentType.Label> {
    /**
     * The label text; max 45 characters
     */
    label: string;
    /**
     * An optional description text for the label; max 100 characters
     */
    description?: string;
    /**
     * The component within the label
     */
    component: APIComponentInLabel;
}
/**
 * @see {@link https://discord.com/developers/docs/components/reference#file-upload}
 */
export interface APIFileUploadComponent extends APIBaseComponent<ComponentType.FileUpload> {
    /**
     * Id for the file upload; max 100 characters
     */
    custom_id: string;
    /**
     * Minimum number of items that must be uploaded (defaults to 1); min 0, max 10
     */
    min_values?: number;
    /**
     * Maximum number of items that can be uploaded (defaults to 1); max 10
     */
    max_values?: number;
    /**
     * Whether the file upload requires files to be uploaded before submitting the modal
     *
     * @defaultValue `true`
     */
    required?: boolean;
}
/**
 * @see {@link https://discord.com/developers/docs/components/reference#radio-group}
 * @unstable This component is not publicly released and is currently in preview.
 */
export interface APIRadioGroupComponent extends APIBaseComponent<ComponentType.RadioGroup> {
    /**
     * Id for the radio group; max 100 characters
     */
    custom_id: string;
    /**
     * 2-10 radio group options within the radio group component
     */
    options: APIRadioGroupOption[];
    /**
     * Whether the radio group requires selection before submitting the modal
     *
     * @defaultValue `true`
     */
    required?: boolean;
}
/**
 * @see {@link https://discord.com/developers/docs/components/reference#radio-group-option-structure}
 * @unstable This feature is not publicly released and is currently in preview.
 */
export interface APIRadioGroupOption {
    /**
     * The value of the radio group option; max 100 characters
     */
    value: string;
    /**
     * The label text; max 100 characters
     */
    label: string;
    /**
     * An optional description for the radio group option; max 100 characters
     */
    description?: string;
    /**
     * Whether this option is selected by default
     *
     * @defaultValue `false`
     */
    default?: boolean;
}
/**
 * @see {@link https://discord.com/developers/docs/components/reference#checkbox-group}
 * @unstable This component is not publicly released and is currently in preview.
 */
export interface APICheckboxGroupComponent extends APIBaseComponent<ComponentType.CheckboxGroup> {
    /**
     * Id for the checkbox group; max 100 characters
     */
    custom_id: string;
    /**
     * Checkbox group option within the component; max 10
     */
    options: APICheckboxGroupOption[];
    /**
     * Minimum number of checkbox that can be selected; min 0, max 10
     *
     * @defaultValue `1`
     */
    min_values?: number;
    /**
     * Maximum number of checkbox that can be selected; min 1, max 10
     *
     * @defaultValue {@link APICheckboxGroupComponent.options} length
     */
    max_values?: number;
    /**
     * Whether selecting within the group is required before submitting the modal
     *
     * @defaultValue `true`
     */
    required?: boolean;
}
/**
 * @see {@link https://discord.com/developers/docs/components/reference#checkbox-group-option-structure}
 * @unstable This feature is not publicly released and is currently in preview.
 */
export interface APICheckboxGroupOption {
    /**
     * The value of the checkbox group option; max 100 characters
     */
    value: string;
    /**
     * The label text; max 100 characters
     */
    label: string;
    /**
     * An optional description for the checkbox group option; max 100 characters;
     */
    description?: string;
    /**
     * Whether this option is selected by default
     *
     * @defaultValue `false`
     */
    default?: boolean;
}
/**
 * @see {@link https://discord.com/developers/docs/components/reference#checkbox}
 * @unstable This component is not publicly released and is currently in preview.
 */
export interface APICheckboxComponent extends APIBaseComponent<ComponentType.Checkbox> {
    /**
     * Id for the checkbox; max 100 characters
     */
    custom_id: string;
    /**
     * Whether this checkbox is selected by default
     *
     * @defaultValue `false`
     */
    default?: boolean;
}
/**
 * @see {@link https://discord.com/developers/docs/resources/message#message-snapshot-object}
 */
export interface APIMessageSnapshot {
    /**
     * Subset of the message object fields
     */
    message: APIMessageSnapshotFields;
    /**
     * Id of the origin message's guild
     *
     * @deprecated This field doesn't accurately reflect the Discord API as it doesn't exist nor is documented and will
     * be removed in the next major version.
     *
     * It was added in {@link https://github.com/discord/discord-api-docs/pull/6833/commits/d18f72d06d62e6b1d51ca2c1ef308ddc29ff3348 | d18f72d}
     * but was later removed before the PR ({@link https://github.com/discord/discord-api-docs/pull/6833 | discord-api-docs#6833}) was merged.
     * @see {@link https://github.com/discordjs/discord-api-types/pull/1084 | discord-api-types#1084} for more information.
     */
    guild_id?: Snowflake;
}
/**
 * All components that can appear in messages.
 *
 * For more specific sets, see {@link APIMessageTopLevelComponent}, {@link APIComponentInMessageActionRow}, {@link APIComponentInContainer}, and {@link APISectionAccessoryComponent}
 *
 * @see {@link https://discord.com/developers/docs/components/reference}
 */
export type APIMessageComponent = APIActionRowComponent<APIComponentInMessageActionRow> | APIButtonComponent | APIContainerComponent | APIFileComponent | APIMediaGalleryComponent | APISectionComponent | APISelectMenuComponent | APISeparatorComponent | APITextDisplayComponent | APIThumbnailComponent;
/**
 * @see {@link https://discord.com/developers/docs/components/reference}
 */
export type APIMessageTopLevelComponent = APIActionRowComponent<APIComponentInMessageActionRow> | APIContainerComponent | APIFileComponent | APIMediaGalleryComponent | APISectionComponent | APISeparatorComponent | APITextDisplayComponent;
/**
 * @see {@link https://discord.com/developers/docs/components/reference}
 */
export type APIModalComponent = APIActionRowComponent<APIComponentInModalActionRow> | APIComponentInLabel | APIComponentInModalActionRow | APILabelComponent;
/**
 * @see {@link https://discord.com/developers/docs/components/reference#action-row}
 */
export type APIComponentInActionRow = APIComponentInMessageActionRow | APIComponentInModalActionRow;
/**
 * @see {@link https://discord.com/developers/docs/components/reference#action-row}
 */
export type APIComponentInMessageActionRow = APIButtonComponent | APISelectMenuComponent;
/**
 * @see {@link https://discord.com/developers/docs/components/reference#action-row}
 * @deprecated
 */
export type APIComponentInModalActionRow = APITextInputComponent;
/**
 * @see {@link https://discord.com/developers/docs/components/reference#label-label-child-components}
 */
export type APIComponentInLabel = APICheckboxComponent | APICheckboxGroupComponent | APIFileUploadComponent | APIRadioGroupComponent | APISelectMenuComponent | APITextInputComponent;
/**
 * @see {@link https://discord.com/developers/docs/components/reference#section}
 */
export type APISectionAccessoryComponent = APIButtonComponent | APIThumbnailComponent;
/**
 * @see {@link https://discord.com/developers/docs/components/reference#container}
 */
export type APIComponentInContainer = APIActionRowComponent<APIComponentInMessageActionRow> | APIFileComponent | APIMediaGalleryComponent | APISectionComponent | APISeparatorComponent | APITextDisplayComponent;
/**
 * https://discord.com/developers/docs/resources/message#message-snapshot-object
 */
export type APIMessageSnapshotFields = Pick<APIMessage, 'attachments' | 'components' | 'content' | 'edited_timestamp' | 'embeds' | 'flags' | 'mention_roles' | 'mentions' | 'sticker_items' | 'stickers' | 'timestamp' | 'type'>;
/**
 * @see {@link https://discord.com/developers/docs/resources/message#message-pin-object}
 */
export interface APIMessagePin {
    /**
     * The time the message was pinned
     */
    pinned_at: string;
    /**
     * The pinned message
     */
    message: APIMessage;
}
//# sourceMappingURL=message.d.ts.map