//#region src/util/type.d.ts
/**
 * 文字列に変換可能な型を表します。
 *
 * このタイプには以下が含まれます：
 * - `string`: 文字列プリミティブ
 * - `number`: 数値
 * - `bigint`: 任意精度の整数
 * - `boolean`: 真/偽の値
 * - `null`: null値
 * - `undefined`: undefined値
 *
 * これらの型はすべて、文字列連結や文字列変換可能な値を期待する関数で安全に使用できます。
 */
type Stringable = string | number | bigint | boolean | null | undefined;
/**
 * ハイフンで結合された2つの文字列変換可能な値、または単一の値を表すユーティリティタイプ。
 *
 * @template T - 文字列に変換可能な型
 * @returns `${T}-${T}` (ハイフンで結合された2つの値) または `${T}` (単一の値)
 *
 * @example
 * type Numbers = 1 | 2 | 3;
 * type JoinedNumbers = Join<Numbers>; // '1' | '2' | '3' | '1-1' | '1-2' | '1-3' | '2-1' | '2-2' | '2-3' | '3-1' | '3-2' | '3-3'
 */
type Join<T extends Stringable> = `${T}-${T}` | `${T}`;
//#endregion
//#region src/params.d.ts
declare const RankingType: {
  readonly Daily: "d";
  readonly Weekly: "w";
  readonly Monthly: "m";
  readonly Quarterly: "q";
};
type RankingType = (typeof RankingType)[keyof typeof RankingType];
/**
 * すべてのAPIで共通のクエリパラメータ
 */
interface ParamsBase {
  /**
   * gzip圧縮してgzipファイルとして返します。
   * gzip圧縮レベルを1～5で指定できます。
   * 転送量上限を減らすためにも推奨
   */
  gzip?: GzipLevel;
  /**
   * 出力形式を指定
   * 本ライブラリはJSONとJSONPのみ対応
   */
  out?: "json" | "jsonp";
}
/**
 * 検索APIで共通のクエリパラメータ
 */
interface ParamsBaseWithOrder<TOrder extends string> extends ParamsBase {
  /**
   * 出力する項目を個別に指定できます。未指定時は全項目出力されます。
   * 転送量軽減のため、このパラメータの使用が推奨されます。
   */
  of?: string;
  /**
   * 最大出力数を指定できます。指定しない場合は20件になります。
   */
  lim?: number;
  /**	表示開始位置の指定です。 */
  st?: number;
  /** 出力順序を指定できます。 */
  order?: TOrder;
}
/**
 * メソッドにパラメータを指定する際のヘルパー。
 * @see https://dev.syosetu.com/man/api/
 * @see https://dev.syosetu.com/xman/atom/
 */
interface SearchParams extends ParamsBaseWithOrder<Order> {
  word?: string;
  notword?: string;
  title?: BooleanNumber;
  ex?: BooleanNumber;
  keyword?: BooleanNumber;
  wname?: BooleanNumber;
  biggenre?: Join<BigGenre> | BigGenre;
  notbiggenre?: Join<BigGenre> | BigGenre;
  genre?: Join<Genre> | Genre;
  notgenre?: Join<Genre> | Genre;
  userid?: Join<number> | number;
  nocgenre?: Join<R18Site> | R18Site;
  notnocgenre?: Join<R18Site> | R18Site;
  xid?: Join<number> | number;
  isr15?: BooleanNumber;
  isbl?: BooleanNumber;
  isgl?: BooleanNumber;
  iszankoku?: BooleanNumber;
  istensei?: BooleanNumber;
  istenni?: BooleanNumber;
  istt?: BooleanNumber;
  notr15?: BooleanNumber;
  notbl?: BooleanNumber;
  notgl?: BooleanNumber;
  notzankoku?: BooleanNumber;
  nottensei?: BooleanNumber;
  nottenni?: BooleanNumber;
  minlen?: number;
  maxlen?: number;
  length?: number | Join<number | "">;
  kaiwaritu?: number | string;
  sasie?: number | string;
  mintime?: number;
  maxtime?: number;
  time?: number | string;
  ncode?: string | Join<string>;
  type?: NovelTypeParam;
  buntai?: BuntaiParam | Join<BuntaiParam>;
  stop?: StopParam;
  ispickup?: typeof BooleanNumber.True;
  lastup?: string;
  lastupdate?: string;
  opt?: Join<OptionalFields>;
}
interface RankingParams extends ParamsBase {
  rtype: `${string}-${RankingType}`;
}
interface RankingHistoryParams extends ParamsBase {
  ncode: string;
}
/**
 * ユーザー検索パラメータ
 */
interface UserSearchParams extends ParamsBaseWithOrder<UserOrder> {
  /** 単語を指定できます。半角または全角スペースで区切るとAND抽出になります。部分一致でHITします。検索の対象はユーザ名とユーザ名のフリガナです。 */
  word?: string;
  /** 含みたくない単語を指定できます。スペースで区切ることにより含ませない単語を増やせます。部分一致で除外されます。除外の対象はユーザ名とユーザ名のフリガナです。 */
  notword?: string;
  /** ユーザIDで抽出可能。 */
  userid?: number;
  /** 抽出するユーザのユーザ名のフリガナの頭文字を指定できます。頭文字はユーザ名のフリガナをひらがなに変換し、最初の1文字が「ぁ」～「ん」の場合に対象となります。 */
  name1st?: string;
  /** 抽出するユーザの小説投稿数の下限を指定できます。小説投稿件数が指定された数値以上のユーザを抽出します。 */
  minnovel?: number;
  /** 抽出するユーザの小説投稿数の上限を指定できます。小説投稿件数が指定された数値以下のユーザを抽出します。 */
  maxnovel?: number;
  /** 抽出するユーザのレビュー投稿数の下限を指定できます。レビュー投稿件数が指定された数値以上のユーザを抽出します。 */
  minreview?: number;
  /** 抽出するユーザのレビュー投稿数の上限を指定できます。レビュー投稿件数が指定された数値以下のユーザを抽出します。 */
  maxreview?: number;
}
declare const BooleanNumber: {
  readonly True: 1;
  readonly False: 0;
};
type BooleanNumber = (typeof BooleanNumber)[keyof typeof BooleanNumber];
type SearchResultFieldNames = keyof NarouSearchResult;
/**
 * なろう小説APIのofパラメータに指定できる出力する項目
 * @see https://dev.syosetu.com/man/api/#output
 */
declare const Fields: {
  /** 小説名 */
  readonly title: "t";
  /** Nコード */
  readonly ncode: "n";
  /** 作者のユーザID(数値) */
  readonly userid: "u";
  /** 作者名 */
  readonly writer: "w";
  /** 小説のあらすじ */
  readonly story: "s";
  /** 大ジャンル */
  readonly biggenre: "bg";
  /** ジャンル */
  readonly genre: "g";
  /** キーワード */
  readonly keyword: "k";
  /** 初回掲載日 */
  readonly general_firstup: "gf";
  /** 最終掲載日 */
  readonly general_lastup: "gl";
  /** 連載の場合は1、短編の場合は2 */
  readonly noveltype: "nt";
  /** 短編小説と完結済小説は0となっています。連載中は1です。 */
  readonly end: "e";
  /** 全掲載部分数 */
  readonly general_all_no: "ga";
  /** 小説文字数 */
  readonly length: "l";
  /** 読了時間(分単位) */
  readonly time: "ti";
  /** 長期連載停止中 */
  readonly isstop: "i";
  /** 登録必須キーワードに「R15」が含まれる場合は1、それ以外は0です。 */
  readonly isr15: "isr";
  /** 登録必須キーワードに「ボーイズラブ」が含まれる場合は1、それ以外は0です。 */
  readonly isbl: "ibl";
  /** 登録必須キーワードに「ガールズラブ」が含まれる場合は1、それ以外は0です。 */
  readonly isgl: "igl";
  /** 登録必須キーワードに「残酷な描写あり」が含まれる場合は1、それ以外は0です。 */
  readonly iszankoku: "izk";
  /** 登録必須キーワードに「異世界転生」が含まれる場合は1、それ以外は0です。 */
  readonly istensei: "its";
  /** 登録必須キーワードに「異世界転移」が含まれる場合は1、それ以外は0です。 */
  readonly istenni: "iti";
  /** 総合評価ポイント */
  readonly global_point: "gp";
  /** 日間ポイント */
  readonly daily_point: "dp";
  /** 週間ポイント */
  readonly weekly_point: "wp";
  /** 月間ポイント */
  readonly monthly_point: "mp";
  /** 四半期ポイント */
  readonly quarter_point: "qp";
  /** 年間ポイント */
  readonly yearly_point: "yp";
  /** ブックマーク数 */
  readonly fav_novel_cnt: "f";
  /** 感想数 */
  readonly impression_cnt: "imp";
  /** レビュー数 */
  readonly review_cnt: "r";
  /** 評価ポイント */
  readonly all_point: "a";
  /** 評価者数 */
  readonly all_hyoka_cnt: "ah";
  /** 挿絵の数 */
  readonly sasie_cnt: "sa";
  /** 会話率 */
  readonly kaiwaritu: "ka";
  /** 小説の更新日時 */
  readonly novelupdated_at: "nu";
  /**
   * 最終更新日時
   * システム用で小説更新時とは関係ありません
   */
  readonly updated_at: "ua";
};
type Fields = (typeof Fields)[keyof Omit<NarouSearchResult, "novel_type" | "weekly_unique" | "nocgenre">];
/**
 * なろうR18小説APIのofパラメータに指定できる出力する項目
 * @see https://dev.syosetu.com/xman/api/#output
 */
declare const R18Fields: {
  /** 小説名 */
  readonly title: "t";
  /** Nコード */
  readonly ncode: "n";
  /** 作者のユーザID(数値) */
  readonly userid: "u";
  /** 作者名 */
  readonly writer: "w";
  /** 小説のあらすじ */
  readonly story: "s";
  /** 掲載サイト */
  readonly nocgenre: "ng";
  /** キーワード */
  readonly keyword: "k";
  /** 初回掲載日 */
  readonly general_firstup: "gf";
  /** 最終掲載日 */
  readonly general_lastup: "gl";
  /** 連載の場合は1、短編の場合は2 */
  readonly noveltype: "nt";
  /** 短編小説と完結済小説は0となっています。連載中は1です。 */
  readonly end: "e";
  /** 全掲載部分数 */
  readonly general_all_no: "ga";
  /** 小説文字数 */
  readonly length: "l";
  /** 読了時間(分単位) */
  readonly time: "ti";
  /** 長期連載停止中 */
  readonly isstop: "i";
  /** 登録必須キーワードに「ボーイズラブ」が含まれる場合は1、それ以外は0です。 */
  readonly isbl: "ibl";
  /** 登録必須キーワードに「ガールズラブ」が含まれる場合は1、それ以外は0です。 */
  readonly isgl: "igl";
  /** 登録必須キーワードに「残酷な描写あり」が含まれる場合は1、それ以外は0です。 */
  readonly iszankoku: "izk";
  /** 登録必須キーワードに「異世界転生」が含まれる場合は1、それ以外は0です。 */
  readonly istensei: "its";
  /** 登録必須キーワードに「異世界転移」が含まれる場合は1、それ以外は0です。 */
  readonly istenni: "iti";
  /** 総合評価ポイント */
  readonly global_point: "gp";
  /** 日間ポイント */
  readonly daily_point: "dp";
  /** 週間ポイント */
  readonly weekly_point: "wp";
  /** 月間ポイント */
  readonly monthly_point: "mp";
  /** 四半期ポイント */
  readonly quarter_point: "qp";
  /** 年間ポイント */
  readonly yearly_point: "yp";
  /** R18ブックマーク数 */
  readonly fav_novel_cnt: "f";
  /** 感想数 */
  readonly impression_cnt: "imp";
  /** レビュー数 */
  readonly review_cnt: "r";
  /** 評価ポイント */
  readonly all_point: "a";
  /** 評価者数 */
  readonly all_hyoka_cnt: "ah";
  /** 挿絵の数 */
  readonly sasie_cnt: "sa";
  /** 会話率 */
  readonly kaiwaritu: "ka";
  /** 小説の更新日時 */
  readonly novelupdated_at: "nu";
  /**
   * 最終更新日時
   * システム用で小説更新時とは関係ありません
   */
  readonly updated_at: "ua";
};
type R18Fields = (typeof R18Fields)[keyof Omit<NarouSearchResult, "novel_type" | "weekly_unique" | "biggenre" | "genre" | "isr15">];
/**
 * オプション項目
 */
declare const OptionalFields: {
  /**
   * 週間ユニークユーザ[項目名:weekly_unique]が追加されます。
   * 週間ユニークユーザは前週の日曜日から土曜日分のユニークの合計です。
   * 毎週火曜日早朝に更新されます。
   */
  readonly weekly_unique: "weekly";
};
type OptionalFields = (typeof OptionalFields)[keyof Pick<NarouSearchResult, "weekly_unique">];
/**
 * ユーザ検索APIのofパラメータに指定できる出力する項目
 * @see https://dev.syosetu.com/man/userapi/#output
 */
declare const UserFields: {
  /** ユーザID */
  readonly userid: "u";
  /** ユーザ名 */
  readonly name: "n";
  /** ユーザ名のフリガナ */
  readonly yomikata: "y";
  /** ユーザ名のフリガナの頭文字 */
  readonly name1st: "1";
  /** 小説投稿数 */
  readonly novel_cnt: "nc";
  /** レビュー投稿数 */
  readonly review_cnt: "rc";
  /** 小説累計文字数 */
  readonly novel_length: "nl";
  /** 総合評価ポイントの合計 */
  readonly sum_global_point: "sg";
};
type UserFields = (typeof UserFields)[keyof UserSearchResult];
/**
 * 出力順序
 */
declare const Order: {
  /** ブックマーク数の多い順 */
  readonly FavoriteNovelCount: "favnovelcnt";
  /** レビュー数の多い順 */
  readonly ReviewCount: "reviewcnt";
  /** 総合ポイントの高い順 */
  readonly HyokaDesc: "hyoka";
  /** 総合ポイントの低い順 */
  readonly HyokaAsc: "hyokaasc";
  /** 感想の多い順 */
  readonly ImpressionCount: "impressioncnt";
  /** 評価者数の多い順 */
  readonly HyokaCountDesc: "hyokacnt";
  /** 評価者数の少ない順 */
  readonly HyokaCountAsc: "hyokacntasc";
  /** 週間ユニークユーザの多い順 */
  readonly Weekly: "weekly";
  /** 小説本文の文字数が多い順 */
  readonly LengthDesc: "lengthdesc";
  /** 小説本文の文字数が少ない順 */
  readonly LengthAsc: "lengthasc";
  /** Nコードが新しい順 */
  readonly NCodeDesc: "ncodedesc";
  /** 新着更新順 */
  readonly New: "new";
  /** 古い順 */
  readonly Old: "old";
  /** 日間ポイントの高い順 */
  readonly DailyPoint: "dailypoint";
  /** 週間ポイントの高い順 */
  readonly WeeklyPoint: "weeklypoint";
  /** 月間ポイントの高い順 */
  readonly MonthlyPoint: "monthlypoint";
  /** 四半期ポイントの高い順 */
  readonly QuarterPoint: "quarterpoint";
  /** 年間ポイントの高い順 */
  readonly YearlyPoint: "yearlypoint";
  /** 初回掲載順 */
  readonly GeneralFirstUp: "generalfirstup";
};
type Order = (typeof Order)[keyof typeof Order];
/** R18掲載サイト */
declare const R18Site: {
  /** ノクターンノベルズ(男性向け) */
  readonly Nocturne: 1;
  /** ムーンライトノベルズ(女性向け) */
  readonly MoonLight: 2;
  /** ムーンライトノベルズ(BL) */
  readonly MoonLightBL: 3;
  /** ミッドナイトノベルズ(大人向け) */
  readonly Midnight: 4;
};
type R18Site = (typeof R18Site)[keyof typeof R18Site];
/** R18掲載サイト表記ヘルパー */
declare const R18SiteNotation: { readonly [K in R18Site]: string };
/** 大ジャンル */
declare const BigGenre: {
  /** 恋愛 */
  readonly Renai: 1;
  /** ファンタジー */
  readonly Fantasy: 2;
  /** 文芸 */
  readonly Bungei: 3;
  /** SF */
  readonly Sf: 4;
  /** その他 */
  readonly Sonota: 99;
  /** ノンジャンル */
  readonly NonGenre: 98;
};
type BigGenre = (typeof BigGenre)[keyof typeof BigGenre];
/** 大ジャンル表記ヘルパー */
declare const BigGenreNotation: { readonly [K in BigGenre]: string };
/** ジャンル */
declare const Genre: {
  /** 異世界〔恋愛〕*/
  readonly RenaiIsekai: 101;
  /** 現実世界〔恋愛〕*/
  readonly RenaiGenjitsusekai: 102;
  /** ハイファンタジー〔ファンタジー〕*/
  readonly FantasyHigh: 201;
  /** ローファンタジー〔ファンタジー〕*/
  readonly FantasyLow: 202;
  /** 純文学〔文芸〕*/
  readonly BungeiJyunbungei: 301;
  /** ヒューマンドラマ〔文芸〕*/
  readonly BungeiHumanDrama: 302;
  /** 歴史〔文芸〕*/
  readonly BungeiHistory: 303;
  /** 推理〔文芸〕*/
  readonly BungeiSuiri: 304;
  /** ホラー〔文芸〕*/
  readonly BungeiHorror: 305;
  /** アクション〔文芸〕*/
  readonly BungeiAction: 306;
  /** コメディー〔文芸〕*/
  readonly BungeiComedy: 307;
  /** VRゲーム〔SF〕*/
  readonly SfVrgame: 401;
  /** 宇宙〔SF〕*/
  readonly SfSpace: 402;
  /** 空想科学〔SF〕*/
  readonly SfKuusoukagaku: 403;
  /** パニック〔SF〕*/
  readonly SfPanic: 404;
  /** 童話〔その他〕*/
  readonly SonotaDouwa: 9901;
  /** 詩〔その他〕*/
  readonly SonotaShi: 9902;
  /** エッセイ〔その他〕*/
  readonly SonotaEssei: 9903;
  /** リプレイ〔その他〕*/
  readonly SonotaReplay: 9904;
  /** その他〔その他〕 */
  readonly SonotaSonota: 9999;
  /** ノンジャンル〔ノンジャンル〕*/
  readonly NonGenre: 9801;
};
type Genre = (typeof Genre)[keyof typeof Genre];
/** ジャンル表記ヘルパー */
declare const GenreNotation: { readonly [K in Genre]: string };
/** 文体指定 */
declare const BuntaiParam: {
  /** 字下げされておらず、連続改行が多い作品 */
  readonly NoJisageKaigyouOoi: 1;
  /** 字下げされていないが、改行数は平均な作品 */
  readonly NoJisageKaigyoHutsuu: 2;
  /** 字下げが適切だが、連続改行が多い作品 */
  readonly JisageKaigyoOoi: 4;
  /** 字下げが適切でかつ改行数も平均な作品 */
  readonly JisageKaigyoHutsuu: 6;
};
type BuntaiParam = (typeof BuntaiParam)[keyof typeof BuntaiParam];
/** 連載停止中指定 */
declare const StopParam: {
  /** 長期連載停止中を除きます */
  readonly NoStopping: 1;
  /** 長期連載停止中のみ取得します */
  readonly Stopping: 2;
};
type StopParam = (typeof StopParam)[keyof typeof StopParam];
/** 小説タイプ指定 */
declare const NovelTypeParam: {
  /** 短編 */
  readonly Short: "t";
  /** 連載中 */
  readonly RensaiNow: "r";
  /** 完結済連載小説 */
  readonly RensaiEnd: "er";
  /** すべての連載小説(連載中および完結済) */
  readonly Rensai: "re";
  /** 短編と完結済連載小説 */
  readonly ShortAndRensai: "ter";
};
type NovelTypeParam = (typeof NovelTypeParam)[keyof typeof NovelTypeParam];
/** 日付指定パラメータ */
declare const DateParam: {
  ThisWeek: string;
  LastWeek: string;
  SevenDays: string;
  ThisMonth: string;
  LastMonth: string;
};
type DateParam = (typeof DateParam)[keyof typeof DateParam];
declare const UserOrder: {
  /** ユーザIDの新しい順 */
  readonly New: "new";
  /** 小説投稿数の多い順 */
  readonly NovelCount: "novelcnt";
  /** レビュー投稿数の多い順 */
  readonly ReviewCount: "reviewcnt";
  /** 小説累計文字数の多い順 */
  readonly NovelLength: "novellength";
  /** 総合評価ポイントの合計の多い順 */
  readonly SumGlobalPoint: "sumglobalpoint";
  /** ユーザIDの古い順 */
  readonly Old: "old";
};
type UserOrder = (typeof UserOrder)[keyof typeof UserOrder];
type GzipLevel = 0 | 1 | 2 | 3 | 4 | 5;
//#endregion
//#region src/narou-search-results.d.ts
/**
 * なろう小説API検索結果
 */
declare class NarouSearchResults<T, TKey extends keyof T> {
  /**
   * 検索結果数
   */
  allcount: number;
  /**
   * 結果表示上限数
   */
  limit: number;
  /**
   * 結果表示開始数
   */
  start: number;
  /**
   * 結果表示の現在ページ(=start/limit)
   */
  page: number;
  /**
   * 今回取得できた検索結果の数
   */
  length: number;
  /**
   * 検索結果
   */
  values: readonly Pick<T, TKey>[];
  /**
   * @constractor
   * @private
   */
  constructor([header, ...result]: [{
    allcount: number;
  }, ...Pick<T, TKey>[]], params: SearchParams | UserSearchParams);
}
/**
 * 小説情報
 * @see https://dev.syosetu.com/man/api/#output
 * @see https://dev.syosetu.com/xman/api/#output
 */
interface NarouSearchResult {
  /** 小説名 */
  title: string;
  /** Nコード */
  ncode: string;
  /** 作者のユーザID(数値) */
  userid: number;
  /** 作者名 */
  writer: string;
  /** 小説のあらすじ */
  story: string;
  /** 掲載サイト */
  nocgenre: R18Site;
  /** 大ジャンル */
  biggenre: BigGenre;
  /** ジャンル */
  genre: Genre;
  /** キーワード */
  keyword: string;
  /** 初回掲載日 YYYY-MM-DD HH:MM:SSの形式 */
  general_firstup: string;
  /** 最終掲載日 YYYY-MM-DD HH:MM:SSの形式 */
  general_lastup: string;
  /** 連載の場合は1、短編の場合は2 */
  novel_type: NovelType;
  /** 連載の場合は1、短編の場合は2 */
  noveltype: NovelType;
  /** 短編小説と完結済小説は0となっています。連載中は1です。 */
  end: End;
  /** 全掲載話数です。短編の場合は1です。 */
  general_all_no: number;
  /** 小説文字数です。スペースや改行は文字数としてカウントしません。 */
  length: number;
  /** 読了時間(分単位)です。読了時間は小説文字数÷500を切り上げした数値です。 */
  time: number;
  /** 長期連載中は1、それ以外は0です。 */
  isstop: BooleanNumber;
  /** 登録必須キーワードに「R15」が含まれる場合は1、それ以外は0です。 */
  isr15: BooleanNumber;
  /** 登録必須キーワードに「ボーイズラブ」が含まれる場合は1、それ以外は0です。 */
  isbl: BooleanNumber;
  /** 登録必須キーワードに「ガールズラブ」が含まれる場合は1、それ以外は0です。 */
  isgl: BooleanNumber;
  /** 登録必須キーワードに「残酷な描写あり」が含まれる場合は1、それ以外は0です。 */
  iszankoku: BooleanNumber;
  /** 登録必須キーワードに「異世界転生」が含まれる場合は1、それ以外は0です。 */
  istensei: BooleanNumber;
  /** 登録必須キーワードに「異世界転移」が含まれる場合は1、それ以外は0です。 */
  istenni: BooleanNumber;
  /** 総合得点(=(ブックマーク数×2)+評価点) */
  global_point: number;
  /**
   * 日間ポイント
   * ランキング集計時点から過去24時間以内で新たに登録されたブックマークや評価が対象
   */
  daily_point: number;
  /**
   * 週間ポイント
   * ランキング集計時点から過去7日以内で新たに登録されたブックマークや評価が対象
   */
  weekly_point: number;
  /**
   * 月間ポイント
   * ランキング集計時点から過去30日以内で新たに登録されたブックマークや評価が対象
   */
  monthly_point: number;
  /**
   * 四半期ポイント
   * ランキング集計時点から過去90日以内で新たに登録されたブックマークや評価が対象
   */
  quarter_point: number;
  /**
   * 年間ポイント
   * ランキング集計時点から過去365日以内で新たに登録されたブックマークや評価が対象
   */
  yearly_point: number;
  /** ブックマーク数 */
  fav_novel_cnt: number;
  /** 感想数 */
  impression_cnt: number;
  /** レビュー数 */
  review_cnt: number;
  /** 評価ポイント */
  all_point: number;
  /** 評価者数 */
  all_hyoka_cnt: number;
  /** 挿絵の数 */
  sasie_cnt: number;
  /**
   * 会話率
   * @see https://dev.syosetu.com/man/kaiwa/
   */
  kaiwaritu: number;
  /**
   * 小説の更新日時
   */
  novelupdated_at: string;
  /**
   * 最終更新日時
   * システム用で小説更新時とは関係ありません
   */
  updated_at: string;
  /** 週間ユニークユーザー数 */
  weekly_unique: number;
}
/**
 * ユーザ情報
 * @see https://dev.syosetu.com/man/userapi/#output
 */
interface UserSearchResult {
  /** ユーザID */
  userid: number;
  /** ユーザ名 */
  name: string;
  /** ユーザ名のフリガナ */
  yomikata: string;
  /**
   * ユーザ名のフリガナの頭文字
   * ひらがな以外の場合はnullまたは空文字となります。
   */
  name1st: string;
  /** 小説投稿数 */
  novel_cnt: number;
  /** レビュー投稿数 */
  review_cnt: number;
  /**
   * 小説累計文字数
   * スペースや改行は文字数としてカウントしません。
   */
  novel_length: number;
  /**
   * 総合評価ポイントの合計
   * 投稿済小説でそれぞれ獲得した総合評価ポイントの合計です。
   */
  sum_global_point: number;
}
/**
 * noveltype/novel_typeの値ヘルパー
 */
declare const NovelType: {
  /** 連載 */
  readonly Rensai: 1;
  /** 短編 */
  readonly Tanpen: 2;
};
type NovelType = typeof NovelType[keyof typeof NovelType];
/**
 * endの値ヘルパー
 */
declare const End: {
  /** 短編小説と完結済小説 */
  readonly KanketsuOrTanpen: 0;
  /** 連載中 */
  readonly Rensai: 1;
};
type End = typeof End[keyof typeof End];
type SearchResultFields<T extends Fields> = { [K in keyof typeof Fields]: typeof Fields[K] extends T ? K : never }[keyof typeof Fields];
type SearchResultOptionalFields<T extends OptionalFields> = { [K in keyof typeof OptionalFields]: typeof OptionalFields[K] extends T ? K : never }[keyof typeof OptionalFields];
type SearchResultR18Fields<T extends R18Fields> = { [K in keyof typeof R18Fields]: typeof R18Fields[K] extends T ? K : never }[keyof typeof R18Fields];
type UserSearchResultFields<T extends UserFields> = { [K in keyof typeof UserFields]: typeof UserFields[K] extends T ? K : never }[keyof typeof UserFields];
type PickedNarouSearchResult<T extends keyof NarouSearchResult> = Pick<NarouSearchResult, T>;
//#endregion
//#region src/search-builder.d.ts
type DefaultSearchResultFields = keyof Omit<NarouSearchResult, "weekly_unique" | "noveltype" | "nocgenre" | "xid">;
declare abstract class SearchBuilderBase<TParams extends ParamsBaseWithOrder<TOrder>, TOrder extends string> {
  protected params: TParams;
  protected api: NarouNovel;
  /**
   * constructor
   * @private
   * @param params クエリパラメータ
   * @param api NarouNovel インスタンス
   */
  constructor(params: TParams | undefined, api: NarouNovel);
  /**
   * 配列から重複を除去する
   * @protected
   * @static
   * @param array 配列
   * @returns 重複を除去した配列
   */
  protected static distinct<T>(array: readonly T[]): T[];
  /**
   * 配列をハイフン区切りの文字列に変換する
   * @protected
   * @static
   * @param n 文字列または数値の配列、あるいは単一の文字列または数値
   * @returns ハイフン区切りの文字列
   */
  protected static array2string<T extends string | number>(n: T | readonly T[]): Join<T>;
  /**
   * 取得件数を指定する (lim)
   * @param num 取得件数 (1-500)
   * @return {this}
   */
  limit(num: number): this;
  /**
   * 取得開始位置を指定する (st)
   * @param num 取得開始位置 (1-)
   * @return {this}
   */
  start(num: number): this;
  /**
   * ページ番号と1ページあたりの件数で取得範囲を指定する
   * @param no ページ番号 (0-)
   * @param count 1ページあたりの件数 (デフォルト: 20)
   * @return {this}
   */
  page(no: number, count?: number): this;
  /**
   * 出力順序を指定する (order)
   * 指定しない場合は新着順となります。
   * @param {TOrder} order 出力順序
   * @return {this}
   */
  order(order: TOrder): this;
  /**
   * gzip圧縮レベルを指定する (gzip)
   *
   * 転送量上限を減らすためにも推奨
   * @param {GzipLevel} level gzip圧縮レベル(1～5)
   * @return {this}
   */
  gzip(level: GzipLevel): this;
  /**
   * クエリパラメータをセットする
   * @protected
   * @param obj セットするパラメータ
   * @return {this}
   */
  protected set(obj: TParams): this;
  /**
   * クエリパラメータを削除する
   * @protected
   * @param key 削除するパラメータのキー
   * @returns {this}
   */
  protected unset(key: keyof TParams): this;
}
declare abstract class NovelSearchBuilderBase<T extends SearchResultFieldNames> extends SearchBuilderBase<SearchParams, Order> {
  /**
   * 検索語を指定します (word)。
   * 半角または全角スペースで区切るとAND抽出になります。部分一致でHITします。
   * @param word 検索語
   * @return {this}
   */
  word(word: string): this;
  /**
   * 除外したい単語を指定します (notword)。
   * スペースで区切ることにより除外する単語を増やせます。部分一致で除外されます。
   * @param word 除外語
   * @return {this}
   */
  notWord(word: string): this;
  /**
   * 検索対象を作品名に限定するかどうかを指定します (title)。
   * @param bool trueの場合、作品名を検索対象とする (デフォルト: true)
   * @return {this}
   */
  byTitle(bool?: boolean): this;
  /**
   * 検索対象をあらすじに限定するかどうかを指定します (ex)。
   * @param bool trueの場合、あらすじを検索対象とする (デフォルト: true)
   * @return {this}
   */
  byOutline(bool?: boolean): this;
  /**
   * 検索対象をキーワードに限定するかどうかを指定します (keyword)。
   * @param bool trueの場合、キーワードを検索対象とする (デフォルト: true)
   * @return {this}
   */
  byKeyword(bool?: boolean): this;
  /**
   * 検索対象を作者名に限定するかどうかを指定します (wname)。
   * @param bool trueの場合、作者名を検索対象とする (デフォルト: true)
   * @return {this}
   */
  byAuthor(bool?: boolean): this;
  /**
   * ボーイズラブ作品を抽出または除外します (isbl/notbl)。
   * @param bool trueの場合、ボーイズラブ作品を抽出する (デフォルト: true)。falseの場合、除外する。
   * @return {this}
   */
  isBL(bool?: boolean): this;
  /**
   * ガールズラブ作品を抽出または除外します (isgl/notgl)。
   * @param bool trueの場合、ガールズラブ作品を抽出する (デフォルト: true)。falseの場合、除外する。
   * @return {this}
   */
  isGL(bool?: boolean): this;
  /**
   * 残酷な描写あり作品を抽出または除外します (iszankoku/notzankoku)。
   * @param bool trueの場合、残酷な描写あり作品を抽出する (デフォルト: true)。falseの場合、除外する。
   * @return {this}
   */
  isZankoku(bool?: boolean): this;
  /**
   * 異世界転生作品を抽出または除外します (istensei/nottensei)。
   * @param bool trueの場合、異世界転生作品を抽出する (デフォルト: true)。falseの場合、除外する。
   * @return {this}
   */
  isTensei(bool?: boolean): this;
  /**
   * 異世界転移作品を抽出または除外します (istenni/nottenni)。
   * @param bool trueの場合、異世界転移作品を抽出する (デフォルト: true)。falseの場合、除外する。
   * @return {this}
   */
  isTenni(bool?: boolean): this;
  /**
   * 異世界転生または異世界転移作品を抽出します (istt)。
   * @return {this}
   */
  isTT(): this;
  /**
   * 抽出する作品の文字数を指定します (length)。
   * 範囲指定する場合は、最小文字数と最大文字数をハイフン(-)記号で区切ってください。
   * @param length 文字数、または[最小文字数, 最大文字数]
   * @return {this}
   */
  length(length: number | readonly number[]): this;
  /**
   * 抽出する作品の会話率を%単位で指定します (kaiwaritu)。
   * @param num 会話率(%)
   * @return {this}
   */
  kaiwaritu(num: number): this;
  /**
   * 抽出する作品の会話率を%単位で範囲指定します (kaiwaritu)。
   * @param min 最低会話率(%)
   * @param max 最高会話率(%)
   * @return {this}
   */
  kaiwaritu(min: number, max: number): this;
  /**
   * 抽出する作品の挿絵数を指定します (sasie)。
   * @param num 挿絵数、または[最小挿絵数, 最大挿絵数]
   * @return {this}
   */
  sasie(num: number | readonly number[]): this;
  /**
   * 抽出する作品の予想読了時間を分単位で指定します (time)。
   * @param num 読了時間(分)、または[最小読了時間, 最大読了時間]
   * @return {this}
   */
  time(num: number | readonly number[]): this;
  /**
   * Nコードを指定して取得します (ncode)。
   * @param ncodes Nコード、またはNコードの配列
   * @return {this}
   */
  ncode(ncodes: string | readonly string[]): this;
  /**
   * 抽出する小説タイプを指定します (type)。
   * @param type 小説タイプ (t: 短編, r: 連載中, er: 完結済連載小説, ter: 短編と完結済連載小説, re: 連載中と完結済連載小説)
   * @return {this}
   */
  type(type: NovelTypeParam): this;
  /**
   * 抽出する作品の文体を指定します (buntai)。
   * 複数指定する場合はハイフン(-)で区切ってください。
   * @param buntai 文体コード、または文体コードの配列
   * @return {this}
   */
  buntai(buntai: BuntaiParam | readonly BuntaiParam[]): this;
  /**
   * 連載停止中作品に関する指定をします (stop)。
   * @param bool trueの場合、長期連載停止中のみ取得する (デフォルト: true)。falseの場合、長期連載停止中を除外する。
   * @return {this}
   */
  isStop(bool?: boolean): this;
  /**
   * ピックアップ作品のみを取得します (ispickup)。
   * @return {this}
   */
  isPickup(): this;
  /**
   * 最終更新日時を指定します (lastup)。
   * @param date 最終更新日時 (YYYYMMDDhhmmss形式またはUNIXタイムスタンプ)
   * @return {this}
   */
  lastUpdate(date: DateParam): this;
  /**
   * 最終更新日時の範囲を指定します (lastup)。
   * @param from 開始日時 (UNIXタイムスタンプ)
   * @param to 終了日時 (UNIXタイムスタンプ)
   * @return {this}
   */
  lastUpdate(from: number, to: number): this;
  /**
   * 最終更新日時の範囲を指定します (lastup)。
   * @param from 開始日時 (Dateオブジェクト)
   * @param to 終了日時 (Dateオブジェクト)
   * @return {this}
   */
  lastUpdate(from: Date, to: Date): this;
  /**
   * 作品の更新日時を指定します (lastupdate)。
   * @param date 作品の更新日時 (YYYYMMDDhhmmss形式またはUNIXタイムスタンプ)
   * @return {this}
   */
  lastNovelUpdate(date: DateParam): this;
  /**
   * 作品の更新日時の範囲を指定します (lastupdate)。
   * @param from 開始日時 (UNIXタイムスタンプ)
   * @param to 終了日時 (UNIXタイムスタンプ)
   * @return {this}
   */
  lastNovelUpdate(from: number, to: number): this;
  /**
   * 作品の更新日時の範囲を指定します (lastupdate)。
   * @param from 開始日時 (Dateオブジェクト)
   * @param to 終了日時 (Dateオブジェクト)
   * @return {this}
   */
  lastNovelUpdate(from: Date, to: Date): this;
  /**
   * なろう小説APIへの検索リクエストを実行する
   * @param options 実行オプション
   * @returns {Promise<NarouSearchResults>} 検索結果
   */
  execute(options?: ExecuteOptions): Promise<NarouSearchResults<NarouSearchResult, T>>;
}
/**
 * 検索ヘルパー
 * @class SearchBuilder
 */
declare class SearchBuilder<T extends keyof NarouSearchResult = DefaultSearchResultFields, TOpt extends keyof NarouSearchResult = never> extends NovelSearchBuilderBase<T | TOpt> {
  /**
   * 大ジャンルを指定して取得します (biggenre)。
   * 複数指定する場合はハイフン(-)で区切ってください。
   * @param genre 大ジャンルコード、または大ジャンルコードの配列
   * @return {this}
   */
  bigGenre(genre: BigGenre | readonly BigGenre[]): this;
  /**
   * 除外したい大ジャンルを指定します (notbiggenre)。
   * 複数指定する場合はハイフン(-)で区切ってください。
   * @param genre 除外する大ジャンルコード、または大ジャンルコードの配列
   * @return {this}
   */
  notBigGenre(genre: BigGenre | readonly BigGenre[]): this;
  /**
   * ジャンルを指定して取得します (genre)。
   * 複数指定する場合はハイフン(-)で区切ってください。
   * @param genre ジャンルコード、またはジャンルコードの配列
   * @return {this}
   */
  genre(genre: Genre | readonly Genre[]): this;
  /**
   * 除外したいジャンルを指定します (notgenre)。
   * 複数指定する場合はハイフン(-)で区切ってください。
   * @param genre 除外するジャンルコード、またはジャンルコードの配列
   * @return {this}
   */
  notGenre(genre: Genre | readonly Genre[]): this;
  /**
   * ユーザIDを指定して取得します (userid)。
   * 複数指定する場合はハイフン(-)で区切ってください。
   * @param ids ユーザID、またはユーザIDの配列
   * @return {this}
   */
  userId(ids: number | readonly number[]): this;
  /**
   * R15作品を抽出または除外します (isr15/notr15)。
   * @param bool trueの場合、R15作品を抽出する (デフォルト: true)。falseの場合、除外する。
   * @return {this}
   */
  isR15(bool?: boolean): this;
  /**
   * 出力する項目を個別に指定します (of)。
   * 未指定時は全項目出力されます。転送量軽減のため、このパラメータの使用が推奨されます。
   * 複数項目を出力する場合はハイフン(-)記号で区切ってください。
   * @param fields 出力するフィールド名、またはフィールド名の配列
   * @return {SearchBuilder<SearchResultFields<TFields>, TOpt>} 型が更新されたビルダー
   */
  fields<TFields extends Fields>(fields: TFields | readonly TFields[]): SearchBuilder<SearchResultFields<TFields>, TOpt>;
  /**
   * 出力オプション項目を指定します (opt)。
   * 複数項目を出力する場合はハイフン(-)記号で区切ってください。
   * @param option 出力するオプションフィールド名、またはオプションフィールド名の配列
   * @return {SearchBuilder<T, SearchResultOptionalFields<TFields>>} 型が更新されたビルダー
   */
  opt<TFields extends OptionalFields>(option: TFields | readonly TFields[]): SearchBuilder<T, SearchResultOptionalFields<TFields>>;
}
//#endregion
//#region src/narou-ranking-results.d.ts
/**
 * ランキングAPIの結果
 * @see https://dev.syosetu.com/man/rankapi/#output
 */
interface NarouRankingResult {
  /** Nコード */
  ncode: string;
  /** 順位 */
  rank: number;
  /** ポイント */
  pt: number;
}
/**
 * ランキングと小説情報をマージした結果
 */
type RankingResult<T extends SearchResultFieldNames = DefaultSearchResultFields> = Partial<PickedNarouSearchResult<T>> & NarouRankingResult;
//#endregion
//#region src/ranking-history.d.ts
interface RankingHistoryRawResult {
  rtype: `${string}-${RankingType}`;
  pt: number;
  rank: number;
}
interface RankingHistoryResult {
  type: RankingType;
  date: Date;
  pt: number;
  rank: number;
}
/**
 * 生のランキング履歴エントリを構造化された形式にフォーマットします。
 *
 * @param rankin - フォーマットする生のランキング履歴データ
 * @returns 日付とタイプが解析されたフォーマット済みランキング履歴
 *
 * @example
 * const rawData = { rtype: "20230101-daily", pt: 500, rank: 10 };
 * const formattedData = formatRankingHistory(rawData);
 * // 返り値: { type: "daily", date: [Dateオブジェクト], pt: 500, rank: 10 }
 */
declare function formatRankingHistory(rankin: RankingHistoryRawResult): RankingHistoryResult;
//#endregion
//#region src/narou.d.ts
/**
 * なろう小説APIへのリクエストパラメータ
 */
type NarouParams = SearchParams | RankingParams | RankingHistoryParams | UserSearchParams;
/**
 * なろう小説APIへのリクエストオプション
 */
interface ExecuteOptions {
  /**
   * fetch関数のオプション
   */
  fetchOptions?: RequestInit;
}
/**
 * なろう小説APIへのリクエストを実行する
 * @class NarouNovel
 * @private
 */
declare abstract class NarouNovel {
  /**
   * なろうAPIへのAPIリクエストを実行する
   * @param params クエリパラメータ
   * @param endpoint APIエンドポイント
   * @returns 実行結果
   */
  protected abstract execute<T>(params: NarouParams, endpoint: string, options?: ExecuteOptions): Promise<T>;
  /**
   * APIへの検索リクエストを実行する
   * @param params クエリパラメータ
   * @param endpoint APIエンドポイント
   * @returns 検索結果
   */
  protected executeSearch<T extends keyof NarouSearchResult>(params: SearchParams, endpoint?: string, options?: ExecuteOptions): Promise<NarouSearchResults<NarouSearchResult, T>>;
  /**
   * 小説APIへの検索リクエストを実行する
   * @param params クエリパラメータ
   * @returns 検索結果
   * @see https://dev.syosetu.com/man/api/
   */
  executeNovel<T extends keyof NarouSearchResult>(params: SearchParams, options?: ExecuteOptions): Promise<NarouSearchResults<NarouSearchResult, T>>;
  /**
   * R18小説APIへの検索リクエストを実行する
   * @param params クエリパラメータ
   * @returns 検索結果
   * @see https://dev.syosetu.com/xman/api/
   */
  executeNovel18<T extends keyof NarouSearchResult>(params: SearchParams, options?: ExecuteOptions): Promise<NarouSearchResults<NarouSearchResult, T>>;
  /**
   * ランキングAPIへのリクエストを実行する
   * @param params クエリパラメータ
   * @returns ランキング結果
   * @see https://dev.syosetu.com/man/rankapi/
   */
  executeRanking(params: RankingParams, options?: ExecuteOptions): Promise<NarouRankingResult[]>;
  /**
   * 殿堂入りAPiへのリクエストを実行する
   * @param params クエリパラメータ
   * @param options 実行オプション
   * @returns ランキング履歴結果
   * @see https://dev.syosetu.com/man/rankinapi/
   */
  executeRankingHistory(params: RankingHistoryParams, options?: ExecuteOptions): Promise<RankingHistoryRawResult[]>;
  /**
   * ユーザー検索APIへのリクエストを実行する
   * @param params クエリパラメータ
   * @returns 検索結果
   * @see https://dev.syosetu.com/man/userapi/
   */
  executeUserSearch<T extends keyof UserSearchResult>(params: UserSearchParams, options?: ExecuteOptions): Promise<NarouSearchResults<UserSearchResult, T>>;
}
//#endregion
//#region src/narou-jsonp.d.ts
/**
 * なろう小説APIへのリクエストを実行する
 */
declare class NarouNovelJsonp extends NarouNovel {
  protected execute<T>(params: NarouParams, endpoint: string, _options?: ExecuteOptions): Promise<T>;
}
//#endregion
//#region src/search-builder-r18.d.ts
type DefaultR18SearchResultFields = keyof Omit<NarouSearchResult, "weekly_unique" | "noveltype" | "biggenre" | "genre" | "isr15" | "id">;
/**
 * 18禁API検索ヘルパー
 * @class SearchBuilderR18
 */
declare class SearchBuilderR18<T extends SearchResultFieldNames = DefaultR18SearchResultFields, TOpt extends keyof NarouSearchResult = never> extends NovelSearchBuilderBase<T | TOpt> {
  /**
   * なろう小説APIへの検索リクエストを実行する
   * @override
   * @param options 実行オプション
   * @returns {Promise<NarouSearchResults>} 検索結果
   */
  execute(options?: ExecuteOptions): Promise<NarouSearchResults<NarouSearchResult, T | TOpt>>;
  /**
   * 抽出するR18サイトを指定します (nocgenre)。
   * @param sites R18サイトコード、またはR18サイトコードの配列 (1: ノクターンノベルズ, 2: ムーンライトノベルズ(男性向け), 3: ムーンライトノベルズ(BL), 4: ミッドナイトノベルズ)
   * @return {this}
   */
  r18Site(sites: R18Site | readonly R18Site[]): this;
  /**
   * X-IDを指定して取得します (xid)。
   * @param ids X-ID、またはX-IDの配列
   * @return {this}
   */
  xid(ids: number | readonly number[]): this;
  /**
   * 出力する項目を個別に指定します (of)。
   * 未指定時は全項目出力されます。転送量軽減のため、このパラメータの使用が推奨されます。
   * @param fields 出力するR18フィールド名、またはR18フィールド名の配列
   * @return {SearchBuilderR18<SearchResultR18Fields<R18Fields>>} 型が更新されたビルダー
   */
  fields<TFields extends R18Fields>(fields: TFields | readonly TFields[]): SearchBuilderR18<SearchResultR18Fields<R18Fields>>;
  /**
   * 出力オプション項目を指定します (opt)。
   * @param option 出力するオプションフィールド名、またはオプションフィールド名の配列
   * @return {SearchBuilderR18<T, SearchResultOptionalFields<TFields>>} 型が更新されたビルダー
   */
  opt<TFields extends OptionalFields>(option: TFields | readonly TFields[]): SearchBuilderR18<T, SearchResultOptionalFields<TFields>>;
}
//#endregion
//#region src/ranking.d.ts
/**
 * なろう小説ランキングAPIのヘルパークラス。
 *
 * ランキング種別や日付を指定してランキングデータを取得します。
 * また、取得したランキングデータに含まれるNコードを元に、
 * なろう小説APIを利用して詳細な小説情報を取得することも可能です。
 *
 * @class RankingBuilder
 * @see https://dev.syosetu.com/man/rankapi/ なろう小説ランキングAPI仕様
 */
declare class RankingBuilder {
  protected params: Partial<RankingParams>;
  protected api: NarouNovel;
  /**
   * ランキング集計対象の日付
   * @protected
   */
  protected date$: Date;
  /**
   * ランキング種別
   * @protected
   */
  protected type$: RankingType;
  /**
   * constructor
   * @param params - 初期クエリパラメータ
   * @param api - API実行クラスのインスタンス
   * @private
   */
  constructor(params: Partial<RankingParams> | undefined, api: NarouNovel);
  /**
   * ランキング集計対象の日付を指定します。
   *
   * - 日間: 任意の日付
   * - 週間: 火曜日の日付
   * - 月間・四半期: 1日の日付
   *
   * @param date 集計対象の日付
   * @returns {RankingBuilder} this
   * @see https://dev.syosetu.com/man/rankapi/
   */
  date(date: Date): this;
  /**
   * ランキング種別を指定します。
   * @param type ランキング種別
   * @returns {RankingBuilder} this
   * @see https://dev.syosetu.com/man/rankapi/
   */
  type(type: RankingType): this;
  /**
   * gzip圧縮する。
   *
   * 転送量上限を減らすためにも推奨
   * @param {GzipLevel} level gzip圧縮レベル(1～5)
   * @return {RankingBuilder} this
   */
  gzip(level: GzipLevel): this;
  /**
   * クエリパラメータを内部的にセットします。
   * @param obj - セットするパラメータオブジェクト
   * @returns {RankingBuilder} this
   * @private
   */
  protected set(obj: Partial<RankingParams>): this;
  /**
   * 設定されたパラメータに基づき、なろう小説ランキングAPIへのリクエストを実行します。
   *
   * 返される結果には、Nコード、ポイント、順位が含まれます。
   * @param options 実行オプション
   * @returns {Promise<NarouRankingResult[]>} ランキング結果の配列
   * @see https://dev.syosetu.com/man/rankapi/#output
   */
  execute(options?: ExecuteOptions): Promise<NarouRankingResult[]>;
  /**
   * ランキングAPIを実行し、取得したNコードを元になろう小説APIで詳細情報を取得して結合します。
   */
  executeWithFields(fields?: never[] | undefined, opt?: never[] | undefined, options?: ExecuteOptions): Promise<RankingResult<DefaultSearchResultFields>[]>;
  /**
   * ランキングAPIを実行し、取得したNコードを元になろう小説APIで詳細情報を取得して結合します。
   *
   * @template TFields - 取得する小説情報のフィールド型
   * @param fields - 取得するフィールドの配列
   * @returns {Promise<RankingResult<SearchResultFields<TFields>>[]>} 詳細情報を含むランキング結果の配列
   */
  executeWithFields<TFields extends Fields>(fields: TFields | TFields[], opt?: never | never[], options?: ExecuteOptions): Promise<RankingResult<SearchResultFields<TFields>>[]>;
  /**
   * ランキングAPIを実行し、取得したNコードを元になろう小説APIで詳細情報を取得して結合します。
   *
   * @param opt - オプショナルな取得フィールド (`weekly` など)
   * @returns {Promise<RankingResult<DefaultSearchResultFields | "weekly_unique">[]>} 詳細情報を含むランキング結果の配列
   */
  executeWithFields(fields: never[], opt: OptionalFields | OptionalFields[], options?: ExecuteOptions): Promise<RankingResult<DefaultSearchResultFields | "weekly_unique">[]>;
  /**
   * ランキングAPIを実行し、取得したNコードを元になろう小説APIで詳細情報を取得して結合します。
   *
   * @template TFields - 取得する小説情報のフィールド型
   * @param fields - 取得するフィールドの配列
   * @param opt - オプショナルな取得フィールド (`weekly` など)
   * @returns {Promise<RankingResult<SearchResultFields<TFields> | "weekly_unique">[]>} 詳細情報を含むランキング結果の配列
   */
  executeWithFields<TFields extends Fields>(fields: TFields | TFields[], opt: OptionalFields | OptionalFields[], options?: ExecuteOptions): Promise<RankingResult<SearchResultFields<TFields> | "weekly_unique">[]>;
}
//#endregion
//#region src/user-search.d.ts
/**
 * なろうユーザ検索API
 * @class UserSearch
 */
declare class UserSearchBuilder<TField extends keyof UserSearchResult = keyof UserSearchResult> extends SearchBuilderBase<UserSearchParams, UserOrder> {
  /**
   * 単語を指定できます。
   * 半角または全角スペースで区切るとAND抽出になります。
   * 部分一致でHITします。検索の対象はユーザ名とユーザ名のフリガナです。
   */
  word(word: string): this;
  /**
   * 含みたくない単語を指定できます。
   * スペースで区切ることにより含ませない単語を増やせます。部分一致で除外されます。
   * 除外の対象はユーザ名とユーザ名のフリガナです。
   */
  notWord(notword: string): this;
  /**
   * ユーザIDで抽出可能。
   */
  userId(userid: number): this;
  /**
   * 抽出するユーザのユーザ名のフリガナの頭文字を指定できます。
   * 頭文字はユーザ名のフリガナをひらがなに変換し、最初の1文字が「ぁ」～「ん」の場合に対象となります。
   * 「ぱ」や「ば」等の半濁音や濁音は清音として扱われます。
   * 漢字や英数字が頭文字のユーザは対象外です。
   */
  name1st(name1st: string): this;
  /**
   * 抽出するユーザの小説投稿数の下限を指定できます。
   * 小説投稿件数が指定された数値以上のユーザを抽出します。
   */
  minNovel(minnovel: number): this;
  /**
   * 抽出するユーザの小説投稿数の上限を指定できます。
   * 小説投稿件数が指定された数値以下のユーザを抽出します。
   */
  maxNovel(maxnovel: number): this;
  /**
   * 抽出するユーザのレビュー投稿数の下限を指定できます。
   * レビュー投稿件数が指定された数値以上のユーザを抽出します。
   */
  minReview(minreview: number): this;
  /**
   * 抽出するユーザのレビュー投稿数の上限を指定できます。
   * レビュー投稿件数が指定された数値以下のユーザを抽出します。
   */
  maxReview(maxreview: number): this;
  /**
   * 出力する項目を個別に指定できます。未指定時は全項目出力されます。転送量軽減のため、このパラメータの使用が推奨されます。
   * @return {SearchBuilder} this
   */
  fields<TFields extends UserFields>(fields: TFields | readonly TFields[]): UserSearchBuilder<UserSearchResultFields<TFields>>;
  /**
   * なろう小説APIへのリクエストを実行する
   * @param options 実行オプション
   * @returns ランキング
   */
  execute(options?: ExecuteOptions): Promise<NarouSearchResults<UserSearchResult, TField>>;
}
//#endregion
export { UserSearchParams as $, BooleanNumber as A, ParamsBase as B, SearchResultFields as C, UserSearchResultFields as D, UserSearchResult as E, GenreNotation as F, RankingHistoryParams as G, R18Fields as H, GzipLevel as I, SearchParams as J, RankingParams as K, NovelTypeParam as L, DateParam as M, Fields as N, BigGenre as O, Genre as P, UserOrder as Q, OptionalFields as R, PickedNarouSearchResult as S, SearchResultR18Fields as T, R18Site as U, ParamsBaseWithOrder as V, R18SiteNotation as W, StopParam as X, SearchResultFieldNames as Y, UserFields as Z, SearchBuilderBase as _, NarouNovelJsonp as a, NarouSearchResults as b, NarouParams as c, formatRankingHistory as d, NarouRankingResult as f, SearchBuilder as g, NovelSearchBuilderBase as h, SearchBuilderR18 as i, BuntaiParam as j, BigGenreNotation as k, RankingHistoryRawResult as l, DefaultSearchResultFields as m, RankingBuilder as n, ExecuteOptions as o, RankingResult as p, RankingType as q, DefaultR18SearchResultFields as r, NarouNovel as s, UserSearchBuilder as t, RankingHistoryResult as u, End as v, SearchResultOptionalFields as w, NovelType as x, NarouSearchResult as y, Order as z };
//# sourceMappingURL=user-search-BY6FdEe7.d.cts.map