import type NarouNovel from "./narou.js";
import type { ExecuteOptions } from "./narou.js";
import NarouNovelJsonp from "./narou-jsonp.js";
import SearchBuilder from "./search-builder.js";
import SearchBuilderR18 from "./search-builder-r18.js";
import RankingBuilder from "./ranking.js";
import { formatRankingHistory, RankingHistoryResult } from "./ranking-history.js";
import UserSearchBuilder from "./user-search.js";

export * from "./index.common.js";
export { NarouNovelJsonp };

const narouNovelJsonp = new NarouNovelJsonp();

/**
 * なろう小説 API で小説を検索する
 * @param {string} [word] 検索ワード
 * @returns {SearchBuilder}
 * @see https://dev.syosetu.com/man/api/
 */
export function search(
  word = "",
  api: NarouNovel = narouNovelJsonp
): SearchBuilder {
  const builder = new SearchBuilder({}, api);
  if (word != "") builder.word(word);
  return builder;
}

/**
 * 18禁小説 API で小説を検索する
 * @param {string} [word] 検索ワード
 * @returns {SearchBuilder}
 * @see https://dev.syosetu.com/xman/api/
 */
export function searchR18(
  word = "",
  api: NarouNovel = narouNovelJsonp
): SearchBuilderR18 {
  const builder = new SearchBuilderR18({}, api);
  if (word != "") builder.word(word);
  return builder;
}

/**
 * なろうユーザ検索 API でユーザを検索する
 * @param {string} [word] - 検索ワード
 * @returns {UserSearchBuilder}
 * @see https://dev.syosetu.com/man/userapi/
 */
export function searchUser(word = "", api: NarouNovel = narouNovelJsonp) {
  const builder = new UserSearchBuilder({}, api);
  if (word != "") builder.word(word);
  return builder;
}


/**
 * なろう小説ランキング API でランキングを取得する
 * @returns {RankingBuilder}
 * @see https://dev.syosetu.com/man/rankapi/
 */
export function ranking(api: NarouNovel = narouNovelJsonp): RankingBuilder {
  const builder = new RankingBuilder({}, api);
  return builder;
}

/**
 * なろう殿堂入り API でランキング履歴を取得する
 * @param {string} ncode 小説のNコード
 * @param {ExecuteOptions} [options] 実行オプション
 * @param {NarouNovel} [api] API実行クラスのインスタンス
 * @see https://dev.syosetu.com/man/rankinapi/
 */
export async function rankingHistory(
  ncode: string,
  options?: ExecuteOptions,
  api: NarouNovel = narouNovelJsonp
): Promise<RankingHistoryResult[]> {
  const result = await api.executeRankingHistory({ ncode }, options);
  if (Array.isArray(result)) {
    return result.map(formatRankingHistory);
  } else {
    throw new Error(result);
  }
}

export default {
  search,
  searchR18,
  searchUser,
  ranking,
  rankingHistory,
};
