#include "dialog/lyricssearch.hpp"
#include "view/sidepanel/view.hpp"

#include <QGroupBox>
#include <QMessageBox>
#include <QPushButton>

Dialog::LyricsSearch::LyricsSearch(const HttpClient &httpClient, QWidget *parent)
	: Base(parent),
	lyrics(httpClient)
{
	lyrics.setAppInfo(QStringLiteral(APP_NAME), QStringLiteral(APP_VERSION),
		QStringLiteral("https://github.com/%1/%2")
		.arg(QStringLiteral(ORG_NAME), QStringLiteral(APP_NAME)));

	auto *layout = Base::layout<QVBoxLayout>();

	searchBox = new QGroupBox(this);
	searchBox->setTitle(QStringLiteral("Query"));
	auto *searchLayout = new QHBoxLayout(searchBox);
	layout->addWidget(searchBox);

	search = new QLineEdit(this);
	searchLayout->addWidget(search);

	auto *searchButton = new QPushButton(this);
	searchButton->setText(QStringLiteral("Search"));
	searchButton->setDefault(true);
	searchLayout->addWidget(searchButton);

	QPushButton::connect(searchButton, &QPushButton::clicked,
		this, &Dialog::LyricsSearch::onSearchClicked);

	results = new QTreeWidget(this);
	results->setHeaderLabels({
		QStringLiteral("ID"),
		QStringLiteral("Title"),
		QStringLiteral("Artist"),
		QStringLiteral("Album"),
	});
	layout->addWidget(results, 1);

	QTreeWidget::connect(results, &QTreeWidget::itemDoubleClicked,
		this, &Dialog::LyricsSearch::onResultsItemDoubleClicked);

	auto *header = results->header();
	header->resizeSection(0, widthId);
	header->resizeSection(1, widthTitle);
	header->resizeSection(2, widthArtist);

	Base::addAction(DialogAction::Ok);
	resize(width, height);
}

void Dialog::LyricsSearch::onSearchClicked(bool /*checked*/)
{
	searchBox->setEnabled(false);
	results->clear();

	const QString query = search->text();
	lyrics.search(query, [this](const Result<QList<Lyrics>> &result)
	{
		if (!result.success())
		{
			QMessageBox::warning(this, QStringLiteral("Error"), result.message());
		}
		else
		{
			for (const auto &lyrics: result.value())
			{
				auto *item = new QTreeWidgetItem(results);
				item->setText(0, QString::number(lyrics.id()));
				item->setText(1, lyrics.trackName());
				item->setText(2, lyrics.artistName());
				item->setText(3, lyrics.albumName());
			}
		}

		searchBox->setEnabled(true);
	});
}

void Dialog::LyricsSearch::onResultsItemDoubleClicked(QTreeWidgetItem *item, int /*column*/)
{
	SidePanel::View *sidePanel = nullptr;
	for (auto *window: QApplication::topLevelWidgets())
	{
		sidePanel = window->findChild<SidePanel::View *>();
		if (sidePanel != nullptr)
		{
			break;
		}
	}

	if (sidePanel == nullptr)
	{
		return;
	}

	const auto lyricsId = item->text(0).toInt();
	sidePanel->openLyrics(lyricsId);
	accept();
}
