import colors from "ansi-colors";
import { AnyObject, AnyString, MaybeArray, MaybeFunc } from "@hey-api/types";

//#region src/refs/types.d.ts

/**
 * Ref wrapper which ensures a stable reference for a value.
 *
 * @example
 * ```ts
 * type NumRef = Ref<number>; // { '~ref': number }
 * const num: NumRef = { '~ref': 42 };
 * console.log(num['~ref']); // 42
 * ```
 */
type Ref<T> = T extends {
  ['~ref']: unknown;
} ? T : {
  '~ref': T;
};
/**
 * Maps every property of `T` to a `Ref` of that property.
 *
 * @example
 * ```ts
 * type Foo = { a: number; b: string };
 * type Refs = Refs<Foo>; // { a: Ref<number>; b: Ref<string> }
 * const refs: Refs = { a: { '~ref': 1 }, b: { '~ref': 'x' } };
 * console.log(refs.a['~ref'], refs.b['~ref']); // 1 'x'
 * ```
 */
type Refs<T> = { [K in keyof T]: Ref<T[K]> };
/**
 * Unwraps a Ref to its value type.
 *
 * @example
 * ```ts
 * type N = FromRef<{ '~ref': number }>; // number
 * ```
 */
type FromRef<T> = T extends {
  '~ref': infer U;
} ? U : T;
/**
 * Maps every property of a Ref-wrapped object back to its plain value.
 *
 * @example
 * ```ts
 * type Foo = { a: number; b: string };
 * type Refs = Refs<Foo>; // { a: Ref<number>; b: Ref<string> }
 * type Foo2 = FromRefs<Refs>; // { a: number; b: string }
 * ```
 */
type FromRefs<T> = { [K in keyof T]: T[K] extends Ref<infer U> ? U : T[K] };
//#endregion
//#region src/extensions.d.ts
/**
 * Arbitrary metadata passed to the project's render function.
 *
 * Implementers should extend this interface for their own needs.
 */
interface IProjectRenderMeta {
  [key: string]: unknown;
}
/**
 * Additional metadata about the symbol.
 *
 * Implementers should extend this interface for their own needs.
 */
interface ISymbolMeta {
  [key: string]: unknown;
}
//#endregion
//#region src/nodes/node.d.ts
type MaybeRef<T> = T | Ref<T>;
type NodeName = MaybeRef<Symbol | string | number>;
type NodeNameSanitizer = (name: string) => string;
type NodeRelationship = 'container' | 'reference';
type NodeScope = 'type' | 'value';
interface INode<T = unknown> {
  /** Perform semantic analysis. */
  analyze(ctx: IAnalysisContext): void;
  /** Create a shallow copy of this node. */
  clone(): this;
  /** Whether this node is exported from its file. */
  exported?: boolean;
  /** The file this node belongs to. */
  file?: File;
  /** The programming language associated with this node */
  language: Language;
  /** The display name of this node. */
  readonly name: Ref<NodeName> & {
    set(value: NodeName): void;
    toString(): string;
  };
  /** Optional function to sanitize the node name. */
  readonly nameSanitizer?: NodeNameSanitizer;
  /** Whether this node is a root node in the file. */
  root?: boolean;
  /** The scope of this node. */
  scope?: NodeScope;
  /** Semantic children in the structure hierarchy. */
  structuralChildren?: Map<INode, NodeRelationship>;
  /** Semantic parents in the structure hierarchy. */
  structuralParents?: Map<INode, NodeRelationship>;
  /** The symbol associated with this node. */
  symbol?: Symbol;
  /** Convert this node into AST representation. */
  toAst(): T;
  /** Brand used for renderer dispatch. */
  readonly '~brand': string;
}
//#endregion
//#region src/symbols/types.d.ts
type BindingKind = 'default' | 'named' | 'namespace';
type ISymbolIdentifier = number | ISymbolMeta;
type SymbolKind = 'class' | 'enum' | 'function' | 'interface' | 'namespace' | 'type' | 'var';
type ISymbolIn = {
  /**
   * Whether this symbol is exported from its own file.
   *
   * @default false
   */
  exported?: boolean;
  /**
   * External module name if this symbol is imported from a module not managed
   * by the project (e.g. "zod", "lodash").
   *
   * @default undefined
   */
  external?: string;
  /**
   * Optional output strategy to override default behavior.
   *
   * @returns The file path(s) that re-export this symbol, or undefined if none.
   */
  getExportFromFilePath?: Symbol['getExportFromFilePath'];
  /**
   * Optional output strategy to override default behavior.
   *
   * @returns The file path to output the symbol to, or undefined to fallback to default behavior.
   */
  getFilePath?: Symbol['getFilePath'];
  /**
   * Kind of import if this symbol represents an import.
   *
   * @default 'named'
   */
  importKind?: BindingKind;
  /**
   * Kind of symbol.
   *
   * @default 'var'
   */
  kind?: SymbolKind;
  /**
   * Arbitrary metadata about the symbol.
   *
   * @default undefined
   */
  meta?: ISymbolMeta;
  /**
   * The intended, user-facing name of the symbol before any conflict resolution.
   * It is **not** guaranteed to be the final emitted name — aliasing may occur if the
   * file contains conflicting local identifiers or other symbols with the same intended name.
   *
   * @example "UserModel"
   */
  name: string;
};
interface ISymbolRegistry {
  /**
   * Get a symbol.
   *
   * @param identifier Symbol identifier to reference.
   * @returns The symbol, or undefined if not found.
   */
  get(identifier: ISymbolIdentifier): Symbol | undefined;
  /**
   * Returns whether a symbol is registered in the registry.
   *
   * @param identifier Symbol identifier to check.
   * @returns True if the symbol is registered, false otherwise.
   */
  isRegistered(identifier: ISymbolIdentifier): boolean;
  /**
   * Returns the current symbol ID and increments it.
   *
   * @returns Symbol ID before being incremented.
   */
  readonly nextId: number;
  /**
   * Queries symbols by metadata filter.
   *
   * @param filter Metadata filter to query symbols by.
   * @returns Array of symbols matching the filter.
   */
  query(filter: ISymbolMeta): ReadonlyArray<Symbol>;
  /**
   * References a symbol.
   *
   * @param meta Metadata filter to reference symbol by.
   * @returns The referenced symbol.
   */
  reference(meta: ISymbolMeta): Symbol;
  /**
   * Register a symbol globally.
   *
   * Deduplicates identical symbols by ID.
   *
   * @param symbol Symbol to register.
   * @returns The registered symbol.
   */
  register(symbol: ISymbolIn): Symbol;
  /**
   * Get all symbols in the order they were registered.
   *
   * @returns Array of all registered symbols, in insert order.
   */
  registered(): IterableIterator<Symbol>;
}
//#endregion
//#region src/symbols/symbol.d.ts
declare class Symbol<Node extends INode = INode> {
  /**
   * Canonical symbol this stub resolves to, if any.
   *
   * Stubs created during DSL construction may later be associated
   * with a fully registered symbol. Once set, all property lookups
   * should defer to the canonical symbol.
   */
  private _canonical?;
  /**
   * True if this symbol is exported from its defining file.
   *
   * @default false
   */
  private _exported;
  /**
   * External module name if this symbol is imported from a module not managed
   * by the project (e.g. "zod", "lodash").
   *
   * @default undefined
   */
  private _external?;
  /**
   * The file this symbol is ultimately emitted into.
   *
   * Only top-level symbols have an assigned file.
   */
  private _file?;
  /**
   * The alias-resolved, conflict-free emitted name.
   */
  private _finalName?;
  /**
   * Custom strategy to determine from which file path(s) this symbol is re-exported.
   *
   * @returns The file path(s) that re-export this symbol, or undefined if none.
   */
  private _getExportFromFilePath?;
  /**
   * Custom strategy to determine file output path.
   *
   * @returns The file path to output the symbol to, or undefined to fallback to default behavior.
   */
  private _getFilePath?;
  /**
   * How this symbol should be imported (namespace/default/named).
   *
   * @default 'named'
   */
  private _importKind;
  /**
   * Kind of symbol (class, type, alias, etc.).
   *
   * @default 'var'
   */
  private _kind;
  /**
   * Arbitrary user metadata.
   *
   * @default undefined
   */
  private _meta?;
  /**
   * Intended user-facing name before conflict resolution.
   *
   * @example "UserModel"
   */
  private _name;
  /**
   * Node that defines this symbol.
   */
  private _node?;
  /** Brand used for identifying symbols. */
  readonly '~brand' = "heyapi.symbol";
  /** Globally unique, stable symbol ID. */
  readonly id: number;
  constructor(input: ISymbolIn, id: number);
  /**
   * Returns the canonical symbol for this instance.
   *
   * If this symbol was created as a stub, this getter returns
   * the fully registered canonical symbol. Otherwise, it returns
   * the symbol itself.
   */
  get canonical(): Symbol;
  /**
   * Indicates whether this symbol is exported from its defining file.
   */
  get exported(): boolean;
  /**
   * External module from which this symbol originates, if any.
   */
  get external(): string | undefined;
  /**
   * Read‑only accessor for the assigned output file.
   *
   * Only top-level symbols have an assigned file.
   */
  get file(): File | undefined;
  /**
   * Read‑only accessor for the resolved final emitted name.
   */
  get finalName(): string;
  /**
   * Custom re-export file path resolver, if provided.
   */
  get getExportFromFilePath(): ((symbol: Symbol) => ReadonlyArray<string> | undefined) | undefined;
  /**
   * Custom file path resolver, if provided.
   */
  get getFilePath(): ((symbol: Symbol) => string | undefined) | undefined;
  /**
   * How this symbol should be imported (named/default/namespace).
   */
  get importKind(): BindingKind;
  /**
   * Indicates whether this is a canonical symbol (not a stub).
   */
  get isCanonical(): boolean;
  /**
   * The symbol's kind (class, type, alias, variable, etc.).
   */
  get kind(): SymbolKind;
  /**
   * Arbitrary user‑provided metadata associated with this symbol.
   */
  get meta(): ISymbolMeta | undefined;
  /**
   * User-intended name before aliasing or conflict resolution.
   */
  get name(): string;
  /**
   * Read‑only accessor for the defining node.
   */
  get node(): Node | undefined;
  /**
   * Marks this symbol as a stub and assigns its canonical symbol.
   *
   * After calling this, all semantic queries (name, kind, file,
   * meta, etc.) should reflect the canonical symbol's values.
   *
   * @param symbol — The canonical symbol this stub should resolve to.
   */
  setCanonical(symbol: Symbol): void;
  /**
   * Marks the symbol as exported from its file.
   *
   * @param exported — Whether the symbol is exported.
   */
  setExported(exported: boolean): void;
  /**
   * Assigns the output file this symbol will be emitted into.
   *
   * This may only be set once.
   */
  setFile(file: File): void;
  /**
   * Assigns the conflict‑resolved final local name for this symbol.
   *
   * This may only be set once.
   */
  setFinalName(name: string): void;
  /**
   * Sets how this symbol should be imported.
   *
   * @param kind — The import strategy (named/default/namespace).
   */
  setImportKind(kind: BindingKind): void;
  /**
   * Sets the symbol's kind (class, type, alias, variable, etc.).
   *
   * @param kind — The new symbol kind.
   */
  setKind(kind: SymbolKind): void;
  /**
   * Updates the intended user‑facing name for this symbol.
   *
   * @param name — The new name.
   */
  setName(name: string): void;
  /**
   * Binds the node that defines this symbol.
   *
   * This may only be set once.
   */
  setNode(node: Node): void;
  /**
   * Returns a debug‑friendly string representation identifying the symbol.
   */
  toString(): string;
  /**
   * Ensures this symbol is canonical before allowing mutation.
   *
   * A symbol that has been marked as a stub (i.e., its `_canonical` points
   * to a different symbol) may not be mutated. This guard throws an error
   * if any setter attempts to modify a stub, preventing accidental writes
   * to non‑canonical instances.
   *
   * @throws {Error} If the symbol is a stub and is being mutated.
   */
  private assertCanonical;
}
//#endregion
//#region src/planner/scope.d.ts
type NameScopes = Map<string, Set<SymbolKind>>;
type Scope = {
  /** Child scopes. */
  children: Array<Scope>;
  /** Resolved names in this scope. */
  localNames: NameScopes;
  /** Parent scope, if any. */
  parent?: Scope;
  /** Symbols registered in this scope. */
  symbols: Array<Ref<Symbol>>;
};
//#endregion
//#region src/planner/types.d.ts
type Input = Ref<object> | object | string | number | undefined;
type NameConflictResolver = (args: {
  attempt: number;
  baseName: string;
}) => string | null;
interface IAnalysisContext {
  /** Register a dependency on another symbol. */
  addDependency(symbol: Ref<Symbol>): void;
  /** Register a dependency on another symbol or analyze further. */
  analyze(input: Input): void;
  /** Get local names in the current scope. */
  localNames(scope: Scope): NameScopes;
  /** Pop the current local scope. */
  popScope(): void;
  /** Push a new local scope. */
  pushScope(): void;
  /** Current local scope. */
  scope: Scope;
  /** Stack of local name scopes. */
  scopes: Scope;
  /** Top-level symbol for the current analysis pass. */
  symbol?: Symbol;
  /** Walks all symbols in the scope tree in depth-first order. */
  walkScopes(callback: (symbol: Ref<Symbol>, scope: Scope) => void, scope?: Scope): void;
}
//#endregion
//#region src/languages/types.d.ts
/**
 * Map of extensions for each language.
 *
 * @example
 * ```ts
 * const exts: Extensions = {
 *   typescript: ['.ts', '.tsx'],
 *   python: ['.py'],
 * };
 * ```
 */
type Extensions = Partial<Record<Language, ReadonlyArray<string>>>;
type Language = 'c' | 'c#' | 'c++' | 'css' | 'dart' | 'go' | 'haskell' | 'html' | 'java' | 'javascript' | 'json' | 'kotlin' | 'lua' | 'markdown' | 'matlab' | 'perl' | 'php' | 'python' | 'r' | 'ruby' | 'rust' | 'scala' | 'shell' | 'sql' | 'swift' | 'typescript' | 'yaml' | AnyString;
/**
 * Map of module entry names for each language.
 *
 * @example
 * ```ts
 * const entries: ModuleEntryNames = {
 *   typescript: 'index',
 *   python: '__init__',
 * };
 * ```
 */
type ModuleEntryNames = Partial<Record<Language, string>>;
type NameConflictResolvers = Partial<Record<Language, NameConflictResolver>>;
//#endregion
//#region src/files/types.d.ts
type FileKeyArgs = Pick<Required<File>, 'logicalFilePath'> & Pick<Partial<File>, 'external' | 'language'>;
type IFileIn = {
  /**
   * Indicates whether the file is external, meaning it is not generated
   * as part of the project but is referenced (e.g., a module from
   * node_modules).
   *
   * @example true
   */
  external?: boolean;
  /**
   * Language of the file.
   *
   * @example "typescript"
   */
  language?: Language;
  /**
   * Logical, extension-free path used for planning and routing.
   *
   * @example "src/models/user"
   */
  logicalFilePath: string;
  /**
   * The desired name for the file within the project. If there are multiple files
   * with the same desired name, this might not end up being the actual name.
   *
   * @example "UserModel"
   */
  name?: string;
};
interface IFileRegistry {
  /**
   * Get a file.
   *
   * @returns The file, or undefined if not found.
   */
  get(args: FileKeyArgs): File | undefined;
  /**
   * Returns whether a file is registered in the registry.
   *
   * @returns True if the file is registered, false otherwise.
   */
  isRegistered(args: FileKeyArgs): boolean;
  /**
   * Returns the current file ID and increments it.
   *
   * @returns File ID before being incremented
   */
  readonly nextId: number;
  /**
   * Register a file globally.
   *
   * @param file File to register.
   * @returns Newly registered file if created, merged file otherwise.
   */
  register(file: IFileIn): File;
  /**
   * Get all files in the order they were registered.
   *
   * @returns Array of all registered files, in insert order.
   */
  registered(): IterableIterator<File>;
}
//#endregion
//#region src/nodes/types.d.ts
interface INodeRegistry {
  /**
   * Register a syntax node.
   *
   * @returns The index of the registered node.
   */
  add(node: INode | null): number;
  /**
   * All nodes in insertion order.
   */
  all(): Iterable<INode>;
  /**
   * Remove a node by its index.
   *
   * @param index Index of the node to remove.
   */
  remove(index: number): void;
  /**
   * Update a node at the given index.
   *
   * @param index Index of the node to update.
   * @param node New node to set.
   */
  update(index: number, node: INode | null): void;
}
//#endregion
//#region src/output.d.ts
interface IOutput {
  /**
   * The main content of the file to output.
   *
   * A raw string representing source code.
   *
   * @example "function foo(): void {\n  // implementation\n}\n"
   */
  content: string;
  /**
   * Logical output path (used for writing the file).
   *
   * @example "models/user.ts"
   */
  path: string;
}
//#endregion
//#region src/renderer.d.ts
interface RenderContext<Node extends INode = INode> {
  /**
   * The current file.
   */
  file: File<Node>;
  /**
   * Arbitrary metadata.
   */
  meta?: IProjectRenderMeta;
  /**
   * The project the file belongs to.
   */
  project: IProject;
}
interface Renderer {
  /** Renders the given file. */
  render(ctx: RenderContext): string;
  /** Returns whether this renderer can render the given file. */
  supports(ctx: RenderContext): boolean;
}
//#endregion
//#region src/project/types.d.ts
/**
 * Represents a code generation project consisting of codegen files.
 *
 * Manages imports, symbols, and output generation across the project.
 */
interface IProject {
  /**
   * The default file to assign symbols without a specific file selector.
   *
   * @default 'main'
   */
  readonly defaultFileName: string;
  /** Default name conflict resolver used when a file has no specific resolver. */
  readonly defaultNameConflictResolver: NameConflictResolver;
  /**
   * Maps language to array of extensions. First element is used by default.
   *
   * @example
   * ```ts
   * const exts: Extensions = {
   *   typescript: ['.ts', '.tsx'],
   *   python: ['.py'],
   * };
   * ```
   */
  readonly extensions: Extensions;
  /**
   * Function to transform file names before they are used.
   *
   * @param name The original file name.
   * @returns The transformed file name.
   */
  readonly fileName?: (name: string) => string;
  /** Centralized file registry for the project. */
  readonly files: IFileRegistry;
  /**
   * Map of module entry names for each language.
   *
   * @example
   * ```ts
   * const entries: ModuleEntryNames = {
   *   typescript: 'index',
   *   python: '__init__',
   * };
   * ```
   */
  readonly moduleEntryNames: ModuleEntryNames;
  /**
   * Map of language-specific name conflict resolvers for files in the project.
   *
   * @example
   * ```ts
   * const resolvers: NameConflictResolvers = {
   *   typescript: myTypeScriptResolver,
   *   python: myPythonResolver,
   * };
   * ```
   */
  readonly nameConflictResolvers: NameConflictResolvers;
  /** Centralized node registry for the project. */
  readonly nodes: INodeRegistry;
  /**
   * Finalizes the project structure, resolving nodes, symbols, and dependencies.
   *
   * @param meta Arbitrary metadata.
   * @returns void
   */
  plan(meta?: IProjectRenderMeta): void;
  /**
   * Produces output representations for all files in the project.
   *
   * @param meta Arbitrary metadata.
   * @returns Array of outputs ready for writing or further processing.
   * @example
   * project.render().forEach(output => writeFile(output));
   */
  render(meta?: IProjectRenderMeta): ReadonlyArray<IOutput>;
  /**
   * List of available renderers.
   *
   * @example
   * [new TypeScriptRenderer()]
   */
  readonly renderers: ReadonlyArray<Renderer>;
  /** The absolute path to the root folder of the project. */
  readonly root: string;
  /** Centralized symbol registry for the project. */
  readonly symbols: ISymbolRegistry;
}
//#endregion
//#region src/files/file.d.ts
declare class File<Node extends INode = INode> {
  /**
   * Exports from this file.
   */
  private _exports;
  /**
   * File extension (e.g. `.ts`).
   */
  private _extension?;
  /**
   * Actual emitted file path, including extension and directories.
   */
  private _finalPath?;
  /**
   * Imports to this file.
   */
  private _imports;
  /**
   * Language of the file.
   */
  private _language?;
  /**
   * Logical, extension-free path used for planning and routing.
   */
  private _logicalFilePath;
  /**
   * Base name of the file (without extension).
   */
  private _name?;
  /**
   * Syntax nodes contained in this file.
   */
  private _nodes;
  /**
   * Renderer assigned to this file.
   */
  private _renderer?;
  /** Brand used for identifying files. */
  readonly '~brand' = "heyapi.file";
  /** All names defined in this file, including local scopes. */
  allNames: NameScopes;
  /** Whether this file is external to the project. */
  external: boolean;
  /** Unique identifier for the file. */
  readonly id: number;
  /** The project this file belongs to. */
  readonly project: IProject;
  /** Names declared at the top level of the file. */
  topLevelNames: NameScopes;
  constructor(input: IFileIn, id: number, project: IProject);
  /**
   * Exports from this file.
   */
  get exports(): ReadonlyArray<ExportModule>;
  /**
   * Read-only accessor for the file extension.
   */
  get extension(): string | undefined;
  /**
   * Read-only accessor for the final emitted path.
   *
   * If undefined, the file has not yet been assigned a final path
   * or is external to the project and should not be emitted.
   */
  get finalPath(): string | undefined;
  /**
   * Imports to this file.
   */
  get imports(): ReadonlyArray<ImportModule>;
  /**
   * Language of the file; inferred from nodes or fallback if not set explicitly.
   */
  get language(): Language | undefined;
  /**
   * Logical, extension-free path used for planning and routing.
   */
  get logicalFilePath(): string;
  /**
   * Base name of the file (without extension).
   *
   * If no name was set explicitly, it is inferred from the logical file path.
   */
  get name(): string;
  /**
   * Syntax nodes contained in this file.
   */
  get nodes(): ReadonlyArray<Node>;
  /**
   * Renderer assigned to this file.
   */
  get renderer(): Renderer | undefined;
  /**
   * Add an export group to the file.
   */
  addExport(group: ExportModule): void;
  /**
   * Add an import group to the file.
   */
  addImport(group: ImportModule): void;
  /**
   * Add a syntax node to the file.
   */
  addNode(node: Node): void;
  /**
   * Sets the file extension.
   */
  setExtension(extension: string): void;
  /**
   * Sets the final emitted path of the file.
   */
  setFinalPath(path: string): void;
  /**
   * Sets the language of the file.
   */
  setLanguage(lang: Language): void;
  /**
   * Sets the name of the file.
   */
  setName(name: string): void;
  /**
   * Sets the renderer assigned to this file.
   */
  setRenderer(renderer: Renderer): void;
  /**
   * Returns a debug‑friendly string representation identifying the file.
   */
  toString(): string;
}
//#endregion
//#region src/bindings.d.ts
interface ExportMember {
  /**
   * Name under which the symbol is exported in this file.
   *
   * export { Foo as Bar } from "./models"
   *
   * exportedName === "Bar"
   */
  exportedName: string;
  /** Whether this export is type-only. */
  isTypeOnly: boolean;
  /** Export flavor. */
  kind: BindingKind;
  /** The exported name of the symbol in its source file. */
  sourceName: string;
}
type ExportModule = Pick<ExportMember, 'isTypeOnly'> & {
  /** Whether this module can export all symbols: `export * from 'module'`. */
  canExportAll: boolean;
  /** Members exported from this module. */
  exports: Array<ExportMember>;
  /** Source file. */
  from: File;
  /** Namespace export: `export * as ns from 'module'`. Mutually exclusive with `exports`. */
  namespaceExport?: string;
};
interface ImportMember {
  /** Whether this import is type-only. */
  isTypeOnly: boolean;
  /**
   * The name this symbol will have locally in this file.
   * This is where aliasing is applied:
   *
   * import { Foo as Foo$2 } from "./x"
   *
   * localName === "Foo$2"
   */
  localName: string;
  /** The exported name of the symbol in its source file. */
  sourceName: string;
}
type ImportModule = Pick<ImportMember, 'isTypeOnly'> & Pick<Partial<ImportMember>, 'localName'> & {
  /** Source file. */
  from: File;
  /** List of symbols imported from this module. */
  imports: Array<ImportMember>;
  /** Import flavor. */
  kind: BindingKind;
};
//#endregion
//#region src/brands.d.ts
declare const nodeBrand = "heyapi.node";
declare const symbolBrand = "heyapi.symbol";
//#endregion
//#region src/config/interactive.d.ts
/**
 * Detect if the current session is interactive based on TTY status and environment variables.
 * This is used as a fallback when the user doesn't explicitly set the interactive option.
 * @internal
 */
declare function detectInteractiveSession(): boolean;
//#endregion
//#region src/logger.d.ts
declare class Logger {
  private events;
  private end;
  /**
   * Recursively end all unended events in the event tree.
   * This ensures all events have end marks before measuring.
   */
  private endAllEvents;
  report(print?: boolean): PerformanceMeasure | undefined;
  private reportEvent;
  private start;
  private storeEvent;
  timeEvent(name: string): {
    mark: PerformanceMark;
    timeEnd: () => void;
  };
}
//#endregion
//#region src/config/load.d.ts
declare function loadConfigFile<T extends AnyObject>({
  configFile,
  logger,
  name,
  userConfig
}: {
  configFile: string | undefined;
  logger: Logger;
  name: string;
  userConfig: T;
}): Promise<{
  configFile: string | undefined;
  configs: ReadonlyArray<T>;
  foundConfig: boolean;
}>;
//#endregion
//#region src/config/merge.d.ts
declare function mergeConfigs<T extends AnyObject>(configA: T | undefined, configB: T | undefined): T;
//#endregion
//#region src/guards.d.ts
declare function isNode(value: unknown): value is INode;
declare function isNodeRef(value: Ref<unknown>): value is Ref<INode>;
declare function isSymbol(value: unknown): value is Symbol;
declare function isSymbolRef(value: Ref<unknown>): value is Ref<Symbol>;
//#endregion
//#region src/languages/extensions.d.ts
declare const defaultExtensions: Extensions;
//#endregion
//#region src/languages/modules.d.ts
declare const defaultModuleEntryNames: ModuleEntryNames;
//#endregion
//#region src/languages/resolvers.d.ts
declare const defaultNameConflictResolvers: NameConflictResolvers;
//#endregion
//#region src/log.d.ts
declare const DebugGroups: {
  readonly analyzer: colors.StyleFunction;
  readonly dsl: colors.StyleFunction;
  readonly file: colors.StyleFunction;
  readonly registry: colors.StyleFunction;
  readonly symbol: colors.StyleFunction;
};
declare const WarnGroups: {
  readonly deprecated: colors.StyleFunction;
};
declare function debug(message: string, group: keyof typeof DebugGroups): void;
declare function warn(message: string, group: keyof typeof WarnGroups): void;
declare function warnDeprecated({
  context,
  field,
  replacement
}: {
  context?: string;
  field: string;
  replacement?: MaybeFunc<(field: string) => MaybeArray<string>>;
}): void;
declare const log: {
  debug: typeof debug;
  warn: typeof warn;
  warnDeprecated: typeof warnDeprecated;
};
//#endregion
//#region src/planner/resolvers.d.ts
declare const simpleNameConflictResolver: NameConflictResolver;
declare const underscoreNameConflictResolver: NameConflictResolver;
//#endregion
//#region src/files/registry.d.ts
type FileId = number;
declare class FileRegistry implements IFileRegistry {
  private _id;
  private _values;
  private readonly project;
  constructor(project: IProject);
  get(args: FileKeyArgs): File | undefined;
  isRegistered(args: FileKeyArgs): boolean;
  get nextId(): FileId;
  register(file: IFileIn): File;
  registered(): IterableIterator<File>;
  private createFileKey;
}
//#endregion
//#region src/nodes/registry.d.ts
declare class NodeRegistry implements INodeRegistry {
  private list;
  add(node: INode | null): number;
  all(): Iterable<INode>;
  remove(index: number): void;
  update(index: number, node: INode | null): void;
}
//#endregion
//#region src/symbols/registry.d.ts
type SymbolId = number;
declare class SymbolRegistry implements ISymbolRegistry {
  private _id;
  private _indices;
  private _queryCache;
  private _queryCacheDependencies;
  private _registered;
  private _stubs;
  private _stubCache;
  private _values;
  get(identifier: ISymbolIdentifier): Symbol | undefined;
  isRegistered(identifier: ISymbolIdentifier): boolean;
  get nextId(): SymbolId;
  query(filter: ISymbolMeta): ReadonlyArray<Symbol>;
  reference(meta: ISymbolMeta): Symbol;
  register(symbol: ISymbolIn): Symbol;
  registered(): IterableIterator<Symbol>;
  private buildCacheKey;
  private buildIndexKeySpace;
  private indexSymbol;
  private invalidateCache;
  private isSubset;
  private replaceStubs;
  private serializeIndexEntry;
}
//#endregion
//#region src/project/project.d.ts
declare class Project implements IProject {
  private _isPlanned;
  readonly files: FileRegistry;
  readonly nodes: NodeRegistry;
  readonly symbols: SymbolRegistry;
  readonly defaultFileName: string;
  readonly defaultNameConflictResolver: NameConflictResolver;
  readonly extensions: Extensions;
  readonly fileName?: (name: string) => string;
  readonly moduleEntryNames: ModuleEntryNames;
  readonly nameConflictResolvers: NameConflictResolvers;
  readonly renderers: ReadonlyArray<Renderer>;
  readonly root: string;
  constructor(args: Pick<Partial<IProject>, 'defaultFileName' | 'defaultNameConflictResolver' | 'extensions' | 'fileName' | 'moduleEntryNames' | 'nameConflictResolvers' | 'renderers'> & Pick<IProject, 'root'>);
  plan(meta?: IProjectRenderMeta): void;
  render(meta?: IProjectRenderMeta): ReadonlyArray<IOutput>;
}
//#endregion
//#region src/refs/refs.d.ts
/**
 * Wraps a single value in a Ref object.
 *
 * If the value is already a Ref, returns it as-is (idempotent).
 *
 * @example
 * ```ts
 * const r = ref(123); // { '~ref': 123 }
 * console.log(r['~ref']); // 123
 *
 * const r2 = ref(r); // { '~ref': 123 } (not double-wrapped)
 * ```
 */
declare const ref: <T>(value: T) => Ref<T>;
/**
 * Converts a plain object to an object of Refs (deep, per property).
 *
 * @example
 * ```ts
 * const obj = { a: 1, b: "x" };
 * const refs = refs(obj); // { a: { '~ref': 1 }, b: { '~ref': "x" } }
 * ```
 */
declare const refs: <T extends Record<string, unknown>>(obj: T) => Refs<T>;
/**
 * Unwraps a single Ref object to its value.
 *
 * @example
 * ```ts
 * const r = { '~ref': 42 };
 * const n = fromRef(r); // 42
 * console.log(n); // 42
 * ```
 */
declare const fromRef: <T extends Ref<unknown> | undefined>(ref: T) => FromRef<T>;
/**
 * Converts an object of Refs back to a plain object (unwraps all refs).
 *
 * @example
 * ```ts
 * const refs = { a: { '~ref': 1 }, b: { '~ref': "x" } };
 * const plain = fromRefs(refs); // { a: 1, b: "x" }
 * ```
 */
declare const fromRefs: <T extends Refs<Record<string, unknown>>>(obj: T) => FromRefs<T>;
/**
 * Checks whether a value is a Ref object.
 *
 * @param value Value to check
 * @returns True if the value is a Ref object.
 */
declare const isRef: <T>(value: unknown) => value is Ref<T>;
//#endregion
//#region src/structure/types.d.ts
interface StructureInsert {
  /** Inserted data. */
  data: unknown;
  /** Locations where the data should be inserted. */
  locations: ReadonlyArray<StructureLocation>;
  /** Source of the inserted data. */
  source: symbol;
}
interface StructureItem extends Pick<StructureInsert, 'data' | 'source'> {
  /** Location of this item within the structure. */
  location: ReadonlyArray<string>;
}
interface StructureLocation {
  /** Path within the structure where the data should be inserted. */
  path: ReadonlyArray<string>;
  /** Shell to apply at this location. */
  shell?: StructureShell;
}
interface StructureShell {
  define: (node: StructureNode) => StructureShellResult;
}
interface StructureShellResult {
  dependencies?: Array<INode>;
  node: INode;
}
//#endregion
//#region src/structure/node.d.ts
declare class StructureNode {
  /** Nested nodes within this node. */
  children: Map<string, StructureNode>;
  /** Items contained in this node. */
  items: Array<StructureItem>;
  /** The name of this node (e.g., "Users", "Accounts"). */
  name: string;
  /** Parent node in the hierarchy. Undefined if this is the root node. */
  parent?: StructureNode;
  /** Shell claimed for this node. */
  shell?: StructureShell;
  /** Source of the claimed shell. */
  shellSource?: symbol;
  /** True if this is a virtual root. */
  virtual: boolean;
  constructor(name: string, parent?: StructureNode, options?: {
    virtual?: boolean;
  });
  get isRoot(): boolean;
  /**
   * Gets or creates a child node.
   *
   * If the child doesn't exist, it's created automatically.
   *
   * @param name - The name of the child node
   * @returns The child node instance
   */
  child(name: string): StructureNode;
  /**
   * Gets the full path of this node in the hierarchy.
   *
   * @returns An array of node names from the root to this node
   */
  getPath(): ReadonlyArray<string>;
  /**
   * Yields items from a specific source with typed data.
   *
   * @param source - The source symbol to filter by
   * @returns Generator of items from that source
   */
  itemsFrom<T = unknown>(source: symbol): Generator<StructureItem & {
    data: T;
  }>;
  /**
   * Walk all nodes in the structure (depth-first, post-order).
   *
   * @returns Generator of all structure nodes
   */
  walk(): Generator<StructureNode>;
}
//#endregion
//#region src/structure/model.d.ts
declare class StructureModel {
  /** Root nodes mapped by their names. */
  private _roots;
  /** Node for data without a specific root. */
  private _virtualRoot?;
  /**
   * Get all root nodes.
   */
  get roots(): ReadonlyArray<StructureNode>;
  /**
   * Insert data into the structure.
   */
  insert(args: StructureInsert): void;
  /**
   * Gets or creates a root by name.
   *
   * If the root doesn't exist, it's created automatically.
   *
   * @param name - The name of the root
   * @returns The root instance
   */
  root(name: string | null): StructureNode;
  /**
   * Walk all nodes in the structure (depth-first, post-order).
   *
   * @returns Generator of all structure nodes
   */
  walk(): Generator<StructureNode>;
}
//#endregion
export { type IAnalysisContext as AnalysisContext, type BindingKind, type ExportMember, type ExportModule, type Extensions, File, type IFileIn as FileIn, type FromRef, type FromRefs, type IProject, type ImportMember, type ImportModule, type Language, Logger, type ModuleEntryNames, type NameConflictResolver, type NameConflictResolvers, type INode as Node, type NodeName, type NodeNameSanitizer, type NodeRelationship, type NodeScope, type IOutput as Output, Project, type IProjectRenderMeta as ProjectRenderMeta, type Ref, type Refs, type RenderContext, type Renderer, type StructureInsert, type StructureItem, type StructureLocation, StructureModel, StructureNode, type StructureShell, type StructureShellResult, Symbol, type ISymbolIdentifier as SymbolIdentifier, type ISymbolIn as SymbolIn, type ISymbolMeta as SymbolMeta, defaultExtensions, defaultModuleEntryNames, defaultNameConflictResolvers, detectInteractiveSession, fromRef, fromRefs, isNode, isNodeRef, isRef, isSymbol, isSymbolRef, loadConfigFile, log, mergeConfigs, nodeBrand, ref, refs, simpleNameConflictResolver, symbolBrand, underscoreNameConflictResolver };
//# sourceMappingURL=index.d.mts.map