"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.connectInParallel = connectInParallel;
exports.connectInSequence = connectInSequence;
exports.lookupAllAddresses = lookupAllAddresses;
var _net = _interopRequireDefault(require("net"));
var _nodeUrl = _interopRequireDefault(require("node:url"));
function _interopRequireDefault(e) { return e && e.__esModule ? e : { default: e }; }
async function connectInParallel(options, lookup, signal) {
  signal.throwIfAborted();
  const addresses = await lookupAllAddresses(options.host, lookup, signal);
  return await new Promise((resolve, reject) => {
    const sockets = new Array(addresses.length);
    const errors = [];
    function onError(err) {
      errors.push(err);
      this.removeListener('error', onError);
      this.removeListener('connect', onConnect);
      this.destroy();
      if (errors.length === addresses.length) {
        signal.removeEventListener('abort', onAbort);
        reject(new AggregateError(errors, 'Could not connect (parallel)'));
      }
    }
    function onConnect() {
      signal.removeEventListener('abort', onAbort);
      for (let j = 0; j < sockets.length; j++) {
        const socket = sockets[j];
        if (this === socket) {
          continue;
        }
        socket.removeListener('error', onError);
        socket.removeListener('connect', onConnect);
        socket.destroy();
      }
      resolve(this);
    }
    const onAbort = () => {
      for (let j = 0; j < sockets.length; j++) {
        const socket = sockets[j];
        socket.removeListener('error', onError);
        socket.removeListener('connect', onConnect);
        socket.destroy();
      }
      reject(signal.reason);
    };
    for (let i = 0, len = addresses.length; i < len; i++) {
      const socket = sockets[i] = _net.default.connect({
        ...options,
        host: addresses[i].address,
        family: addresses[i].family
      });
      socket.on('error', onError);
      socket.on('connect', onConnect);
    }
    signal.addEventListener('abort', onAbort, {
      once: true
    });
  });
}
async function connectInSequence(options, lookup, signal) {
  signal.throwIfAborted();
  const errors = [];
  const addresses = await lookupAllAddresses(options.host, lookup, signal);
  for (const address of addresses) {
    try {
      return await new Promise((resolve, reject) => {
        const socket = _net.default.connect({
          ...options,
          host: address.address,
          family: address.family
        });
        const onAbort = () => {
          socket.removeListener('error', onError);
          socket.removeListener('connect', onConnect);
          socket.destroy();
          reject(signal.reason);
        };
        const onError = err => {
          signal.removeEventListener('abort', onAbort);
          socket.removeListener('error', onError);
          socket.removeListener('connect', onConnect);
          socket.destroy();
          reject(err);
        };
        const onConnect = () => {
          signal.removeEventListener('abort', onAbort);
          socket.removeListener('error', onError);
          socket.removeListener('connect', onConnect);
          resolve(socket);
        };
        signal.addEventListener('abort', onAbort, {
          once: true
        });
        socket.on('error', onError);
        socket.on('connect', onConnect);
      });
    } catch (err) {
      // If the signal was aborted, re-throw the error.
      signal.throwIfAborted();
      errors.push(err);
      continue;
    }
  }
  throw new AggregateError(errors, 'Could not connect (sequence)');
}

/**
 * Look up all addresses for the given hostname.
 */
async function lookupAllAddresses(host, lookup, signal) {
  signal.throwIfAborted();
  if (_net.default.isIPv6(host)) {
    return [{
      address: host,
      family: 6
    }];
  } else if (_net.default.isIPv4(host)) {
    return [{
      address: host,
      family: 4
    }];
  } else {
    return await new Promise((resolve, reject) => {
      const onAbort = () => {
        reject(signal.reason);
      };
      signal.addEventListener('abort', onAbort);
      const domainInASCII = _nodeUrl.default.domainToASCII(host);
      lookup(domainInASCII === '' ? host : domainInASCII, {
        all: true
      }, (err, addresses) => {
        signal.removeEventListener('abort', onAbort);
        err ? reject(err) : resolve(addresses);
      });
    });
  }
}
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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