"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = void 0;
var _core = require("@js-joda/core");
var _writableTrackingBuffer = _interopRequireDefault(require("../tracking-buffer/writable-tracking-buffer"));
function _interopRequireDefault(e) { return e && e.__esModule ? e : { default: e }; }
const EPOCH_DATE = _core.LocalDate.ofYearDay(1, 1);
const NULL_LENGTH = Buffer.from([0x00]);
const DateTimeOffset = {
  id: 0x2B,
  type: 'DATETIMEOFFSETN',
  name: 'DateTimeOffset',
  declaration: function (parameter) {
    return 'datetimeoffset(' + this.resolveScale(parameter) + ')';
  },
  resolveScale: function (parameter) {
    if (parameter.scale != null) {
      return parameter.scale;
    } else if (parameter.value === null) {
      return 0;
    } else {
      return 7;
    }
  },
  generateTypeInfo(parameter) {
    return Buffer.from([this.id, parameter.scale]);
  },
  generateParameterLength(parameter, options) {
    if (parameter.value == null) {
      return NULL_LENGTH;
    }
    switch (parameter.scale) {
      case 0:
      case 1:
      case 2:
        return Buffer.from([0x08]);
      case 3:
      case 4:
        return Buffer.from([0x09]);
      case 5:
      case 6:
      case 7:
        return Buffer.from([0x0A]);
      default:
        throw new Error('invalid scale');
    }
  },
  *generateParameterData(parameter, options) {
    if (parameter.value == null) {
      return;
    }
    const value = parameter.value;
    let scale = parameter.scale;
    const buffer = new _writableTrackingBuffer.default(16);
    scale = scale;
    let timestamp;
    timestamp = ((value.getUTCHours() * 60 + value.getUTCMinutes()) * 60 + value.getUTCSeconds()) * 1000 + value.getMilliseconds();
    timestamp = timestamp * Math.pow(10, scale - 3);
    timestamp += (value.nanosecondDelta != null ? value.nanosecondDelta : 0) * Math.pow(10, scale);
    timestamp = Math.round(timestamp);
    switch (scale) {
      case 0:
      case 1:
      case 2:
        buffer.writeUInt24LE(timestamp);
        break;
      case 3:
      case 4:
        buffer.writeUInt32LE(timestamp);
        break;
      case 5:
      case 6:
      case 7:
        buffer.writeUInt40LE(timestamp);
    }
    const date = _core.LocalDate.of(value.getUTCFullYear(), value.getUTCMonth() + 1, value.getUTCDate());
    const days = EPOCH_DATE.until(date, _core.ChronoUnit.DAYS);
    buffer.writeUInt24LE(days);
    const offset = -value.getTimezoneOffset();
    buffer.writeInt16LE(offset);
    yield buffer.data;
  },
  validate: function (value, collation, options) {
    if (value == null) {
      return null;
    }
    if (!(value instanceof Date)) {
      value = new Date(Date.parse(value));
    }
    value = value;
    let year;
    if (options && options.useUTC) {
      year = value.getUTCFullYear();
    } else {
      year = value.getFullYear();
    }
    if (year < 1 || year > 9999) {
      throw new TypeError('Out of range.');
    }
    if (isNaN(value)) {
      throw new TypeError('Invalid date.');
    }
    return value;
  }
};
var _default = exports.default = DateTimeOffset;
module.exports = DateTimeOffset;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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