"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.sendInParallel = sendInParallel;
exports.sendMessage = sendMessage;
var _dgram = _interopRequireDefault(require("dgram"));
var _net = _interopRequireDefault(require("net"));
var _nodeUrl = _interopRequireDefault(require("node:url"));
function _interopRequireDefault(e) { return e && e.__esModule ? e : { default: e }; }
async function sendInParallel(addresses, port, request, signal) {
  signal.throwIfAborted();
  return await new Promise((resolve, reject) => {
    const sockets = [];
    let errorCount = 0;
    const onError = err => {
      errorCount++;
      if (errorCount === addresses.length) {
        signal.removeEventListener('abort', onAbort);
        clearSockets();
        reject(err);
      }
    };
    const onMessage = message => {
      signal.removeEventListener('abort', onAbort);
      clearSockets();
      resolve(message);
    };
    const onAbort = () => {
      clearSockets();
      reject(signal.reason);
    };
    const clearSockets = () => {
      for (const socket of sockets) {
        socket.removeListener('error', onError);
        socket.removeListener('message', onMessage);
        socket.close();
      }
    };
    signal.addEventListener('abort', onAbort, {
      once: true
    });
    for (let j = 0; j < addresses.length; j++) {
      const udpType = addresses[j].family === 6 ? 'udp6' : 'udp4';
      const socket = _dgram.default.createSocket(udpType);
      sockets.push(socket);
      socket.on('error', onError);
      socket.on('message', onMessage);
      socket.send(request, 0, request.length, port, addresses[j].address);
    }
  });
}
async function sendMessage(host, port, lookup, signal, request) {
  signal.throwIfAborted();
  let addresses;
  if (_net.default.isIP(host)) {
    addresses = [{
      address: host,
      family: _net.default.isIPv6(host) ? 6 : 4
    }];
  } else {
    addresses = await new Promise((resolve, reject) => {
      const onAbort = () => {
        reject(signal.reason);
      };
      const domainInASCII = _nodeUrl.default.domainToASCII(host);
      lookup(domainInASCII === '' ? host : domainInASCII, {
        all: true
      }, (err, addresses) => {
        signal.removeEventListener('abort', onAbort);
        err ? reject(err) : resolve(addresses);
      });
    });
  }
  return await sendInParallel(addresses, port, request, signal);
}
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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